// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_CACHE_H_
#define CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_CACHE_H_
#pragma once

// A class that gets malware details from the HTTP Cache.
// An instance of this class is generated by MalwareDetails.

#include <string>
#include <vector>

#include "base/hash_tables.h"
#include "base/memory/linked_ptr.h"
#include "base/memory/ref_counted.h"
#include "chrome/browser/safe_browsing/report.pb.h"
#include "content/common/url_fetcher.h"
#include "net/base/completion_callback.h"

namespace net {
class URLRequestContext;
}

class MalwareDetailsFactory;

namespace safe_browsing {

// Maps a URL to its Resource.
typedef base::hash_map<
  std::string,
  linked_ptr<safe_browsing::ClientMalwareReportRequest::Resource> > ResourceMap;
}

class MalwareDetailsCacheCollector
    : public base::RefCountedThreadSafe<MalwareDetailsCacheCollector>,
      public URLFetcher::Delegate {

 public:
  MalwareDetailsCacheCollector();
  virtual ~MalwareDetailsCacheCollector();

  // We use |request_context_getter|, we modify |resources| and
  // |result|, and we call |callback|, so they must all remain alive
  // for the lifetime of this object.
  void StartCacheCollection(
      net::URLRequestContextGetter* request_context_getter,
      safe_browsing::ResourceMap* resources,
      bool* result,
      Task* callback);

  // Returns whether or not StartCacheCollection has been called.
  bool HasStarted();

 protected:
  // Implementation of URLFetcher::Delegate. Called after the request
  // completes (either successfully or with failure).
  virtual void OnURLFetchComplete(const URLFetcher* source,
                                  const GURL& url,
                                  const net::URLRequestStatus& status,
                                  int response_code,
                                  const net::ResponseCookies& cookies,
                                  const std::string& data);

 private:
  // Points to the url for which we are fetching the HTTP cache entry or
  // redirect chain.
  safe_browsing::ResourceMap::iterator resources_it_;

  // Points to the resources_ map in the MalwareDetails.
  safe_browsing::ResourceMap* resources_;

  // Points to the cache_result_ in the MalwareDetails.
  bool* result_;

  // Method we call when we are done. The caller must be alive for the
  // whole time, we are modifying its state (see above).
  Task* callback_;

  // Set to true as soon as StartCacheCollection is called.
  bool has_started_;

  // Used to get a pointer to the current request context.
  scoped_refptr<net::URLRequestContextGetter> request_context_getter_;

  // The current URLFetcher.
  scoped_ptr<URLFetcher> current_fetch_;

  // Returns the resource from resources_ that corresponds to |url|
  safe_browsing::ClientMalwareReportRequest::Resource* GetResource(
      const GURL& url);

  // Creates a new URLFetcher and starts it.
  void OpenEntry();

  // Read the HTTP response from |source| and add it to |pb_resource|.
  void ReadResponse(
      safe_browsing::ClientMalwareReportRequest::Resource* pb_resource,
      const URLFetcher* source);

  // Read the body |data| and add it to |pb_resource|.
  void ReadData(
      safe_browsing::ClientMalwareReportRequest::Resource* pb_resource,
      const std::string& data);

  // Called when we are done.
  void AllDone(bool success);

  // Advances to the next entry in resources_it_.
  void AdvanceEntry();
};

#endif  // CHROME_BROWSER_SAFE_BROWSING_MALWARE_DETAILS_CACHE_H_
