// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "media/filters/omx_video_decoder.h"

#include "base/bind.h"
#include "base/callback.h"
#include "base/message_loop.h"
#include "media/base/callback.h"
#include "media/base/filter_host.h"
#include "media/base/limits.h"
#include "media/ffmpeg/ffmpeg_common.h"
#include "media/video/omx_video_decode_engine.h"

namespace media {

OmxVideoDecoder::OmxVideoDecoder(
    MessageLoop* message_loop,
    VideoDecodeContext* context)
    : message_loop_(message_loop),
      decode_engine_(new OmxVideoDecodeEngine()),
      decode_context_(context) {
  DCHECK(decode_engine_.get());
  memset(&info_, 0, sizeof(info_));
}

OmxVideoDecoder::~OmxVideoDecoder() {
  // TODO(hclam): Make sure OmxVideoDecodeEngine is stopped.
}

void OmxVideoDecoder::Initialize(DemuxerStream* demuxer_stream,
                                 FilterCallback* callback,
                                 StatisticsCallback* stats_callback) {
  if (MessageLoop::current() != message_loop_) {
    message_loop_->PostTask(
        FROM_HERE,
        NewRunnableMethod(this,
                          &OmxVideoDecoder::Initialize,
                          make_scoped_refptr(demuxer_stream),
                          callback, stats_callback));
    return;
  }

  DCHECK_EQ(message_loop_, MessageLoop::current());
  DCHECK(!demuxer_stream_);
  DCHECK(!initialize_callback_.get());

  initialize_callback_.reset(callback);
  statistics_callback_.reset(stats_callback);
  demuxer_stream_ = demuxer_stream;

  // We require bit stream converter for openmax hardware decoder.
  demuxer_stream->EnableBitstreamConverter();

  AVStream* av_stream = demuxer_stream->GetAVStream();
  if (!av_stream) {
    VideoCodecInfo info = {0};
    OnInitializeComplete(info);
    return;
  }

  pts_stream_.Initialize(GetFrameDuration(av_stream));

  int width = av_stream->codec->coded_width;
  int height = av_stream->codec->coded_height;
  if (width > Limits::kMaxDimension ||
      height > Limits::kMaxDimension ||
      (width * height) > Limits::kMaxCanvas) {
    VideoCodecInfo info = {0};
    OnInitializeComplete(info);
    return;
  }

  VideoDecoderConfig config(CodecIDToVideoCodec(av_stream->codec->codec_id),
                            width, height,
                            av_stream->r_frame_rate.num,
                            av_stream->r_frame_rate.den,
                            av_stream->codec->extradata,
                            av_stream->codec->extradata_size);
  decode_engine_->Initialize(message_loop_, this, NULL, config);
}

void OmxVideoDecoder::OnInitializeComplete(const VideoCodecInfo& info) {
  // TODO(scherkus): Dedup this from FFmpegVideoDecoder::OnInitializeComplete.
  DCHECK_EQ(MessageLoop::current(), message_loop_);
  DCHECK(initialize_callback_.get());

  info_ = info;
  AutoCallbackRunner done_runner(initialize_callback_.release());

  if (info.success) {
    media_format_.SetAsInteger(MediaFormat::kWidth,
                               info.stream_info.surface_width);
    media_format_.SetAsInteger(MediaFormat::kHeight,
                               info.stream_info.surface_height);
    media_format_.SetAsInteger(
        MediaFormat::kSurfaceType,
        static_cast<int>(info.stream_info.surface_type));
    media_format_.SetAsInteger(
        MediaFormat::kSurfaceFormat,
        static_cast<int>(info.stream_info.surface_format));
  } else {
    host()->SetError(PIPELINE_ERROR_DECODE);
  }
}

void OmxVideoDecoder::Stop(FilterCallback* callback) {
  if (MessageLoop::current() != message_loop_) {
    message_loop_->PostTask(FROM_HERE,
                             NewRunnableMethod(this,
                                               &OmxVideoDecoder::Stop,
                                               callback));
    return;
  }

  DCHECK_EQ(MessageLoop::current(), message_loop_);
  DCHECK(!uninitialize_callback_.get());

  uninitialize_callback_.reset(callback);
  decode_engine_->Uninitialize();
}

void OmxVideoDecoder::OnUninitializeComplete() {
  DCHECK_EQ(MessageLoop::current(), message_loop_);
  DCHECK(uninitialize_callback_.get());

  AutoCallbackRunner done_runner(uninitialize_callback_.release());

  // TODO(jiesun): Destroy the decoder context.
}

void OmxVideoDecoder::Flush(FilterCallback* callback) {
  if (MessageLoop::current() != message_loop_) {
    message_loop_->PostTask(FROM_HERE,
                             NewRunnableMethod(this,
                                               &OmxVideoDecoder::Flush,
                                               callback));
    return;
  }

  DCHECK_EQ(MessageLoop::current(), message_loop_);
  DCHECK(!flush_callback_.get());

  flush_callback_.reset(callback);

  decode_engine_->Flush();
}


void OmxVideoDecoder::OnFlushComplete() {
  DCHECK(flush_callback_.get());

  AutoCallbackRunner done_runner(flush_callback_.release());

  pts_stream_.Flush();
}

void OmxVideoDecoder::Seek(base::TimeDelta time, const FilterStatusCB& cb) {
  if (MessageLoop::current() != message_loop_) {
     message_loop_->PostTask(FROM_HERE,
                              NewRunnableMethod(this,
                                                &OmxVideoDecoder::Seek,
                                                time,
                                                cb));
     return;
  }

  DCHECK_EQ(MessageLoop::current(), message_loop_);
  DCHECK(seek_cb_.is_null());

  pts_stream_.Seek(time);
  seek_cb_ = cb;
  decode_engine_->Seek();
}

void OmxVideoDecoder::OnSeekComplete() {
  DCHECK_EQ(MessageLoop::current(), message_loop_);
  DCHECK(!seek_cb_.is_null());

  ResetAndRunCB(&seek_cb_, PIPELINE_OK);
}

void OmxVideoDecoder::OnError() {
  NOTIMPLEMENTED();
}
void OmxVideoDecoder::OnFormatChange(VideoStreamInfo stream_info) {
  NOTIMPLEMENTED();
}

void OmxVideoDecoder::ProduceVideoSample(scoped_refptr<Buffer> buffer) {
  DCHECK_EQ(message_loop_, MessageLoop::current());

  // Issue more demux.
  demuxer_stream_->Read(base::Bind(&OmxVideoDecoder::DemuxCompleteTask, this));
}

void OmxVideoDecoder::ConsumeVideoFrame(scoped_refptr<VideoFrame> frame,
                                        const PipelineStatistics& statistics) {
  DCHECK_EQ(message_loop_, MessageLoop::current());
  statistics_callback_->Run(statistics);

  if (frame.get()) {
    pts_stream_.UpdatePtsAndDuration(frame.get());

    frame->SetTimestamp(pts_stream_.current_pts());
    frame->SetDuration(pts_stream_.current_duration());
  }

  VideoFrameReady(frame);
}

void OmxVideoDecoder::ProduceVideoFrame(scoped_refptr<VideoFrame> frame) {
  DCHECK(decode_engine_.get());
  message_loop_->PostTask(
     FROM_HERE,
     NewRunnableMethod(decode_engine_.get(),
                       &VideoDecodeEngine::ProduceVideoFrame, frame));
}

bool OmxVideoDecoder::ProvidesBuffer() {
  DCHECK(info_.success);
  return info_.provides_buffers;
}

const MediaFormat& OmxVideoDecoder::media_format() {
  return media_format_;
}

void OmxVideoDecoder::DemuxCompleteTask(Buffer* buffer) {
  // We simply delicate the buffer to the right message loop.
  scoped_refptr<Buffer> ref_buffer = buffer;
  DCHECK(decode_engine_.get());
  message_loop_->PostTask(
      FROM_HERE,
      NewRunnableMethod(this,
                        &OmxVideoDecoder::ConsumeVideoSample, ref_buffer));
}

void OmxVideoDecoder::ConsumeVideoSample(scoped_refptr<Buffer> buffer) {
  if (buffer.get())
    pts_stream_.EnqueuePts(buffer.get());
  decode_engine_->ConsumeVideoSample(buffer);
}

}  // namespace media

// Disable refcounting for the decode engine because it only lives on the
// video decoder thread.
DISABLE_RUNNABLE_METHOD_REFCOUNT(media::VideoDecodeEngine);
