/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2011 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <ctype.h> /* isprint() */

#include "cdw_debug.h"
#include "cdw_form.h"
#include "cdw_widgets.h"
#include "cdw_string.h"

static void cdw_form_driver_focus_off(cdw_form_t *cdw_form, int fi);
static void cdw_form_driver_focus_on(cdw_form_t *cdw_form, int fi);
static void cdw_form_driver_toggle_focus(cdw_form_t *cdw_form, int fi, bool off);
static void cdw_form_driver_toggle_checkbox(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_return_key(cdw_form_t *form, int key);
static int  cdw_form_handle_enter(cdw_form_t *cdw_form, int fi);

static bool cdw_form_is_checkbox_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_dropdown_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_button_field_index(cdw_form_t *cdw_form, int fi);
static bool cdw_form_is_input_field_index(cdw_form_t *cdw_form, int fi);





/**
   \brief Constructor

   \param n_fields - expected maximal number of fields in a form
*/
cdw_form_t *cdw_form_new(int n_fields)
{
	cdw_form_t *cdw_form = (cdw_form_t *) malloc(sizeof (cdw_form_t));
	if (cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for new cdw form\n");
		return (cdw_form_t *) NULL;
	}
	cdw_form->n_fields = n_fields;

	cdw_form->field_widget_types = malloc((size_t) cdw_form->n_fields * sizeof (cdw_id_t));
	if (cdw_form->field_widget_types ==  NULL) {
		cdw_vdm ("ERROR: failed to allocate fields widget ids table\n");
		free(cdw_form);
		cdw_form = (cdw_form_t *) NULL;
		return (cdw_form_t *) NULL;
	}

	cdw_form->window = (WINDOW *) NULL;
	cdw_form->subwindow = (WINDOW *) NULL;
	cdw_form->form = (FORM *) NULL;
	cdw_form->fields = (FIELD **) NULL;

	cdw_form->form_id = 0;
	cdw_form->n_return_keys = 0;
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		cdw_form->return_keys[i] = 0;
	}

	/* may be overwritten by form-specific handler */
	cdw_form->handle_enter = cdw_form_handle_enter;

	return cdw_form;
}





/**
   \brief cdw form destructor - stage 2
*/
void cdw_form_delete(cdw_form_t **cdw_form)
{
	cdw_assert (cdw_form != (cdw_form_t **) NULL, "ERROR: incorrect argument\n");
	if (*cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("WARNING: passing NULL form to function\n");
		return;
	}

	if ((*cdw_form)->field_widget_types != (cdw_id_t *) NULL) {
		free((*cdw_form)->field_widget_types);
		(*cdw_form)->field_widget_types = (cdw_id_t *) NULL;
	}

	free(*cdw_form);
	*cdw_form = (cdw_form_t *) NULL;

	return;
}





/**
   \brief cdw form destructor - stage 1

   Variable of type cdw_form_t needs to be deallocated
   in stages, this is first of them;
   second stage is cdw_form_delete()
*/
void cdw_form_delete_form_objects(cdw_form_t *cdw_form)
{
	if (cdw_form == (cdw_form_t *) NULL) {
		cdw_vdm ("WARNING: passing NULL form to function\n");
		return;
	}
	/* from ncurses docs: "The functions free_field() and
	   free_form are available to free field and form objects.
	   It is an error to attempt to free a field connected to
	   a form, but not vice-versa; thus, you will generally
	   free your form objects first. */
	if (cdw_form->form != (FORM *) NULL) {
		unpost_form(cdw_form->form);
		free_form(cdw_form->form);
		cdw_form->form = (FORM *) NULL;
	}

	for (int i = 0; i < cdw_form->n_fields; i++) {
		if (cdw_form->fields[i] != (FIELD *) NULL) {
			if (cdw_form->field_widget_types[i] == CDW_WIDGET_CHECKBOX) {
				CDW_CHECKBOX *cb = (CDW_CHECKBOX *) field_userptr(cdw_form->fields[i]);
				cdw_assert (cb != (CDW_CHECKBOX *) NULL, "ERROR: field #%d is checkbox, but the pointer is null\n", i);
				cdw_checkbox_free(cb);
			} else if (cdw_form->field_widget_types[i] == CDW_WIDGET_BUTTON) {
				CDW_BUTTON *b = (CDW_BUTTON *) field_userptr(cdw_form->fields[i]);
				cdw_assert (b != (CDW_BUTTON *) NULL, "ERROR: field #%d is button, but the pointer is null\n", i);
				cdw_button_free(b);
			} else if (cdw_form->field_widget_types[i] == CDW_WIDGET_DROPDOWN) {
				CDW_DROPDOWN *d = (CDW_DROPDOWN *) field_userptr(cdw_form->fields[i]);
				cdw_assert (d != (CDW_DROPDOWN *) NULL, "ERROR: field #%d is dropdown, but the pointer is null\n", i);
				cdw_dropdown_free(d);
			} else {
				;
			}
			free_field(cdw_form->fields[i]);
			cdw_form->fields[i] = (FIELD *) NULL;
		}
	}

	return;
}





int cdw_form_driver(cdw_form_t *cdw_form, int initial_fi)
{
	cdw_assert (cdw_form->handle_enter != (int (*)(cdw_form_t *, int)) NULL,
		    "ERROR: this form has no Enter handler set up\n");

	cdw_form_driver_go_to_field(cdw_form, initial_fi);
	initial_fi = field_index(current_field(cdw_form->form));
	cdw_form_driver_focus_on(cdw_form, initial_fi);

	form_driver(cdw_form->form, REQ_END_LINE);
	wrefresh(cdw_form->subwindow);

	int key = KEY_END; /* safe initial value */
	do {
		wrefresh(cdw_form->subwindow);
		key = wgetch(cdw_form->subwindow);

		if (cdw_form_is_movement_key(key)) {
			int fi = field_index(current_field(cdw_form->form));
			cdw_form_driver_focus_off(cdw_form, fi);
		}

		if (key == KEY_HOME) {
			form_driver(cdw_form->form, REQ_BEG_LINE);
		} else if (key == KEY_END) {
			form_driver(cdw_form->form, REQ_END_LINE);
		} else if (key == KEY_LEFT) {
			form_driver(cdw_form->form, REQ_PREV_CHAR);
		} else if (key == KEY_RIGHT) {
			form_driver(cdw_form->form, REQ_NEXT_CHAR);
		} else if (key == KEY_DOWN || key == CDW_KEY_TAB) {
			form_driver(cdw_form->form, REQ_NEXT_FIELD);
			form_driver(cdw_form->form, REQ_END_LINE);
		} else if (key == KEY_UP || key == KEY_BTAB) {
			form_driver(cdw_form->form, REQ_PREV_FIELD);
			form_driver(cdw_form->form, REQ_END_LINE);
		} else if (key == KEY_BACKSPACE) {
			form_driver(cdw_form->form, REQ_DEL_PREV);
		} else if (key == KEY_DC) {
			form_driver(cdw_form->form, REQ_DEL_CHAR);
		} else if (key == CDW_KEY_ENTER) {
			int fi = field_index(current_field(cdw_form->form));
			/* the function may return dummy key 'a' which means
			   to continue in driver loop */
			key = cdw_form->handle_enter(cdw_form, fi);
		} else {
			int fi = field_index(current_field(cdw_form->form));
			if (cdw_form_is_checkbox_field_index(cdw_form, fi)) {
				if (key == ' ' || key == 'x' || key == 'X') {
					cdw_form_driver_toggle_checkbox(cdw_form, fi);
				}
			} else if (cdw_form_is_dropdown_field_index(cdw_form, fi)) {
				/* dropdown in its initial state reacts only
				   to enter, and this has already been handled
				   elsewhere by cdw_form->handle_enter() */
				;
			} else {
				/* normal text/input field, pass char from keyboard to field */
				form_driver(cdw_form->form, key);
			}
		}
		if (cdw_form_is_movement_key(key)) {
			int fi = field_index(current_field(cdw_form->form));
			cdw_form_driver_focus_on(cdw_form, fi);
		}
		if (cdw_form_is_return_key(cdw_form, key)) {
			int fi = field_index(current_field(cdw_form->form));
			if (isprint(key)
			    && cdw_form_is_input_field_index(cdw_form, fi)) {
				/* keys like 'q'/'Q' may be configured
				   as return keys, but they shouldn't be
				   considered as such in regular text
				   input fields */
				;
			} else {
				break;
			}
		}


	} while (key != CDW_KEY_ESCAPE);

	form_driver(cdw_form->form, REQ_VALIDATION);
	curs_set(0); /* turn cursor off */
	return key;
}





bool cdw_form_is_return_key(cdw_form_t *form, int key)
{
	for (int i = 0; i < N_RETURN_KEYS_MAX; i++) {
		if (form->return_keys[i] == 0) {
			break;
		} else if (form->return_keys[i] == key) {
			return true;
		} else {
			;
		}
	}
	return false;
}





void cdw_form_add_return_char(cdw_form_t *form, int key)
{
	cdw_assert (form != (cdw_form_t *) NULL, "ERROR: \"form\" argument is NULL\n");
	cdw_assert (form->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    form->n_return_keys, N_RETURN_KEYS_MAX);
	cdw_assert (key != 0, "ERROR: trying to add key '0', but '0' is an initializer value\n");

	if (cdw_form_is_return_key(form, key)) {
		cdw_vdm ("WARNING: attempting to add key %d / \"%s\", but it is already on the list of return keys\n",
			 key, cdw_ncurses_key_label(key));
	} else {
		form->return_keys[form->n_return_keys++] = key;
	}

	return;
}





void cdw_form_add_return_chars(cdw_form_t *form, ...)
{
	cdw_assert (form != (cdw_form_t *) NULL, "ERROR: \"form\" argument is NULL\n");
	cdw_assert (form->n_return_keys < N_RETURN_KEYS_MAX,
		    "ERROR: there are already %d / %d return keys in the form, can't add another one\n",
		    form->n_return_keys, N_RETURN_KEYS_MAX);

	va_list ap;
	va_start(ap, form);
	int key = 'a';
	while ((key = va_arg(ap, int)) != 0) {
		cdw_form_add_return_char(form, key);
	}
	va_end(ap);

	return;
}





cdw_rv_t cdw_form_description_to_fields(cdw_form_descr_t descr[], cdw_form_t *cdw_form)
{
	FIELD **fields = cdw_form->fields;
	/* "<=" - to set proper value of guard in fields table */
	for (int i = 0; i <= cdw_form->n_fields; i++) {
		fields[i] = (FIELD *) NULL;
	}

	cdw_assert (cdw_form->field_widget_types != (cdw_id_t *) NULL, "ERROR: table of widget ids is NULL\n");

	for (int i = 0; i < cdw_form->n_fields; i++) {
		cdw_id_t widget_type = descr[i].widget_type;
		cdw_assert (i == descr[i].field_enum, "ERROR: order of items! field iterator (%d) != field enum (%d)\n", i, descr[i].field_enum);
		if (widget_type == -1) { /* guard in form description table */
			cdw_assert (0, "ERROR: reached descr guard, so loop condition is incorrect; i = %d, n_fields = %d\n", i, cdw_form->n_fields);
			break;
		} else if (widget_type == CDW_WIDGET_LABEL) {
			cdw_vdm ("INFO: creating field #%d: label with string = \"%s\"\n", i, (char *) descr[i].data1);
			fields[i] = cdw_ncurses_new_label_field(descr[i].n_lines, descr[i].n_cols,
								descr[i].begin_y, descr[i].begin_x,
								(char *) descr[i].data1);
		} else if (widget_type == CDW_WIDGET_CHECKBOX) {
			cdw_vdm ("INFO: creating field #%d: checkbox (new)\n", i);
			fields[i] = cdw_ncurses_new_checkbox_field(descr[i].begin_y, descr[i].begin_x,
								   (char *) NULL);
			/* I don't know why, but sometimes moving into
			   a checkbox field erases content displayed
			   in overlaying window; turning off O_PUBLIC
			   seems to fix this problem */
			field_opts_off(fields[i], O_PUBLIC);
			CDW_CHECKBOX *cb = cdw_checkbox_new(cdw_form->subwindow, descr[i].begin_y, descr[i].begin_x, descr[i].data2);
			int rv = set_field_userptr(fields[i], (void *) cb);
			cdw_assert (rv == E_OK, "ERROR: can't set userptr for field #%d\n", i);
		} else if (widget_type == CDW_WIDGET_INPUT) {
			cdw_vdm ("INFO: creating field #%d: input with initial string = \"%s\"\n", i, (char *) descr[i].data1);
			fields[i] = cdw_ncurses_new_input_field((size_t) descr[i].n_lines,
								descr[i].n_cols,
								descr[i].begin_y,
								descr[i].begin_x,
								(char *) descr[i].data1,
								descr[i].data2,
								CDW_NCURSES_INPUT_NONE,
								CDW_COLORS_INPUT);
		} else if (widget_type == CDW_WIDGET_BUTTON) {
			cdw_vdm ("INFO: creating field #%d: button\n", i);
			fields[i] = cdw_ncurses_new_input_field((size_t) descr[i].n_lines,
								descr[i].n_cols,
								descr[i].begin_y,
								descr[i].begin_x,
								(char *) NULL,
								0, //descr[i].data2,
								CDW_NCURSES_INPUT_NONE,
								CDW_COLORS_INPUT);
			/* to avoid situation when user presses some
			   alphanumeric key and the key is entered
			   into field under a button */
			field_opts_off(fields[i], O_EDIT);
			CDW_BUTTON *b = cdw_button_new(cdw_form->subwindow, descr[i].begin_y, descr[i].begin_x, descr[i].data1, descr[i].data2);
			if (b == (CDW_BUTTON *) NULL) {
				cdw_vdm ("ERROR: failed to create \"log path\" button for field #%d\n", i);
				return CDW_ERROR;
			}
			int rv = set_field_userptr(fields[i], (void *) b);
			cdw_assert (rv == E_OK, "ERROR: can't set userptr for field #%d\n", i);
		} else if (widget_type == CDW_WIDGET_DROPDOWN) {
			cdw_vdm ("INFO: creating field #%d: dropdown (new)\n", i);
			fields[i] = cdw_ncurses_new_input_field((size_t) descr[i].n_lines,
								descr[i].n_cols,
								descr[i].begin_y,
								descr[i].begin_x,
								(char *) NULL,
								0,
								CDW_NCURSES_INPUT_NONE,
								CDW_COLORS_INPUT);

			cdw_form_dropdown_maker_t *makers = (cdw_form_dropdown_maker_t *) descr[i].data1;
			int ind = (int) descr[i].data2;
			CDW_DROPDOWN *dd = makers[ind](cdw_form->subwindow, descr[i].begin_y, descr[i].begin_x,
						       descr[i].n_cols);
			int rv = set_field_userptr(fields[i], (void *) dd);
			cdw_assert (rv == E_OK, "ERROR: can't set userptr, i = %d\n", i);
		} else if (widget_type == CDW_WIDGET_TEXT) {
			cdw_vdm ("INFO: creating field #%d: text area\n", i);

			cdw_form_text_maker_t *makers = (cdw_form_text_maker_t *) descr[i].data1;
			int ind = (int) descr[i].data2;
			char *message = makers[ind]((int) descr[i].n_cols);

			fields[i] = cdw_ncurses_new_label_field(descr[i].n_lines, descr[i].n_cols,
								descr[i].begin_y, descr[i].begin_x,
								message);
			free(message);
			message = (char *) NULL;

			field_opts_off(fields[i], O_ACTIVE);
			field_opts_off(fields[i], O_EDIT);
		} else {
			cdw_assert (0, "ERROR: unknown widget type: %lld (field #%d)\n", widget_type, i);
		}

		cdw_form->field_widget_types[i] = widget_type;

		if (fields[i] == (FIELD *) NULL) {
			cdw_vdm ("ERROR: failed to initialize field #%d\n", i);
			return CDW_ERROR;
		}
	}

	return CDW_OK;
}





bool cdw_form_is_movement_key(int key)
{
	if (key == KEY_DOWN
	    || key == CDW_KEY_TAB
	    || key == KEY_UP
	    || key == KEY_BTAB
	    || key == KEY_BACKSPACE) { /* in some configurations backspace
					  may be a movement key, i.e. when
					  cursor is at the beginning of input field */
		return true;
	} else {
		return false;
	}
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_focus_off(cdw_form_t *cdw_form, int fi)
{
	cdw_form_driver_toggle_focus(cdw_form, fi, true);
	return;
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_focus_on(cdw_form_t *cdw_form, int fi)
{
	cdw_form_driver_toggle_focus(cdw_form, fi, false);
	return;
}





/* this function is called when a movement key has been detected;
   this function checks if current focus is on a widget that
   implements highlighted/non-highlighted states, and if it is,
   then highlight of current widget is turned off */
void cdw_form_driver_toggle_focus(cdw_form_t *cdw_form, int fi, bool off)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);

	if (cdw_form_is_button_field_index(cdw_form, fi)) {
		CDW_BUTTON *button = cdw_form_get_button(cdw_form, fi);
		if (button != (CDW_BUTTON *) NULL) {
			void (*function)(CDW_BUTTON *) = (void (*)(CDW_BUTTON *)) NULL;
			if (off) {
				function = cdw_button_unfocus;
				/* moving away from button, display blinking cursor */
				curs_set(1);
			} else {
				function = cdw_button_focus;
				/* moving into button, don't display blinking cursor */
				curs_set(0);
			}
			function(button);
		}
	} else if (cdw_form_is_dropdown_field_index(cdw_form, fi)) {
		CDW_DROPDOWN *dropdown = cdw_form_get_dropdown(cdw_form, fi);
		if (dropdown != (CDW_DROPDOWN *) NULL) {
			void (*function)(CDW_DROPDOWN *) = (void (*)(CDW_DROPDOWN *)) NULL;
			if (off) {
				function = cdw_dropdown_unfocus;
				/* moving away from dropdown, display blinking cursor */
				curs_set(1);
			} else {
				function = cdw_dropdown_focus;
				/* moving into dropdown, don't display blinking cursor */
				curs_set(0);
			}
			function(dropdown);
		}
	} else if (cdw_form_is_checkbox_field_index(cdw_form, fi)
		   || cdw_form_is_input_field_index(cdw_form, fi)) {
		if (off) {
			/* moving away from checkbox/input, don't display blinking cursor */
			curs_set(0);
		} else {
			/* moving away from checkbox/input, display blinking cursor */
			curs_set(1);
		}
	} else {
		;
	}

	//form_driver(cdw_form->form, REQ_INS_MODE);
	wrefresh(cdw_form->subwindow);

	return;
}





CDW_DROPDOWN *cdw_form_get_dropdown(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	FIELD *field = cdw_form->fields[fi];
	cdw_assert (field != (FIELD *) NULL, "ERROR: field %d is NULL\n", fi);

	CDW_DROPDOWN *dropdown = (CDW_DROPDOWN *) NULL;
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_DROPDOWN) {
		dropdown = (CDW_DROPDOWN *) field_userptr(field);
		cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: field user pointer is NULL dropdown (fi = %d)\n", fi);
	}

	return dropdown;
}





const char *cdw_form_get_string(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	FIELD *field = cdw_form->fields[fi];
	cdw_assert (field != (FIELD *) NULL, "ERROR: field %d is NULL\n", fi);

	char *string = (char *) NULL;
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_INPUT) {
		string = field_buffer(cdw_form->fields[fi], 0);
		cdw_string_rtrim(string);
		cdw_assert (string != (char *) NULL, "ERROR: field user pointer is NULL char (fi = %d)\n", fi);
	}

	return string;
}





CDW_BUTTON *cdw_form_get_button(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	FIELD *field = cdw_form->fields[fi];
	cdw_assert (field != (FIELD *) NULL, "ERROR: field %d is NULL\n", fi);

	CDW_BUTTON *button = (CDW_BUTTON *) NULL;
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_BUTTON) {
		button = (CDW_BUTTON *) field_userptr(field);
		cdw_assert (button != (CDW_BUTTON *) NULL, "ERROR: field user pointer is NULL button (fi = %d)\n", fi);
	}

	return button;
}





CDW_CHECKBOX *cdw_form_get_checkbox(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	FIELD *field = cdw_form->fields[fi];
	cdw_assert (field != (FIELD *) NULL, "ERROR: field %d is NULL\n", fi);

	CDW_CHECKBOX *checkbox = (CDW_CHECKBOX *) NULL;
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_CHECKBOX) {
		checkbox = (CDW_CHECKBOX *) field_userptr(field);
		cdw_assert (checkbox != (CDW_CHECKBOX *) NULL, "ERROR: field user pointer is NULL checkbox (fi = %d)\n", fi);
	}

	return checkbox;
}





bool cdw_form_is_checkbox_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	cdw_id_t type = cdw_form->field_widget_types[fi];

	if (type == CDW_WIDGET_CHECKBOX) {
		return true;
	} else {
		return false;
	}
}





bool cdw_form_is_dropdown_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	if (cdw_form->field_widget_types[fi] == CDW_WIDGET_DROPDOWN) {
		return true;
	} else {
		return false;
	}
}





bool cdw_form_is_button_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	return cdw_form->field_widget_types[fi] == CDW_WIDGET_BUTTON ? true : false;
}





bool cdw_form_is_input_field_index(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);
	return cdw_form->field_widget_types[fi] == CDW_WIDGET_INPUT ? true : false;
}





/**
   \brief Toggle state of checkbox

   Toggle state of checkbox that is placed in field with index\p fi.
   Make sure that this is really a checkbox before calling the function.

   Additionally the function shows/hides widgets in "tools" page as
   state of related checkbox changes.

   \param cdw_form - form in which a checkbox is placed
   \param fi - field index, index of field of form, that is related to the checkbox
*/
void cdw_form_driver_toggle_checkbox(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_CHECKBOX,
		    "ERROR: called the function for non-checkbox\n");
	cdw_assert (fi < cdw_form->n_fields,
		    "ERROR: fi is larger than number of fields: %d / %d\n",
		    fi, cdw_form->n_fields);

	CDW_CHECKBOX *checkbox = cdw_form_get_checkbox(cdw_form, fi);
	cdw_checkbox_toggle(checkbox);
	if (checkbox->on_toggle != (cdw_form_widget_function_t) NULL) {
		bool state = cdw_checkbox_get_state(checkbox);
		checkbox->on_toggle(cdw_form, &state);
	}
	return;
}





void cdw_form_driver_go_to_field(cdw_form_t *cdw_form, int fi)
{
	FORM *form = cdw_form->form;
	if (fi < 0 || fi >= cdw_form->n_fields) {
		cdw_vdm ("WARNING: fi = %d out of bounds (n_fields = %d)\n", fi, cdw_form->n_fields);
		fi = 0;
	}

	int rv = form_driver(form, REQ_FIRST_FIELD);
	if (rv != E_OK) {
		cdw_vdm ("ERROR: form_driver() returns \"%s\"\n", cdw_ncurses_error_string(rv));
	}

	/* initial field may not be first when driver needs to switch
	   to field in which there is an error */
	/* TODO: check why using simple for() loop fails: iterating stops
	   before correct field */
	int i = field_index(current_field(form));
	if (i == fi) {
		;
	} else {
		while (i < fi && i < cdw_form->n_fields - 1) {
			rv = form_driver(form, REQ_NEXT_FIELD);
			if (rv != E_OK) {
				cdw_vdm ("ERROR: form_driver() returns \"%s\"\n", cdw_ncurses_error_string(rv));
				break;
			} else {
				i = field_index(current_field(form));
			}
		}
	}

	cdw_form_driver_focus_on(cdw_form, fi);

	return;
}





void cdw_form_redraw_widgets(cdw_form_t *cdw_form)
{
	for (int i = 0; i < cdw_form->n_fields; i++) {
		cdw_id_t id = cdw_form->field_widget_types[i];
		if (id == CDW_WIDGET_CHECKBOX) {
			CDW_CHECKBOX *cb = cdw_form_get_checkbox(cdw_form, i);
			cdw_checkbox_draw(cb);
		} else if (id == CDW_WIDGET_BUTTON) {
			CDW_BUTTON *b = cdw_form_get_button(cdw_form, i);
			cdw_button_unfocus(b);
		} else if (id == CDW_WIDGET_DROPDOWN) {
			CDW_DROPDOWN *dd = cdw_form_get_dropdown(cdw_form, i);
			cdw_dropdown_display_current_item(dd);
		} else {
			;
		}

	}
	return;
}





bool cdw_form_get_checkbox_state(cdw_form_t *cdw_form, int fi)
{
	cdw_assert (cdw_form->field_widget_types[fi] == CDW_WIDGET_CHECKBOX,
		    "ERROR: trying to get state of field which isn't a CHECKBOX\n");

	CDW_CHECKBOX *cb = cdw_form_get_checkbox(cdw_form, fi);
	return cdw_checkbox_get_state(cb);
}





void *cdw_form_set_function(cdw_form_t *cdw_form, int fi, cdw_form_widget_function_t function)
{
	cdw_id_t id = cdw_form->field_widget_types[fi];
	if (id == CDW_WIDGET_BUTTON) {
		CDW_BUTTON *b = field_userptr(cdw_form->fields[fi]);
		b->on_click = function;
		set_field_userptr(cdw_form->fields[fi], (void *) b);
		return (void *) b;
	} else if (id == CDW_WIDGET_CHECKBOX) {
		CDW_CHECKBOX *cb = field_userptr(cdw_form->fields[fi]);
		cb->on_toggle = function;
		set_field_userptr(cdw_form->fields[fi], (void *) cb);
		return (void *) cb;
	} else {
		cdw_assert(0, "ERROR: called the function for widget, which doesn't have a function to set\n");
		return (void *) NULL;
	}
}





int cdw_form_handle_enter(cdw_form_t *cdw_form, int fi)
{
	if (cdw_form_is_button_field_index(cdw_form, fi)) {
		CDW_BUTTON *b = cdw_form_get_button(cdw_form, fi);
		return b->on_click((void *) cdw_form, (void *) NULL);
	} else if (cdw_form_is_dropdown_field_index(cdw_form, fi)) {
		CDW_DROPDOWN *dropdown = cdw_form_get_dropdown(cdw_form, fi);
		cdw_assert (dropdown != (CDW_DROPDOWN *) NULL, "ERROR: dropdown is NULL\n");
		cdw_rv_t crv = cdw_dropdown_driver(dropdown);
		if (crv == CDW_OK) {
			const char *label = cdw_dropdown_get_current_item_label(dropdown);
			FIELD **fields = form_fields(cdw_form->form);
			/* setting buffer makes field to display
			   string, but slightly moved to left */
			set_field_buffer(*(fields + fi), 0, label);

			/* call to cdw_dropdown_focus() fixes it */
			cdw_dropdown_focus(dropdown);

			form_driver(cdw_form->form, REQ_VALIDATION);
		} else {
			;
		}
		return 'a';
	} else {
		return 'a';
	}
}
