# bzr-dbus: dbus support for bzr/bzrlib.
# Copyright (C) 2007,2009 Canonical Limited.
#   Author: Robert Collins
#   Author: Markus Korn (zeitgeist support)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 

"""System wide hooks to trigger dbus events from bzr activity."""


def get_activity():
    import activity
    import dbus
    try:
        return activity.Activity()
    except dbus.DBusException, e:
        from bzrlib import trace
        trace.mutter("Unable to connect to dbus, won't send events."
                     "Reason: '%s'" % e)
        return None


def on_post_change_branch_tip(params):
    """Announce the new head revision of the branch to dbus."""
    activity = get_activity()
    if activity is None:
        return
    activity.advertise_branch(params.branch)


def on_server_start(local_urls, public_url):
    """Add the servers local and public urls to the session Broadcaster."""
    activity = get_activity()
    if activity is None:
        return
    for local_url in local_urls:
        activity.add_url_map(local_url, public_url)


def on_server_stop(local_urls, public_url):
    """The server has shutdown, so remove the servers local and public urls."""
    activity = get_activity()
    if activity is None:
        return
    for local_url in local_urls:
        activity.remove_url_map(local_url, public_url)


_checked_zeitgeist_client = False
_zeitgeist_client = None


def get_zeitgeist_client():
    global _zeitgeist_client, _checked_zeitgeist_client
    if _checked_zeitgeist_client:
        return _zeitgeist_client
    _checked_zeitgeist_client = True
    from bzrlib import trace
    import dbus
    try:
        from zeitgeist.client import ZeitgeistClient
    except ImportError, e:
        trace.mutter("Zeitgeist not present: %s" % e)
        return None
    try:
        _zeitgeist_client = ZeitgeistClient()
    except RuntimeError, e:
        trace.warn("Unable to connect to Zeitgeist, won't send events."
                   "Reason: '%s'" % e)
        return None
    except dbus.DBusException, e:
        trace.mutter("Unable to connect to Zeitgeist, won't send events."
                     "Reason: '%s'" % e)
        return None
    else:
        return _zeitgeist_client


def on_post_commit(local, master, old_revno, old_revid, new_revno, new_revid):
    """A commit was done. Inform zeitgeist if possible."""
    client = get_zeitgeist_client()
    if client is None:
        return
    import time
    from zeitgeist.datamodel import (Event, Subject, Interpretation,
                                     Manifestation)
    revision = master.repository.get_revision(new_revid)
    if new_revno == 1:
        interpretation = Interpretation.CREATE_EVENT
    else:
        interpretation = Interpretation.MODIFY_EVENT
    subject = Subject.new_for_values(
        uri=unicode(master.base),
        interpretation=unicode(Interpretation.DATA_CONTAINER),
        manifestation=unicode(Manifestation.FILE_DATA_OBJECT),
        text=unicode(revision.message),
        origin=unicode(master.base),
    )
    event = Event.new_for_values(
        timestamp=int(time.time()*1000),
        interpretation=unicode(interpretation),
        manifestation=unicode(Manifestation.USER_ACTIVITY),
        actor="application://bzr.desktop", #something usefull here, always olive-gtk?
        subjects=[subject,]
    )
    client.insert_event(event)


