/*
 * out_html.c                      HTML Output
 *
 * $Id: out_html.c 1 2004-10-17 17:32:34Z tgr $
 *
 * Copyright (c) 2001-2004 Thomas Graf <tgraf@suug.ch>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <bmon/bmon.h>
#include <bmon/node.h>
#include <bmon/output.h>
#include <bmon/graph.h>
#include <bmon/utils.h>

static const char *c_path;
static int c_graph_height = 6;

static FILE *
open_file(const char *p)
{
	FILE *fd;

	if (!(fd = fopen(p, "w")))
		quit("fopen(%s) failed: %s\n", p, strerror(errno));

	return fd;
}

static void
write_css(const char *path)
{
    char ofile[FILENAME_MAX];
    FILE *f;

	snprintf(ofile, sizeof(ofile), "%s/layout.css", path);
	
	/* do not overwrite an existing .css */
	if (!access(ofile, R_OK))
		return;

	f = open_file(ofile);
	fprintf(f, 
		"/*\n" \
		" * Generated by %s\n" \
		" */\n" \
		"\n" \
		"body {\n" \
		"	background: #f7f7f7;\n" \
		"	margin-top: 20px;\n" \
		"}\n" \
		"\n" \
		"p, b {\n" \
		"	color: #000000;\n" \
		"	font-family: Verdana, Arial, Helvetica, sans-serif;\n" \
		"	font-size: 10pt;\n" \
		"}\n" \
		"\n" \
		"a:link,a:visited,a:hover {\n" \
		"  color: #4E7AAF;\n" \
		"  text-decoration: underline;\n" \
		"}\n" \
		"\n" \
		".p_banner {\n" \
		"	margin-top: 8px;\n" \
		"	margin-bottom: 8px;\n" \
		"	text-align: center;\n" \
		"	font-size: 18pt;\n" \
		"	font-weight: bold;\n" \
		"}\n" \
		"\n" \
		".p_intf_title {\n" \
		"	font-weight: bold;\n" \
		"	font-size: 12pt;\n" \
		"}\n" \
		"\n" \
		".p_nodes_title {\n" \
		"	font-weight: bold;\n" \
		"	margin-bottom: 1px;\n" \
		"}\n" \
		".p_node_title {\n" \
		"	text-align: center;\n" \
		"	margin: 0pt;\n" \
		"	margin-bottom: 10px;\n" \
		"	font-weight: bold;\n" \
		"	font-size: 18pt;\n" \
		"}\n" \
		"\n" \
		".p_title {\n" \
		"	margin: 0pt;\n" \
		"	margin-top: 5px;\n" \
		"	text-align: center;\n" \
		"	font-weight: bold;\n" \
		"	font-size: 14pt;\n" \
		"}\n" \
		"\n" \
		"a.a_node_link:link,a.a_node_link:visited,a.a_node_link:hover {\n" \
		"	color: #000;\n" \
		"	text-decoration: none;\n" \
		"}\n" \
		"\n" \
		"a.a_node_link:hover {\n" \
		"	text-decoration: underline;\n" \
		"}\n" \
		".p_node_list {\n" \
		"	margin-left: 10px;\n" \
		"	margin-top: 0px;\n" \
		"	padding: 0px;\n" \
		"	font-weight: bold;\n" \
		"}\n" \
		"\n" \
		".p_selection {\n" \
		"	text-align: center;\n" \
		"	margin: 0pt;\n" \
		"}\n" \
		"\n" \
		"#tbl_intf_list {\n" \
		"	width: 100%%;\n" \
		"\n" \
		"	padding-top: 5px;\n" \
		"	border-top: 1px dashed #cdcdcd;\n" \
		"\n" \
		"	padding-bottom: 5px;\n" \
		"	border-bottom: 1px dashed #cdcdcd;\n" \
		"}\n" \
		"\n" \
		"#tr_intf_list_hdr {\n" \
		"	font-weight: bold;\n" \
		"}\n" \
		"\n" \
		"#tbl_details {\n" \
		"	width: 100%%;\n" \
		"	border-top: 1px dashed #cdcdcd;\n" \
		"}\n" \
		"\n" \
		"#tr_details_hdr {\n" \
		"	font-weight: bold;\n" \
		"}\n" \
		"\n" \
		"#tbl_overall {\n" \
		"	width: 750px;\n" \
		"	border-spacing: 0px;\n" \
		"}\n" \
		"\n" \
		"#td_header {\n" \
		"	background: #b8c4db;\n" \
		"	text-align: center;\n" \
		"	border: 1px solid #000;\n" \
		"}\n" \
		"\n" \
		"#td_left_col {\n" \
		"	background: #eeeeee;\n" \
		"	padding-left: 10px;\n" \
		"	border-left: 1px solid #cdcdcd;\n" \
		"	border-right: 1px solid #cdcdcd;\n" \
		"	border-bottom: 1px solid #cdcdcd;\n" \
		"	vertical-align: top;\n" \
		"	width: 130px;\n" \
		"}\n" \
		"\n" \
		"#td_right_col {\n" \
		"	background: #fff;\n" \
		"	padding: 20px;\n" \
		"	padding-top: 10px;\n" \
		"	border-bottom: 1px solid #cdcdcd;\n" \
		"	border-right: 1px solid #cdcdcd;\n" \
		"	vertical-align: top;\n" \
		"}\n", PACKAGE_STRING);
	fclose(f);
}

static void
list_add(node_t *node, void *arg)
{
	FILE *fd = (FILE *) arg;

	fprintf(fd, "<a class=\"a_node_link\" href=\"%s.html\">%s</a><br />\n",
		node->n_name, node->n_name);
}

static void
write_header(FILE *fd, const char *title)
{
	fprintf(fd,
		"<!DOCTYPE html PUBLIC \"-//W3c/DTD XHTML 1.0 Strict//EN\"\n"         \
		"    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n"        \
		"<html xmlns=\"http://www.w3.org/1999/xhtml\" lang=\"en\"\n"          \
		"       xml:lang=\"en\">\n"                                           \
		"<head>\n"                                                            \
        "<title>%s</title>\n"                                                 \
        "<link rel=\"stylesheet\" type=\"text/css\" href=\"layout.css\" />\n" \
        "</head>\n"                                                           \
        "<body>\n"                                                            \
		"<table id=\"tbl_overall\" align=\"center\">\n"                       \
		"<tr>\n"                                                              \
		"<td colspan=\"2\" id=\"td_header\">\n"                               \
		"<p class=\"p_banner\">Bandwidth Statistics</p>\n"                    \
		"</p></td>\n"                                                         \
		"</tr><tr>\n"                                                         \
		"<td id=\"td_left_col\">\n"                                           \
		"<p class=\"p_nodes_title\">Nodes:</p>\n"                             \
		"<p class=\"p_node_list\">\n",
		title);

	foreach_node(list_add, fd);

	fprintf(fd,
		"</p></td>\n"                                                         \
		"<td id=\"td_right_col\">\n");
}

static void
write_footer(FILE *fd)
{
	fprintf(fd,
		"</td>\n" \
		"</tr>\n" \
		"</table>\n" \
		"</body>\n" \
		"</htlm>\n");
}

struct html_xdata {
	FILE *fd;
	node_t *node;
};

static void
write_interface_entry(FILE *fd, intf_t *intf, node_t *node)
{
	int i;
	double rx, tx;
	char *rx_u, *tx_u;
	
	rx = sumup(intf->i_rx_bytes.r_tps, &rx_u);
	tx = sumup(intf->i_tx_bytes.r_tps, &tx_u);

	fprintf(fd,
		"<tr>\n" \
		"<td>%d</td>\n" \
		"<td>",
		intf->i_index);

	for (i = 0; i < intf->i_level; i++)
		fprintf(fd, "&nbsp;&nbsp;");

	fprintf(fd,
		"<a class=\"a_intf\" href=\"%s.%d.s.html\">%s</a></td>\n" \
		"<td>%.2f %s</td>\n" \
		"<td>%u</td>\n" \
		"<td>%.2f %s</td>\n" \
		"<td>%u</td>\n" \
		"</tr>\n",
		node->n_name, intf->i_index, intf->i_name,
		rx, rx_u, intf->i_rx_packets.r_tps,
		tx, tx_u, intf->i_tx_packets.r_tps);
}

static void
handle_child(intf_t *intf, void *arg)
{
	struct html_xdata *x = arg;

	write_interface_entry(x->fd, intf, x->node);
	foreach_child(x->node, intf, handle_child, x);
}

static void
add_to_interace_list(FILE *fd, node_t *node, intf_t *intf)
{
	struct html_xdata x = {
		.fd = fd,
		.node = node,
	};

	if (intf->i_parent)
		return;

	write_interface_entry(fd, intf, node);
	foreach_child(node, intf, handle_child, &x);
}

static void
write_interface_list(FILE *fd, node_t *node)
{
	int i;

	fprintf(fd,
		"<p class=\"p_node_title\">%s</p>\n" \
		"<table id=\"tbl_intf_list\">\n" \
		"<tr id=\"tr_intf_list_hdr\">\n" \
		"<td id=\"td_intf_list_hdr_idx\">#</td>\n" \
		"<td id=\"td_intf_list_hdr_name\">Name</td>\n" \
		"<td id=\"td_intf_list_hdr_rx\">RX</td>\n" \
		"<td id=\"td_intf_list_hdr_rxp\">RX #</td>\n" \
		"<td id=\"td_intf_list_hdr_tx\">TX</td>\n" \
		"<td id=\"td_intf_list_hdr_txp\">TX #</td>\n",
		node->n_name);

	for (i = 0; i < node->n_nintf; i++)
		if (node->n_intf[i].i_name[0])
			add_to_interace_list(fd, node, &node->n_intf[i]);

	fprintf(fd,
		"</table>\n");
}

static void
write_graph(FILE *fd, node_t *node, intf_t *intf, hist_elem_t *src, const char *x_unit)
{
	int w;
	graph_t *g = create_graph(src, c_graph_height);

	fprintf(fd,
		"<p class=\"p_selection\">[");

	if (get_read_interval() != 1.0f)
		fprintf(fd, 
			"<a class=\"a_selection\" href=\"%s.%d.r.html\">Read interval</a> ] [",
				node->n_name, intf->i_index);

	fprintf(fd, 
		"<a class=\"a_selection\" href=\"%s.%d.s.html\">Seconds</a> ] [" \
		"<a class=\"a_selection\" href=\"%s.%d.m.html\">Minutes</a> ] [" \
		"<a class=\"a_selection\" href=\"%s.%d.h.html\">Hours</a> ] [" \
		"<a class=\"a_selection\" href=\"%s.%d.d.html\">Days</a> ]</p>\n" \
		"<pre class=\"pre_graph\">\n" \
		"RX   %s\n",
		node->n_name, intf->i_index,
		node->n_name, intf->i_index,
		node->n_name, intf->i_index,
		node->n_name, intf->i_index,
		g->g_rx.t_y_unit);
	
	for (w = (c_graph_height - 1); w >= 0; w--)
		fprintf(fd, "%8.2f %s\n", g->g_rx.t_y_scale[w], (char *) g->g_rx.t_data + (w * (HISTORY_SIZE + 1)));
	
	fprintf(fd, "         1   5   10   15   20   25   30   35   40   " \
		"45   50   55   60 %s\n", x_unit);

	fprintf(fd, "TX   %s\n", g->g_tx.t_y_unit);
	
	for (w = (c_graph_height - 1); w >= 0; w--)
		fprintf(fd, "%8.2f %s\n", g->g_tx.t_y_scale[w], (char *) g->g_tx.t_data + (w * (HISTORY_SIZE + 1)));
	
	fprintf(fd, "         1   5   10   15   20   25   30   35   40   " \
		"45   50   55   60 %s\n", x_unit);

	fprintf(fd,
		"</pre>\n");

	free_graph(g);
}

static void
print_attr_detail(intf_attr_t *a, void *arg)
{
	FILE *fd = (FILE *) arg;

	fprintf(fd,
		"<tr id=\"tr_details\">\n" \
		"<td id=\"td_details_name\">%s</td>\n" \
		"<td id=\"td_details_rx\">%llu</td>\n" \
		"<td id=\"td_details_tx\">%llu</td>\n" \
		"</tr>\n",
		type2name(a->a_type), a->a_rx, a->a_tx);
}

static void
write_details(FILE *fd, intf_t *intf)
{
	double rx, tx;
	char *rx_u, *tx_u;
	
	rx = sumup(intf->i_rx_bytes.r_total, &rx_u);
	tx = sumup(intf->i_tx_bytes.r_total, &tx_u);

	fprintf(fd,
		"<table id=\"tbl_details\">\n" \
		"<tr id=\"tr_details_hdr\">\n" \
		"<td id=\"td_details_hdr_name\">Attribute</td>\n" \
		"<td id=\"td_details_hdr_rx\">RX</td>\n" \
		"<td id=\"td_details_hdr_tx\">TX</td>\n" \
		"</tr>\n" \
		"<tr id=\"tr_details\">\n" \
		"<td id=\"td_details_name\">Bytes</td>\n" \
		"<td id=\"td_details_rx\">%.2f %s</td>\n" \
		"<td id=\"td_details_tx\">%.2f %s</td>\n" \
		"</tr>\n" \
		"<tr id=\"tr_details\">\n" \
		"<td id=\"td_details_name\">Packets</td>\n" \
		"<td id=\"td_details_rx\">%llu</td>\n" \
		"<td id=\"td_details_tx\">%llu</td>\n" \
		"</tr>\n",
		rx, rx_u, tx, tx_u,
		intf->i_rx_packets.r_total,
		intf->i_tx_packets.r_total);
	
	foreach_attr(intf, print_attr_detail, (void *) fd);
}

static void
__write_per_intf(intf_t *intf, node_t *node, hist_elem_t *src, char *x_unit)
{
	char outf[FILENAME_MAX];
	FILE *fd;

	snprintf(outf, sizeof(outf), "%s/%s.%d.%s.html",
		c_path, node->n_name, intf->i_index, x_unit);
	fd = open_file(outf);
	write_header(fd, node->n_name);

	write_interface_list(fd, node);
	fprintf(fd, "<p class=\"p_title\">%s</p>\n", intf->i_name);
	write_graph(fd, node, intf, src, x_unit);
	write_details(fd, intf);
	
	write_footer(fd);
	fclose(fd);
}

static void
write_per_intf(intf_t *intf, void *arg)
{
	node_t *node = (node_t *) arg;

	if (get_read_interval() != 1.0f)
		__write_per_intf(intf, node, &intf->i_bytes_hist.h_read, "r");
	__write_per_intf(intf, node, &intf->i_bytes_hist.h_sec, "s");
	__write_per_intf(intf, node, &intf->i_bytes_hist.h_min, "m");
	__write_per_intf(intf, node, &intf->i_bytes_hist.h_hour, "h");
	__write_per_intf(intf, node, &intf->i_bytes_hist.h_day, "d");
}


static void
write_per_node(node_t *node, void *arg)
{
	char outf[FILENAME_MAX];
	FILE *fd;

	snprintf(outf, sizeof(outf), "%s/%s.html", c_path, node->n_name);
	fd = open_file(outf);
	write_header(fd, node->n_name);
	write_interface_list(fd, node);
	write_footer(fd);
	fclose(fd);

	foreach_intf(node, write_per_intf, (void *) node);
}


void
html_draw(void)
{
	char outf[FILENAME_MAX];
	FILE *fd;

	umask(0133);
	write_css(c_path);
	
	foreach_node(write_per_node, NULL);

	snprintf(outf, sizeof(outf), "%s/index.html", c_path);
	fd = open_file(outf);
	write_header(fd, "Bandwidth Statistics");
	write_footer(fd);
	fclose(fd);
}

static void
print_module_help(void)
{
	printf(
		"HTML - HTML Output\n" \
		"\n" \
		"  Lightweight HTML output with CSS configuration.\n" \
		"  Author: Thomas Graf <tgraf@suug.ch>\n" \
		"\n" \
		"  Options:\n" \
		"    path=PATH        Output directory\n" \
		"    height=NUM       Height of graphical statistics (default: 6)\n");
}

static void
html_set_opts(tv_t *attrs)
{
	while (attrs) {
		if (!strcasecmp(attrs->type, "path") && attrs->value)
			c_path = attrs->value;
		if (!strcasecmp(attrs->type, "height") && attrs->value)
			c_graph_height = strtol(attrs->value, NULL, 0);
		else if (!strcasecmp(attrs->type, "help")) {
			print_module_help();
			exit(0);
		}
		
		attrs = attrs->next;
	}
}

static int
html_probe(void)
{
	if (NULL == c_path)
		quit("You must specify a path (-O html:path=DIR)\n");

	return 1;
}

static struct output_module html_ops = {
	.om_name = "html",
	.om_draw = html_draw,
	.om_set_opts = html_set_opts,
	.om_probe = html_probe,
};

static void __init
html_init(void)
{
	register_secondary_output_module(&html_ops);
}
