/*
 * Bickley - a meta data management framework, the file system spider.
 *
 * Copyright © 2007-2008 Øyvind Kolås <pippin@gimp.org>
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <glib.h>
#include <gio/gio.h>

#include "bkl-orbiter.h"
#include "bkl-path-finder.h"

/*#define REQUIRED_FIELDS "*" */
#define REQUIRED_FIELDS "standard::type,standard::name,time::modified,standard::content-type"

typedef struct _BklPathFinderDir {
    char *uri;

    GFileMonitor *dir_monitor;
    GHashTable *entries; /* Of BklPathFinderEntry */
} BklPathFinderDir;

typedef struct _BklPathFinderEntry {
    char *uri;
    gboolean directory;
} BklPathFinderEntry;

static void
free_entry (BklPathFinderEntry *entry)
{
    g_free (entry->uri);
    g_slice_free (BklPathFinderEntry, entry);
}

static void
free_directory (gpointer data)
{
    BklPathFinderDir *pfd = (BklPathFinderDir *) data;

    g_hash_table_destroy (pfd->entries);

    g_object_unref (pfd->dir_monitor);

    g_free (pfd->uri);
    g_slice_free (BklPathFinderDir, pfd);
}

static void
unref_pending (gpointer data,
               gpointer userdata)
{
    g_object_unref ((GObject *) data);
}

void
bkl_path_finder_destroy (BklFinder *finder)
{
    BklPathFinder *self = (BklPathFinder *) finder;

    g_free (finder->base_uri);
    g_object_unref (self->base_file);

    g_queue_foreach (self->pending, unref_pending, NULL);
    g_queue_free (self->pending);

    g_hash_table_destroy (self->directories);
    g_free (self);
}

static void
remove_entries_in_directory (BklPathFinder    *finder,
                             BklPathFinderDir *pfd)
{
    GPtrArray *fa;
    GList *l, *entries;
    int i;

    fa = g_ptr_array_new ();

    entries = g_hash_table_get_values (pfd->entries);
    for (l = entries; l; l = l->next) {
        BklPathFinderEntry *entry = (BklPathFinderEntry *) l->data;

        if (entry->directory) {
            BklPathFinderDir *subdir;

            subdir = g_hash_table_lookup (finder->directories, entry->uri);
            if (subdir == NULL) {
                g_warning ("Could not find %s in directories", entry->uri);
            } else {
                /* Recurse into subdirs */
                /* g_print ("Removing dir %s\n", uri); */
                remove_entries_in_directory (finder, subdir);
                g_hash_table_remove (finder->directories, entry->uri);
            }
        } else {
            GFile *file = g_file_new_for_uri (entry->uri);

            /* g_print ("Removing %s\n", uri); */
            g_ptr_array_add (fa, file);
        }
    }
    g_list_free (entries);

    bkl_source_remove_files (((BklFinder *) finder)->source, fa);

    for (i = 0; i < fa->len; i++) {
        g_object_unref (fa->pdata[i]);
    }
    g_ptr_array_free (fa, TRUE);
}

static void
remove_file_from_parent (BklPathFinder *finder,
                         GFile         *file)
{
    BklPathFinderDir *pfd;
    GFile *parent = g_file_get_parent (file);
    char *uri, *parent_uri;

    parent_uri = g_file_get_uri (parent);
    pfd = g_hash_table_lookup (finder->directories, parent_uri);
    g_free (parent_uri);

    g_object_unref (parent);

    /* @file may not have a parent in @finder->directories if it is the
       top level that we are watching */
    if (pfd == NULL) {
        return;
    }

    uri = g_file_get_uri (file);
    g_hash_table_remove (pfd->entries, uri);
    g_free (uri);
}

static void
directory_changed_cb (GFileMonitor     *monitor,
                      GFile            *file,
                      GFile            *other_file,
                      GFileMonitorEvent event,
                      BklPathFinder    *finder)
{
    GFileInfo *info;
    GError *error = NULL;
    GPtrArray *fa;
    char *uri, *basename;
    BklPathFinderDir *pfd;

    basename = g_file_get_basename (file);
    if (basename && basename[0] == '.') {
        /* Ignore anything on files that begin with a .
           Some programs (like sound-juicer) use .files as temporary
           work files and then copy the .file into the correct place */
        g_free (basename);
        return;
    }

    g_free (basename);

    switch (event) {
    case G_FILE_MONITOR_EVENT_CHANGED:
        break;

    case G_FILE_MONITOR_EVENT_CREATED:
    case G_FILE_MONITOR_EVENT_CHANGES_DONE_HINT:
        info = g_file_query_info (file, "standard::type",
                                  G_FILE_QUERY_INFO_NONE, NULL, &error);
        if (error != NULL) {
            uri = g_file_get_uri (file);

            g_warning ("Error getting info for %s: %s", uri, error->message);
            g_error_free (error);

            g_free (uri);
            return;
        }

        if (event == G_FILE_MONITOR_EVENT_CHANGES_DONE_HINT &&
            info && g_file_info_get_file_type (info) == G_FILE_TYPE_DIRECTORY) {
            /* We ignore directories as all the changes will be dealt with in
               other events */
            g_object_unref (info);
            return;
        }

        g_object_unref (info);
        g_queue_push_tail (finder->pending, g_object_ref (file));

        /* Tell the orbiter that we have more work to do */
        ((BklFinder *) finder)->source->more_work = TRUE;
        bkl_source_in_progress (((BklFinder *) finder)->source);
        bkl_orbiter_start_worker ();
        break;

    case G_FILE_MONITOR_EVENT_DELETED:

        uri = g_file_get_uri (file);
        pfd = g_hash_table_lookup (finder->directories, uri);

        /* We can't call g_file_info_query because @file points at a
           non-existing file. Which means we can't find out if it was a dir
           that way, so we check if its in the directories hash table */
        if (pfd) {
            remove_entries_in_directory (finder, pfd);
            g_hash_table_remove (finder->directories, uri);
        } else {
            fa = g_ptr_array_sized_new (1);
            g_ptr_array_add (fa, file);

            bkl_source_remove_files (((BklFinder *) finder)->source, fa);
            g_ptr_array_free (fa, TRUE);
        }

        remove_file_from_parent (finder, file);

        g_free (uri);
        break;

    case G_FILE_MONITOR_EVENT_ATTRIBUTE_CHANGED:
        break;

    case G_FILE_MONITOR_EVENT_PRE_UNMOUNT:
    case G_FILE_MONITOR_EVENT_UNMOUNTED:
    default:
        break;
    }
}

static gboolean
entry_in_directory (BklPathFinderDir *pfd,
                    GFile            *file)
{
    gboolean exists;
    char *uri;

    uri = g_file_get_uri (file);
    exists = (g_hash_table_lookup (pfd->entries, uri) != NULL);
    g_free (uri);

    return exists;
}

static void
process_file (BklPathFinder *finder,
              GFile         *file)
{
    BklPathFinderDir *current_pfd;
    GError *error = NULL;
    GFileInfo *info;
    GFileType type;
    GFileEnumerator *fe;
    InvestigateUri *iu;
    GPtrArray *flist = NULL;
    int i;
    char *uri, *parent_uri;
    GFile *parent_file;
    GPtrArray *album_art;
    gboolean is_music_dir = FALSE; /* We set this when we find
                                      a music file */

    info = g_file_query_info (file, REQUIRED_FIELDS,
                              G_FILE_QUERY_INFO_NONE, NULL, &error);
    if (info == NULL) {
        char *uri;

        uri = g_file_get_uri (file);
        g_warning ("Error querying info for %s: %s", uri, error->message);
        g_free (uri);

        g_error_free (error);

        return;
    }

    type = g_file_info_get_file_type (info);

    switch (type) {
    case G_FILE_TYPE_REGULAR:

        /* FIXME: Need to add this to the parent's entry list */
        flist = g_ptr_array_sized_new (1);

        iu = g_slice_new (InvestigateUri);
        iu->file = g_object_ref (file);
        g_file_info_get_modification_time (info, &iu->mtime);
        g_ptr_array_add (flist, iu);

        g_object_unref (info);

        parent_file = g_file_get_parent (file);
        parent_uri = g_file_get_uri (parent_file);
        g_object_unref (parent_file);

        current_pfd = g_hash_table_lookup (finder->directories, parent_uri);
        if (current_pfd && entry_in_directory (current_pfd, file) == FALSE) {
            BklPathFinderEntry *entry = g_slice_new0 (BklPathFinderEntry);

            entry->uri = g_file_get_uri (file);
            entry->directory = FALSE;

            g_hash_table_insert (current_pfd->entries, entry->uri, entry);
        }
        g_free (parent_uri);
        break;

    case G_FILE_TYPE_DIRECTORY:

        g_object_unref (info);

        uri = g_file_get_uri (file);

        current_pfd = g_hash_table_lookup (finder->directories, uri);
        if (current_pfd == NULL) {
            current_pfd = g_slice_new0 (BklPathFinderDir);

            current_pfd->uri = g_strdup (uri);
            current_pfd->dir_monitor = g_file_monitor_directory
                (file, G_FILE_MONITOR_NONE, NULL, &error);

            if (current_pfd->dir_monitor == NULL) {
                g_warning ("Error adding monitor for %s: %s", uri,
                           error->message);
                g_error_free (error);
            } else {
                g_signal_connect (current_pfd->dir_monitor, "changed",
                                  G_CALLBACK (directory_changed_cb), finder);
            }

            current_pfd->entries = g_hash_table_new_full
                (g_str_hash, g_str_equal, NULL, (GDestroyNotify) free_entry);

            /* Store the directory so we can delete it and all the files
               contained in it when the directory is deleted from the
               file system */
            g_hash_table_insert (finder->directories, g_strdup (uri),
                                 current_pfd);
        }

        fe = g_file_enumerate_children (file, REQUIRED_FIELDS,
                                        G_FILE_QUERY_INFO_NONE, NULL, &error);
        if (fe == NULL) {
            g_warning ("Error enumerating children of %s: %s", uri,
                       error->message);
            g_free (uri);

            g_error_free (error);

            return;
        }

        flist = g_ptr_array_new ();

        /* We collect all images in a directory seperately.
           If the directory has music in it, then we ignore the images
           otherwise we treat them as normal.
           FIXME: Is this an acceptable way to work it out */
        album_art = g_ptr_array_new ();

        while ((info = g_file_enumerator_next_file (fe, NULL, NULL))) {
            BklPathFinderEntry *pfe;
            GFile *subfile;
            const char *filename, *mimetype;

            filename = g_file_info_get_name (info);

            /* Skip hidden files */
            if (filename && filename[0] == '.') {
                g_object_unref (info);
                continue;
            }

            mimetype = g_file_info_get_content_type (info);
            if (is_music_dir == FALSE) {
                is_music_dir = g_str_has_prefix (mimetype, "audio");
            } else {
                /* Once we've found a music dir, we can ignore images */
                if (g_str_has_prefix (mimetype, "image")) {
                    g_object_unref (info);
                    continue;
                }
            }

            if (is_music_dir && album_art) {
                /* Now we've found the music dir, free the pending art */
                for (i = 0; i < album_art->len; i++) {
                    InvestigateUri *iu = album_art->pdata[i];

                    g_object_unref (iu->file);
                    g_slice_free (InvestigateUri, iu);
                }
                g_ptr_array_free (album_art, TRUE);
                album_art = NULL;
            }

            subfile = g_file_get_child (file, filename);

            pfe = g_slice_new0 (BklPathFinderEntry);
            pfe->uri = g_file_get_uri (subfile);

            type = g_file_info_get_file_type (info);
            if (type == G_FILE_TYPE_REGULAR) {
                gboolean is_image;

                is_image = g_str_has_prefix (mimetype, "image");
                if (album_art && is_image) {
                    InvestigateUri *iu = g_slice_new (InvestigateUri);

                    iu->file = subfile;
                    g_file_info_get_modification_time (info, &iu->mtime);

                    g_ptr_array_add (album_art, iu);
                } else if (!is_image) {
                    InvestigateUri *iu = g_slice_new (InvestigateUri);

                    iu->file = subfile;
                    g_file_info_get_modification_time (info, &iu->mtime);

                    g_ptr_array_add (flist, iu);
                }

                pfe->directory = FALSE;
            } else if (type == G_FILE_TYPE_DIRECTORY) {
                g_queue_push_tail (finder->pending, subfile);
                pfe->directory = TRUE;
            }

            g_object_unref (info);

            if (entry_in_directory (current_pfd, subfile) == FALSE) {
                g_hash_table_insert (current_pfd->entries, pfe->uri, pfe);
            }
        }

        g_object_unref (fe);

        /* If we're here and album art isn't NULL, then this didn't have
           any music in it. So add the image files to the flist */
        if (album_art) {
            for (i = 0; i < album_art->len; i++) {
                g_ptr_array_add (flist, album_art->pdata[i]);
            }

            g_ptr_array_free (album_art, TRUE);
        }

        g_free (uri);
        break;

    default:
        /* not a regular file or directory */
        g_object_unref (info);
    }

    if (flist == NULL) {
        return;
    }

    /* Tell the investigator to do something with the files */
    bkl_source_investigate_files (((BklFinder *) finder)->source, flist);

    /* Unref all the new files now that they've been sent to the investigator */
    for (i = 0; i < flist->len; i++) {
        InvestigateUri *iu = flist->pdata[i];

        g_object_unref (iu->file);
        g_slice_free (InvestigateUri, iu);
    }
    g_ptr_array_free (flist, TRUE);
}

/* return TRUE if there is more work */
gboolean
bkl_path_finder_lazy_dig (BklFinder *finder)
{
    BklPathFinder *self = (BklPathFinder *) finder;
    GFile *next;

    /* Pluck an item from from the pending queue and process it
       If it is a file then we tell the inspector to look at it
       If it is a directory, then we add all the children directories
       to the pending queue, and tell the inspector about all the
       direct children files */
    next = g_queue_pop_head (self->pending);
    if (next == NULL) {
        return FALSE;
    }

    process_file (self, next);

    g_object_unref (next);
    return (g_queue_peek_head (self->pending) != NULL);
}

BklFinder *
bkl_path_finder_new (BklSource  *source,
                     const char *uri)
{
    BklPathFinder *path_finder;
    BklFinder *finder;

    path_finder = g_new0 (BklPathFinder, 1);
    finder = (BklFinder *) path_finder;

    finder->destroy = bkl_path_finder_destroy;
    finder->lazy_dig = bkl_path_finder_lazy_dig;

    finder->base_uri = g_strdup (uri);
    finder->source = source;

    path_finder->base_file = g_file_new_for_uri (uri);

    path_finder->pending = g_queue_new ();
    g_queue_push_tail (path_finder->pending,
                       g_object_ref (path_finder->base_file));

    path_finder->directories = g_hash_table_new_full (g_str_hash, g_str_equal,
                                                      g_free, free_directory);

    return finder;
}
