/*
Copyright (c) 2007, 2008 by Juliusz Chroboczek

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <net/if.h>
#include <arpa/inet.h>

#include "babeld.h"
#include "util.h"
#include "kernel.h"
#include "network.h"
#include "neighbour.h"
#include "message.h"
#include "route.h"
#include "configuration.h"

struct network *networks = NULL;

static struct network *
last_network(void)
{
    struct network *net = networks;

    if(!net)
        return NULL;

    while(net->next)
        net = net->next;

    return net;
}

struct network *
add_network(char *ifname, struct network_conf *conf)
{
    struct network *net;

    assert(!conf || strcmp(ifname, conf->ifname) == 0);

    FOR_ALL_NETS(net) {
        if(strcmp(net->ifname, ifname) == 0) {
            assert(conf == NULL);
            return net;
        }
    }

    net = malloc(sizeof(struct network));
    if(net == NULL)
        return NULL;

    memset(net, 0, sizeof(struct network));
    strncpy(net->ifname, ifname, IF_NAMESIZE);
    net->conf = conf;
    net->activity_time = now.tv_sec;
    net->bucket_time = now.tv_sec;
    net->bucket = BUCKET_TOKENS_MAX;
    net->hello_seqno = (random() & 0xFFFF);

    if(networks == NULL)
        networks = net;
    else
        last_network()->next = net;

    return net;
}

int
network_idle(struct network *net)
{
    return (idle_hello_interval > 0 &&
            net->activity_time < now.tv_sec - idle_time);
}

int
update_hello_interval(struct network *net)
{
    int rc = 0;
    unsigned short interval;

    if(network_idle(net))
        interval = idle_hello_interval;
    else if(NET_CONF(net, hello_interval) > 0)
        interval = NET_CONF(net, hello_interval);
    else if((net->flags & NET_WIRED))
        interval = wired_hello_interval;
    else
        interval = wireless_hello_interval;

    if(net->hello_interval != interval) {
        net->hello_interval = interval;
            rc = 1;
    }

    return rc;
}

/* This should be no more than half the hello interval, so that hellos
   aren't sent late.  The result is in milliseconds. */
unsigned
jitter(struct network *net, int urgent)
{
    unsigned interval = net->hello_interval;
    if(urgent)
        interval = MIN(interval, 100);
    else
        interval = MIN(interval, 4000);
    return roughly(interval) / 4;
}

unsigned
update_jitter(struct network *net, int urgent)
{
    unsigned interval = net->hello_interval;
    if(urgent)
        interval = MIN(interval, 100);
    else
        interval = MIN(interval, 4000);
    return roughly(interval);
}

void
set_timeout(struct timeval *timeout, int msecs)
{
    timeval_add_msec(timeout, &now, roughly(msecs));
}

static int
check_network_ipv4(struct network *net)
{
    unsigned char ipv4[4];
    int rc;

    if(net->ifindex > 0)
        rc = kernel_interface_ipv4(net->ifname, net->ifindex, ipv4);
    else
        rc = 0;

    if(rc > 0) {
        if(!net->ipv4 || memcmp(ipv4, net->ipv4, 4) != 0) {
            debugf("Noticed IPv4 change for %s.\n", net->ifname);
            flush_network_routes(net, 0);
            if(!net->ipv4)
                net->ipv4 = malloc(4);
            if(net->ipv4)
                memcpy(net->ipv4, ipv4, 4);
            return 1;
        }
    } else {
        if(net->ipv4) {
            debugf("Noticed IPv4 change for %s.\n", net->ifname);
            flush_network_routes(net, 0);
            free(net->ipv4);
            net->ipv4 = NULL;
            return 1;
        }
    }
    return 0;
}

int
check_network_channel(struct network *net)
{
    int channel = NET_CONF(net, channel);

    if(channel == NET_CHANNEL_UNKNOWN) {
        if((net->flags & NET_WIRED)) {
            channel = NET_CHANNEL_NONINTERFERING;
        } else {
            channel = kernel_interface_channel(net->ifname, net->ifindex);
            if(channel < 0)
                fprintf(stderr,
                        "Couldn't determine channel of interface %s: %s.\n",
                       net->ifname, strerror(errno));
            if(channel <= 0)
                channel = NET_CHANNEL_INTERFERING;
        }
    }

    if(net->channel != channel) {
        net->channel = channel;
        return 1;
    }
    return 0;
}

int
network_up(struct network *net, int up)
{
    int mtu, rc, wired;
    struct ipv6_mreq mreq;

    if((!!up) == net_up(net))
        return 0;

    if(up)
        net->flags |= NET_UP;
    else
        net->flags &= ~NET_UP;

    if(up) {
        struct kernel_route ll[32];
        if(net->ifindex <= 0) {
            fprintf(stderr,
                    "Upping unknown interface %s.\n", net->ifname);
            return network_up(net, 0);
        }

        rc = kernel_setup_interface(1, net->ifname, net->ifindex);
        if(rc < 0) {
            fprintf(stderr, "kernel_setup_interface(%s, %d) failed.\n",
                    net->ifname, net->ifindex);
            return network_up(net, 0);
        }

        mtu = kernel_interface_mtu(net->ifname, net->ifindex);
        if(mtu < 0) {
            fprintf(stderr, "Warning: couldn't get MTU of interface %s (%d).\n",
                    net->ifname, net->ifindex);
            mtu = 1280;
        }

        /* We need to be able to fit at least two messages into a packet,
           so MTUs below 116 require lower layer fragmentation. */
        /* In IPv6, the minimum MTU is 1280, and every host must be able
           to reassemble up to 1500 bytes, but I'd rather not rely on this. */
        if(mtu < 128) {
            fprintf(stderr, "Suspiciously low MTU %d on interface %s (%d).\n",
                    mtu, net->ifname, net->ifindex);
            mtu = 128;
        }

        if(net->sendbuf)
            free(net->sendbuf);

        /* 40 for IPv6 header, 8 for UDP header, 12 for good luck. */
        net->bufsize = mtu - sizeof(packet_header) - 60;
        net->sendbuf = malloc(net->bufsize);
        if(net->sendbuf == NULL) {
            fprintf(stderr, "Couldn't allocate sendbuf.\n");
            net->bufsize = 0;
            return network_up(net, 0);
        }

        resize_receive_buffer(mtu);

        if(NET_CONF(net, wired) == CONFIG_NO) {
            wired = 0;
        } else if(NET_CONF(net, wired) == CONFIG_YES) {
            wired = 1;
        } else if(all_wireless) {
            wired = 0;
        } else {
            rc = kernel_interface_wireless(net->ifname, net->ifindex);
            if(rc < 0) {
                fprintf(stderr,
                        "Warning: couldn't determine whether %s (%d) "
                        "is a wireless interface.\n",
                        net->ifname, net->ifindex);
                wired = 0;
            } else {
                wired = !rc;
            }
        }

        if(wired) {
            net->flags |= NET_WIRED;
            net->cost = NET_CONF(net, cost);
            if(net->cost <= 0) net->cost = 96;
            if(NET_CONF(net, split_horizon) == CONFIG_NO)
                net->flags &= ~NET_SPLIT_HORIZON;
            else if(NET_CONF(net, split_horizon) == CONFIG_YES)
                net->flags |= NET_SPLIT_HORIZON;
            else if(split_horizon)
                net->flags |= NET_SPLIT_HORIZON;
            else
                net->flags &= ~NET_SPLIT_HORIZON;
            if(NET_CONF(net, lq) == CONFIG_YES)
                net->flags |= NET_LQ;
            else
                net->flags &= ~NET_LQ;
        } else {
            net->flags &= ~NET_WIRED;
            net->cost = NET_CONF(net, cost);
            if(net->cost <= 0) net->cost = 256;
            if(NET_CONF(net, split_horizon) == CONFIG_YES)
                net->flags |= NET_SPLIT_HORIZON;
            else
                net->flags &= ~NET_SPLIT_HORIZON;
            if(NET_CONF(net, lq) == CONFIG_NO)
                net->flags &= ~NET_LQ;
            else
                net->flags |= NET_LQ;
        }

        if(NET_CONF(net, faraway) == CONFIG_YES)
            net->flags |= NET_FARAWAY;

        net->activity_time = now.tv_sec;
        update_hello_interval(net);
        /* Since the interface was marked as active above, the
           idle_hello_interval cannot be the one being used here. */
        net->update_interval =
            NET_CONF(net, update_interval) > 0 ?
            NET_CONF(net, update_interval) :
            net->hello_interval * 4;

        memset(&mreq, 0, sizeof(mreq));
        memcpy(&mreq.ipv6mr_multiaddr, protocol_group, 16);
        mreq.ipv6mr_interface = net->ifindex;

        rc = setsockopt(protocol_socket, IPPROTO_IPV6, IPV6_JOIN_GROUP,
                        (char*)&mreq, sizeof(mreq));
        if(rc < 0) {
            perror("setsockopt(IPV6_JOIN_GROUP)");
            /* This is probably due to a missing link-local address,
               so down this network, and wait until the main loop
               tries to up it again. */
            return network_up(net, 0);
        }

        if(net->ll)
            free(net->ll);
        net->numll = 0;
        net->ll = NULL;
        rc = kernel_addresses(net->ifname, net->ifindex, 1, ll, 32);
        if(rc < 0) {
            perror("kernel_ll_addresses");
        } else if(rc > 0) {
            net->ll = malloc(16 * rc);
            if(net->ll == NULL) {
                perror("malloc(ll)");
            } else {
                int i;
                for(i = 0; i < rc; i++)
                    memcpy(net->ll[i], ll[i].prefix, 16);
                net->numll = rc;
            }
        }
        set_timeout(&net->hello_timeout, net->hello_interval);
        set_timeout(&net->update_timeout, net->update_interval);
        send_hello(net);
        send_request(net, NULL, 0);
    } else {
        flush_network_routes(net, 0);
        net->buffered = 0;
        net->bufsize = 0;
        free(net->sendbuf);
        net->num_buffered_updates = 0;
        net->update_bufsize = 0;
        if(net->buffered_updates)
            free(net->buffered_updates);
        net->buffered_updates = NULL;
        net->sendbuf = NULL;
        if(net->ifindex > 0) {
            memset(&mreq, 0, sizeof(mreq));
            memcpy(&mreq.ipv6mr_multiaddr, protocol_group, 16);
            mreq.ipv6mr_interface = net->ifindex;
            rc = setsockopt(protocol_socket, IPPROTO_IPV6, IPV6_LEAVE_GROUP,
                            (char*)&mreq, sizeof(mreq));
            if(rc < 0)
                perror("setsockopt(IPV6_LEAVE_GROUP)");
            kernel_setup_interface(0, net->ifname, net->ifindex);
        }
        if(net->ll)
            free(net->ll);
        net->ll = NULL;
        net->numll = 0;
    }

    check_network_channel(net);
    update_network_metric(net);
    rc = check_network_ipv4(net);

    debugf("Upped network %s (%s, cost=%d, channel=%d%s).\n",
           net->ifname,
           (net->flags & NET_WIRED) ? "wired" : "wireless",
           net->cost,
           net->channel,
           net->ipv4 ? ", IPv4" : "");

    if(up && rc > 0)
        send_update(net, 0, NULL, 0);

    return 1;
}

int
network_ll_address(struct network *net, const unsigned char *address)
{
    int i;

    if(!net_up(net))
        return 0;

    for(i = 0; i < net->numll; i++)
        if(memcmp(net->ll[i], address, 16) == 0)
           return 1;

    return 0;
}

void
check_networks(void)
{
    struct network *net;
    int rc, ifindex, ifindex_changed = 0;

    FOR_ALL_NETS(net) {
        ifindex = if_nametoindex(net->ifname);
        if(ifindex != net->ifindex) {
            debugf("Noticed ifindex change for %s.\n", net->ifname);
            net->ifindex = 0;
            network_up(net, 0);
            net->ifindex = ifindex;
            ifindex_changed = 1;
        }

        if(net->ifindex > 0)
            rc = kernel_interface_operational(net->ifname, net->ifindex);
        else
            rc = 0;
        if((rc > 0) != net_up(net)) {
            debugf("Noticed status change for %s.\n", net->ifname);
            network_up(net, rc > 0);
        }

        check_network_channel(net);
        rc = check_network_ipv4(net);
        if(rc > 0) {
            send_request(net, NULL, 0);
            send_update(net, 0, NULL, 0);
         }
    }

    if(ifindex_changed)
        renumber_filters();
}
