# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::annotate;
use base 'AXP::Command::tree';

my $TERM_COLS = 80;

use Arch::Tree;

sub infoline {
	"show descriptions of ancestry revisions from logs"
}

sub optusage {
	"[options] file"
}

sub options {
	(
		$_[0]->tree_options,
		reverse  => { sh => 'r', desc => "sort from oldest to newest" },
		summary  => { sh => 's', desc => "show summary of each revision" },
		creator  => { sh => 'c', desc => "show creator of each revision" },
		date     => { sh => 'D', desc => "show date of each revision" },
		no_revs  => { sh => 'V', desc => "do not append revision descriptions at all" },
		no_desc  => { sh => 'S', desc => "do not assume the default -Dcs and auto -n" },
		no_full  => { sh => 'F', desc => "show short revision names" },
		filename => { sh => 'n', desc => "show historical file names (default: auto)" },
		no_email => { sh => 'E', desc => "don't show email of creator" },
		full_history => { sh => 'a', desc => "list revisions that didn't add current lines too" },
		one_version => { sh => 'o', desc => "don't follow tags from other versions" },
		no_progress => { sh => 'P', desc => "don't show progress even if stderr is tty" },
		no_lines => { sh => 'L', desc => "don't show file lines, just revisions" },
		linenums => { sh => 'l', type => "=s", desc => "limit to line number(s), like: 12-24,50,75-" },
		match    => { sh => 'm', type => "=s", desc => "limit to lines matching RE, like '^sub '", arg => 'RE' },
		no_unan_lines => { sh => 'U', desc => "don't show unannotated lines" },
		highlight => { sh => 'H', desc => "syntax-highlight the lines using markup" },
		group    => { sh => 'g', desc => "group lines (annotate once per group)" },
		delim    => { desc => 'specify delimiter (default: " | ")', type => "=s" },
		format   => { sh => 'f', desc => 'specify format (default: "%num%delim%line")', type => "=s" },
		show_linenums => { sh => 'N', desc => 'prepend "%linenum: " to format' },
		cvs      => { sh => 'C', desc => '-f "%0revision  (%-8*8username %10*~10date): %line" -V' },
	)
}   
 
sub helptext {
	q{
		Produce annotated output of the given tree file, similarly to
		"cvs annotate", with a thorough configurability.

		All file lines are printed to stdout, prefixed by index of
		revision that last modified the line, and the indexed
		revision descriptions follow the file content.

		The options and the format of revision descriptions are
		similar to 'axp history' command, but the default is --desc.

		The format of the annotated lines is configurable using options
		--show-linenums, --no-email, --highlight, --group, --delim D,
		--cvs and --format F.  The available format variables are:
		    %num      - padded revision number (1 is normally the latest)
		    %delim    - delimiter
		    %line     - plain or highlight'd line
		    %linenum  - padded line number starting with 1
		    %groupnum - padded group number starting with 1
		    %revision, %version, %name, %summary, %date, %age, %filename,
		    %creator, %email, %realname, %realname1, %realname2, %username
		The available width specifiers for any format variables:
		    %10var  %-10var  - left (right) pad with up to 10 spaces
		    %*20var %*-20var - cut at 20 first (last) characters
		    %-10*-20var      - use 20 last characters, padded to at least 10
		    %~var  %~-10var  - strip surrounding spaces from the value
		    %*~8var  %*8var  - just cut at 8, or cut at 5 with ellipsis "..."
	}
}

sub print_annotated_line ($$$;$) {
	my ($self, $index, $line, $is_continuation) = @_;

	my $num = $index;
	my $revision_desc;

	goto SKIP_LINE if !defined $index && $self->{skip_unannotated};

	if ($is_continuation) {
		$num = " " x (length $self->{no_num});
	} elsif (!defined $index) {
		$num = $self->{no_num};
	} else {
		$revision_desc = $self->{revision_descs}->[$index];
		$num = $revision_desc->{num};
	}
	my %values = (
		num => $num, delim => $self->{delim}, line => $line,
		linenum  => sprintf("%$self->{linenumlen}d",  $self->{linenum}),
		groupnum => sprintf("%$self->{groupnumlen}d", $self->{groupnum}),
	);

	my $annotated_line = $self->{format};
	$annotated_line =~ s/%(~?)(-?\d+)?(\*(~?)(-?)(\d+))?(?:{(\w+)}|(\w+))/
		my $name = $7 || $8;
		my $value = exists $values{$name}? $values{$name}:
			$revision_desc? defined $revision_desc->{$name}?
			$revision_desc->{$name}: "*no-$name*": "";
		(undef, $value) = split(m!(\S(?:.*\S)?)!, $value, -1) if $1;
		my ($elen, $ellipsis) = $4? (0, ""): (3, "...");
		$value = !$3 || $6 <= $elen || $6 >= length($value)? $value: $5?
			$ellipsis . substr($value, -$6 + $elen, $6 - $elen):
			substr($value, 0, $6 - $elen) . $ellipsis;
		$2? sprintf("%$2s", $value): $value;
	/ge;

	print $annotated_line, "\n";

	$self->{groupnum}++ unless $is_continuation;
	SKIP_LINE:
	$self->{linenum}++;
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	$opt{date} = $opt{creator} = $opt{summary} = 1
		unless $opt{no_desc} || $opt{date} || $opt{creator} || $opt{summary};

	my $tree = $self->tree;
	my $filepath = shift(@ARGV);
	warn "Post file-name parameters (@ARGV) are ignored\n" if @ARGV;

	my %args = (match_re => $opt{match});
	foreach (qw(one_version linenums highlight group full_history)) {
		$args{$_} = $opt{$_} if $opt{$_};
	}

	my $nr = 0;  # print revision to fetch and its number
	if (!$opt{no_progress} && -t STDERR) {
		$args{prefetch_callback} = sub ($$) {
			print STDERR "\010 \010" x $TERM_COLS if $nr;
			my $revision = substr($_[0], 0, $TERM_COLS - 11);
			printf STDERR "%4d: %s ... ", ++$nr, $revision;
		};
	}

	my ($lines, $line_rd_indexes, $revision_descs) =
		$tree->get_annotate_revision_descs($filepath, %args);

	print STDERR "\010 \010" x $TERM_COLS if $nr;
	@$revision_descs = reverse @$revision_descs if $opt{reverse};
	goto REVISION_OUTPUT if $opt{no_lines};

	my $num = 0;
	my $digits = length(0 + @$revision_descs);
	my $no_num = sprintf "%${digits}s", '?';
	my $max_rev_len = 0;

	foreach my $revision_desc (@$revision_descs) {
		$opt{filename} ||= $_->{is_filepath_renamed}
			unless $opt{no_desc} || $opt{filename};
		$revision_desc->{realname} = $revision_desc->{creator};
		$revision_desc->{realname} =~ /^(.*?)(?: (.*))?$/;
		$revision_desc->{realname1} = $1 || "_none_";
		$revision_desc->{realname2} = $2 || "_none_";
		$revision_desc->{creator} .= " <" . $revision_desc->{email} . ">"
			unless $opt{no_email};
		$revision_desc->{num} = sprintf "%${digits}d", ++$num;
		$revision_desc->{revision} = (
			$opt{no_full} ? "" : $revision_desc->{version} . "--"
		) . $revision_desc->{name};
		$max_rev_len = length($revision_desc->{revision})
			if length($revision_desc->{revision}) > $max_rev_len;
	}

	my $format = $opt{format} || "%num%delim%line";
	$format = "%linenum: $format" if $opt{show_linenums};
	my $delim = $opt{delim} || " | ";

	if ($opt{cvs}) {
		$opt{no_revs} = 1;
		$format = "%${max_rev_len}revision  (%-8*8username %10*~10date): %line";
	}

	my $linenum = 1;
	my $linenumlen = length($opt{group}? do { my $i = 0; map { $i += @$_ } @$lines; $i }: @$lines);
	my $groupnum = 1;
	my $groupnumlen = length(@$lines);
	my $skip_unannotated = $opt{no_unan_lines};

	foreach (
		qw(revision_descs format delim no_num skip_unannotated),
		qw(linenum linenumlen groupnum groupnumlen)
	) {
		$self->{$_} = eval "\$$_";
	}

	for (my $i = 0; $i < @$lines; $i++) {
		my $index = $line_rd_indexes->[$i];
		my ($line, @rest_lines) = $lines->[$i];
		($line, @rest_lines) = @$line if $opt{group};
		$index = @$revision_descs - $index - 1 if $opt{reverse} && defined $index;
		$self->print_annotated_line($index, $line);
		$self->print_annotated_line($index, $_, 1)
			foreach @rest_lines;
	}

	return if $opt{no_revs};

	print "-" x $TERM_COLS, "\n";

	REVISION_OUTPUT:
	$nr = 0;

	foreach my $revision_desc (@$revision_descs) {
		my $creator_line = "";
		$creator_line .= $revision_desc->{date} if $opt{date};
		if ($opt{creator}) {
			$creator_line .= "      " if $opt{date};
			$creator_line .= $revision_desc->{creator};
		}

		printf "%${digits}s: ", ++$nr unless $opt{no_lines};
		print "$revision_desc->{version}--" unless $opt{no_full};
		print $revision_desc->{name}, "\n";
		print "    ", $revision_desc->{filepath}, "\n" if $opt{filename};
		print "    $creator_line\n" if $creator_line;
		print "    ", $revision_desc->{summary}, "\n" if $opt{summary};
	}
}

1;
