#!/usr/bin/env python
# coding: utf-8

# Copyright © 2011 Julian Mehnle <julian@mehnle.net>,
# Copyright © 2011 Scott Kitterman <scott@kitterman.com>
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#  http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Module for parsing ``Authentication-Results`` headers as defined in RFC 5451.
"""

MODULE = 'authres'

__author__  = 'Julian Mehnle, Scott Kitterman'
__email__   = 'julian@mehnle.net'
__version__ = '0.2'

import re

# Helper functions
###############################################################################

retype = type(re.compile(''))

def isre(obj):
    return isinstance(obj, retype)

# Patterns
###############################################################################

RFC2045_TOKEN_PATTERN       = r"[A-Za-z0-9!#$%&'*+.^_`{|}~-]+"    # Printable ASCII w/o tspecials
RFC5234_WSP_PATTERN         = r'[\t ]'
RFC5234_VCHAR_PATTERN       = r'[\x21-\x7e]'                      # Printable ASCII
RFC5322_QUOTED_PAIR_PATTERN = r'\\[\t \x21-\x7e]'
RFC5322_FWS_PATTERN         = r'(?:%s*(?:\r\n|\n))?%s+' % (RFC5234_WSP_PATTERN, RFC5234_WSP_PATTERN)
RFC5322_CTEXT_PATTERN       = r'[\x21-\x27\x2a-\x5b\x5d-\x7e]'    # Printable ASCII w/o ()\
RFC5322_ATEXT_PATTERN       = r"[A-Za-z0-9!#$%&'*+/=?^_`{|}~-]"   # Printable ASCII w/o specials
RFC5322_QTEXT_PATTERN       = r'[\x21\x23-\x5b\x5d-\x7e]'         # Printable ASCII w/o "\
KTEXT_PATTERN               = r"[A-Za-z0-9!#$%&'*+?^_`{|}~-]"     # Like atext, w/o /=
PTEXT_PATTERN               = r"[A-Za-z0-9!#$%&'*+/=?^_`{|}~.@-]"

# Exceptions
###############################################################################

class AuthResError(Exception):
    "Generic exception generated by the `authres` module"

    def __init__(self, message = None):
        Exception.__init__(self, message)
        self.message = message

class SyntaxError(AuthResError):
    "Syntax error while parsing ``Authentication-Results`` header"

    def __init__(self, message = None, parse_text = None):
        AuthResError.__init__(self, message)
        if parse_text is None or len(parse_text) <= 40:
            self.parse_text = parse_text
        else:
            self.parse_text = parse_text[0:40] + '...'

    def __str__(self):
        if self.message and self.parse_text:
            return 'Syntax error: {0} at: {1}'.format(self.message, self.parse_text)
        elif self.message:
            return 'Syntax error: {0}'.format(self.message)
        elif self.parse_text:
            return 'Syntax error at: {0}'.format(self.parse_text)
        else:
            return 'Syntax error'

class UnsupportedVersionError(AuthResError):
    "Unsupported ``Authentication-Results`` header version"

    def __init__(self, message = None, version = None):
        message = message or \
            'Unsupported Authentication-Results header version: %s' % version
        AuthResError.__init__(self, message)
        self.version = version

# Main classes
###############################################################################

# AuthenticationResultProperty class
# =============================================================================

class AuthenticationResultProperty(object):
    """
    A property (``type.name=value``) of a result clause of an
    ``Authentication-Results`` header
    """

    def __init__(self, type, name, value = None):
        self.type  = type.lower()
        self.name  = name.lower()
        if value:
            self.value = value

    def __str__(self):
        return '%s.%s=%s' % (self.type, self.name, self.value)

# AuthenticationResult and related classes
# =============================================================================

class BaseAuthenticationResult(object): pass

class NoneAuthenticationResult(BaseAuthenticationResult):
    "Sole ``none`` clause of an empty ``Authentication-Results`` header"

    def __init__(self):
        pass

    def __str__(self):
        return 'none'

# Clarification of identifier naming:
# The following function acts as a factory for Python property attributes to
# be bound to a class, so it is named `authres_class_property`.  Its nested
# `getter` and `setter` functions use the identifier `authres_property` to
# refer to an instance of the `AuthenticationResultProperty` class.
def authres_class_property(type, name):
    """
    Return a property attribute to be bound to an `AuthenticationResult` class
    for accessing the `AuthenticationResultProperty` objects in its `properties`
    attribute.
    """

    def getter(self, type = type, name = name):
        authres_property = self._find_first_property(type, name)
        return authres_property and authres_property.value

    def setter(self, value, type = type, name = name):
        authres_property = self._find_first_property(type, name)
        if not authres_property:
            authres_property = AuthenticationResultProperty(type, name)
            self.properties.append(authres_property)
        authres_property.value = value

    return property(getter, setter)

class AuthenticationResult(BaseAuthenticationResult):
    "Generic result clause of an ``Authentication-Results`` header"

    def __init__(self, method, version = None, result = None, reason = None, properties = []):
        self.method     = method.lower()
        self.version    = version and version.lower()
        self.result     = result.lower()
        if not self.result:
            raise ArgumentError('Required result argument missing or None or empty')
        self.reason     = reason and re.sub(r'[^\x20-\x7e]', '?', reason)
            # Remove unprintable characters
        self.properties = properties

    def __str__(self):
        strs = []
        strs.append(self.method)
        if self.version:
            strs.append('/')
            strs.append(self.version)
        strs.append('=')
        strs.append(self.result)
        if self.reason:
            strs.append(' reason="')
            strs.append(re.sub(r'(["\\])', r'\\\1', self.reason))  # Escape "\
            strs.append('"')
        for property_ in self.properties:
            strs.append(' ')
            strs.append(str(property_))
        return ''.join(strs)

    def _find_first_property(self, type, name):
        properties = [
            property
            for property
            in self.properties
            if property.type == type and property.name == name
        ]
        return properties[0] if properties else None

class DKIMAuthenticationResult(AuthenticationResult):
    "DKIM result clause of an ``Authentication-Results`` header"

    def __init__(self, version = None, result = None, reason = None, properties = [],
        header_d = None, header_i = None
    ):
        AuthenticationResult.__init__(self, 'dkim', version, result, reason, properties)
        if header_d:
            self.header_d = header_d
        if header_i:
            self.header_i = header_i

    header_d = authres_class_property('header', 'd')
    header_i = authres_class_property('header', 'i')

class DomainKeysAuthenticationResult(AuthenticationResult):
    "DomainKeys result clause of an ``Authentication-Results`` header"

    def __init__(self, version = None, result = None, reason = None, properties = [],
        header_d = None, header_from = None, header_sender = None
    ):
        AuthenticationResult.__init__(self, 'domainkeys', version, result, reason, properties)
        if header_d:
            self.header_d = header_d
        if header_from:
            self.header_from = header_from
        if header_sender:
            self.header_sender = header_sender

    header_d      = authres_class_property('header', 'd')
    header_from   = authres_class_property('header', 'from')
    header_sender = authres_class_property('header', 'sender')

class SPFAuthenticationResult(AuthenticationResult):
    "SPF result clause of an ``Authentication-Results`` header"

    def __init__(self, version = None, result = None, reason = None, properties = [],
        smtp_helo = None, smtp_mailfrom = None
    ):
        AuthenticationResult.__init__(self, 'spf', version, result, reason, properties)
        if smtp_helo:
            self.smtp_helo = smtp_helo
        if smtp_mailfrom:
            self.smtp_mailfrom = smtp_mailfrom

    smtp_helo     = authres_class_property('smtp', 'helo')
    smtp_mailfrom = authres_class_property('smtp', 'mailfrom')

class SenderIDAuthenticationResult(AuthenticationResult):
    "Sender ID result clause of an ``Authentication-Results`` header"

    def __init__(self, version = None, result = None, reason = None, properties = [],
        header_from        = None, header_sender        = None,
        header_resent_from = None, header_resent_sender = None
    ):
        AuthenticationResult.__init__(self, 'sender-id', version, result, reason, properties)
        if header_from:
            self.header_from = header_from
        if header_sender:
            self.header_sender = header_sender
        if header_resent_from:
            self.header_resent_from = header_resent_from
        if header_resent_sender:
            self.header_resent_sender = header_resent_sender

    header_from          = authres_class_property('header', 'from')
    header_sender        = authres_class_property('header', 'sender')
    header_resent_from   = authres_class_property('header', 'resent-from')
    header_resent_sender = authres_class_property('header', 'resent-sender')

    @property
    def header_pra(self):
        return (
            self.header_resent_sender or
            self.header_resent_from   or
            self.header_sender        or
            self.header_from
        )

class IPRevAuthenticationResult(AuthenticationResult):
    "iprev result clause of an ``Authentication-Results`` header"

    def __init__(self, version = None, result = None, reason = None, properties = [],
        policy_iprev = None
    ):
        AuthenticationResult.__init__(self, 'iprev', version, result, reason, properties)
        if policy_iprev:
            self.policy_iprev = policy_iprev

    policy_iprev = authres_class_property('policy', 'iprev')

class SMTPAUTHAuthenticationResult(AuthenticationResult):
    "SMTP AUTH result clause of an ``Authentication-Results`` header"

    def __init__(self, version = None, result = None, reason = None, properties = [],
        smtp_auth = None
    ):
        AuthenticationResult.__init__(self, 'auth', version, result, reason, properties)
        if smtp_auth:
            self.smtp_auth = smtp_auth

    smtp_auth = authres_class_property('smtp', 'auth')

AUTHRES_CLASS_BY_AUTH_METHOD = {
    'dkim':         DKIMAuthenticationResult,
    'domainkeys':   DomainKeysAuthenticationResult,
    'spf':          SPFAuthenticationResult,
    'sender-id':    SenderIDAuthenticationResult,
    'iprev':        IPRevAuthenticationResult,
    'auth':         SMTPAUTHAuthenticationResult
}

def authres(method, version = None, result = None, reason = None, properties = []):
    try:
        return AUTHRES_CLASS_BY_AUTH_METHOD[method](version, result, reason, properties)
    except KeyError:
        return AuthenticationResult(method, version, result, reason, properties)

# AuthenticationResultsHeader class
# =============================================================================

class AuthenticationResultsHeader(object):
    VERSIONS = ['1']

    NONE_RESULT = NoneAuthenticationResult()

    HEADER_FIELD_NAME = 'Authentication-Results'
    HEADER_FIELD_PATTERN = re.compile(r'^Authentication-Results:\s*', re.I)

    @classmethod
    def parse(self, string):
        """
        Creates an `AuthenticationResultsHeader` object by parsing an ``Authentication-
        Results`` header (expecting the field name at the beginning).  Expects the
        header to have been unfolded and any RFC 5322 header field comments to have
        been removed.
        """
        string, n = self.HEADER_FIELD_PATTERN.subn('', string, 1)
        if n == 1:
            return self.parse_value(string)
        else:
            raise SyntaxError('parse_with_name', 'Not an "Authentication-Results" header field: {0}'.format(string))

    @classmethod
    def parse_value(self, string):
        """
        Creates an `AuthenticationResultsHeader` object by parsing an ``Authentication-
        Results`` header value.  Expects the header value to have been unfolded.
        """
        header = self()
        header._parse_text = string.rstrip('\r\n\t ')
        header._parse()
        return header

    def __init__(self, authserv_id = None, version = None, results = []):
        """

        Examples:
        RFC 5451 B.2
        >>> str(AuthenticationResultsHeader('test.example.org'))
        'Authentication-Results: test.example.org; none'

        >>> str(AuthenticationResultsHeader('test.example.org', version=1))
        'Authentication-Results: test.example.org 1; none'

        RFC 5451 B.3
        >>> str(AuthenticationResultsHeader(authserv_id = 'example.com', 
        ... results = [SPFAuthenticationResult(result = 'pass', 
        ... smtp_mailfrom = 'example.net')]))
        'Authentication-Results: example.com; spf=pass smtp.mailfrom=example.net'

        >>> arobj = AuthenticationResultsHeader.parse('Authentication-Results: example.com; spf=pass smtp.mailfrom=example.net')
        >>> str(arobj.authserv_id)
        'example.com'
        >>> str(arobj.results[0])
        'spf=pass smtp.mailfrom=example.net'
        >>> str(arobj.results[0].method)
        'spf'
        >>> str(arobj.results[0].result)
        'pass'
        >>> str(arobj.results[0].smtp_mailfrom)
        'example.net'
        >>> str(arobj.results[0].smtp_helo)
        'None'
        """
        self.authserv_id = authserv_id and authserv_id.lower()
        self.version     = version     and str(version).lower()
        if self.version and not self.version in self.VERSIONS:
            raise UnsupportedVersionError(version = self.version)
        self.results     = results

    def __str__(self):
        strs = []
        strs.append(self.HEADER_FIELD_NAME)
        strs.append(': ')
        strs.append(self.authserv_id)
        if self.version:
            strs.append(' ')
            strs.append(self.version)
        if len(self.results):
            for result in self.results:
                strs.append('; ')
                strs.append(str(result))
        else:
            strs.append('; ')
            strs.append(str(self.NONE_RESULT))
        return ''.join(strs)

    # Principal parser methods
    # =========================================================================

    def _parse(self):
        authserv_id = self._parse_authserv_id()
        if not authserv_id:
            raise SyntaxError('Expected authserv-id', self._parse_text)

        self._parse_rfc5322_cfws()

        version = self._parse_version()
        if version and not version in self.VERSIONS:
            raise UnsupportedVersionError(version = version)

        self._parse_rfc5322_cfws()

        results = []
        result = True
        while result:
            result = self._parse_resinfo()
            if result:
                results.append(result)
                if result == self.NONE_RESULT:
                    break
        if not len(results):
            raise SyntaxError('Expected "none" or at least one resinfo', self._parse_text)
        elif results == [self.NONE_RESULT]:
            results = []

        self._parse_rfc5322_cfws()
        self._parse_end()

        self.authserv_id = authserv_id.lower()
        self.version     = version and version.lower()
        self.results     = results

    def _parse_authserv_id(self):
        return self._parse_rfc5322_dot_atom()

    def _parse_version(self):
        version_match = self._parse_pattern(r'\d+')
        self._parse_rfc5322_cfws()
        return version_match and version_match.group()

    def _parse_resinfo(self):
        self._parse_rfc5322_cfws()
        if not self._parse_pattern(r';'):
            return
        self._parse_rfc5322_cfws()
        if self._parse_pattern(r'none'):
            return self.NONE_RESULT
        else:
            method, version, result = self._parse_methodspec()
            self._parse_rfc5322_cfws()
            reason = self._parse_reasonspec()
            properties = []
            property_ = True
            while property_:
                self._parse_rfc5322_cfws()
                property_ = self._parse_propspec()
                if property_:
                    properties.append(property_)
            return authres(method, version, result, reason, properties)

    def _parse_methodspec(self):
        self._parse_rfc5322_cfws()
        method, version = self._parse_method()
        self._parse_rfc5322_cfws()
        if not self._parse_pattern(r'='):
            raise SyntaxError('Expected "="', self._parse_text)
        self._parse_rfc5322_cfws()
        result = self._parse_rfc5322_dot_atom()
        if not result:
            raise SyntaxError('Expected result', self._parse_text)
        return (method, version, result)

    def _parse_method(self):
        method = self._parse_dot_key_atom()
        if not method:
            raise SyntaxError('Expected method', self._parse_text)
        self._parse_rfc5322_cfws()
        if not self._parse_pattern(r'/'):
            return (method, None)
        self._parse_rfc5322_cfws()
        version_match = self._parse_pattern(r'\d+')
        if not version_match:
            raise SyntaxError('Expected version', self._parse_text)
        return (method, version_match.group())

    def _parse_reasonspec(self):
        if self._parse_pattern(r'reason'):
            self._parse_rfc5322_cfws()
            if not self._parse_pattern(r'='):
                raise SyntaxError('Expected "="', self._parse_text)
            self._parse_rfc5322_cfws()
            reasonspec = self._parse_rfc2045_value()
            if not reasonspec:
                raise SyntaxError('Expected reason', self._parse_text)
            return reasonspec

    def _parse_propspec(self):
        ptype = self._parse_key_atom()
        if not ptype:
            return
        elif ptype.lower() not in ['smtp', 'header', 'body', 'policy']:
            raise SyntaxError('Invalid ptype; expected any of "smtp", "header", "body", "policy", got "%s"' % ptype, self._parse_text)
        self._parse_rfc5322_cfws()
        if not self._parse_pattern(r'\.'):
            raise SyntaxError('Expected "."', self._parse_text)
        self._parse_rfc5322_cfws()
        property_ = self._parse_dot_key_atom()
        self._parse_rfc5322_cfws()
        if not self._parse_pattern(r'='):
            raise SyntaxError('Expected "="', self._parse_text)
        pvalue = self._parse_pvalue()
        if not pvalue:
            raise SyntaxError('Expected pvalue', self._parse_text)
        return AuthenticationResultProperty(ptype, property_, pvalue)

    def _parse_pvalue(self):
        self._parse_rfc5322_cfws()

        # The original rule is (modulo CFWS):
        #
        #     pvalue = [ [local-part] "@" ] domain-name / value
        #     value  = token / quoted-string
        #
        # Distinguishing <token> and <domain-name> may require backtracking,
        # and in order to avoid the need for that, the following is a simpli-
        # fication of the <pvalue> rule from RFC 5451, erring on the side of
        # laxity.
        #
        # Since <local-part> is either a <quoted-string> or <dot-atom>, and
        # <value> is either a <quoted-string> or a <token>, and <dot-atom> and
        # <token> are very similar (<dot-atom> is a superset of <token> except
        # that it multiple dots may not be adjacent), we allow a union of ".",
        # "@" and <atext> characters (jointly denoted <ptext>) in the place of
        # <dot-atom> and <token>.
        #
        # We then allow four patterns:
        #
        #     pvalue = quoted-string                 /
        #              quoted-string "@" domain-name /
        #                            "@" domain-name /
        #              1*ptext

        quoted_string_match = self._parse_rfc5322_quoted_string()
        if quoted_string_match:
            if self._parse_pattern(r'@'):
                # quoted-string "@" domain-name
                domain_name = self._parse_rfc5322_dot_atom()
                self._parse_rfc5322_cfws()
                if domain_name:
                    return '%s@%s' % (quoted_string, domain_name)
            else:
                # quoted-string
                self._parse_rfc5322_cfws()
                # Look ahead to see whether pvalue terminates after quoted-string as expected:
                if re.match(r';|$', self._parse_text):
                    return quoted_string
        else:
            if self._parse_pattern(r'@'):
                # "@" domain-name
                domain_name = self._parse_rfc5322_dot_atom()
                self._parse_rfc5322_cfws()
                if domain_name:
                    return '@' + domain_name
            else:
                # 1*ptext
                pvalue_match = self._parse_pattern(r'%s+' % PTEXT_PATTERN)
                self._parse_rfc5322_cfws()
                if pvalue_match:
                    return pvalue_match.group()

    def _parse_end(self):
        if self._parse_text == '':
            return True
        else:
            raise SyntaxError('Expected end of text', self._parse_text)

    # Generic grammar parser methods
    # =========================================================================

    def _parse_pattern(self, pattern):
        match = [None]

        def matched(m):
            match[0] = m
            return ''

        # TODO: This effectively recompiles most patterns on each use, which
        #       is far from efficient.  This should be rearchitected.
        regexp = pattern if isre(pattern) else re.compile(r'^' + pattern, re.I)
        self._parse_text = regexp.sub(matched, self._parse_text, 1)
        return match[0]

    def _parse_rfc2045_value(self):
        return self._parse_rfc2045_token() or self._parse_rfc5322_quoted_string()

    def _parse_rfc2045_token(self):
        token_match = self._parse_pattern(RFC2045_TOKEN_PATTERN)
        return token_match and token_match.group()

    def _parse_rfc5322_quoted_string(self):
        self._parse_rfc5322_cfws()
        if not self._parse_pattern(r'^"'):
            return
        all_qcontent = ''
        qcontent = True
        while qcontent:
            fws_match = self._parse_pattern(RFC5322_FWS_PATTERN)
            if fws_match:
                all_qcontent += fws_match.group()
            qcontent = self._parse_rfc5322_qcontent()
            if qcontent:
                all_qcontent += qcontent
        self._parse_pattern(RFC5322_FWS_PATTERN)
        if not self._parse_pattern(r'"'):
            raise SyntaxError('Expected <">', self._parse_text)
        self._parse_rfc5322_cfws()
        return all_qcontent

    def _parse_rfc5322_qcontent(self):
        qtext_match = self._parse_pattern(r'%s+' % RFC5322_QTEXT_PATTERN)
        if qtext_match:
            return qtext_match.group()
        quoted_pair_match = self._parse_pattern(RFC5322_QUOTED_PAIR_PATTERN)
        if quoted_pair_match:
            return quoted_pair_match.group()

    def _parse_rfc5322_dot_atom(self):
        self._parse_rfc5322_cfws()
        dot_atom_text_match = self._parse_pattern(r'%s+(?:\.%s+)*' %
            (RFC5322_ATEXT_PATTERN, RFC5322_ATEXT_PATTERN))
        self._parse_rfc5322_cfws()
        return dot_atom_text_match and dot_atom_text_match.group()

    def _parse_dot_key_atom(self):
        # Like _parse_rfc5322_dot_atom, but disallows "/" (forward slash) and
        # "=" (equal sign).
        self._parse_rfc5322_cfws()
        dot_atom_text_match = self._parse_pattern(r'%s+(?:\.%s+)*' %
            (KTEXT_PATTERN, KTEXT_PATTERN))
        self._parse_rfc5322_cfws()
        return dot_atom_text_match and dot_atom_text_match.group()

    def _parse_key_atom(self):
        # Like _parse_dot_key_atom, but also disallows "." (dot).
        self._parse_rfc5322_cfws()
        dot_atom_text_match = self._parse_pattern(r'%s+' % KTEXT_PATTERN)
        self._parse_rfc5322_cfws()
        return dot_atom_text_match and dot_atom_text_match.group()

    def _parse_rfc5322_cfws(self):
        fws_match     = False
        comment_match = True
        while comment_match:
            fws_match     = fws_match or self._parse_pattern(RFC5322_FWS_PATTERN)
            comment_match = self._parse_rfc5322_comment()
        fws_match = fws_match or self._parse_pattern(RFC5322_FWS_PATTERN)
        return fws_match or comment_match

    def _parse_rfc5322_comment(self):
        if self._parse_pattern(r'\('):
            while self._parse_pattern(RFC5322_FWS_PATTERN) or self._parse_rfc5322_ccontent(): pass
            if self._parse_pattern(r'^\)'):
                return True
            else:
                raise SyntaxError('comment: expected FWS or ccontent or ")"', self._parse_text)

    def _parse_rfc5322_ccontent(self):
        if self._parse_pattern(r'%s+' % RFC5322_CTEXT_PATTERN):
            return True
        elif self._parse_pattern(RFC5322_QUOTED_PAIR_PATTERN):
            return True
        elif self._parse_rfc5322_comment():
            return True

def _test():
    import doctest
    import authres
    return doctest.testmod(authres)

if __name__ == '__main__':
    _test()

# vim:sw=4 sts=4
