#!/usr/bin/python
# -*- coding: ISO-8859-1
#   Copyright © 2011 Scott Kitterman <scott@kitterman.com>

#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at

#    http://www.apache.org/licenses/LICENSE-2.0

#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
"""
    Module for RFC 5451 Authentication Results Header processing.
"""
__author__ = "Scott Kitterman"
__email__ = "scott@kitterman.com"
__version__ = "0.1: March 14, 2011"
MODULE = 'authres'

from email.header import Header

__metaclass__ = type
__all__ = ['build', 'MakeHeader', 'UnknownVersionError',
    ]
ptypes = ['smtp', 'header', 'body', 'policy']


class UnknownVersionError(Exception):
    "Error due to trying to use functionality that's not implemented"
    def __init__(self, function, message=None):
        Exception.__init__(self, function, message)
        self.function = function
        self.message = message
    def __str__(self):
        if self.message:
            return 'Not implemented {0}: {1}'.format(self.function, self.message)
        return 'Not implemented {0}'.format(self.function)


def build(authservid, services=None, version=None):
    """
    Build an authentication-results header using the specified authservid and
    list of authentication services utilized.

    Examples:
    >>> build('test.example.org')
    ['Authentication-Results: test.example.org; none']

    >>> build('test.example.org', version=1)
    ['Authentication-Results: test.example.org 1; none']

    >>> try: build('test.example.org', version=2)
    ... except UnknownVersionError as x: print(x)
    Not implemented version: Reported version was 2: Only version 1 supported

    >>> try: build('test.example.org', version='fred')
    ... except ValueError as x: print(x)
    Invalid version: fred

    >>> build('example.com', [{'method': 'spf', 'result': 'pass', \
            'ptype': 'smtp', 'prop': 'mailfrom', 'pvalue': 'example.net'}])
    ['Authentication-Results: example.com; spf=pass smtp.mailfrom=example.net']

    >>> build('example.com', [{'method': 'auth', 'result': 'pass', \
            'reasonspec': 'cram-md5', 'ptype': 'smtp', 'prop': 'auth', \
            'pvalue': 'sender@example.com'}, {'method': 'spf', \
            'result': 'pass', 'ptype': 'smtp', 'prop': 'mailfrom', \
            'pvalue': 'example.com'}]) #doctest: +NORMALIZE_WHITESPACE
    ['Authentication-Results: example.com;\\n auth=pass (cram-md5) \
        smtp.auth=sender@example.com;\\n spf=pass smtp.mailfrom=example.com']

    >>> build('example.com', [{'method': 'auth', 'result': 'pass', \
            'reasonspec': 'cram-md5', 'ptype': 'smtp', 'prop': 'auth', \
            'pvalue': 'sender@example.com'}, {'method': 'spf', \
            'result': 'pass', 'ptype': 'smtp', 'prop': 'mailfrom', 'pvalue': \
            'example.com'}, {'method': 'sender-id', 'result': 'pass', \
            'ptype': 'header', 'prop': 'from', 'pvalue': 'example.com'}]) \
            #doctest: +NORMALIZE_WHITESPACE
    ['Authentication-Results: example.com;\\n auth=pass (cram-md5) \
        smtp.auth=sender@example.com;\\n spf=pass smtp.mailfrom=example.com', \
        'Authentication-Results: example.com; sender-id=pass \
        header.from=example.com']

    More RFC examples:

    >>> build('example.com', [{'method': 'sender-id', 'result': 'hardfail', \
        'ptype': 'header', 'prop': 'from', 'pvalue': 'example.com'}, \
        {'method': 'dkim', 'result': 'pass', 'reasonspec': 'good signature', \
        'ptype': 'header', 'prop': 'i', 'pvalue': 'sender@example.com'}]) \
        #doctest: +NORMALIZE_WHITESPACE
    ['Authentication-Results: example.com;\\n sender-id=hardfail \
    header.from=example.com;\\n dkim=pass (good signature) \
    header.i=sender@example.com']

    >>> build('example.com', [{'method': 'auth', 'result': 'pass', \
        'reasonspec': 'cram-md5', 'ptype': 'smtp', 'prop': 'auth', \
        'pvalue': 'sender@example.com'}, {'method': 'spf', 'result': 'hardfail', \
        'ptype': 'smtp', 'prop': 'mailfrom', 'pvalue': 'example.com'}]) \
        #doctest: +NORMALIZE_WHITESPACE
    ['Authentication-Results: example.com;\\n auth=pass (cram-md5) \
        smtp.auth=sender@example.com;\\n spf=hardfail smtp.mailfrom=example.com'] 

    >>> build('example.com', [{'method': 'dkim', 'result': 'pass', \
        'reasonspec': 'good signature', 'ptype': 'header', 'prop': 'i', \
        'pvalue': '@mail-router.example.net'}, {'method': 'dkim', 'result': 'fail', \
        'reasonspec': 'bad signature', 'ptype': 'header', 'prop': 'i', \
        'pvalue': '@newyork.example.com'}])  #doctest: +NORMALIZE_WHITESPACE
    ['Authentication-Results: example.com;\\n dkim=pass (good signature) \
        header.i=@mail-router.example.net;\\n dkim=fail (bad signature) \
        header.i=@newyork.example.com']

    >>> build('example.net', [{'method': 'dkim', 'result': 'pass', \
        'reasonspec': 'good signature', 'ptype': 'header', 'prop': 'i', \
        'pvalue': '@newyork.example.com'}]) #doctest: +NORMALIZE_WHITESPACE
    ['Authentication-Results: example.net;\\n dkim=pass (good signature) \
        header.i=@newyork.example.com']  

    >>> try: build('example.net', [{'method': 'dkim', 'result': 'pass', \
        'reasonspec': 'good signature©', 'ptype': 'header', 'prop': 'i', \
        'pvalue': '@newyork.example.com'}]) #doctest: +NORMALIZE_WHITESPACE
    ... except UnicodeDecodeError as x: print(x)
    'ascii' codec can't decode byte 0xc2 in position 62: ordinal not in range(128)
    """
    if services is None:
        services=[{},]
    header = MakeHeader(authservid, services, version).construct()
    return header


class MakeHeader:
    """
    A MakeHeader object takes authentication inputs and returns an RFC 5451
    Authentication Results header string.

    authservid: Unique identifier for auth service (RFC 5451 para 2.3)
    version: Optional and assumed to be 1 if not provided
    services: [{
    method: Authentication method used,
    result: Type of result for an auth method, para 2.4,
    reasonspec: Comment associated with the result,
    propspec: Which properties of the message were evaluated,
    ptype: "smtp" / "header" / "body" / "policy",
    prop: SMTP stage from which smtp ptype result was gotten,
    pvalue: Identity information associate with the ptype
    },]

    Except as noted, these are defined in RFC 5451 para 2.2
    """
    def __init__(self, authservid, services=None, version=None):
        self.authservid = authservid
        if services is None:
            services=[{},]
        self.services = services
        self.version = version

    def _validate_version(self):
        if type(self.version) is int:
            if self.version > 1 and self.version < 10:
                errstring = \
                    'Reported version was {0}: Only version 1 supported'.format(self.version)
                raise UnknownVersionError('version', errstring)
        if self.version != 1:
            errstring = 'Invalid version: {0}'.format(str(self.version))
            raise ValueError(errstring)

    def _validate_ptypes(self, ptypes_used):
        for ptype in ptypes_used:
            if ptype not in ptypes:
                errstring = \
                    'Unknown ptype: {0}, "smtp" / "header" / "body" / "policy" are known.'.format(ptype)
                raise ValueError(errstring)

    def _parse_services(self):
        resinfolist = []
        ptypes_used = set()
        for service in self.services:
            ptypes_used.add(service['ptype'])
        self._validate_ptypes(ptypes_used)
        # use ptypes here to get a consistent ordering so tests reliably pass
        for ptype in ptypes:
            resinfos = ''
            if ptype in ptypes_used:
                for service in self.services:
                    if service['ptype'] == ptype:
                        methodspec = '{0}={1}'.format(service['method'],\
                            service['result'])
                        if 'reasonspec' in service:
                            reasonspec = '({0}) '.format(service['reasonspec']) 
                        else:
                            reasonspec = ''
                        propspec = '{0}.{1}={2}'.format(service['ptype'], \
                            service['prop'], service['pvalue'])
                        resinfo = '{0} {1}{2}'.format(methodspec, reasonspec, \
                            propspec)
                        resinfos += '; {0}'.format(resinfo)
                resinfolist.append(resinfos)
        return(resinfolist)

    def construct(self):
        headers = []
        if not self.authservid:
            # This is the only validation test for authservid.
            raise ValueError('Required authservid missing, see RFC 5451, para 2.3.')
        if 'method' not in self.services[0]:
            resinfos = ['; none']
        else:
            resinfos = self._parse_services()
        if not self.version:
            version = ''
        else:
            self._validate_version()
            version = ' {0}'.format(str(self.version))
        for resinfo in resinfos:
            headerin = 'Authentication-Results: {0}{1}{2}'.format(self.authservid, \
                version, resinfo)
            header = Header(headerin, charset='us-ascii')
            headers.append(str(header))
        return headers


def _test():
    import doctest
    import authres
    return doctest.testmod(authres)


if __name__ == '__main__':
    _test()
