/** \file compose.hh
 * String composition library.
 * Defines compose::compose(fmt, arg...) for easy, i18n-friendly
 * composition of strings.
 * 
 * Basic usage is like
 *
 * \code compose::compose("This is a %1x%2 matrix.", rows, cols); \endcode
 *
 * \sa See http://www.cs.aau.dk/~olau/compose/ or the included README for more
 * details. Note that these refer to the official version, which uses different
 * naming than this modified version.
 *
 * Modified from compose version 1.0.4.
 */
/* This file is part of libmisc, an assortment of code for reuse.
 *
 * Copyright (c) 2006-2007 Kevin Daughtridge <kevin@kdau.com>
 * Copyright (c) 2002-2004 Ole Laursen <olau@hardworking.dk>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef LIBMISC__COMPOSE_HH
#define LIBMISC__COMPOSE_HH

#include <sstream>
#include <string>
#include <list>
#include <map>

/******************************************************************************/
/** String composition library.
 * A series of functions which accept a format string on the form "text %1
 * more %2 less %3" and a number of templated parameters and spits out the
 * composited string. */
namespace compose {
/******************************************************************************/

using std::string;

/******************************************************************************/
/** The actual composition class.
 * Using compose::compose is cleaner, so we hide it here. */
class Composition {
/******************************************************************************/

public:

	/** Initialise and prepare format string on the form "foo %1 bar %2 etc". */
	explicit Composition(std::string fmt);

	/** Supply a replacement argument starting from %1. */
	template<typename T>
	Composition& arg(const T &obj)
	{
		os << obj;
		do_arg(os.str());
		return *this;
	}

	/** Compose and return string. */
	string str() const;

	/** Set format flags for future arguments. */
	Composition& setf(std::ios_base::fmtflags set);

	/** Unset format flags for future arguments. */
	Composition& unsetf(std::ios_base::fmtflags unset);
	
	/** Set precision for future arguments. */
	Composition& precision(std::streamsize prec);

private:

	void do_arg(const string &rep);

	std::ostringstream os;
	int arg_no;

	/**
	 * We store the output as a list - when the output string is requested, the
	 * list is concatenated to a string; this way we can keep iterators into
	 * the list instead of into a string where they're possibly invalidated
	 * when inserting a specification string. */
	typedef std::list<std::string> output_list;
	output_list output;

	/**
	 * The initial parse of the format string fills in the specification map
	 * with positions for each of the various %?s. */
	typedef std::multimap<int, output_list::iterator> specification_map;
	specification_map specs;

}; /* class Composition */


/******************************************************************************/
/** \name Composition Functions
 * @{ */
/******************************************************************************/

template<typename  T1>
inline string
compose(const string &fmt,
        const  T1  &o1)
{
	Composition c(fmt);
	c.arg(o1);
	return c.str();
}

template<typename  T1, typename  T2>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2)
{
	Composition c(fmt);
	c.arg(o1).arg(o2);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9, typename T10>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9, const T10 &o10)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9)
		.arg(o10);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9, typename T10,
         typename T11>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9, const T10 &o10, const T11 &o11)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9)
		.arg(o10).arg(o11);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9, typename T10,
         typename T11, typename T12>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9, const T10 &o10, const T11 &o11, const T12 &o12)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9)
		.arg(o10).arg(o11).arg(o12);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9, typename T10,
         typename T11, typename T12, typename T13>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9, const T10 &o10, const T11 &o11, const T12 &o12,
        const T13 &o13)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9)
		.arg(o10).arg(o11).arg(o12).arg(o13);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9, typename T10,
         typename T11, typename T12, typename T13, typename T14>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9, const T10 &o10, const T11 &o11, const T12 &o12,
        const T13 &o13, const T14 &o14)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9)
		.arg(o10).arg(o11).arg(o12).arg(o13).arg(o14);
	return c.str();
}

template<typename  T1, typename  T2, typename  T3, typename  T4, typename  T5,
         typename  T6, typename  T7, typename  T8, typename  T9, typename T10,
         typename T11, typename T12, typename T13, typename T14, typename T15>
inline string
compose(const string &fmt,
        const  T1  &o1, const  T2  &o2, const  T3  &o3, const  T4  &o4,
        const  T5  &o5, const  T6  &o6, const  T7  &o7, const  T8  &o8,
        const  T9  &o9, const T10 &o10, const T11 &o11, const T12 &o12,
        const T13 &o13, const T14 &o14, const T15 &o15)
{
	Composition c(fmt);
	c.arg(o1).arg(o2).arg(o3).arg(o4).arg(o5).arg(o6).arg(o7).arg(o8).arg(o9)
		.arg(o10).arg(o11).arg(o12).arg(o13).arg(o14).arg(o15);
	return c.str();
}

/** @} */

} /* namespace compose */

#endif /* LIBMISC__COMPOSE_HH */
