// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __gnu_c_tree_h__
#define __gnu_c_tree_h__

/** \file
 *  GNU C/C++ specific syntax tree classes. */

namespace Puma {

// Syntax tree node hierarchy:
class   CT_GnuAsmSpec;       // derived from CTree
class   CT_GnuAsmDef;        // derived from CT_AsmDef
class   CT_GnuAsmOperand;    // derived from CTree
class     CT_GnuAsmOperands; // derived from CT_List
class     CT_GnuAsmClobbers; // derived from CT_List
class   CT_GnuStatementExpr; // derived from CT_Expression
class   CT_GnuTypeof;        // derived from CT_DeclSpec

} // namespace Puma

#include "Puma/CTree.h"

namespace Puma {


/** \class CT_GnuAsmSpec GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing an extended inline assembly specifier. 
 *  Example: \code asm("r0") \endcode */
class CT_GnuAsmSpec : public CTree {
  CTree *_key_asm;    // CT_Token
  CTree *_open;   // CT_Token
  CTree *_expr;
  CTree *_close;  // CT_Token

public:
  /** Constructor.
   *  \param a The keyword 'asm'.
   *  \param o Left parenthesis before the assembly.
   *  \param e The assembly instructions.
   *  \param c Right parenthesis behind the assembly. */
  CT_GnuAsmSpec (CTree *a, CTree *o, CTree *e, CTree *c) :
    _key_asm (a), _open (o), _expr (e), _close (c) {}
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); } 
  /** Get the number of sons. */
  int Sons () const { return 4; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const {
    switch (n) {
      case 0: return _key_asm;
      case 1: return _open;
      case 2: return _expr;
      case 3: return _close;
      default: return (CTree*)0;
    }
  }
  /** Get the assembly instructions. */
  CT_Expression *Expr () const { return (CT_Expression*)_expr; }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) {
    if (old_son == _expr) _expr = new_son;
    else if (old_son == _key_asm) _key_asm = new_son;
    else if (old_son == _open) _open = new_son;
    else if (old_son == _close) _close = new_son;
  }
};

/** \class CT_GnuAsmDef GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing an extended inline assembly definition. 
 *  Example: \code asm("fsinx %1,%0" : "=f" (result) : "f" (angle)); \endcode */
class CT_GnuAsmDef : public CT_AsmDef {
  CTree *_cvqual;
  CTree *_operands0;
  CTree *_operands1;
  CTree *_clobbers;

public:
  /** Constructor.
   *  \param a The keyword 'asm'.
   *  \param cv Optional const/volatile qualifier sequence.
   *  \param o Left parenthesis before the assembly.
   *  \param s The assembly instructions.
   *  \param op0 First operand.
   *  \param op1 Second operand.
   *  \param cl Clobbers.
   *  \param c Right parenthesis behind the assembly.
   *  \param sc The trailing semi-colon. */
  CT_GnuAsmDef (CTree *a, CTree *cv, CTree *o, CTree *s, 
                CTree *op0, CTree *op1, CTree *cl, CTree *c, CTree *sc) :
    CT_AsmDef (a, o, s, c, sc) {
      _cvqual = cv; _operands0 = op0; _operands1 = op1; _clobbers = cl;
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }  
  /** Get the number of sons. */
  virtual int Sons () const {
    return 5 +
           (_cvqual ? 1 : 0) +
           (_operands0 ? 1 : 0) +
           (_operands1 ? 1 : 0) +
           (_clobbers ? 1 : 0);
  }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  virtual CTree *Son (int n) const {
    int have_cv = _cvqual ? 1 : 0;
    int gnu_sons = Sons () - have_cv - 5;
    int gnu_first = 3 + have_cv;
    if (n == 0) return CT_AsmDef::Son (0);
    else if (n == 1 && _cvqual) return _cvqual;
    else if ((n == 1 && !_cvqual) || (n == 2 && _cvqual))
       return CT_AsmDef::Son (1);
    else if ((n == 2 && !_cvqual) || (n == 3 && _cvqual))
       return CT_AsmDef::Son (2);
    else if (n == gnu_first && gnu_sons >= 1) return _operands0;
    else if (n == gnu_first + 1 && gnu_sons >= 2) return _operands1;
    else if (n == gnu_first + 2 && gnu_sons >= 3) return _clobbers;
    else if (n == Sons () - 2)
       return CT_AsmDef::Son (3);
    else if (n == Sons () - 1)
       return CT_AsmDef::Son (4);
    else return (CTree*)0;
  }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  virtual void ReplaceSon (CTree *old_son, CTree *new_son) {
    if (old_son == _cvqual) _cvqual = new_son;
    else if (old_son == _operands0) _operands0 = new_son;
    else if (old_son == _operands1) _operands1 = new_son;
    else if (old_son == _clobbers) _clobbers = new_son;
    else CT_AsmDef::ReplaceSon (old_son, new_son);
  }
};

/** \class CT_GnuAsmOperand GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing an extended inline assembly operand. 
 *  Example: \code "=f" (result) \endcode */
class CT_GnuAsmOperand : public CTree {
  CTree *_string;
  CTree *_open;   // CT_Token
  CTree *_expr;
  CTree *_close;  // CT_Token

public:
  /** Constructor.
   *  \param s The operand constraint string.
   *  \param o Left parenthesis before the operand.
   *  \param e The operand.
   *  \param c Right parenthesis behind the operand. */
  CT_GnuAsmOperand (CTree *s, CTree *o, CTree *e, CTree *c) :
    _string (s), _open (o), _expr (e), _close (c) {}
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); } 
  /** Get the number of sons. */
  int Sons () const { return 4; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const {
    switch (n) {
      case 0: return _string;
      case 1: return _open;
      case 2: return _expr;
      case 3: return _close;
      default: return (CTree*)0;
    }
  }
  /** Get the operand expression. */
  CT_Expression *Expr () const { return (CT_Expression*)_expr; }
  /** Get the operand constraint string. */
  CT_String *String () const { return (CT_String*)_string; }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) {
    if (old_son == _expr) _expr = new_son;
    else if (old_son == _string) _string = new_son;
    else if (old_son == _open) _open = new_son;
    else if (old_son == _close) _close = new_son;
  }
};

/** \class CT_GnuAsmOperands GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing a list of extended inline assembly operands. 
 *  Example: \code : "=f" (result) : "f" (angle) \endcode */
class CT_GnuAsmOperands : public CT_List {
public:
  /** Constructor. */
  CT_GnuAsmOperands () { AddProperties (OPEN | SEPARATORS); }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }  
};

/** \class CT_GnuAsmClobbers GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing a list of extended inline assembly clobbers. 
 *  Example: \code : "r1", "r2", "r3", "r4", "r5" \endcode */
class CT_GnuAsmClobbers : public CT_List {
public:
  /** Constructor. */
  CT_GnuAsmClobbers () { AddProperties (OPEN | SEPARATORS); }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }  
};

/*****************************************************************************/
/*                                                                           */
/*                              Expressions                                  */
/*                                                                           */
/*****************************************************************************/

/** \class CT_GnuStatementExpr GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing a statement expression. 
 *  Example: \code ({ int i = 0; i++; }) \endcode */
class CT_GnuStatementExpr : public CT_Expression {
  CTree *_open;
  CTree *_stmt;
  CTree *_close;
  
public:
  /** Constructor.
   *  \param o Left parenthesis before the statement.
   *  \param s The statement. 
   *  \param c Right parenthesis behind the statement. */
  CT_GnuStatementExpr (CTree *o, CTree *s, CTree *c) :
    _open (o), _stmt (s), _close (c) {}
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }  
  /** Get the number of sons. */
  int Sons () const { return 3; }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const {
    switch (n) {
      case 0: return _open;
      case 1: return _stmt;
      case 2: return _close;
      default: return (CTree*)0;
    }
  }
  /** Get the statement. */
  CT_CmpdStmt *CmpdStmt () const { return (CT_CmpdStmt*)Son (1); }
};

/*****************************************************************************/
/*                                                                           */
/*                         Declaration specifiers                            */
/*                                                                           */
/*****************************************************************************/

/** \class CT_GnuTypeof GnuCTree.h Puma/GnuCTree.h
 *  Tree node representing a typeof expression. 
 *  Example: \code typeof(a+b) \endcode */
class CT_GnuTypeof : public CT_DeclSpec, public CSemValue {
  CTree *sons[5]; // key, open, type, close, expr

public:
  /** Constructor.
   *  \param k The keyword 'typeof'.
   *  \param o Left parenthesis before the type.
   *  \param t The type from which to get the type string.
   *  \param c Right parenthesis behind the type. */
  CT_GnuTypeof (CTree *k, CTree *o, CTree *t, CTree *c) {
    sons[0] = k; sons[1] = o; sons[2] = t; sons[3] = c; sons[4] = 0;
  }
  /** Constructor.
   *  \param k The keyword 'typeof'.
   *  \param e The expression from which to get the type string. */
  CT_GnuTypeof (CTree *k, CTree *e) {
    sons[0] = k; sons[1] = 0; sons[2] = 0; sons[3] = 0; sons[4] = e;
  }
  /** Get the identifier for this node type. Can be compared with NodeName(). */
  static const char *NodeId ();
  /** Get the name of the node. Can be compared with NodeId(). */
  const char *NodeName () const { return NodeId (); }
  /** Get the number of sons. */
  int Sons () const { return CTree::Sons (sons, 5); }
  /** Get the n-th son.
   *  \param n The index of the son.
   *  \return The n-th son or NULL. */
  CTree *Son (int n) const { return CTree::Son (sons, 5, n); }
  /** Replace a son.
   *  \param old_son The son to replace.
   *  \param new_son The new son. */
  void ReplaceSon (CTree *old_son, CTree *new_son) { 
    CTree::ReplaceSon (sons, 5, old_son, new_son);
  }
  /** Get the expression from which to get the type string. */
  CTree *Expr () const { return sons[4]; }
  /** Get the type from which to get the type string. */
  CT_NamedType *TypeName () const { return (CT_NamedType*)sons[2]; }
  /** Get the semantic value information. */
  CSemValue *SemValue () const { return (CSemValue*)this; }
  /** Get the resulting type. */
  CTypeInfo *Type () const { return type; }
};

} // namespace Puma

#endif /* __gnu_c_tree_h__ */
