------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--              A S I S _ U L . G L O B A L _ S T A T E . C G               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                    Copyright (C) 2007-2008, AdaCore                      --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not,  write  to the  Free Software Foundation,  51 Franklin  --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

with Asis;                           use Asis;
with Asis.Declarations;              use Asis.Declarations;
with Asis.Definitions;               use Asis.Definitions;
with Asis.Elements;                  use Asis.Elements;
with Asis.Expressions;               use Asis.Expressions;
with Asis.Extensions;
with Asis.Extensions.Flat_Kinds;     use Asis.Extensions.Flat_Kinds;
with Asis.Iterator;                  use Asis.Iterator;

with Asis.Set_Get;                   use Asis.Set_Get;

with ASIS_UL.Common;
with ASIS_UL.Options;                use ASIS_UL.Options;
with ASIS_UL.Output;                 use ASIS_UL.Output;
with ASIS_UL.Utilities;              use ASIS_UL.Utilities;

with ASIS_UL.Global_State.Utilities; use ASIS_UL.Global_State.Utilities;

package body ASIS_UL.Global_State.CG is

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Process_Call
     (Element : Asis.Element;
      At_SLOC : String_Loc := Nil_String_Loc);
   --  Analyzes a subprogram call. If the call cannot be statically analyzed,
   --  generates the corresponding diagnostic message in case if ??? is ON.
   --  IF At_SLOC is equal to Nil_String_Loc, the SLOC of the call is the SLOC
   --  of the argument Element, otherwise At_SLOC is used as the SLOC of the
   --  call (see the documentation for Add_CG_Info).

   procedure Process_Callable_Entity (El : Asis.Element);
   --  Stores (if needed) in the call graph the information about the
   --  callable entity. In case of a single task declaration, this procedure
   --  also stores the call link from the current scope to the task entity

   procedure Process_Elaboration_Calls (Element : Asis.Element);
   --  For the argument Element that should be
   --  ASIS_UL.Utilities.May_Contain_Elaboration_Calls, tries to find implicit
   --  calls that are made during the elaboration and for each of these calls
   --  processes this call as a regular call.

   procedure Process_Type_Default_Expressions
     (Type_To_Analyze : Asis.Element;
      Call_At_SLOC    : String_Loc);
   --  Implements a part of the functionality of Process_Elaboration_Calls
   --  Recursively traverses the type structure of the type represented by
   --  Type_To_Analyze argument (note that this type should not be private or
   --  derived type!) and adds all the fucntion calls from the component
   --  expressions in the call graph. At_SLOC parameter represents the location
   --  of the calls to store (because these calls are issued as a part of
   --  object declaration elaboration declaration or allocator evaluation).

   procedure Process_Init_Expressions_In_Record_Components
     (Component_List : Asis.Element_List;
      Call_At_SLOC   : String_Loc);
   --  Implements a part of the functionality of
   --  Process_Type_Default_Expressions. Traverses the argument list and do
   --  the following:
   --
   --  - if a list element is a component definition and it contains an
   --    initialization expression, traverses this expression to locate
   --    function calls;
   --
   --  - if a list element is a component definition and it does not contain an
   --    initialization expression, analyzes the component type to get
   --    initialization expressions for suncomponents and to extract function
   --    calls from them;
   --
   --  - if a list elemen is a variant part, recursively gets into the variant
   --    part strcture to get and to analyze the variant components;
   --
   --  For errey compoenets, the component type is analyzed for possible
   --  default initialization expressions.

   procedure Process_Renaming_As_Body (El : Asis.Element);
   --  If we have renaming-as-body, this means that we have the corresponding
   --  subprigram deckaration, so - the corresponding node in the call graph.
   --  This subprogram detects (and creates, if needed) the corresponding node
   --  in the call graph and sets for this node Is_Renaming ON. Then in tries
   --  to unwind the renaming, and if the renamed entity can be statically
   --  defined, stores the ID of this entity in the Calls_Chain for the node.
   --  (That is, if we have a subprogram that has renaming-as-body as its
   --  completion, we represent this in the call graph as if this subprogram
   --  calls the renamed subprogram. The case of renaming a task entry as a
   --  subprogram is not implemented yet.)

   procedure Process_Task_Components
     (Type_Decl    : Asis.Element;
      Call_At_SLOC : String_Loc);
   --  Analyze the argument type declaration and defines the tasks that are
   --  created when creating the value of this type. It is supposed that
   --  Get_Type_Structure function has already been applied to the argument
   --  type declaration. The actual for Call_At_SLOC should indicate the source
   --  location of the construct that initiate task creations (e.g. SLOC of an
   --  object declaration that contains task components).

   procedure Process_Record_Task_Components
     (Component_List : Asis.Element_List;
      Call_At_SLOC   : String_Loc);
   --  Similar to the Process_Task_Components procedure, but works on a list of
   --  record components (more exactly, on the list returned by the
   --  Asis.Definitions.Record_Components query.

   procedure Process_Task_Creation (El : Asis.Element);
   --  Supposing that Can_Create_Tasks (El), recursively traverse the type
   --  declaration of the object or value representing by El and stores the
   --  information about all the tasks (if any) that are created when the
   --  object/value is cretaed.
   --  (Suppose we have:
   --
   --     task type T is ,.. end T;
   --     type Rec is record
   --         Comp_I : Integer;
   --         Comp_T : T;
   --     end record;
   --
   --     Var : Rec; --  here a task of type T is created,
   --
   --  This procedure should get from the declaration of Var the information
   --  that a task of the type T is created as a result of elaboration this
   --  declaration.

   procedure Process_Scope (El : Asis.Element);
   --  Stores in the call graph the information about the scope (that is -
   --  about the body of a callable entity) and updates Current_Scope and
   --  the scope stack.

   procedure Store_Arc (Called_Entity : Asis.Element; At_SLOC : String_Loc);
   --  Supposing that Called_Entity is an Element that can be stored as a node
   --  of the Call Graph (that is, Corresponding_Element has already been
   --  applied to it), stores the call arc from the current scope to the node
   --  corresponding to this element using At_SLOC as the SLOC of the place
   --  where the call takes place. Only one (the first) call from the scope to
   --  the given Element is stored.

   procedure Traverse_Renamings;
   --  This procedure goes trough all the Call Graph renamings nodes and
   --  decides for which renaming nodes we can say that the corresponding
   --  body is processed.

   procedure Check_Call_Graph_Completeness;
   --  Checks if the call information stored in the global data structure is
   --  complete and allows to construct the full Call Graph. Generates a
   --  diagnostic message each time when any incompleteness is detected.

   procedure Set_Is_Renaming (N : GS_Node_Id; Val : Boolean := True);
   --  Set the flag indicating if the callable entity is a renaming of another
   --  callable entity (only renamings-as-bodies are represented in the call
   --  graph),

   function First_Direct_Call (N : GS_Node_Id) return GS_Node_Id;
   --  Returns the first node from the direct call list of N. Returns
   --  No_GS_Node if the list of direcr calls for N is empty

   procedure Traverse_Construct_For_CG_Info is new Traverse_Element
     (State_Information => String_Loc,
      Pre_Operation     => Add_CG_Info_Pre_Op,
      Post_Operation    => Complete_CG_Info_Post_Op);
   --  Traverses the argument Element in ordrer to collect call graph
   --  information. Usded as internal traversal routine for the implementation
   --  of Collect_CG_Info_From_Construct.

   -----------------
   -- Add_CG_Info --
   -----------------

   procedure Add_CG_Info
     (Element : Asis.Element;
      At_SLOC : String_Loc := Nil_String_Loc)
   is
   begin

      if Can_Have_Elaboration_Calls (Element) then
         --  Is_Call and Can_Create_Tasks Elements can have elaboration calls,
         --  so we have to process elaboration calls in a separate IF
         --  statement.
         Process_Elaboration_Calls (Element);
      end if;

      if Is_Scope (Element) then
         Process_Scope (Element);

      elsif Is_Declaration_Of_Callable_Entity (Element) then
         Process_Callable_Entity (Element);
         --  Potential main subprograms !!!!!!!!!!!

      elsif Asis.Extensions.Is_Renaming_As_Body (Element) then
         Process_Renaming_As_Body (Element);
         --  At the moment, we just unwind renamings to the called subprogram

      elsif Is_Call (Element) then
         Process_Call (Element, At_SLOC => At_SLOC);
      elsif Can_Create_Tasks (Element) then
         Process_Task_Creation (Element);
      end if;

   end Add_CG_Info;

   ------------------------
   -- Add_CG_Info_Pre_Op --
   ------------------------

   procedure Add_CG_Info_Pre_Op
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out String_Loc)
   is
      pragma Unreferenced (Control);
      pragma Unmodified  (State);

   begin
      Add_CG_Info (Element, State);
   end Add_CG_Info_Pre_Op;

   -------------------
   -- Body_Analyzed --
   -------------------

   function Body_Analyzed (N : GS_Node_Id) return Boolean is
   begin
      pragma Assert (GS_Node_Kind (N) in  Callable_Nodes);
      return Table (N).Bool_Flag_1;
   end Body_Analyzed;

   -----------------------------------
   -- Check_Call_Graph_Completeness --
   -----------------------------------

   procedure Check_Call_Graph_Completeness is
   begin

      for Node in First_GS_Node .. Last_Node loop

         if Is_Callable_Node (Node)
           and then
            not Body_Analyzed (Node)
         then
            ASIS_UL.Output.Error
              ("body is not analyzed for " &
               Get_String (GS_Node_SLOC (Node)));
         end if;

      end loop;

   end Check_Call_Graph_Completeness;

   ------------------------------------
   -- Collect_CG_Info_From_Construct --
   ------------------------------------

   procedure Collect_CG_Info_From_Construct
     (Element : Asis.Element;
      At_SLOC : String_Loc := Nil_String_Loc)
   is
      State   : String_Loc       := At_SLOC;
      Control : Traverse_Control := Continue;
   begin
      Traverse_Construct_For_CG_Info (Element, Control, State);
   end Collect_CG_Info_From_Construct;

   ----------------------
   -- Complete_CG_Info --
   ----------------------

   procedure Complete_CG_Info (El : Asis.Element) is
   begin

      if Is_Scope (El) then
         Remove_Current_Scope;
      end if;

   end Complete_CG_Info;

   ------------------------------
   -- Complete_CG_Info_Post_Op --
   ------------------------------

   procedure Complete_CG_Info_Post_Op
     (Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out String_Loc)
   is
      pragma Unreferenced (Control, State);
   begin
      Complete_CG_Info (Element);
   end Complete_CG_Info_Post_Op;

   -----------------------
   -- First_Direct_Call --
   -----------------------

   function First_Direct_Call (N : GS_Node_Id) return GS_Node_Id is
      Result : GS_Node_Id := No_GS_Node;
   begin

      pragma Assert (Is_Callable_Node (N));

      if not SLOC_Node_Lists.Is_Empty (Table (N).SLOC_Node_List_1) then
         Result :=
           SLOC_Node_Lists.First_Element (Table (N).SLOC_Node_List_1).Node;
      end if;

      return Result;
   end First_Direct_Call;

   --------------------
   -- GS_Is_Renaming --
   --------------------

   function GS_Is_Renaming (N : GS_Node_Id) return Boolean is
   begin
      pragma Assert (GS_Node_Kind (N) in  Callable_Nodes);
      return Table (N).Bool_Flag_2;
   end GS_Is_Renaming;

   ---------------------
   -- GS_Is_Task_Type --
   ---------------------

   function GS_Is_Task_Type (N : GS_Node_Id) return Boolean is
   begin
      pragma Assert (GS_Node_Kind (N) in  Callable_Nodes);

      return GS_Node_Kind (N) = A_Task
            and then
             Table (N).Bool_Flag_3;
   end GS_Is_Task_Type;

   -----------------------
   -- Is_Recursive_Node --
   -----------------------

   function Is_Recursive_Node (N : GS_Node_Id) return Boolean is
   begin

      return Node_Lists.Contains
               (Container => Table (N).Node_List_1, -- all calls
                Item      => N);
   end Is_Recursive_Node;

   ------------------
   -- Process_Call --
   ------------------

   procedure Process_Call
     (Element : Asis.Element;
      At_SLOC : String_Loc := Nil_String_Loc)
   is
      Called_El : Asis.Element := Get_Called_Element (Element);
   begin

      if Is_Nil (Called_El) then

         if Is_Call_To_Predefined_Operation (Element)
           or else
            Is_Call_To_Attribute_Subprogram (Element)
         then
            --  We do not consider such calls at all
            return;
         elsif Generate_Global_Structure_Warnings then
            ASIS_UL.Output.Error (Build_GNAT_Location (Element) &
                   ": call can not be resolved statically");
         end if;

      elsif Declaration_Kind (Called_El) =
            An_Enumeration_Literal_Specification
      then
         --  This may happen in instantiation if an enumeration literal is
         --  used as an actual for a formal function.
         return;
      else

         if Is_Predefined_Operation_Renaming (Called_El) then
            --  We do not consider such calls at all
            return;
         end if;

         Called_El := Corresponding_Element (Called_El);

         if Is_Nil (Called_El) then
            --  subprogram renaming cannot be resolved statically, diagnstic
            --  should be generated
            raise ASIS_UL.Common.Non_Implemented_Error;
         elsif Expression_Kind (Called_El) = An_Attribute_Reference
              or else
               Expression_Kind (Called_El) = An_Enumeration_Literal
         then
            --  These calls are of no interest
            return;
         end if;

         pragma Assert
           (Is_Declaration_Of_Callable_Entity (Called_El)
           or else
            Is_Scope (Called_El));

         if At_SLOC = Nil_String_Loc then
            Store_Arc
              (Called_Entity => Called_El,
               At_SLOC       => Build_GNAT_Location (Element));
         else
            Store_Arc
              (Called_Entity => Called_El,
               At_SLOC       => At_SLOC);
         end if;

      end if;

   end Process_Call;

   -----------------------------
   -- Process_Callable_Entity --
   -----------------------------

   procedure Process_Callable_Entity (El : Asis.Element) is
      Tmp : GS_Node_Id;
      pragma Unreferenced (Tmp);
   begin
      Tmp := Corresponding_Node (El, Current_Scope);

      if Declaration_Kind (El) = A_Single_Task_Declaration then
         Store_Arc (Called_Entity => El, At_SLOC => Build_GNAT_Location (El));
      end if;

   end Process_Callable_Entity;

   -------------------------------
   -- Process_Elaboration_Calls --
   -------------------------------

   procedure Process_Elaboration_Calls (Element : Asis.Element) is
      Arg_Kind : constant Flat_Element_Kinds := Flat_Element_Kind (Element);
      Call_AT_SLOC : constant String_Loc     := Build_GNAT_Location (Element);

      Type_To_Analyze : Asis.Element := Nil_Element;
      --  To be set to point to the (full) type declaration of the type
      --  for that we have to process default (sub)component initialization
      --  expressions

      Tmp_El : Asis.Element;

      Process_Discriminants : Boolean := False;
      --  In case if the discriminant constraint is present, we do not have to
      --  process default expressions for discriminants

   begin

      case Arg_Kind is
         when A_Variable_Declaration |
              An_Allocation_From_Subtype =>

            if Arg_Kind = A_Variable_Declaration then
               Type_To_Analyze := Object_Declaration_View (Element);
            else
               Type_To_Analyze := Allocator_Subtype_Indication (Element);
            end if;

            if Type_Kind (Type_To_Analyze) in
                 An_Unconstrained_Array_Definition ..
                  A_Constrained_Array_Definition
            then
               Type_To_Analyze := Array_Component_Definition (Type_To_Analyze);
               Type_To_Analyze :=
                 Component_Subtype_Indication (Type_To_Analyze);
            end if;

            Tmp_El := Asis.Definitions.Subtype_Mark (Type_To_Analyze);

            if Expression_Kind (Tmp_El) /= An_Attribute_Reference then
               --  In case of a attribute reference as a subtype mark the
               --  only possible case is 'Base, so we have a scalar type
               --  here, therefore it can be no default initialization

               Process_Discriminants :=
                  Is_Nil (Subtype_Constraint (Type_To_Analyze))
                 and then
                  Is_Indefinite_Subtype (Tmp_El);
            else
               Process_Discriminants := False;
            end if;

            Type_To_Analyze := Get_Subtype_Structure (Type_To_Analyze);

            --  First, check discriminants:

            if Process_Discriminants then
               Tmp_El := Discriminant_Part (Type_To_Analyze);

               if Definition_Kind (Tmp_El) = A_Known_Discriminant_Part then

                  declare
                     Discr_Specs : constant Asis.Element_List :=
                       Discriminants (Tmp_El);
                  begin

                     for J in Discr_Specs'Range loop

                        Tmp_El := Initialization_Expression (Discr_Specs (J));

                        if Is_Nil (Tmp_El) then
                           exit;
                        end if;

                        Collect_CG_Info_From_Construct
                          (Element => Tmp_El,
                           At_SLOC => Call_AT_SLOC);

                     end loop;

                  end;

               end if;

            end if;

            --  Now, check if we have record components with defaul
            --  initialization expressions

            Process_Type_Default_Expressions
              (Type_To_Analyze => Type_To_Analyze,
               Call_At_SLOC    => Call_AT_SLOC);

         when An_Entry_Call_Statement    |
              A_Procedure_Call_Statement |
              A_Function_Call            =>

            declare
               Call_Parameters : constant Asis.Element_List :=
                 Get_Call_Parameters (Element, Normalized => True);
               --  Note that if Elemnent is a dispatching or dynamic call,
               --  Call_Parameters are Nil_Element_List!
            begin

               for J in Call_Parameters'Range loop

                  if Is_Defaulted_Association (Call_Parameters (J)) then
                     Tmp_El := Actual_Parameter (Call_Parameters (J));

                     Collect_CG_Info_From_Construct
                       (Element => Tmp_El,
                        At_SLOC => Call_AT_SLOC);
                  end if;

               end loop;

            end;

         when A_Procedure_Instantiation |
              A_Function_Instantiation  =>

            declare
               Inst_Parameters : constant Asis.Element_List :=
                 Generic_Actual_Part (Element, Normalized => True);
            begin

               for J in Inst_Parameters'Range loop

                  if Is_Defaulted_Association (Inst_Parameters (J))
                    and then
                     Declaration_Kind (Enclosing_Element
                       (Formal_Parameter (Inst_Parameters (J)))) =
                          A_Formal_Object_Declaration
                  then
                     --  Note the condition expression: we check that we have
                     --  an association corresponding to formal object by
                     --  querying the kind of Enclosing_Element of a formal,
                     --  but not actual parameter of the association, because
                     --  the ASIS Standard does not define exactly the effect
                     --  of Enclosing_Element for an actual parameter from a
                     --  normalized association

                     Tmp_El := Actual_Parameter (Inst_Parameters (J));

                     Collect_CG_Info_From_Construct
                       (Element => Tmp_El,
                        At_SLOC => Call_AT_SLOC);
                  end if;

               end loop;

            end;

         when others =>
            null;
            --  Not implemented yet
      end case;

   end Process_Elaboration_Calls;

   ---------------------------------------------------
   -- Process_Init_Expressions_In_Record_Components --
   ---------------------------------------------------

   procedure Process_Init_Expressions_In_Record_Components
     (Component_List : Asis.Element_List;
      Call_At_SLOC   : String_Loc)
   is
      Comp_Def : Asis.Element;
   begin

      for J in Component_List'Range loop

         case Flat_Element_Kind (Component_List (J)) is
            when Flat_Clause_Kinds |
                 A_Null_Component  =>
               null;
            when A_Variant_Part =>

               Process_Init_Expressions_In_Record_Components
                 (Component_List =>
                    Asis.Definitions.Variants (Component_List (J)),
                  Call_At_SLOC => Call_At_SLOC);

            when A_Variant =>

               Process_Init_Expressions_In_Record_Components
                 (Component_List =>
                    Asis.Definitions.Record_Components (Component_List (J)),
                  Call_At_SLOC => Call_At_SLOC);

            when A_Component_Declaration =>

               Comp_Def := Initialization_Expression (Component_List (J));

               if Is_Nil (Comp_Def) then
                  --  No initialization here, but we have to go down the
                  --  component structure:

                  Comp_Def := Object_Declaration_View (Component_List (J));
                  Comp_Def := Component_Subtype_Indication (Comp_Def);
                  Comp_Def := Get_Subtype_Structure (Comp_Def);

                  Process_Type_Default_Expressions
                    (Type_To_Analyze => Comp_Def,
                     Call_At_SLOC    => Call_At_SLOC);
               else
                  Collect_CG_Info_From_Construct
                    (Element => Comp_Def,
                     At_SLOC => Call_At_SLOC);
               end if;

            when others =>
               --  Just in case...
               pragma Assert (False);
               null;
         end case;
      end loop;

   end Process_Init_Expressions_In_Record_Components;

   ------------------------------------
   -- Process_Record_Task_Components --
   ------------------------------------

   procedure Process_Record_Task_Components
     (Component_List : Asis.Element_List;
      Call_At_SLOC   : String_Loc)
   is
      Comp_Def : Asis.Element;
   begin

      for J in Component_List'Range loop

         case Flat_Element_Kind (Component_List (J)) is
            when Flat_Clause_Kinds |
                 A_Null_Component  =>
               null;
            when A_Variant_Part =>

               Process_Record_Task_Components
                 (Component_List =>
                    Asis.Definitions.Variants (Component_List (J)),
                  Call_At_SLOC => Call_At_SLOC);

            when A_Variant =>

               Process_Record_Task_Components
                 (Component_List =>
                    Asis.Definitions.Record_Components (Component_List (J)),
                  Call_At_SLOC => Call_At_SLOC);

            when A_Component_Declaration =>
               Comp_Def := Object_Declaration_View (Component_List (J));
               Comp_Def := Component_Subtype_Indication (Comp_Def);
               Comp_Def := Get_Subtype_Structure (Comp_Def);

               Process_Task_Components
                 (Type_Decl    => Comp_Def,
                  Call_At_SLOC => Call_At_SLOC);
            when others =>
               --  Just in case...
               pragma Assert (False);
               null;
         end case;
      end loop;

   end Process_Record_Task_Components;

   ------------------------------
   -- Process_Renaming_As_Body --
   ------------------------------

   procedure Process_Renaming_As_Body (El : Asis.Element) is
      Subprogram_Node : constant GS_Node_Id :=
        Corresponding_Node (Corresponding_Declaration (El));

      Renamed_Subprogram : constant Asis.Element :=
        Get_Renamed_Subprogram (El);

      Renamed_Subprogram_Node : GS_Node_Id;

      Is_Of_No_Interest : Boolean := True;
   begin
      Set_Is_Renaming (Subprogram_Node);

      case Declaration_Kind (Renamed_Subprogram) is

         when A_Procedure_Declaration      |
              A_Function_Declaration       |
              A_Procedure_Body_Declaration |
              A_Function_Body_Declaration  |
              A_Procedure_Body_Stub        |
              A_Function_Body_Stub         |
              A_Procedure_Instantiation    |
              A_Function_Instantiation     =>
            Is_Of_No_Interest := False;

         when An_Entry_Declaration    =>
            --  Task entry is renamed as a subprogram - we cannot process
            --  this case yet:
            Set_Is_Of_No_Interest (Subprogram_Node);
            raise ASIS_UL.Common.Non_Implemented_Error;

         when others =>
            --  Is_Of_No_Interest remains ON. Here we have all the cases of
            --  attrubute subprogram renamings
            null;
      end case;

      if Is_Of_No_Interest then
         Set_Is_Of_No_Interest (Subprogram_Node);
      else
         Renamed_Subprogram_Node := Corresponding_Node (Renamed_Subprogram);

         --  Add the "call" from a renaming to the renamed subprogram
         Add_Link_To_SLOC_List
           (To_Node     => Subprogram_Node,
            To_List     => Calls,
            Link_To_Add => (Node => Renamed_Subprogram_Node,
                            SLOC => Build_GNAT_Location (El)));
      end if;

   end Process_Renaming_As_Body;

   -------------------
   -- Process_Scope --
   -------------------

   procedure Process_Scope (El : Asis.Element) is
      Tmp      : GS_Node_Id;
      Scope_El : Asis.Element;
   begin

      Scope_El := Corresponding_Element (El);
      Tmp      := Corresponding_Node (Scope_El, Current_Scope);

      if Declaration_Kind (El) = A_Task_Body_Declaration
        and then
          Declaration_Kind (Corresponding_Declaration (Scope_El)) =
            A_Task_Type_Declaration
      then
         --  Task type differs from a single anonymiosily typed task object in
         --  respect of the scope node. For a task object, the front-end
         --  creates an inplicit task type using the defining identifier node
         --  from the task body as the defining identifier node for this type,
         --  so the defining identifier from the body works as a top of the
         --  scope for bodies corresponding to single task declarations. But
         --  for a body that corresponds to a task type we have to go to the
         --  task type declaration to get the scope node.

         Scope_El := Corresponding_Declaration (Scope_El);
      end if;

      Scope_El := First_Name (Scope_El);

      Set_Current_Scope (Tmp, Node (Scope_El));
      Set_Body_Analyzed (Tmp);

   end Process_Scope;

   -----------------------------
   -- Process_Task_Components --
   -----------------------------

   procedure Process_Task_Components
     (Type_Decl    : Asis.Element;
      Call_At_SLOC : String_Loc)
   is
      T_Def : Asis.Element;
      Tmp   : Asis.Element;
   begin

      case Declaration_Kind (Type_Decl) is
         when A_Task_Type_Declaration =>
            Store_Arc
              (Called_Entity => Type_Decl,
               At_SLOC       => Call_At_SLOC);

         when A_Protected_Type_Declaration =>
            null;
         when An_Ordinary_Type_Declaration =>
            T_Def := Type_Declaration_View (Type_Decl);

            case Type_Kind (T_Def) is
               when A_Derived_Record_Extension_Definition =>

                  Tmp := Asis.Definitions.Record_Definition (T_Def);

                  if Definition_Kind (Tmp) /= A_Null_Record_Definition then
                     Process_Record_Task_Components
                       (Component_List => Record_Components (Tmp),
                        Call_At_SLOC   => Call_At_SLOC);
                  end if;

                  Tmp := Parent_Subtype_Indication (T_Def);
                  Tmp := Get_Subtype_Structure (Tmp);
                  Process_Task_Components (Tmp, Call_At_SLOC => Call_At_SLOC);

               when An_Unconstrained_Array_Definition |
                    A_Constrained_Array_Definition    =>

                  Tmp := Array_Component_Definition (T_Def);
                  Tmp := Component_Definition_View  (Tmp);

                  if Definition_Kind (Tmp) = A_Subtype_Indication then
                     --  we are not interested in components that are defined
                     --  by An_Access_Definition
                     Tmp := Get_Subtype_Structure (Tmp);

                     Process_Task_Components
                       (Tmp,
                        Call_At_SLOC => Call_At_SLOC);
                  end if;

               when A_Record_Type_Definition |
                    A_Tagged_Record_Type_Definition =>

                  --  Note: we do not process discriminant components!

                  Tmp := Asis.Definitions.Record_Definition (T_Def);

                  if Definition_Kind (Tmp) /= A_Null_Record_Definition then
                     Process_Record_Task_Components
                       (Component_List => Record_Components (Tmp),
                        Call_At_SLOC   => Call_At_SLOC);
                  end if;

               when A_Derived_Type_Definition =>
                  --  Just in case...
                  pragma Assert (False);
                  null;

               when others =>
                  null;
            end case;

         when others =>
            pragma Assert (False);
            null;
      end case;

   end Process_Task_Components;

   ---------------------------
   -- Process_Task_Creation --
   ---------------------------

   procedure Process_Task_Creation (El : Asis.Element) is
      Type_To_Analyze : Asis.Element;
   begin

      case Flat_Element_Kind (El) is
         when A_Variable_Declaration |
              A_Constant_Declaration =>
            Type_To_Analyze := Object_Declaration_View (El);

            if Type_Kind (Type_To_Analyze) in
                 An_Unconstrained_Array_Definition ..
                  A_Constrained_Array_Definition
            then
               Type_To_Analyze := Array_Component_Definition (Type_To_Analyze);
               Type_To_Analyze :=
                 Component_Subtype_Indication (Type_To_Analyze);
            end if;

         when An_Allocation_From_Subtype =>
            Type_To_Analyze := Allocator_Subtype_Indication (El);
         when others =>
            pragma Assert (False);
            null;
      end case;

      Type_To_Analyze := Get_Subtype_Structure (Type_To_Analyze);

      Process_Task_Components
        (Type_To_Analyze,
         Call_At_SLOC => Build_GNAT_Location (El));
   end Process_Task_Creation;

   --------------------------------------
   -- Process_Type_Default_Expressions --
   --------------------------------------

   procedure Process_Type_Default_Expressions
     (Type_To_Analyze : Asis.Element;
      Call_At_SLOC    : String_Loc)
   is
      Type_Def : constant Asis.Element :=
        Type_Declaration_View (Type_To_Analyze);

      Tmp : Asis.Element;
   begin

      --  Note: we do not process discriminant components!

      case Definition_Kind (Type_Def) is
         when A_Protected_Definition =>
            Process_Init_Expressions_In_Record_Components
              (Component_List => Private_Part_Items (Type_Def),
               Call_At_SLOC   => Call_At_SLOC);

         when A_Type_Definition =>

            case Type_Kind (Type_Def) is

               when A_Derived_Record_Extension_Definition =>

                  Tmp := Asis.Definitions.Record_Definition (Type_Def);

                  if Definition_Kind (Tmp) = A_Null_Record_Definition then
                     Process_Init_Expressions_In_Record_Components
                       (Component_List => Record_Components (Tmp),
                        Call_At_SLOC   => Call_At_SLOC);
                  end if;

                  Tmp := Parent_Subtype_Indication (Type_Def);
                  Tmp := Get_Subtype_Structure (Tmp);

                  Process_Type_Default_Expressions
                    (Type_To_Analyze => Tmp,
                     Call_At_SLOC    => Call_At_SLOC);

               when An_Unconstrained_Array_Definition |
                    A_Constrained_Array_Definition    =>

                  Tmp := Array_Component_Definition (Type_Def);
                  Tmp := Component_Definition_View  (Tmp);

                  if Definition_Kind (Tmp) = A_Subtype_Indication then
                     --  we are not interested in components that are defined
                     --  by An_Access_Definition
                     Tmp := Get_Subtype_Structure (Tmp);

                     Process_Type_Default_Expressions
                       (Type_To_Analyze => Tmp,
                        Call_At_SLOC    => Call_At_SLOC);
                  end if;

               when A_Record_Type_Definition |
                    A_Tagged_Record_Type_Definition =>

                  Tmp := Asis.Definitions.Record_Definition (Type_Def);

                  if Definition_Kind (Tmp) /= A_Null_Record_Definition then
                     Process_Init_Expressions_In_Record_Components
                       (Component_List => Record_Components (Tmp),
                        Call_At_SLOC   => Call_At_SLOC);
                  end if;

               when others =>
                  --  No default initialization expression in this case!
                  null;
            end case;

         when others =>
            --  No default initialization expression in this case!
            null;
      end case;

   end Process_Type_Default_Expressions;

   -----------------------
   -- Set_Body_Analyzed --
   -----------------------

   procedure Set_Body_Analyzed (N : GS_Node_Id; Val : Boolean := True) is
   begin
      pragma Assert (GS_Node_Kind (N) in  Callable_Nodes);
      Set_Bool_Flag_1 (N, Val);
   end Set_Body_Analyzed;

   ---------------------
   -- Set_Is_Renaming --
   ---------------------

   procedure Set_Is_Renaming (N : GS_Node_Id; Val : Boolean := True) is
   begin
      pragma Assert (GS_Node_Kind (N) in  Callable_Nodes);
      Set_Bool_Flag_2 (N, Val);
   end Set_Is_Renaming;

   ----------------------
   -- Set_Is_Task_Type --
   ----------------------

   procedure Set_Is_Task_Type (N : GS_Node_Id; Val : Boolean := True) is
   begin
      pragma Assert (GS_Node_Kind (N) = A_Task);
      Set_Bool_Flag_3 (N, Val);
   end Set_Is_Task_Type;

   ---------------
   -- Store_Arc --
   ---------------

   procedure Store_Arc (Called_Entity : Asis.Element; At_SLOC : String_Loc) is
      Called_Node : constant GS_Node_Id := Corresponding_Node (Called_Entity);
   begin
      pragma Assert
        (First_GS_Node < Called_Node
       and then
         Called_Node <= Last_Node);

      Add_Link_To_SLOC_List
        (To_Node     => Current_Scope,
         To_List     => Calls,
         Link_To_Add => (Node => Called_Node, SLOC => At_SLOC));

   end Store_Arc;

   ------------------------
   -- Transitive_Closure --
   ------------------------

   procedure Transitive_Closure is
      procedure Close_Node (Node : GS_Node_Id);
      --  Creates a list of all the nodes called by the given node using
      --  workpile algorithm.
      --
      --  The following variables are used by this procedure, we define them
      --  as global to avoid elaboration expances for each call of Close_Node.

      New_Set   : Node_Lists.Set;
      --  A set of nodes that are added to All_Call. For each of the nodes
      --  from this set we should analyse its direct calls and then remove
      --  the node fron this set. We stop the loop for the next node when
      --  this set is empty,

      Newer_Set : Node_Lists.Set;
      --  Nodes that are added for All_Call at the last iteration of the
      --  processing of New_Set for the given node. They should be added to
      --  New_Set to process their direct calls.

      Next_Direct_Call : Node_Lists.Cursor;
      Next_Call        : SLOC_Node_Lists.Cursor;

      ----------------
      -- Close_Node --
      ----------------

      procedure Close_Node (Node : GS_Node_Id) is
      begin

         --  Node_List_1 <--> Direct calls
         --  Node_List_2 <--> All calls

         Node_Lists.Clear (New_Set);
         Node_Lists.Clear (Newer_Set);

         Add_SLOC_Node_List_To_Node_List
           (Table (Node).Node_List_1,
            Table (Node).SLOC_Node_List_1);

         Add_SLOC_Node_List_To_Node_List
           (New_Set,
            Table (Node).SLOC_Node_List_1);

         while not Node_Lists.Is_Empty (New_Set) loop
            Next_Direct_Call := Node_Lists.First (New_Set);

            Next_Call :=
              SLOC_Node_Lists.First
                (Table (Node_Lists.Element (Next_Direct_Call)).
                   SLOC_Node_List_1);

            while SLOC_Node_Lists.Has_Element (Next_Call) loop

               if not Node_Lists.Contains
                 (Table (Node).Node_List_1,
                  SLOC_Node_Lists.Element (Next_Call).Node)
               then
                  Node_Lists.Insert
                    (Newer_Set, SLOC_Node_Lists.Element (Next_Call).Node);
               end if;

               Next_Call := SLOC_Node_Lists.Next (Next_Call);
            end loop;

            Node_Lists.Delete_First (New_Set);

            if not Node_Lists.Is_Empty (Newer_Set) then
               Node_Lists.Union (Table (Node).Node_List_1,
                                 Newer_Set);
               Node_Lists.Union (New_Set,   Newer_Set);
               Node_Lists.Clear (Newer_Set);
            end if;

         end loop;

      end Close_Node;

   begin

      Traverse_Renamings;
      Check_Call_Graph_Completeness;

      for Node in First_GS_Node .. Last_Node loop

         if Is_Callable_Node (Node) then
            Close_Node (Node);
         end if;

      end loop;

   end Transitive_Closure;

   ------------------------
   -- Traverse_Renamings --
   ------------------------

   procedure Traverse_Renamings is
      Already_Processed_Renamings : Node_Lists.Set;

      procedure Process_Renaming (Node : GS_Node_Id);
      --  Processes one renaming node and after that add node to
      --  Already_Processed_Renamings set. This procedure recursively traverses
      --  renaming chains.

      procedure Process_Renaming (Node : GS_Node_Id) is
         Renamed_Node : constant GS_Node_Id := First_Direct_Call (Node);
      begin

         if Is_Of_No_Interest (Renamed_Node) then
            Set_Is_Of_No_Interest (Node);
            return;
         end if;

         if GS_Is_Renaming (Renamed_Node)
           and then
             not Node_Lists.Contains
                   (Already_Processed_Renamings, Renamed_Node)
         then
            Process_Renaming (Renamed_Node);
            --  This may define that Renamed_Node is of no interest, so:

            if Is_Of_No_Interest (Renamed_Node) then
               Set_Is_Of_No_Interest (Node);
               return;
            end if;

         end if;

         Set_Body_Analyzed (Node, Body_Analyzed (Renamed_Node));

      end Process_Renaming;

   begin
      Node_Lists.Clear (Already_Processed_Renamings);

      for Node in First_GS_Node .. Last_Node loop

         if Is_Callable_Node (Node)
           and then
             GS_Is_Renaming (Node)
           and then
            not Is_Of_No_Interest (Node)
           and then
             not Node_Lists.Contains (Already_Processed_Renamings, Node)
         then
            Process_Renaming (Node);
         end if;

      end loop;

   end Traverse_Renamings;

end ASIS_UL.Global_State.CG;
