/*
 * Argus Software.  Argus files - main argus processing
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* 
 * $Id: $
 * $DateTime: $
 * $Change: $
 */

/*
 * argus - Audit Record Generation and Utilization System
 *
 * written by Carter Bullard
 * QoSient LLC
 *
 */

#if !defined(Argus)
#define Argus
#endif

#include <stdlib.h>
#include <syslog.h>
#include <stdarg.h>

#include <argus.h>
#include <compat.h>
#include <ArgusModeler.h>
#include <ArgusOutput.h>
#include <ArgusSource.h>
#include <ArgusUtil.h>

void ArgusParseResourceFile (struct ArgusModelerStruct *, char *);


void
usage(void)
{
   extern char version[];

   fprintf (stderr, "Argus Version %s\n", version);
   fprintf (stderr, "usage: %s [options] [-i interface] [filter-expression] \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options]  -r packetfile [filter-expression] \n\n", ArgusProgramName);

   fprintf (stderr, "options: -A                   Generate application byte metrics.\n");
   fprintf (stderr, "         -b                   dump filter compiler output.\n");
   fprintf (stderr, "         -B <addr>            specify bind interface address.\n");
   fprintf (stderr, "         -c <dir>             daemon chroot directory.\n");
   fprintf (stderr, "         -d                   run Argus in daemon mode.\n");

#if defined(ARGUSDEBUG)
   fprintf (stderr, "         -D <level>           set debug reporting <level>.\n");
#endif

   fprintf (stderr, "         -e <value>           specify Argus Identifier <value>.\n");
   fprintf (stderr, "         -h                   print help.\n");
   fprintf (stderr, "         -F <conffile>        read configuration from <conffile>.\n");
   fprintf (stderr, "         -J                   generate packet performance data.\n");
   fprintf (stderr, "         -M <secs>            set MAR Status Report Time Interval (300s).\n");
   fprintf (stderr, "         -m                   turn on MAC Layer Reporting.\n");
   fprintf (stderr, "         -O                   turn off filter optimizer.\n");
   fprintf (stderr, "         -p                   don't go into promiscuous mode.\n");
   fprintf (stderr, "         -P <portnum>         enable remote access on <portnum> (561).\n");
   fprintf (stderr, "         -r <file file ...>   use packet file as data source.\n");
   fprintf (stderr, "         -R                   generate response time data.\n");
   fprintf (stderr, "         -S <secs>            set FAR Status Report Time Interval (60s).\n");
   fprintf (stderr, "         -t                   indicate that packetfile is MOAT Tsh format. \n");
   fprintf (stderr, "         -u <userid>          specify user id for daemon.\n");
   fprintf (stderr, "         -g <groupid>         specify group id for daemon.\n");
   fprintf (stderr, "         -U <bytes>           specify the number of user bytes to capture.\n");
   fprintf (stderr, "         -w <file [\"filter\"]> write output to <file>, or '-', for stdout,\n");
   fprintf (stderr, "                              against optional filter expression.\n");
   fprintf (stderr, "         -X                   reset argus configuration.\n");
   fprintf (stderr, "         -Z                   generate packet size data.\n");
   exit (-1);
}


/*
 *  Argus main routine 
 *
 *  Argus main will:
 *       simply instantiate the source, modeler, and output tasks,
 *       parse out the command line options,
 *       initalize the tasks and then loop.
 *       Afterwards, it will delete all the tasks and exit();
 *
 */

#define ArgusEnvItems      2

char *ArgusResourceEnvStr [] = {
   "ARGUSHOME",
   "HOME",
};


#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>


static char ArgusPidFileName[MAXPATHNAMELEN];
char * ArgusCreatePIDFile (struct ArgusSourceStruct *, char *, char *);

#define ARGUS_MAX_INSTANCES	5


char *
ArgusCreatePIDFile (struct ArgusSourceStruct *src, char *pidpath, char *appname)
{
   FILE *fd;
   char pidstrbuf[128], *pidstr = pidstrbuf;
   char *retn = NULL, *dev, *devstr;
   int i, pid;
   struct stat statbuf;

   if (pidpath == NULL)
      if (stat ("/var/run", &statbuf) == 0)
         pidpath = "/var/run";

   if ((appname != NULL) && ((dev = getArgusDevice(src)) != NULL) && (pidpath != NULL)) {
      if ((devstr = strrchr(dev, (int)'/')) != NULL)
         devstr++;
      else
         devstr = dev;

      for (i = 0; i < ARGUS_MAX_INSTANCES; i++) {
         snprintf (ArgusPidFileName, MAXPATHNAMELEN - 1, "%s/%s.%s.%d.pid", pidpath, appname, devstr, i);
         retn = ArgusPidFileName;

         if ((stat (retn, &statbuf)) == 0) {
            if ((fd = fopen (ArgusPidFileName, "r")) != NULL) {
               if ((pidstr = fgets (pidstrbuf, 128, fd)) != NULL) {
                  if ((pid = strtol(pidstr, (char **)NULL, 10)) > 0) {
                     if (pid < (int) LONG_MAX) {
                        if ((kill (pid, 0)) == 0) {
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusCreatePIDFile(%s, %s) pid %d is running\n", pidpath, appname, pid);
#endif 
                           retn = NULL;
                        }
                     }
                  }
               }

               fclose (fd);
            }
         }
 
         if (retn != NULL)
            break;
      }

      if (retn && ((fd = fopen (retn, "w+")) != NULL)) {
         pid = getpid();
         fprintf (fd, "%d\n", pid);
         fclose (fd);
      } else
         retn = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusCreatePIDFile(0x%x, 0x%x) returning %s\n", pidpath, appname, retn);
#endif 

   return (retn);
}


char *ArgusPidFile = NULL;
pid_t ArgusSessionId = 0;
struct timeval ArgusNowTime;
char *ArgusPidPath = NULL;

int
main (int argc, char *argv[])
{
   int commandlinew = 0, doconf = 0;
   static char path[MAXPATHNAMELEN];
   int dodebug = 0, i, pid = 0;
   char *tmparg, *filter;
   extern char *optarg;
   struct stat statbuf;
   extern int optind, opterr;
   int op, commandlinei = 0;
#if defined(ARGUS_THREADS)
   pthread_attr_t attr;
   int thread_policy;
   struct sched_param thread_param;
#if defined(HAVE_SCHED_GET_PRIORITY_MIN)
   int rr_min_priority, rr_max_priority;
#endif
   int status;
#endif

   ArgusUid = getuid();
   
   gettimeofday(&ArgusNowTime, 0L);

   uflag = 0;
   pflag = 6;

   if (strchr (argv[0], '/')) {
      strcpy(path, argv[0]);
      argv[0] = strrchr(argv[0], '/') + 1;
   }

   ArgusProgramName = argv[0];

#ifdef ARGUS_SYSLOG
#ifndef LOG_PERROR
#define LOG_PERROR      LOG_CONS
#endif

   openlog (ArgusProgramName, LOG_PID | LOG_PERROR, LOG_DAEMON);
#endif

   for (i = 1; (i < argc); i++) {
      char *ptr = argv[i]; 
      if (ptr != NULL) {
         if (*ptr == '-') {
            ptr++;
            if (isspace((int)*ptr))
               break;
            do {
               switch (*ptr) {
                  case 'D': 
                     if (isdigit((int)*++ptr)) {
                        setArgusdflag (ArgusModel, atoi (ptr));
                     } else {
                        if (isdigit((int)*argv[i + 1]))
                           setArgusdflag (ArgusModel, atoi (argv[++i]));
                        else
                           break;
                     }
                     break;
                  case 'X': 
                  case 'F': 
                     doconf++; 
                     break; 
                   
                  default: {
                     if (dodebug) {
                        if (isdigit((int)*ptr)) {
                           setArgusdflag (ArgusModel, atoi (ptr));
                           dodebug = 0;
                        }
                     }
                  }
               }

            } while (isalpha((int)*++ptr));
         }
      }
   }

   if ((ArgusModel = ArgusNewModeler()) == NULL)
      ArgusLog (LOG_ERR, "Error Creating Modeler: Exiting.\n");

   if ((ArgusSourceTask = ArgusNewSource(ArgusModel)) == NULL)
      ArgusLog (LOG_ERR, "Error Creating Source Task: Exiting.\n");

   if ((ArgusOutputTask = ArgusNewOutput(ArgusSourceTask, ArgusModel)) == NULL)
      ArgusLog (LOG_ERR, "Error Creating Output Thread: Exiting.\n");

   ArgusModel->ArgusSrc = ArgusSourceTask;

   setArgusFarReportInterval (ArgusModel, ARGUS_FARSTATUSTIMER);
   setArgusMarReportInterval (ArgusOutputTask,ARGUS_MARSTATUSTIMER);

   if (!doconf) {
      snprintf (path, MAXPATHNAMELEN - 1, "/etc/argus.conf");
      if (stat (path, &statbuf) == 0) {
         ArgusParseResourceFile (ArgusModel, path);
      }
   }

   optind = 1, opterr = 0;

   while ((op = getopt (argc, argv, "AbB:c:dD:e:F:g:i:JmM:N:OP:pRr:S:stT:u:U:w:XZh")) != EOF) {
      switch (op) {
         case 'A': setArgusAflag(ArgusModel, 1); break;
         case 'b': setArgusbpflag (ArgusSourceTask, 1); break;
         case 'B': setArgusBindAddr (ArgusOutputTask, optarg); break;
         case 'c': 
            if ((chroot_dir = strdup(optarg)) == NULL)
                ArgusLog (LOG_ERR, "strdup %s", strerror(errno));
            break;

         case 'd': daemonflag = daemonflag ? 0 : 1; break;
         case 'D': setArgusdflag (ArgusModel, atoi (optarg)); break;
         case 'e': ArgusParseSourceID(ArgusModel, optarg); break;

         case 'i':
            if (!commandlinei++)
               clearArgusDevice(ArgusSourceTask);
            setArgusDevice (ArgusSourceTask, optarg);
            break;

         case 'g': {
            struct group *gr;
            if ((gr = getgrnam(optarg)) == NULL)
                ArgusLog (LOG_ERR, "unknown group \"%s\"\n", optarg);
            new_gid = gr->gr_gid;
            endgrent();
            break;
         }

         case 'F': ArgusParseResourceFile (ArgusModel, optarg); break;
         case 'J': setArgusGenerateTime  (ArgusModel, 1); break;
         case 'm': setArgusmflag (ArgusModel, 1); break;
         case 'M': setArgusMarReportInterval (ArgusOutputTask, optarg); break;
         case 'N': {
            char *ptr = NULL;

            if ((ptr = strchr (optarg, '-')) != NULL) {
               char *eptr = ptr + 1;
               ArgusSourceTask->sNflag = strtol(optarg, (char **)&ptr, 10);
               if (ptr == optarg)
                  usage ();
               ArgusSourceTask->eNflag = strtol(eptr, (char **)&ptr, 10);
               if (ptr == eptr)
                  usage ();

            } else {
               ArgusSourceTask->sNflag = 0;
               ArgusSourceTask->eNflag = strtol(optarg, (char **)&ptr, 10);
               if (ptr == optarg)
                  usage ();
            }
            break;
         }

         case 'O': setArgusOflag  (ArgusSourceTask, 0); break;
         case 'P': setArgusPortNum(ArgusOutputTask,atoi (optarg)); break;
         case 'p': setArguspflag (ArgusSourceTask, 1); break;
         case 'R': setArgusResponseStatus (ArgusModel, 1); break;
         case 'r': {
            char Rstr[1024], *Rptr = Rstr;
            do {
               if (*optarg == '"') {
                  if (Rptr[1] != '\0')
                     snprintf (Rstr, 1024, "%s", (&Rptr[1]));

                  while ((Rptr = strchr (Rstr, '"')) == NULL) {
                     if ((optarg = argv[optind]) != NULL) {
                        strncat (Rstr, optarg, (1024 - strlen(Rstr)));
                        optind++;
                     } else
                        break;
                  }
                  optarg = Rstr;
               }
               setArgusrfile  (ArgusSourceTask, optarg);
               if ((optarg = argv[optind]) != NULL)
                  if (*optarg != '-')
                     optind++;
            } while (optarg && (*optarg != '-'));
            break;
         }

         case 'S': setArgusFarReportInterval (ArgusModel, optarg); break;
         case 't': setArgusMoatTshFile (ArgusSourceTask, 1); break;
         case 'T': {
            float value;
            char *nptr = optarg, *endptr;
            
            value = strtod(nptr, &endptr);
            if (nptr != endptr)
               setArgusRealTime (ArgusSourceTask, value);
            else
               usage ();
            break;
         }
         case 'u': {
            struct passwd *pw;
            if ((pw = getpwnam(optarg)) == NULL)
               ArgusLog (LOG_ERR, "unknown user \"%s\"\n", optarg);
            new_uid = pw->pw_uid;
            endpwent();
            break;
         }
         case 'U': 
            setArgusUserDataLen (ArgusModel, atoi (optarg));
            if (getArgusSnapLen(ArgusSourceTask) != ARGUS_MAXSNAPLEN)
               setArgusSnapLen (ArgusSourceTask, atoi(optarg) + ARGUS_MINSNAPLEN);
            break;
         case 'w':
            if (!commandlinew++)
               clearArgusWfile();

            if ((tmparg = optarg) != NULL) {
               if ((*tmparg != '-') || ((*tmparg == '-') &&
                                       (!(strcmp (tmparg, "-"))))) {
                  if (argc == optind)
                     filter = NULL;
                  else {
                     filter = argv[optind];
                     if (*filter == '-') {
                        filter = NULL;
                     } else
                        optind++;
                     }
                  setArgusWfile (tmparg, filter);
                  break;
               }
            }
         case 'X': clearArgusConfiguration (ArgusModel); break;
         case 'Z': setArgusGeneratePacketSize(ArgusModel, 1); break;
         case 'h':
         default:
            usage ();
      }
   }

   setArgusArgv   (ArgusSourceTask, argv);
   setArgusOptind (ArgusSourceTask, optind);

   if (getArgusrfile(ArgusSourceTask) != NULL) {
      if (!(getArgusRealTime(ArgusSourceTask))) {
         setArgusPortNum(ArgusOutputTask, 0);
         setArgusBindAddr(ArgusOutputTask, NULL);
         daemonflag = 0;
      }
   }

   setArgusInterfaceStatus(ArgusSourceTask, 1);

   if (daemonflag) {
      if ((pid = fork ()) < 0) {
         ArgusLog (LOG_ERR, "Can't fork daemon %s", strerror(errno));
      } else {
         if (pid)
            exit (0);
         else {
            ArgusSessionId = setsid();

            if ((freopen ("/dev/null", "w", stdout)) == NULL)
               ArgusLog (LOG_ERR, "Cannot map stdout to /dev/null");

            if ((freopen ("/dev/null", "w", stderr)) == NULL)
               ArgusLog (LOG_ERR, "Cannot map stderr to /dev/null");
         }
      }
   }

#if defined(ARGUS_THREADS)
#if defined(sun)
   thr_setconcurrency (2);
#endif
   if ((status = pthread_attr_init(&attr)) != 0)
      ArgusLog (LOG_ERR, "pthreads init error");
 
#if defined(_POSIX_THREAD_PRIORITY_SCHEDULING) && !defined(sun) && !defined(CYGWIN) && !defined(OpenBSD)
   if ((status = pthread_attr_getschedpolicy(&attr, &thread_policy)) != 0)
      ArgusLog (LOG_ERR, "pthreads get policy error");
   if ((status = pthread_attr_getschedparam(&attr, &thread_param)) != 0)
      ArgusLog (LOG_ERR, "pthreads get sched params error");
   if ((status = pthread_attr_setschedpolicy(&attr, SCHED_RR)) != 0)
      ArgusLog (LOG_ERR, "pthreads set SCHED_RR error");

#if defined(HAVE_SCHED_GET_PRIORITY_MIN)
   if ((rr_min_priority = sched_get_priority_min(SCHED_RR)) == -1)
      ArgusLog (LOG_ERR, "pthreads get priority min error");
   if ((rr_max_priority = sched_get_priority_max(SCHED_RR)) == -1)
      ArgusLog (LOG_ERR, "pthreads get priority max error");
 
   thread_param.sched_priority = (rr_max_priority + rr_min_priority)/2 + 1;
 
   if (thread_param.sched_priority > rr_max_priority)
      thread_param.sched_priority = rr_max_priority;
   if (thread_param.sched_priority < (rr_max_priority - 8))
      thread_param.sched_priority = rr_max_priority - 8;
 
   if ((status = pthread_attr_setschedparam(&attr, &thread_param)) != 0)
      ArgusLog (LOG_ERR, "pthreads set sched param error");
#endif
#else
   pthread_attr_setschedpolicy(&attr, SCHED_RR);
#endif
 
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
#endif

   ArgusInitSource (ArgusSourceTask);
   ArgusInitOutput (ArgusOutputTask);

#if defined(HAVE_SOLARIS)
   sigignore(SIGPIPE);
#else
   (void) signal (SIGPIPE, SIG_IGN);
#endif
 
   (void) signal (SIGHUP,  (void (*)(int)) ArgusScheduleShutDown);
   (void) signal (SIGINT,  (void (*)(int)) ArgusScheduleShutDown);
   (void) signal (SIGTERM, (void (*)(int)) ArgusScheduleShutDown);
   (void) signal (SIGUSR1, (void (*)(int)) ArgusUsr1Sig);
   (void) signal (SIGUSR2, (void (*)(int)) ArgusUsr2Sig);

   if (!(ArgusSourceTask->ArgusReadingOffLine))
      ArgusLog(LOG_WARNING, "started");

   if (daemonflag)
      if (getArguspidflag() && ((ArgusPidFile = ArgusCreatePIDFile (ArgusSourceTask, ArgusPidPath, argv[0])) == NULL))
         ArgusLog (LOG_ERR, "daemon cannot create pid file");

   ArgusInitModeler(ArgusModel);

   ArgusGetPackets(ArgusSourceTask);

#ifdef ARGUSDEBUG
   ArgusDebug (1, "main() ArgusGetPackets returned: shuting down");
#endif 

   ArgusShutDown(0);

#ifdef ARGUS_SYSLOG
   closelog();
#endif
   exit(0);
}


void
ArgusComplete ()
{
/*
#define ARGUSPERFMETRICS		1
*/
#if defined(ARGUSPERFMETRICS)
   long long ArgusTotalPkts = 0, ArgusTotalIPPkts = 0;
   long long ArgusTotalNonIPPkts = 0;
   struct timeval timediff;
   double totaltime;
   int i, len;
   char buf[256];

   long long ArgusTotalNewFlows;
   long long ArgusTotalClosedFlows;
   long long ArgusTotalSends;
   long long ArgusTotalBadSends;
   long long ArgusTotalUpdates;
   long long ArgusTotalCacheHits;

   char *ArgusIntStr[ARGUS_MAXINTERFACE];

   bzero(ArgusIntStr, sizeof(ArgusIntStr));
#endif

#if defined(ARGUSPERFMETRICS)
   for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
      if (ArgusSourceTask->ArgusInterface[i].ArgusDevice != NULL) {
         ArgusTotalPkts      += ArgusSourceTask->ArgusInterface[i].ArgusTotalPkts;
         ArgusTotalIPPkts    += ArgusSourceTask->ArgusInterface[i].ArgusTotalIPPkts;
         ArgusTotalNonIPPkts += ArgusSourceTask->ArgusInterface[i].ArgusTotalNonIPPkts;
      }
   }
   if (ArgusSourceTask->ArgusEndTime.tv_sec == 0)
      gettimeofday (&ArgusSourceTask->ArgusEndTime, 0L);

   if (ArgusSourceTask->ArgusStartTime.tv_sec == 0)
      ArgusSourceTask->ArgusStartTime = ArgusSourceTask->ArgusEndTime;

   bzero(buf, sizeof(buf));

   timediff.tv_sec  = ArgusSourceTask->ArgusEndTime.tv_sec  - ArgusSourceTask->ArgusStartTime.tv_sec;
   timediff.tv_usec = ArgusSourceTask->ArgusEndTime.tv_usec - ArgusSourceTask->ArgusStartTime.tv_usec;
 
   if (timediff.tv_usec < 0) {
      timediff.tv_usec += 1000000;
      timediff.tv_sec--;
   }
 
   totaltime = (double) timediff.tv_sec + (((double) timediff.tv_usec)/1000000.0);

   for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
      char sbuf[MAXSTRLEN];
      if (ArgusSourceTask->ArgusInterface[i].ArgusDevice != NULL) {
         sprintf (sbuf, "%s\n    Total Pkts %8lld  Rate %f\n",
                     ArgusSourceTask->ArgusInterface[i].ArgusDevice->name, ArgusSourceTask->ArgusInterface[i].ArgusTotalPkts,
                     ArgusSourceTask->ArgusInterface[i].ArgusTotalPkts/totaltime);
         ArgusIntStr[i] = strdup(sbuf);
      }
   }

#endif

   ArgusCloseSource (ArgusSourceTask);
   ArgusCloseModeler (ArgusModel);
   ArgusCloseOutput (ArgusOutputTask);

#if defined(ARGUSPERFMETRICS)
   ArgusTotalNewFlows    = ArgusModel->ArgusTotalNewFlows;
   ArgusTotalClosedFlows = ArgusModel->ArgusTotalClosedFlows;
   ArgusTotalSends       = ArgusModel->ArgusTotalSends;
   ArgusTotalBadSends    = ArgusModel->ArgusTotalBadSends;
   ArgusTotalUpdates     = ArgusModel->ArgusTotalUpdates;
   ArgusTotalCacheHits   = ArgusModel->ArgusTotalCacheHits;
#endif

   ArgusFree(ArgusOutputTask);
   ArgusFree(ArgusModel);
   ArgusFree(ArgusSourceTask);

#if defined(ARGUSPERFMETRICS)
   len = strlen(ArgusProgramName);
   for (i = 0; i < len; i++)
      buf[i] = ' ';

   if (ArgusTotalNewFlows > 0) {
      extern int ArgusAllocTotal, ArgusFreeTotal, ArgusAllocMax;

      fprintf (stderr, "%s: Time %d.%06d Flows %-8lld  Closed %-8lld  Sends %-8lld  BSends %-8lld\n",
                         ArgusProgramName, (int)timediff.tv_sec, (int)timediff.tv_usec,
                         ArgusTotalNewFlows,  ArgusTotalClosedFlows,
                         ArgusTotalSends, ArgusTotalBadSends);
      fprintf (stderr, "%*s  Updates %-8lld Cache %-8lld\n", (int)strlen(ArgusProgramName), " ",
                         ArgusTotalUpdates, ArgusTotalCacheHits);
      fprintf (stderr, "%*s  Total Memory %-8d Free %-8d MaxBytes %d\n", (int)strlen(ArgusProgramName), " ",
                         ArgusAllocTotal, ArgusFreeTotal, ArgusAllocMax);
   }
   for (i = 0; i < ARGUS_MAXINTERFACE; i++) {
      if (ArgusIntStr[i] != NULL) {
         fprintf (stderr, "Source: %s\n", ArgusIntStr[i]);
         free(ArgusIntStr[i]);
      }
   }

#endif
}


char *ArgusSignalTable [] = { "Normal Shutdown",
"SIGHUP", "SIGINT", "SIGQUIT", "SIGILL", "SIGTRAP",
"SIGABRT", "SIGBUS", "SIGFPE", "SIGKILL", "SIGUSR1",
"SIGSEGV", "SIGUSR2", "SIGPIPE", "SIGALRM", "SIGTERM",
"SIGSTKFLT", "SIGCHLD", "SIGCONT", "SIGSTOP", "SIGTSTP",
"SIGTTIN", "SIGTTOU", "SIGURG", "SIGXCPU", "SIGXFSZ",
"SIGVTALRM", "SIGPROF", "SIGWINCH", "SIGIO",
};

int ArgusShutDownFlag = 0;

void
ArgusScheduleShutDown (int sig)
{
   ArgusShutDownFlag++;

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusScheduleShutDown(%d)\n", sig);
#endif 
}

void
ArgusShutDown (int sig)
{
   ArgusShutDownFlag++;

   if (sig < 0) {
#ifdef ARGUSDEBUG
      ArgusDebug (1, "ArgusShutDown(ArgusError)\n");
#endif 
      exit(0);
   }

#ifdef ARGUSDEBUG
   if (Argusdflag >= 1)
      fprintf(stderr, "\n");

   ArgusDebug (1, "ArgusShutDown(%s)\n\n", ArgusSignalTable[sig]);
#endif 

   if (!(ArgusShutDownStarted++)) {
      ArgusComplete ();

   } else {
#ifdef ARGUSDEBUG
   ArgusDebug (2, "ArgusShutDown() returning\n");
#endif 
      return;
   }

   if (ArgusPidFile)
      unlink (ArgusPidFile);

   if (daemonflag)
      ArgusLog(LOG_WARNING, "stopped");

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusShutDown()\n");
#endif 

   exit(0);
}


void
setArgusBindAddr (struct ArgusOutputStruct *output, char *value)
{
   if (output->ArgusBindAddr != NULL)
      free(output->ArgusBindAddr);

   if (value != NULL)
      output->ArgusBindAddr = strdup(value);
   else
      output->ArgusBindAddr = NULL;
}

char *
getArgusBindAddr (struct ArgusOutputStruct *output)
{
   return(output->ArgusBindAddr);
}

void
setArguspidflag (int value)
{
   pidflag = value;
}

int
getArguspidflag ()
{
   return (pidflag);
}

#define ARGUS_RCITEMS				33

#define ARGUS_DAEMON				0
#define ARGUS_MONITOR_ID			1
#define ARGUS_ACCESS_PORT			2
#define ARGUS_INTERFACE				3
#define ARGUS_OUTPUT_FILE			4
#define ARGUS_SET_PID 				5
#define ARGUS_PID_PATH				6
#define ARGUS_GO_PROMISCUOUS			7
#define ARGUS_FLOW_STATUS_INTERVAL		8
#define ARGUS_MAR_STATUS_INTERVAL		9
#define ARGUS_CAPTURE_DATA_LEN			10
#define ARGUS_GENERATE_START_RECORDS		11
#define ARGUS_GENERATE_RESPONSE_TIME_DATA	12
#define ARGUS_GENERATE_JITTER_DATA		13
#define ARGUS_GENERATE_MAC_DATA			14
#define ARGUS_DEBUG_LEVEL			15
#define ARGUS_FILTER_OPTIMIZER			16
#define ARGUS_FILTER				17
#define ARGUS_PACKET_CAPTURE_FILE		18
#define ARGUS_BIND_IP				19
#define ARGUS_MIN_SSF				20
#define ARGUS_MAX_SSF				21
#define ARGUS_COLLECTOR				22
#define ARGUS_FLOW_TYPE				23
#define ARGUS_FLOW_KEY				24
#define ARGUS_GENERATE_APPBYTE_METRIC		25
#define ARGUS_CHROOT_DIR			26
#define ARGUS_SETUSER_ID			27
#define ARGUS_SETGROUP_ID			28
#define ARGUS_GENERATE_TCP_PERF_METRIC		29
#define ARGUS_GENERATE_BIDIRECTIONAL_TIMESTAMPS 30
#define ARGUS_GENERATE_PACKET_SIZE		31
#define ARGUS_ENV				32


char *ArgusResourceFileStr [] = {
   "ARGUS_DAEMON=",
   "ARGUS_MONITOR_ID=",
   "ARGUS_ACCESS_PORT=",
   "ARGUS_INTERFACE=",
   "ARGUS_OUTPUT_FILE=",
   "ARGUS_SET_PID=",
   "ARGUS_PID_PATH=",
   "ARGUS_GO_PROMISCUOUS=",
   "ARGUS_FLOW_STATUS_INTERVAL=",
   "ARGUS_MAR_STATUS_INTERVAL=",
   "ARGUS_CAPTURE_DATA_LEN=",
   "ARGUS_GENERATE_START_RECORDS=",
   "ARGUS_GENERATE_RESPONSE_TIME_DATA=",
   "ARGUS_GENERATE_JITTER_DATA=",
   "ARGUS_GENERATE_MAC_DATA=",
   "ARGUS_DEBUG_LEVEL=",
   "ARGUS_FILTER_OPTIMIZER=",
   "ARGUS_FILTER=",
   "ARGUS_PACKET_CAPTURE_FILE=",
   "ARGUS_BIND_IP=",
   "ARGUS_MIN_SSF=",
   "ARGUS_MAX_SSF=",
   "ARGUS_COLLECTOR=",
   "ARGUS_FLOW_TYPE=",
   "ARGUS_FLOW_KEY=",
   "ARGUS_GENERATE_APPBYTE_METRIC=",
   "ARGUS_CHROOT_DIR=",
   "ARGUS_SETUSER_ID=",
   "ARGUS_SETGROUP_ID=",
   "ARGUS_GENERATE_TCP_PERF_METRIC=",
   "ARGUS_GENERATE_BIDIRECTIONAL_TIMESTAMPS=",
   "ARGUS_GENERATE_PACKET_SIZE=",
   "ARGUS_ENV=",
};



extern pcap_dumper_t *ArgusPcapOutFile;
extern char *ArgusWriteOutPacketFile;

void
ArgusParseResourceFile (struct ArgusModelerStruct *model, char *file)
{
   FILE *fd;
   char strbuf[MAXSTRLEN], *str = strbuf, *optarg;
   char result[MAXSTRLEN], *ptr;
   int i, len, done = 0, linenum = 0;
   int interfaces = 0, outputfiles = 0;

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         while ((fgets(str, MAXSTRLEN, fd)) != NULL)  {
            done = 0;
            linenum++;
            while (*str && isspace((int)*str))
                str++;

            if (*str && (*str != '#') && (*str != '\n') && (*str != '!')) {
               for (i = 0; i < ARGUS_RCITEMS && !done; i++) {
                  len = strlen(ArgusResourceFileStr[i]);
                  if (!(strncmp (str, ArgusResourceFileStr[i], len))) {
                     optarg = &str[len];
                     if (*optarg == '\"') optarg++;
                     while (optarg[strlen(optarg) - 1] == '\n')
                        optarg[strlen(optarg) - 1] = '\0';
                     while (optarg[strlen(optarg) - 1] == '\"')
                        optarg[strlen(optarg) - 1] = '\0';

                     switch (i) {
                        case ARGUS_DAEMON: 
                           if (!(strncasecmp(optarg, "yes", 3)))
                              daemonflag = 1;
                           else
                              daemonflag = 0;
                           break;

                        case ARGUS_MONITOR_ID: 
                           if (optarg && (*optarg == '`')) {
                              if (optarg[strlen(optarg) - 1] == '`') {
                                 FILE *fd;

                                 optarg++;
                                 optarg[strlen(optarg) - 1] = '\0';
                                 if (!(strcmp (optarg, "hostname"))) {
                                    if ((fd = popen("hostname", "r")) != NULL) {
                                       ptr = NULL;
                                       clearerr(fd);
                                       while ((ptr == NULL) && !(feof(fd)))
                                          ptr = fgets(result, MAXSTRLEN, fd);

                                       if (ptr == NULL)
                                          ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) `hostname` failed %s.\n", file, strerror(errno));

                                       optarg = ptr;
                                       optarg[strlen(optarg) - 1] = '\0';
                                       pclose(fd);

                                       if ((ptr = strstr(optarg, ".local")) != NULL) {
                                          if (strlen(ptr) == strlen(".local"))
                                             *ptr = '\0';
                                       }

                                    } else
                                       ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) System error: popen() %s\n", file, strerror(errno));
                                 } else
                                    ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) unsupported command `%s` at line %d.\n", file, optarg, linenum);
                              } else
                                 ArgusLog (LOG_ERR, "ArgusParseResourceFile(%s) syntax error line %d\n", file, linenum);
                           }

                           ArgusParseSourceID(ArgusModel, optarg);
                           break;
                           
                        case ARGUS_ACCESS_PORT:
                           setArgusPortNum(ArgusOutputTask, atoi(optarg));
                           break;

                        case ARGUS_OUTPUT_FILE:
                           ptr = NULL;
                           if ((ptr = strchr (optarg, '"')) != NULL)
                              *ptr = '\0';

                           if ((ptr = strchr (optarg, ' ')) != NULL) {
                              *ptr++ = '\0';
                           }
                   
                           if (!outputfiles++)
                              clearArgusWfile();
                           setArgusWfile (optarg, ptr);
                           break;

                        case ARGUS_INTERFACE:
                           if (!interfaces++)
                              clearArgusDevice(ArgusSourceTask);
                           setArgusDevice (ArgusSourceTask, optarg);
                           break;

                        case ARGUS_SET_PID:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArguspidflag  (1);
                           else
                              setArguspidflag  (0);
                           break;

                        case ARGUS_PID_PATH: {
                           ArgusPidPath = strdup(optarg);
                           break;
                        }

                        case ARGUS_GO_PROMISCUOUS:
                           if ((strncasecmp(optarg, "yes", 3)))
                              setArguspflag  (ArgusSourceTask, 1);
                           else
                              setArguspflag  (ArgusSourceTask, 0);
                           break;

                        case ARGUS_FLOW_STATUS_INTERVAL:
                           setArgusFarReportInterval (model, optarg);
                           break;

                        case ARGUS_MAR_STATUS_INTERVAL:
                           setArgusMarReportInterval (ArgusOutputTask, optarg);
                           break;

                        case ARGUS_CAPTURE_DATA_LEN:
                           setArgusUserDataLen (model, atoi(optarg));
                           if (getArgusSnapLen(ArgusSourceTask) != ARGUS_MAXSNAPLEN)
                              setArgusSnapLen (ArgusSourceTask, atoi(optarg) + ARGUS_MINSNAPLEN);
                           break;

                        case ARGUS_GENERATE_START_RECORDS: {
                           extern int ArgusGenerateStartRecords;

                           if ((!strncasecmp(optarg, "yes", 3)))
                              ArgusGenerateStartRecords++;
                           else
                              ArgusGenerateStartRecords = 0;
                           break;
                        }

                        case ARGUS_GENERATE_RESPONSE_TIME_DATA:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusResponseStatus  (model, 1);
                           else
                              setArgusResponseStatus  (model, 0);
                           break;

                        case ARGUS_GENERATE_JITTER_DATA:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusGenerateTime  (model, 1);
                           else
                              setArgusGenerateTime  (model, 0);
                           break;

                        case ARGUS_GENERATE_MAC_DATA:
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusmflag (model, 1);
                           else
                              setArgusmflag (model, 0);
                           break;

                        case ARGUS_DEBUG_LEVEL:
                           setArgusdflag (model, atoi(optarg));
                           break;
                        
                        case ARGUS_FILTER_OPTIMIZER:
                           if ((strncasecmp(optarg, "yes", 3)))
                              setArgusOflag  (ArgusSourceTask, 1);
                           else
                              setArgusOflag  (ArgusSourceTask, 0);
                           break;

                        case ARGUS_FILTER:
                           if ((ArgusSourceTask->ArgusInputFilter = ArgusCalloc (1, MAXSTRLEN)) != NULL) {
                              ptr = ArgusSourceTask->ArgusInputFilter;
                              str = optarg;
                              while (*str) {
                                 if ((*str == '\\') && (str[1] == '\n')) {
                                    fgets(str, MAXSTRLEN, fd);
                                    while (*str && (isspace((int)*str) && (str[1] && isspace((int)str[1]))))
                                       str++;
                                 }
                                 
                                 if ((*str != '\n') && (*str != '"'))
                                    *ptr++ = *str++;
                                 else
                                    str++;
                              }
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusParseResourceFile: ArgusFilter \"%s\" \n", ArgusSourceTask->ArgusInputFilter);
#endif 
                           }
                           break;

                        case ARGUS_PACKET_CAPTURE_FILE:
                           if (*optarg != '\0')
                              setArgusWriteOutPacketFile (ArgusSourceTask, optarg);
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusParseResourceFile: ArgusPacketCaptureFile \"%s\" \n", ArgusSourceTask->ArgusWriteOutPacketFile);
#endif 
                           break;

                        case ARGUS_BIND_IP:
                           if (*optarg != '\0')
                              setArgusBindAddr (ArgusOutputTask, optarg);
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusParseResourceFile: ArgusBindAddr \"%s\" \n", ArgusBindAddr);
#endif 
                           break;

                        case ARGUS_MIN_SSF:
                           if (*optarg != '\0') {
#ifdef ARGUS_SASL
                              ArgusMinSsf = atoi(optarg);
#ifdef ARGUSDEBUG
                           ArgusDebug (1, "ArgusParseResourceFile: ArgusMinSsf \"%s\" \n", ArgusMinSsf);
#endif 
#endif 
                           }
                           break;

                        case ARGUS_MAX_SSF:
                           if (*optarg != '\0') {
#ifdef ARGUS_SASL
                              ArgusMaxSsf = atoi(optarg);
#ifdef ARGUSDEBUG
                              ArgusDebug (1, "ArgusParseResourceFile: ArgusMaxSsf \"%s\" \n", ArgusMaxSsf);
#endif 
#endif 
                           }
                           break;

                        case ARGUS_COLLECTOR:
                           break;

                        case ARGUS_FLOW_TYPE:
                           if (!(strncasecmp(optarg, "Uni", 3)))
                              setArgusFlowType (model, ARGUS_UNIDIRECTIONAL);
                           else
                           if (!(strncasecmp(optarg, "Bi", 2)))
                              setArgusFlowType (model, ARGUS_BIDIRECTIONAL);
                           break;

                        case ARGUS_FLOW_KEY: {
                           char *tok = NULL;

                           while ((tok = strtok(optarg, " +\t")) != NULL) {
                              if (!(strncasecmp(tok, "CLASSIC_5_TUPLE", 14)))
                                 setArgusFlowKey (model, ARGUS_FLOW_KEY_CLASSIC5TUPLE);
                              else
                              if (!(strncasecmp(tok, "LAYER_2_MATRIX", 14)))
                                 setArgusFlowKey (model, ARGUS_FLOW_KEY_LAYER_2_MATRIX);
                              else
                              if (!(strncasecmp(tok, "LAYER_3_MATRIX", 14)))
                                 setArgusFlowKey (model, ARGUS_FLOW_KEY_LAYER_3_MATRIX);
                              else
                              if (!(strncasecmp(tok, "LOCAL_MPLS", 10)))
                                 setArgusFlowKey (model, ARGUS_FLOW_KEY_LOCAL_MPLS);
                              else
                              if (!(strncasecmp(tok, "COMPLETE_MPLS", 10)))
                                 setArgusFlowKey (model, ARGUS_FLOW_KEY_COMPLETE_MPLS);
                              else
                              if (!(strncasecmp(tok, "VLAN", 4)))
                                 setArgusFlowKey (model, ARGUS_FLOW_KEY_VLAN);

                              optarg = NULL;
                           }
                           break;
                        }

                        case ARGUS_GENERATE_TCP_PERF_METRIC: {
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusTCPflag(model, 1);
                           else
                              setArgusTCPflag(model, 0);
                           break;
                        }

                        case ARGUS_GENERATE_BIDIRECTIONAL_TIMESTAMPS: {
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusTimeReport(model, 1);
                           else
                              setArgusTimeReport(model, 0);
                           break;
                        }

                        case ARGUS_GENERATE_APPBYTE_METRIC: {
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusAflag(model, 1);
                           else
                              setArgusAflag(model, 0);
                           break;
                        }

                        case ARGUS_GENERATE_PACKET_SIZE: {
                           if (!(strncasecmp(optarg, "yes", 3)))
                              setArgusGeneratePacketSize(model, 1);
                           else
                              setArgusGeneratePacketSize(model, 0);
                           break;
                        }

                        case ARGUS_CHROOT_DIR: {
                           if (chroot_dir != NULL)
                              free(chroot_dir);
                           chroot_dir = strdup(optarg);
                           break;
                        }
                        case ARGUS_SETUSER_ID: {
                           struct passwd *pw;
                           if ((pw = getpwnam(optarg)) == NULL)
                              ArgusLog (LOG_ERR, "unknown user \"%s\"\n", optarg);
                           new_uid = pw->pw_uid;
                           endpwent();
                           break;
                        }
                        case ARGUS_SETGROUP_ID: {
                           struct group *gr;
                           if ((gr = getgrnam(optarg)) == NULL)
                               ArgusLog (LOG_ERR, "unknown group \"%s\"\n", optarg);
                           new_gid = gr->gr_gid;
                           endgrent();
                           break;
                        }
                        case ARGUS_ENV: {
                           if (putenv(optarg))
                              ArgusLog (LOG_ERR, "Argus set env \"%s\" error %s\n", optarg, strerror(errno));
                           break;
                        }
                     }

                     done = 1;
                     break;
                  }
               }
            }
         }
      } else {
#ifdef ARGUSDEBUG
         ArgusDebug (1, "ArgusParseResourceFile: open %s %s\n", file, strerror(errno));
#endif 
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "ArgusParseResourceFile (%s) returning\n", file);
#endif 

}

void
clearArgusConfiguration (struct ArgusModelerStruct *model)
{
   daemonflag = 0;
   setArgusID (model, 0);
   setArgusIDType (model, 0);
   clearArgusWfile ();
   clearArgusDevice (ArgusSourceTask);
   setArgusPortNum(ArgusOutputTask, 0);
   setArgusBindAddr (ArgusOutputTask, NULL);
   setArguspidflag  (0);
   setArguspflag  (ArgusSourceTask, 0);
   setArgusFarReportInterval (model, ARGUS_FARSTATUSTIMER);
   setArgusMarReportInterval (ArgusOutputTask, ARGUS_MARSTATUSTIMER);
   setArgusUserDataLen (model, 0);
   setArgusSnapLen (ArgusSourceTask, ARGUS_MINSNAPLEN);
   setArgusResponseStatus (model, 0);
   setArgusGenerateTime (model, 0);
   setArgusmflag (model, 0);
   setArgusOflag (ArgusSourceTask, 1);
   setArgusAflag(model, 0);
   setArgusTimeReport(model, 0);

   if (ArgusSourceTask->ArgusWriteOutPacketFile) {
      if (ArgusSourceTask->ArgusWriteOutPacketFile) {
         if (ArgusSourceTask->ArgusPcapOutFile != NULL) {
            pcap_dump_close(ArgusSourceTask->ArgusPcapOutFile);
            ArgusSourceTask->ArgusPcapOutFile = NULL;
         }
         ArgusSourceTask->ArgusWriteOutPacketFile = NULL;
      }
   }

   if (ArgusSourceTask->ArgusInputFilter) {
      ArgusFree(ArgusSourceTask->ArgusInputFilter);
      ArgusSourceTask->ArgusInputFilter = NULL;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "clearArgusConfiguration () returning\n");
#endif 
}

int
getArgusPortNum(struct ArgusOutputStruct *output)
{
   return(output->ArgusPortNum);
}
 
void
setArgusPortNum(struct ArgusOutputStruct *output, int value)
{
   output->ArgusPortNum = value;
 
#ifdef ARGUSDEBUG
   ArgusDebug (2, "setArgusPortNum(%d) returning\n", value);
#endif
}


void
clearArgusWfile(void)
{
   ArgusDeleteList (ArgusOutputTask->ArgusWfileList, ARGUS_WFILE_LIST);
   ArgusOutputTask->ArgusWfileList = NULL;
}

void
setArgusWfile(char *file, char *filter)
{
   struct ArgusWfileStruct *wfile = NULL;

   if (ArgusOutputTask->ArgusWfileList == NULL)
      ArgusOutputTask->ArgusWfileList = ArgusNewList();

   if (file) {
      if ((wfile = (struct ArgusWfileStruct *) ArgusCalloc (1, sizeof (*wfile))) != NULL) {
         wfile->filename = strdup(file);
         if (filter)
            wfile->filter = strdup(filter);
         ArgusPushFrontList(ArgusOutputTask->ArgusWfileList, (struct ArgusListRecord *) wfile, ARGUS_LOCK);

      } else
         ArgusLog (LOG_ERR, "setArgusWfile, ArgusCalloc %s\n", strerror(errno));
   } else
      ArgusLog (LOG_ERR, "setArgusWfile, file is null\n");
}

 
void
ArgusUsr1Sig (int sig)
{
#ifdef ARGUSDEBUG
   Argusdflag = (Argusdflag++ > 30) ? 30 : Argusdflag;
 
   ArgusDebug (0, "ArgusUsr1Sig: debug %d enabled\n", Argusdflag);
#endif
}
 
void
ArgusUsr2Sig (int sig)
{
#ifdef ARGUSDEBUG
   Argusdflag = 0;
 
   ArgusDebug (0, "ArgusUsr2Sig: debug disabled\n");
#endif
}
 
int
getArgusControlMonitor (struct ArgusModelerStruct *model)
{
   return (model->ArgusControlMonitor);
}

void
setArgusControlMonitor (struct ArgusModelerStruct *model)
{
   model->ArgusControlMonitor++;
}

