/*
 * Argus Software
 * Copyright (c) 2000-2008 QoSient, LLC
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * rahisto.c  - histogram tracking.
 *       
 * written by Carter Bullard
 * QoSient, LLC
 */


#if defined(CYGWIN)
#define USE_IPV6
#endif

#include <unistd.h>
#include <stdlib.h>

#include <compat.h>

#include <argus_util.h>
#include <argus_client.h>
#include <argus_main.h>
#include <argus_filter.h>
 
#include <rabins.h>
#include <rasplit.h>
#include <argus_sort.h>
#include <argus_cluster.h>
 
#include <signal.h>
#include <ctype.h>
#include <math.h>
 
struct RaBinProcessStruct *RaBinProcess = NULL;
int ArgusProcessOutLayers = 0;
int ArgusProcessNoZero = 0;


void
ArgusClientInit (struct ArgusParserStruct *parser)
{
   struct ArgusModeStruct *mode = NULL;
   parser->RaWriteOut = 0;
 
   if (!(parser->RaInitialized)) {
      (void) signal (SIGHUP,  (void (*)(int)) RaParseComplete);
      (void) signal (SIGTERM, (void (*)(int)) RaParseComplete);
      (void) signal (SIGQUIT, (void (*)(int)) RaParseComplete);
      (void) signal (SIGINT,  (void (*)(int)) RaParseComplete);
 
      if ((mode = parser->ArgusModeList) != NULL) {
         while (mode) {
            if (!(strncasecmp (mode->mode, "nozero", 6)))
               ArgusProcessNoZero = 1;
            if (!(strncasecmp (mode->mode, "outlayer", 8)))
               ArgusProcessOutLayers = 1;

            mode = mode->nxt;
         }
      }

      if (ArgusParser->RaSOptionStrings[0] == NULL) {
         int i = 0;
         while (parser->RaPrintAlgorithmList[i] != NULL) {
           ArgusFree(parser->RaPrintAlgorithmList[i]);
           parser->RaPrintAlgorithmList[i] = NULL;
           i++;
         }
      }
 
      if ((parser->ArgusAggregator = ArgusNewAggregator(parser, NULL)) == NULL)
         ArgusLog (LOG_ERR, "ArgusClientInit: ArgusNewAggregator error");
 
      if (parser->Hstr)
         if (!(ArgusHistoMetricParse (parser, parser->ArgusAggregator)))
            usage ();
 
      if (parser->vflag)
         ArgusReverseSortDir++;
 
      parser->RaInitialized++;
   }
}


void RaArgusInputComplete (struct ArgusInput *input) { return; }


void
RaParseComplete (int sig)
{
   struct ArgusParserStruct *parser = ArgusParser;
   struct ArgusRecordStruct *ns = NULL;
   struct ArgusAgrStruct *agr = NULL;
   int i, freq, cum = 0, class = 0, start = 999999999, end = 0;
   float rel, relcum = 0.0;
   double bs, be, bf;

   if (sig >= 0) {
      if ((!parser->RaParseCompleting++) && parser->RaHistoRecords) {
         for (i = 0; i < parser->RaHistoBins + 2; i++) {
            struct ArgusRecordStruct *argus = parser->RaHistoRecords[i];
            if ((!ArgusProcessOutLayers && ((i > 0) && (i <= parser->RaHistoBins))) || ArgusProcessOutLayers) {
               if (argus) {
                  if (i < start) start = i;
                  if (i > end)   end   = i;
                  if (ns == NULL)
                     ns = ArgusCopyRecordStruct (argus);
                  else
                     ArgusMergeRecords (parser->ArgusAggregator, ns, argus);
               }
            }
         }

         if (ns != NULL) {
            double start, bsize;
            char buf[MAXSTRLEN];
            agr = &ns->canon.agr;

            if (parser->ArgusWfileList == NULL) {
               printf (" N = %-d  mean = %-.*f  stddev = %-.*f  max = %-.*f  min %-.*f\n",
                                agr->act.n, parser->pflag, agr->act.meanval, 
                                            parser->pflag, agr->act.stdev, 
                                            parser->pflag, agr->act.maxval, 
                                            parser->pflag, agr->act.minval);
            }

            if (parser->RaHistoMetricLog) {
               start = parser->RaHistoStartLog;
            } else {
               start = parser->RaHistoStart;
            }
            bsize = parser->RaHistoBinSize;

            for (i = 0; i < parser->RaHistoBins + 2; i++) {
               struct ArgusRecordStruct *argus = parser->RaHistoRecords[i];

               if (i == 0) {
                  bs = 0;
                  be = start;
                  if (parser->RaHistoMetricLog) 
                     be = pow(10.0, be);

               } else {
                  bs = (start + ((i - 1) * bsize));
                  if (i > parser->RaHistoBins) {
                     be = bs;
                  } else {
                     be = (start +  (i * bsize));
                  }
                  if (parser->RaHistoMetricLog) {
                     bs = pow(10.0, bs);
                     be = pow(10.0, be);
                  }
               }

               if ((!ArgusProcessOutLayers && ((i > 0) && (i <= parser->RaHistoBins))) || ArgusProcessOutLayers) {
                  if (!ArgusProcessNoZero || (ArgusProcessNoZero && ((i >= start) && (i <= end)))) {
                     if (parser->ArgusWfileList != NULL) {
                        if (argus) {
                           struct ArgusWfileStruct *wfile = NULL;
                           struct ArgusListObjectStruct *lobj = NULL; 
                           int i, count = parser->ArgusWfileList->count; 
                           double value, frac;

                           frac = modf(bs, &value);
                           argus->canon.time.src.start.tv_sec  = value;
                           argus->canon.time.src.start.tv_usec = frac * 1000000;

                           frac = modf(be, &value);
                           argus->canon.time.src.end.tv_sec    = value;
                           argus->canon.time.src.end.tv_usec   = frac * 1000000;
                           argus->canon.time.hdr.subtype       = ARGUS_TIME_RELATIVE_TIMESTAMP;
 
                           if ((lobj = parser->ArgusWfileList->start) != NULL) {
                              for (i = 0; i < count; i++) {
                                 if ((wfile = (struct ArgusWfileStruct *) lobj) != NULL) {
                                    int pass = 1;
                                    if (wfile->filterstr) {
                                       struct nff_insn *wfcode = wfile->filter.bf_insns;
                                       pass = ArgusFilterRecord (wfcode, argus);
                                    }

                                    if (pass != 0) {
                                       if ((parser->exceptfile == NULL) || strcmp(wfile->filename, parser->exceptfile)) {
                                          struct ArgusRecord *argusrec = NULL;
                                          static char sbuf[0x10000];
                                          if ((argusrec = ArgusGenerateRecord (argus, 0L, sbuf)) != NULL) {
#ifdef _LITTLE_ENDIAN
                                             ArgusHtoN(argusrec);
#endif
                                             ArgusWriteNewLogfile (parser, argus->input, wfile, argusrec);
                                          }
                                       }
                                    }
                                 }

                                 lobj = lobj->nxt;
                              }
                           }
                        }

                     } else {
                        int size = parser->pflag;

                        bzero(buf, MAXSTRLEN);
                        if (argus != NULL) {
                           if (ArgusParser->RaLabel == NULL) {
                              char rangebuf[128];
                              int rblen = 0;

                              sprintf (rangebuf, "%*.*e-%*.*e ", size, size, be, size, size, be);
                              rblen = (strlen(rangebuf) - strlen("Interval"))/2;

                              ArgusParser->RaLabel = ArgusGenerateLabel(ArgusParser, argus);
                              printf (" Class  %*.*s%s%*.*s     Freq    Rel.Freq     Cum.Freq    %s\n",
                                       rblen, rblen, " ", "Interval", rblen, rblen, " ", ArgusParser->RaLabel);
                           }

                           freq =  argus->canon.agr.count;
                           rel  = (argus->canon.agr.count * 1.0)/(ns->canon.agr.count * 1.0);
                           ArgusPrintRecord (parser, buf, argus, MAXSTRLEN);

                        } else {
                           freq = 0;
                           rel  = 0.0;
                        }

                        cum    += freq;
                        relcum += rel;

                        if (i > parser->RaHistoBins) {
                           if (argus && (argus->canon.agr.count > 0)) {
                              bf = be;
                              do {
                                 if (parser->RaHistoMetricLog)
                                    bf += pow(10.0, parser->RaHistoBinSize);
                                 else
                                    bf += parser->RaHistoBinSize;
                              } while (!(bf >= agr->act.maxval));

                              printf ("%6d*  %*.*e-%*.*e %8d   %8.4f%%    %8.4f%%    %s\n",
                                         class++, size, size, be, size, size, bf, freq, rel * 100.0, relcum * 100.0, buf);
                           }

                        } else {
                           if (!ArgusProcessNoZero || (ArgusProcessNoZero && (argus && (argus->canon.agr.count > 0)))) {
                              if (i == 0) {
                                 if (be != bs)
                                    printf ("%6d*  %*.*e-%*.*e %8d   %8.4f%%    %8.4f%%    %s\n",
                                            class++, size, size, bs, size, size, be, freq, rel * 100.0, relcum * 100.0, buf);
                              } else
                                 printf ("%6d   %*.*e-%*.*e %8d   %8.4f%%    %8.4f%%    %s\n",
                                            class++, size, size, bs, size, size, be, freq, rel * 100.0, relcum * 100.0, buf);
                           }
                        }
                     }
                  }
               }
            }
         }
      }

      if ((sig == SIGINT) || (sig == SIGQUIT))
         exit(0);
   }
}


void
ArgusClientTimeout ()
{
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout()\n");
#endif
}

void
parse_arg (int argc, char**argv)
{}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rahisto Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusParser->ArgusProgramName);
   fprintf (stderr, "usage: %s [raoptions] -H metric bins[L]:[range | size]\n", ArgusParser->ArgusProgramName);

   fprintf (stderr, "options: -H metric bins[L]:[range | size] \n");
   fprintf (stderr, "            metric - any metric in argus data record\n");
   fprintf (stderr, "              bins - number of bins to use in histogram\n");
   fprintf (stderr, "               [L] - optionally specify logorithmic bin sizes\n");
   fprintf (stderr, "             range - minimum and maxium values for histogram bins\n");
   fprintf (stderr, "                syntax:  value-value\n");
   fprintf (stderr, "                         value = %%f[umsMHD] | %%f[umKMG] depending on metric type\n");
   fprintf (stderr, "              size - single numeric for size of each bin\n");
   fprintf (stderr, "         -M [nozero | outlayer]\n");
   fprintf (stderr, "             nozero - don't print bins that have zero frequency\n");
   fprintf (stderr, "           outlayer - accumlate bins that are outside bin range\n");

#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>         specify debug level\n");
#endif
   exit(1);
}


void
RaProcessRecord (struct ArgusParserStruct *parser, struct ArgusRecordStruct *argus)
{
   if (argus->hdr.type & ARGUS_MAR) {
      
   } else {
      struct ArgusAggregatorStruct *agg = parser->ArgusAggregator;
      if (agg && (agg->RaMetricFetchAlgorithm != ArgusFetchTransactions)) {
         struct ArgusAgrStruct *agr = NULL;
         if ((agr = (struct ArgusAgrStruct *) argus->dsrs[ARGUS_AGR_INDEX]) != NULL) {
            agr->count = 1;
         }
         argus->dsrs[ARGUS_AGR_INDEX] = NULL;
      }

      if (agg && agg->RaMetricFetchAlgorithm != NULL) {
         ArgusHistoTallyMetric (parser, argus);
      }
   }
}

int RaSendArgusRecord(struct ArgusRecordStruct *argus) {return 0;}

void ArgusWindowClose(void);

void ArgusWindowClose(void) { 
#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusWindowClose () returning\n"); 
#endif
}
