// -*- C++ -*- (c) 2007, 2008 Petr Rockai <me@mornfall.net>

#include <ept/core/apt/action.h>
#include <ept/core/desktop.h>

#include <adept/util.h>
#include <adept/packagedata.h>
#include <adept/tokenmodel.h>
#include <adept/packagelist.h>

#include <adept/sidebar.h>

#include <KLineEdit>
#include <KIcon>
#include <KLocale>
#include <KVBox>
#include <KPageWidget>

#ifndef ADEPT_APPLICATIONSWIDGET_H
#define ADEPT_APPLICATIONSWIDGET_H

namespace adept {

using namespace ept::core;
namespace str = wibble::str;

struct AppGroupPolicy : desktop::GroupPolicy {
    QString u8group( const desktop::Entry &e ) {
        /* for ( Range< std::string > i = r; i != i.end(); ++i ) {
            if ( i->find( "X-KDE-settings" ) != std::string::npos )
                return "Settings";
            if ( i->find( "X-KDE-information" ) != std::string::npos )
                return "Settings";
                } */
        if ( e.inCategory( "Development" ) ) return i18n( "Development" );
        if ( !e.inCategory( "Education" ) ) {
            if ( e.inCategory( "Astronomy" )
                 || e.inCategory( "Biology" )
                 || e.inCategory( "Chemistry" )
                 || e.inCategory( "Geology" )
                 || e.inCategory( "MedicalSoftware" )
                 || e.inCategory( "Physics" )
                 || e.inCategory( "Math" )
                 || e.inCategory( "Science" ) )
                return i18n( "Science" );
        } else {
            return i18n( "Edutainment" );
        }
        if ( e.inCategory( "Game" ) ) return i18n( "Games" );
        if ( e.inCategory( "Graphics" ) ) return i18n( "Graphics" );
        if ( e.inCategory( "Network" ) ) return i18n( "Internet" );
        if ( e.inCategory( "AudioVideo" ) ) return i18n( "Multimedia" );
        if ( e.inCategory( "Office" ) ) return i18n( "Office" );
        if ( e.inCategory( "Settings" ) ) return i18n( "Settings" );
        if ( e.inCategory( "System" ) ) return i18n( "System" );
        if ( e.inCategory( "Utility" ) ) {
            if ( e.inCategory( "Accessibility" ) )
                return i18n( "Accessibility" );
            return i18n( "Utilities" );
        }
        return i18n( "Others" );
    }

    std::string group( const desktop::Entry &e ) {
        return s8( u8group( e ) );
    }

    virtual ~AppGroupPolicy() {}
};

struct IconPolicy {
    virtual QString iconForGroup( QString g ) const {
        if ( g == u8( "Settings" ) ) return u8( "preferences-system" );
        if ( g == u8( "Edutainment" ) ) return u8( "applications-accessories" );
        if ( g == u8( "Accessibility" ) ) return u8( "preferences-desktop-accessibility" );
        if ( g == u8( "Others" ) ) return u8( "applications-other" );
        return QString( "applications-" ) + g.toLower();
    }
    virtual ~IconPolicy() {}
};

class ApplicationsWidget : public KVBox
{
    Q_OBJECT

    KLineEdit *m_searchLine;
    KHBox *hbox;
    SidebarListWidget *m_groups;
    PackageListView *m_list;

    PackageData &d;
    typedef std::map< std::string, TokenModel * > Models;
    Models all, visible;
    std::string empty, current;

    std::set< ept::Token > m_xapianSet;
    Validator validator;

    AppGroupPolicy groupPol;
    IconPolicy iconPol;

public:

    struct Matches {
        ApplicationsWidget *m_apps;
        Matches( ApplicationsWidget *a ) : m_apps( a ) {}
        bool operator()( ept::Token t, std::string n ) const {
            std::string m = str::tolower( s8( m_apps->m_searchLine->text() ) );
            std::string name = str::tolower(
                m_apps->d.desk.get< desktop::Name >( t ) );
            n = str::tolower( n );
            if ( n.find( m ) != std::string::npos )
                return true;
            if ( name.find( m ) != std::string::npos )
                return true;
            ept::Token pkg = m_apps->d.desk.get< desktop::Package >( t );
            if ( m_apps->m_xapianSet.count( pkg ) )
                return true;
            return false;
        }
    };

    ept::Token validate( ept::Token t ) {
        ept::Token pkg = d.desk.get< desktop::Package >( t );
        if ( validator.validate( pkg ).valid() )
            return t;
        return ept::Token();
    }

    TokenModel *model( std::string g ) {
        if ( all.find( g ) == all.end() ) {
            TokenModel *m = new TokenModel();
            all.insert( std::make_pair( g, m ) );
        }
        return all.find( g )->second;
    }

    template< typename L >
    void fillList( L l ) {
        ProcessEvents::restart();
        QApplication::setOverrideCursor( QCursor( Qt::BusyCursor ) );

        for ( Models::iterator i = all.begin(); i != all.end(); ++i )
            i->second->rewind();

        visible.clear();

        while ( !l.empty() ) {
            ProcessEvents::check( this );

            ept::Token t = validate( l.head().token() );

            if ( !t.valid() ) {
                l = l.tail();
                continue;
            }

            std::string g = l.head().template get< desktop::Group >();
            l = l.tail();

            model( g )->appendToken( t );

            if ( visible.find( g ) == visible.end() ) {
                visible[ g ] = model( g );
                // visible.insert( visible.begin(), model( g ) );
                updateVisible();
            }
        }

        setUpdatesEnabled( true );

        for ( Models::iterator i = all.begin(); i != all.end(); ++i )
            i->second->end();
        QApplication::restoreOverrideCursor();

        if ( !visible.count( current ) || current == empty ) {
            if ( !visible.empty() )
                current = visible.begin()->first;
            else
                current = empty;
        }

        QTimer::singleShot( 0, this, SLOT( updateVisible() ) );
    }

public Q_SLOTS:
    void updateVisible() {
        bool nocurrent = false;
        if ( visible.empty() ) {
            visible[ current ] = model( current );
        } else {
            if ( ! (visible.size() == 1 && visible.begin()->first == empty) )
                visible.erase( empty );
        }
        if ( !visible.count( current ) ) {
            visible[ current ] = model( current );
            nocurrent = true;
        }
        m_groups->clear();
        std::cerr << "-------------- (current = " << current << ") --"
                  << std::endl;
        for ( Models::iterator i = visible.begin(); i != visible.end(); ++i ) {
            std::cerr << "Adding " << i->first << std::endl;
            QIcon icon = QIcon( KIcon(
                                    iconPol.iconForGroup( u8( i->first ) ) ) );
            SidebarItem *it =
                new SidebarItem( 0, icon, u8( i->first ) );
            m_groups->addItem( it );
            if ( i->first == current ) {
                m_groups->setCurrentItem( it );
            }
        }
        if ( nocurrent )
            visible.erase( current );
        updateWidths();
        updateModel();
    }

public:
    void updateWidths() {
        int w = 0;
        for ( int i = 0; i < m_groups->count(); ++i )
        {
            QSize s = m_groups->sizeHintForIndex(
                m_groups->model()->index( i, 0 ) );
            if ( s.width() > w )
                w = s.width();
        }
        m_groups->setFixedWidth( w + 30 );
    }

    ApplicationsWidget( PackageData &_d, QWidget *p = 0 )
        : KVBox( p ), d( _d ),
          empty( s8( i18n( "No matches found" ) ) ),
          validator( _d )
    {
        KGlobal::dirs()->addResourceDir(
            "desktopicon",
            u8( "/usr/share/app-install/icons" ) );
        d.desk.setGroupPolicy( &groupPol );
        m_searchLine = new KLineEdit( this );
        m_searchLine->setClearButtonShown( true );
        m_searchLine->setClickMessage( i18n( "Search applications" ) );
        hbox = new KHBox( this );
        m_groups = new SidebarListWidget( hbox );

        // FUNK
        SidebarDelegate *deleg = new SidebarDelegate( m_groups );
        deleg->setShowText( true ); // TODO: SETTINGS
        m_groups->setItemDelegate( deleg );
        m_groups->setIconSize( QSize( 48, 48 ) );
        m_groups->setUniformItemSizes( true );
        // END FUNK

        m_list = setupList( this, hbox, model( empty ), d );

        connect( m_groups, SIGNAL( itemClicked( QListWidgetItem* ) ),
                 this, SLOT( selectGroup( QListWidgetItem* ) ) );
        connect( m_searchLine, SIGNAL( returnPressed() ),
                 this, SLOT( search() ) );

        current = empty;
        updateVisible();
    }

Q_SIGNALS:
    void refreshSig();

public Q_SLOTS:
    void updateModel() {
        std::cerr << "updateModel: '" << current << "' -> " <<
            (void*) model( current ) << std::endl;
        m_list->setModel( model( empty ) ); // work around something that looks
                                            // like a QT bug to me
        m_list->setModel( model( current ) );
    }

    void selectGroup( QListWidgetItem *i ) {
        current = s8( i->text() );
        updateModel();
    }

    void search() {
        m_xapianSet.clear();
        list::output(
            list::map(
                list::take( 50, d.xap.query( s8( m_searchLine->text() ) ).results() ),
                std::mem_fun_ref( &xapian::List::token ) ),
            std::inserter( m_xapianSet, m_xapianSet.begin() ) );
        fillList( d.desk.propertyFilter< desktop::ShortDescription >(
                      Matches( this ) ) );
    }

    void delayed() {
        search();
    }

    void refresh() {
        refreshSig();
    }
};

}

#endif
