// -*- C++ -*- (c) 2008 Petr Rockai <me@mornfall.net>

#include <sys/socket.h>
#include <sys/un.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <string>
#include <iostream>
#include <map>
#include <set>
#include <deque>

#include <wibble/exception.h>
#include <wibble/test.h> // for assert
#include <wibble/regexp.h>

#ifndef ADEPT_DEBCONF_H
#define ADEPT_DEBCONF_H

namespace adept {

namespace wexcept = wibble::exception;

struct UnixSocket {
    struct sockaddr_un addr;
    int sock;
    std::string path;

    UnixSocket( std::string p, bool rem = false )
        : path( p )
    {
        sock = socket( PF_UNIX, SOCK_STREAM, 0 );
        if ( !sock )
            throw wexcept::System( "creating unix socket for " + path );
        fcntl( sock, F_SETFL, O_NONBLOCK );

        memset( &addr, 0, sizeof(struct sockaddr_un) );
        addr.sun_family = AF_UNIX;
        assert( path.length() < sizeof( addr.sun_path ) );
        strcpy( addr.sun_path, path.c_str() );

        if ( rem && unlink( path.c_str() ) ) {
            if ( errno != ENOENT )
                throw wexcept::System( "unlinking " + path );
        }

        if ( bind( sock, (sockaddr *) &addr, sizeof( struct sockaddr_un ) ) )
            throw wexcept::System( "binding unix socket to " + path );
    }

    ~UnixSocket() {
        // probably better not throw here...
        unlink( path.c_str() );
    }

    void listen( int back = 1 ) {
        if ( ::listen( sock, back ) )
            throw wexcept::System( "listening on unix socket " + path );
    }

    int accept() {
        int fd = ::accept( sock, NULL, NULL );
        /* if ( fd == -1 )
           throw wexcept::System( "accept on unix socket " + path );
        */
        return fd;
    }
};

struct Pipe {
    typedef std::deque< char > Buffer;
    Buffer buffer;
    int fd;
    bool _eof;

    Pipe( int p ) : fd( p ), _eof( false )
    {
        if ( p == -1 )
            return;
        if ( fcntl( fd, F_SETFL, O_NONBLOCK ) == -1 )
            throw wexcept::System( "fcntl on a pipe" );
    }
    Pipe() : fd( -1 ), _eof( false ) {}

    void write( std::string what ) {
        ::write( fd, what.c_str(), what.length() );
    }

    bool active() {
        return fd != -1 && !_eof;
    }

    bool eof() {
        return _eof;
    }

    int readMore() {
        char _buffer[1024];
        int r = ::read( fd, _buffer, 1023 );
        if ( r == -1 && errno != EAGAIN )
            throw wexcept::System( "reading from pipe" );
        else if ( r == -1 )
            return 0;
        if ( r == 0 )
            _eof = true;
        else
            std::copy( _buffer, _buffer + r, std::back_inserter( buffer ) );
        return r;
    }

    std::string nextLine() {
        Buffer::iterator nl =
            std::find( buffer.begin(), buffer.end(), '\n' );
        while ( nl == buffer.end() && readMore() )
            ;
        nl = std::find( buffer.begin(), buffer.end(), '\n' );
        if ( nl == buffer.end() )
            return "";

        std::string line( buffer.begin(), nl );
        ++ nl;
        buffer.erase( buffer.begin(), nl );
        return line;
    }
};

inline std::pair< std::string, std::string > split( std::string what,
                                                    std::string where )
{
    std::string::size_type spl = what.find( where );
    std::string
        f( what, 0, spl ),
        s( what, spl + where.length(), std::string::npos );
    return std::make_pair( f, spl == std::string::npos ? "" : s );
}

struct DebconfFrontend {
    enum PropertyKey { Choices, Description, ExtendedDescription, Type,
                       UnknownProperty = -1 };
    enum TypeKey { String, Password, Boolean, Select, MultiSelect,
                   Note, Error, Title, UnknownType = -1 };

    static const char *types[];
    static const char *properties[];

    typedef std::pair< std::string, std::string > SP;
    typedef std::map< PropertyKey, std::string > Properties;
    typedef std::map< std::string, std::string > Substitutions;

    bool go_back;
    bool backupEnabled;
    std::string title;
    std::set< std::string > input;
    std::map< std::string, std::string > values;
    std::map< std::string, Properties > data;
    std::map< std::string, Substitutions > subst;
    UnixSocket sock;
    Pipe pipe;

    std::string value( std::string k ) {
        return values[ k ];
    }

    int findId( std::string x, const char **f ) {
        int i = 0;
        while ( true ) {
            if ( f[ i ] == 0 )
                return -1;
            if ( x == f[ i ] )
                return i;
            ++ i;
        }
        return -1;
    }

    PropertyKey propertyKeyFromString( std::string s ) {
        return static_cast< PropertyKey >( findId( s, properties ) );
    }

    TypeKey typeFromString( std::string s ) {
        return static_cast< TypeKey >( findId( s, types ) );
    }

    TypeKey type( std::string s ) {
        return typeFromString( property( s, Type ) );
    }

    DebconfFrontend() : sock( "/tmp/adept.debconf", true ) {
        putenv( const_cast< char * >( "DEBIAN_FRONTEND=passthrough" ) );
        putenv( const_cast< char * >( "DEBCONF_DEBUG=developer" ) );
        putenv( const_cast< char * >( "DEBCONF_PIPE=/tmp/adept.debconf" ) );
        sock.listen();
    }

    virtual ~DebconfFrontend() {}

    virtual void reset() {
        backupEnabled = false;
    }

    void say( std::string s ) {
        std::cerr << "ADEPT ---> " << s << std::endl;
        pipe.write( s + "\n" );
    }

    // TODO this is apparently very much untested
    std::string substitute( std::string key, std::string rest ) {
        Substitutions sub = subst[ key ];
        std::string result, var, escape;
        wibble::ERegexp re( "^(.*?)(\\\\)?\\$\\{([^{}]+)\\}(.*)$", 5 );
        while( re.match( rest ) ) {
            result += re[1];
            escape = re[2];
            var = re[3];
            rest = re[4];
            if ( !escape.empty() ) {
                result += "${" + var + "}";
            } else {
                result += sub[ var ];
            }
        }
        return result + rest;

    }

    std::string property( std::string key, PropertyKey p ) {
        std::string r = data[ key ][ p ];
        if ( p == Description || p == Choices )
            return substitute( key, r );
        return r;
    }

    virtual void cmd_capb( std::string caps ) {
        wibble::Splitter split( ", ", 0 );
        wibble::Splitter::const_iterator i = split.begin( caps );
        for ( ; i != split.end(); ++i ) {
            if ( *i == "backup" )
                backupEnabled = true;
        }
        say( "0 backup" );
    }

    virtual void cmd_set( std::string param ) {
        SP p = split( param, " " );
        values[ p.first ] = p.second;
        std::cerr << "# SET: [" << p.first << "] " << p.second << std::endl;
        say( "0 ok" );
    }

    virtual void cmd_get( std::string param ) {
        say( "0 " + values[ param ] );
    }

    virtual void cmd_input( std::string param ) {
        SP p = split( param, " " );
        input.insert( p.second );
        say( "0 will ask" );
    }

    virtual void cmd_go( std::string ) {
        std::cerr << "# GO" << std::endl;
        for ( std::set< std::string >::iterator i = input.begin();
              i != input.end(); ++ i )
        {
            std::cerr << "# TITLE: " << title << std::endl;
            std::cerr << "# DESCRIPTION: "
                      << property( *i, ExtendedDescription ) << std::endl;
            if ( type( *i ) == Select || type( *i ) == MultiSelect )
                std::cerr << "# CHOICES: " << property( *i, Choices )
                          << std::endl;
            std::cerr << "# QUERY: " << property( *i, Description ) << std::endl;
            std::cout << *i << ": [" << values[ *i ] << "] ";
            std::string answer;
            std::getline( std::cin, answer );
            if ( !answer.empty() )
                values[ *i ] = answer;
        }
        if ( go_back )
            back();
        else
            next();
    }

    virtual void cleanup() {
        input.clear();
        go_back = false;
    }

    virtual void next() {
        cleanup();
        say( "0 ok, got the answers" );
    }

    virtual void back() {
        cleanup();
        say( "30 go back" );
    }

    virtual void done() {
    }

    virtual void cmd_title( std::string param ) {
        std::cerr << "ADEPT: TITLE " << param << std::endl;
        title = param;
        say( "0 ok" );
    }
 
    virtual void cmd_data( std::string param ) {
        SP p = split( param, " " );
        SP q = split( p.second, " " );
        data[ p.first ][ propertyKeyFromString( q.first ) ] = q.second;
        std::cerr << "# NOTED: [" << p.first << "] [" << q.first << "] "
                  << q.second << std::endl;
        say( "0 ok" );
    }

    virtual void cmd_subst( std::string param ) {
        SP p = split( param, " " );
        SP q = split( p.second, " " );
        subst[ p.first ][ q.first ] = q.second;
        std::cerr << "# SUBST: [" << p.first << "] [" << q.first << "] "
                  << q.second << std::endl;
        say( "0 ok" );
    }

    struct Cmd {
        const char *cmd;
        void (DebconfFrontend::*run)( std::string );
    };

    static const Cmd commands[];

    void listen() {
        reset();
        pipe = Pipe( sock.accept() );
    }

    // non-blockingly process a single line from debconf
    bool process() {
        if ( pipe.eof() )
            done();
        if ( !pipe.active() )
            listen();
        if ( !pipe.active() )
            return false;

        std::string line = pipe.nextLine();
        if ( line.empty() )
            return false;

        std::pair< std::string, std::string > cmd = split( line, " " );
        std::cerr << "ADEPT <--- [" << cmd.first << "] "
                  << cmd.second << std::endl;
        const Cmd *c = commands;
        while ( c->cmd != 0 ) {
            if ( cmd.first == c->cmd ) {
                (this->*(c->run))( cmd.second );
                return true;
            }
            ++ c;
        }
        return true;
    }

    void processAvailable() {
        while( process() )
            ;
    }
};

}

#endif
