/*
 *  aatv - a program to watch TV on a text-based console
 *  Copyright (C) 2000 Florent de Lamotte
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <aalib.h>

#include <asm/types.h>
#include <linux/videodev.h>

#include "aatv.h"

char *grab_device = "/dev/video0";
int   grab_input = 0;
int   grab_norm  = VIDEO_MODE_SECAM;
unsigned long presets [10] = {0,679250,655250,631250,0,703250,727250,0,0,0};

void
parse_config (void)
{
	char line [256];     
	char filename [100];
	char *buf;
     	int chan;
     	FILE *fp;
	
	sprintf (filename, "%s/%s", getenv ("HOME"), ".aatv");
	
	if ((fp = fopen(filename, "r")) == NULL) 
		if ((fp = fopen(AATV_CONFIGFILE, "r")) == NULL) 
			return;
	

     	/* parse file */
     	while ((fgets(line, 255, fp)) != NULL) {
	  	buf = strtok (line, " ");
	  	if (buf[0] == '\n' || buf[0] == '#') 
	       	continue; /* empty or commented */
	  	if (!strcasecmp(buf, "channel")) {
	       		chan = atoi(strtok(NULL," "));
	       	if (chan >= 0 && chan <= 9)
		    	presets [chan] =
                                (unsigned long) atol(strtok(NULL," "));
	  	} else if (!strcasecmp (buf, "vidmode")) {
	       		buf = strtok (NULL, " ");
	       		if (!strcasecmp (buf, "PAL"))
		    		grab_norm = VIDEO_MODE_PAL;
	       		else if (!strcasecmp (buf, "SECAM"))
		    		grab_norm = VIDEO_MODE_SECAM;
	       		else if (!strcasecmp (buf, "NTSC"))
		    		grab_norm = VIDEO_MODE_NTSC;
		}
     	}	       
     	fclose (fp);
}

/* for the window */
aa_context *c;
aa_renderparams *p;

void
exit_aatv (int return_code)
{
	aa_uninitmouse(c);
	aa_close (c);
	grab_free ();
	printf("\n");
	exit (return_code);
}

int
(*show_image) (unsigned char *data, int width, int height);

int 
show_image_std (unsigned char *data, int width, int height)
{
	int i, j;

	/*
	 * here we use a double loop because with webcam for example
	 * capture width may not be wanted width
	 */
        switch (capture_type->type) {
        case VIDEO_PALETTE_GREY:
		for (i=0;i<width;i++)
		for (j=0;j<height;j++)
			(c->imagebuffer)[i+j*width] = data[i+j*width];
		break;
                
        case VIDEO_PALETTE_RGB24:
		for (i=0;i<width;i++)
		for (j=0;j<height;j++)	
			(c->imagebuffer)[i+j*width] = (int)rint(
				data[3*(i+j*width)] * 0.3
				+ data[3*(i+j*width)+1] * 0.59
				+ data[3*(i+j*width)+2] * 0.11);
                break;

	case VIDEO_PALETTE_RGB32:
		for (i=0;i<width;i++)
		for (j=0;j<height;j++)	
			(c->imagebuffer)[i+j*width] = (int)rint(
				data[4*(i+j*width)] * 0.3
				+ data[4*(i+j*width)+1] * 0.59
				+ data[4*(i+j*width)+2] * 0.11);
		break;

        case VIDEO_PALETTE_RGB565:
                for (i = 0; i < width; i++)
                        for (j = 0; j < height; j++) {
                                unsigned short pixel;

                                pixel = *(unsigned short *)
                                        (data + 2 * (i + j * width));

                                c->imagebuffer[i + j * width] = (int)rint(
					((pixel >> 11) << 3) * 0.3
					+ (((pixel >> 5) & 0x3f) << 2) * 0.59
					+ ((pixel & 0x1f) << 3) * 0.11);
                        }
                break;
        default:
                return -1;
        }

        return 0;
}

int
show_image_invert (unsigned char *data, int width, int height)
{
	int i,j,k;
	for (i=width-1,k=0;i>=0;i--,k++) 
		for (j=0;j<height;j++)
	       		aa_putpixel (c,i,j,data[k+j*width]);
     	return 0;
}
     

int
main(int argc, char *argv[])
{
     	unsigned char *image;
     	int width, height;
     	int inverse;
     	int cont = 1;		/* flag to end main loop */
     	int key;
     	int fast = 0;		/* fast render ON/OFF */
     	char buf [10];		/* buffer for interactive commands */
    

     	parse_config ();

     	/* initialize AA-lib */
     	if (!aa_parseoptions(NULL, NULL, &argc, argv) || argc != 1) {
	  	printf("%s\n", aa_help);
	  	exit(1);
     	}

     	c = aa_autoinit(&aa_defparams);
     	if (c == NULL) {
	  	printf("Can not intialize aalib\n");
	  	exit(1);
     	}
     	if (!aa_autoinitkbd(c,0)) {
	  	printf("Can not intialize keyboard\n");
	  	aa_close(c);
	  	exit(1);
     	}
     	aa_hidemouse (c);

     	/* init grabber */
     	p = aa_getrenderparams();
     	grab_width = aa_imgwidth (c);
     	grab_height = aa_imgheight (c);
     	show_image = &show_image_std;
     	inverse=0;
      	
	/* grab_width and grab_height should be changed after that call */
     	grab_init();
     	get_picture_properties ();

     	/* main loop */
     	while (cont) {
	  	image = grab_one(&width,&height);
	  	(*show_image)(image,width,height);
	  	if (fast)
	       		aa_fastrender(c, 0, 0, aa_scrwidth(c), aa_scrheight(c));
	  	else
	       		aa_render(c, p, 0, 0, aa_scrwidth(c), aa_scrheight(c));
	  	aa_flush(c);

	  	if ((key=aa_getkey(c,0)) == AA_NONE)
	       		continue;

	  	switch (key) {
	  	case 'q':
	       		cont = 0;
	       		break;
	  	case 'v': 
	       		if ((inverse=!inverse))
		    		show_image=&show_image_invert;
	       		else
		    		show_image=&show_image_std;
	       		break;
	  	case 'c':
	       		if (!get_video_source()) {
				set_video_source (1);
		    		set_video_norm (VIDEO_MODE_PAL);
	       		} else { /* switch from camera to TV */
		    		set_video_source (0);
				set_video_norm (grab_norm);
	       		}
	       		break;

	  	case 's' :
	       		if (++grab_norm == 3)
		    		grab_norm = 0;
	      	 	set_video_norm (grab_norm);
	       		break;
	       
	       		/* AA image controls */
	  	case 'f':
	       		fast=!fast;
	       		break;
	  	case 'a':
	       		p->dither=AA_NONE;
	       		break;
	  	case 'z':
	       		p->dither=1;
	       		break;
	  	case 'e':
	       		p->dither=2;
	       		break;
	  	case 'p':		/* contrast */
	       		sprintf(buf,"%d",p->contrast);
	       		aa_edit(c,0,0,5,buf,5);
	       		p->contrast=atol(buf);
	       		break; 
	  	case '!':		/* contrast */
	       		p->contrast--;
	       		break;
	  	case 'm':		/* contrast */
	       		p->contrast++;
	       		break;		/* brightness */
	  	case 'o':
	       		sprintf(buf,"%d",p->bright);
	       		aa_edit(c,0,0,5,buf,5);
	       		p->bright=atol(buf);
	       		break;
	  	case 'l':		/* brightness */
	       		p->bright++;
	       		break;
	  	case ':':		/* brightness */
	       		p->bright--;
	       		break;

	       		/* Tuner Image control */
	  	case 'i':		/* contrast */
	       		sprintf(buf,"%d",grab_pict.contrast);
	       		aa_edit(c,0,0,6,buf,6);
	       		grab_pict.contrast=atoi(buf);
	       		set_picture_properties ();
	       		break;
	  	case 'k':		/* contrast */
	       		if (grab_pict.contrast<65535)
		    	grab_pict.contrast++;
	       		set_picture_properties ();
	       		break;
	  	case ';':		/* contrast */
	       		if (grab_pict.contrast>0)
		    	grab_pict.contrast--;
	       		set_picture_properties ();
	       		break;
	  	case 'u':		/* brightness */
	       		sprintf(buf,"%d",grab_pict.brightness);
	       		aa_edit(c,0,0,6,buf,6);
	       		grab_pict.brightness=atoi(buf);
	       		set_picture_properties ();
	       		break;
	  	case 'j':		/* brightness */
	       		if (grab_pict.brightness<65535)
		    	grab_pict.brightness++;
	       		set_picture_properties ();
	       		break;
	  	case ',':		/* brightness */
	       		if (grab_pict.brightness>0)
		    	grab_pict.brightness--;
	       		set_picture_properties ();
	       		break;
	  	case 'y':		/* whiteness */
	       		sprintf(buf,"%d",grab_pict.whiteness);
	       		aa_edit(c,0,0,6,buf,6);
	       		grab_pict.whiteness=atoi(buf);
	       		set_picture_properties ();
	       		break;
	  	case 'h':		/* whiteness */
	       		if (grab_pict.whiteness<65535)
		    	grab_pict.whiteness++;
	       		set_picture_properties ();
	      	 	break;
	  	case 'n':		/* whiteness */
	       		if (grab_pict.whiteness>0)
		    	grab_pict.whiteness--;
	       		set_picture_properties ();
	       		break;

	       		/* frequencies */
	  	case 'b':
	       		sprintf(buf,"%ld",get_freq());
	       		aa_edit (c, 0,0,7,buf,7);
	       		change_freq (strtoul(buf,NULL,10));
	       		break;
	  	default :
	       		if ((key >= '0') && (key <= '9') && (presets [key-'0']!=0)) {
		    		change_freq (presets [key - '0']);
	       		}
	       		break;
	  	}
     	}

     	aa_uninitmouse(c);
     	aa_close (c);

     	grab_free ();
     	
	return EXIT_SUCCESS;
}
