/*
 *   pcmfade.c
 *
 *   Oliver Fromme  <olli@fromme.com>
 *
 *   Copyright (C) 1997,1998,1999
 *        Oliver Fromme.  All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of the author nor the names of any co-contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY OLIVER FROMME AND CONTRIBUTORS ``AS IS'' AND
 *   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *   ARE DISCLAIMED.  IN NO EVENT SHALL OLIVER FROMME OR CONTRIBUTORS BE LIABLE
 *   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *   SUCH DAMAGE.
 *
 *   @(#)$Id: pcmfade.c,v 1.3 1999/01/01 23:31:52 olli Exp $
 */

static const char cvsid[]
    = "@(#)$Id: pcmfade.c,v 1.3 1999/01/01 23:31:52 olli Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "utils.h"

void usage (char *dummy)
{
	fprintf (stderr, "Usage:  %s <0%% offset> <100%% offset> <PCM file>\n", me);
	fprintf (stderr, "The following formats can be used for offsets:\n");
	fprintf (stderr, "   [-][<minutes>]:<seconds>\n");
	fprintf (stderr, "   [-]<samples>\n");
	fprintf (stderr, "<minutes> is an integer value, default is 0.\n");
	fprintf (stderr, "<seconds> is an integer OR floating point value (can be >= 60).\n");
	fprintf (stderr, "<samples> is an integer value (1 sample = 4 bytes).\n");
	fprintf (stderr, "Negative values are relative to the end of the file.\n");
	fprintf (stderr, "if <0%% offset>  <  <100%% offset>, a fade-in is created.\n");
	fprintf (stderr, "if <0%% offset>  >  <100%% offset>, a fade-out is created.\n");
	fprintf (stderr, "Note: the samples are assumed to be 44.1 kHz signed 16 bit stereo:\n");
	fprintf (stderr, "      1 second = 44100 samples = 176400 bytes.\n");
	fprintf (stderr, "WARNING: the <PCM file> is modified in place, so make a backup copy!\n");
	exit (0);
}

char *samples2time (long samples)
{
	static char time[24];
	long mins, secs;

	secs = samples / 44100;
	samples -= secs * 44100;
	mins = secs / 60;
	secs -= mins * 60;
	samples = (samples * 2000 + 441) / 882;
	sprintf (time, "%2ld:%02ld.%05ld", mins, secs, samples);
	return (time);
}

long get_filesize (int fd)
{
	struct stat fdstat;

	if (fstat(fd, &fdstat))
		return (-1);
	return (fdstat.st_size);
}

void skip_whitespace (char **c)
{
	*c += strspn(*c, " \t\r\n");
} 

long get_offset (char *s, long samples)
{
	char *ends, *dpp;
	int neg = FALSE;
	long off;

	skip_whitespace (&s);
	switch (*s) {
		case '-':
			neg = TRUE;
		case '+':
			s++;
			skip_whitespace (&s);
	}
	if (!*s)
		return (-1);
	if ((dpp = strchr(s, ':'))) {
		long mins;
		double secs;

		if (dpp++ == s)
			mins = 0;
		else {
			mins = strtoul(s, &ends, 10);
			skip_whitespace (&ends);
			if (*ends != ':')
				return (-1);
		}
		skip_whitespace (&dpp);
		if (*dpp) {
			secs = strtod(dpp, &ends);
			skip_whitespace (&ends);
			if (*ends)
				return (-1);
		}
		else
			secs = 0.0;
		off = (mins * 60 + secs) * 44100 + 0.5;
	}
	else {
		off = strtoul(s, &ends, 0);
		skip_whitespace (&ends);
		if (*ends || off >= samples || (neg && !off))
			return (-1);
	}
	return (neg ? samples - off : off);
}

#define BUFFERSAMPLES 16384
#define BUFFERSHORTS (BUFFERSAMPLES * 2)
#define BUFFERSIZE (BUFFERSHORTS * sizeof(short))

void fade (int fd, long maxlevel, long ostart, int lstart, long oend, int lend)
{
	long off, olen = oend - ostart, olen2 = olen >> 1;
	int level = lstart, llen = lend - lstart;
	int inc, bres = 0;

	short *buffer;
	long bindex = 0;

	buffer = tmalloc(BUFFERSIZE);
	if (lseek(fd, ostart * 4, SEEK_SET) == -1)
		die ("lseek()");
	if (read(fd, buffer, BUFFERSIZE) == -1)
		die ("read()");
	for (off = ostart; off <= oend; off++) {
		buffer[bindex] = ((long) buffer[bindex] * level) / maxlevel;
		bindex++;
		buffer[bindex] = ((long) buffer[bindex] * level) / maxlevel;
		bindex++;
		if (bindex >= BUFFERSHORTS) {
			if (lseek(fd, ostart * 4, SEEK_SET) == -1)
				die ("lseek()");
			if (write(fd, buffer, BUFFERSIZE) == -1)
				die ("write()");
			ostart += BUFFERSAMPLES;
			if (read(fd, buffer, BUFFERSIZE) == -1)
				die ("read()");
			bindex = 0;
		}
		bres += llen;
		if (abs(bres) + olen2 >= olen) {
			if (bres > 0)
				inc = (bres + olen2) / olen;
			else
				inc = (bres - olen2) / olen;
			level += inc;
			bres -= (inc * olen);
		}
	}
	if (bindex) {
		if (lseek(fd, ostart * 4, SEEK_SET) == -1)
			die ("lseek()");
		if (write(fd, buffer, bindex * sizeof(short)) == -1)
			die ("write()");
	}
	free (buffer);
}

#define MAXLEVEL 0x10000

int main (int argc, char *argv[])
{
	int fd;
	int lstart, lend;
	long filesize;
	long s0, s100, ostart, oend;

	utils_init (argv[0]);
	if (argc != 4)
		usage (NULL);
	if ((fd = open(argv[3], O_RDWR, 0)) == -1)
		die (argv[3]);
	if ((filesize = get_filesize(fd)) == -1)
		die (argv[3]);
	if (filesize % 3)
		fprintf (stderr,
			"%s: Warning: file size is not a multiple of 4.\n",
			me);
	filesize >>= 2;
	if ((s0 = get_offset(argv[1], filesize)) == -1
			|| (s100 = get_offset(argv[2], filesize)) == -1) {
		fprintf (stderr,
			"%s: Offset specification invalid or out of range.\n",
			me);
		exit (1);
	}
	if (s0 < s100) {
		ostart = s0;
		oend = s100;
		lstart = 0;
		lend = MAXLEVEL;
	}
	else if (s0 > s100) {
		ostart = s100;
		oend = s0;
		lstart = MAXLEVEL;
		lend = 0;
	}
	else {
		fprintf (stderr,
			"%s: start offset and end offset must be different.\n",
			me);
		close (fd);
		exit (1);
	}
	printf ("File:  %s\n", argv[3]);
	printf ("Size (in samples):  %8ld / %s\n", filesize, samples2time(filesize));
	printf ("Fading from %3d%% at %8ld / %s\n",
		lstart ? 100 : 0, ostart, samples2time(ostart));
	printf ("         to %3d%% at %8ld / %s\n",
		lend ? 100 : 0, oend, samples2time(oend));
	printf ("Duration of fade:   %8ld / %s\n",
		oend - ostart, samples2time(oend - ostart));
	fade (fd, MAXLEVEL, ostart, lstart, oend, lend);
	close (fd);
	exit (0);
}

/* EOF */
