/*

Copyright (C) 2000 - 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <time.h>
#include <stdio.h>
#include <errno.h>

#include <nd_tpm_gui.h>
#include <nd_trace_registry.h>
#include <interface.h>
#include <nd_gui.h>
#include <callbacks.h>


static GtkWidget              *nav_dialog = NULL;
static LND_TPM_NavMode         nav_mode = LND_TPM_TIME;

static void
tpm_time_scale_cb(GtkAdjustment *adj)
{
  LND_Trace          *trace;
  GtkWidget          *w;
  char                buf[MAXPATHLEN];
  struct bpf_timeval  delta, sum;

  return_if_no_current_trace(trace);

  pcapnav_timeval_sub(&trace->tpm->base->end_ts,
		      &trace->tpm->base->start_ts,
		      &delta);  

  delta.tv_sec  = delta.tv_sec * adj->value;
  delta.tv_usec = delta.tv_usec * adj->value;
  pcapnav_timeval_add(&delta, &trace->tpm->base->start_ts, &sum);

  ND_GTK_GET(w, nav_dialog, "nav_time_sec_entry");
  g_snprintf(buf, MAXPATHLEN, "%lu", (unsigned long) sum.tv_sec);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_nav_time_entries_changed, NULL);
  gtk_entry_set_text(GTK_ENTRY(w), buf);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_nav_time_entries_changed, NULL);

  ND_GTK_GET(w, nav_dialog, "nav_time_usec_entry");
  g_snprintf(buf, MAXPATHLEN, "%lu", (unsigned long) sum.tv_usec);
  gtk_signal_handler_block_by_func(GTK_OBJECT(w), on_nav_time_entries_changed, NULL);
  gtk_entry_set_text(GTK_ENTRY(w), buf);  
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(w), on_nav_time_entries_changed, NULL);

  ND_GTK_GET(w, nav_dialog, "nav_time_label");
  libnd_misc_ctime(&sum, buf, MAXPATHLEN, TRUE, TRUE);
  gtk_label_set_text(GTK_LABEL(w), buf);
}


static void
tpm_space_scale_cb(GtkAdjustment *adj)
{
  GtkWidget      *w;
 
  ND_GTK_GET(w, nav_dialog, "nav_frac_spinbutton");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), adj->value);
}


static void
tpm_space_spin_cb(GtkAdjustment *adj)
{
  GtkWidget      *w;
  GtkAdjustment  *adj_scale;

  ND_GTK_GET(w, nav_dialog, "nav_frac_hscale");
  adj_scale = gtk_range_get_adjustment(GTK_RANGE(w));

  adj_scale->value = adj->value;
  gtk_adjustment_value_changed(adj_scale);
}


static gboolean
tpm_dialog_get_timeval(struct bpf_timeval *tv)
{
  GtkWidget *w;
  char      *errptr;

  if (!tv || !nav_dialog)
    return FALSE;

  ND_GTK_GET(w, nav_dialog, "nav_time_sec_entry");
  tv->tv_sec = strtoul(gtk_entry_get_text(GTK_ENTRY(w)), &errptr, 10);
      
  if ((*errptr != '\0') || (errno == ERANGE))
    return FALSE;

  ND_GTK_GET(w, nav_dialog, "nav_time_usec_entry");
  tv->tv_usec = strtoul(gtk_entry_get_text(GTK_ENTRY(w)), &errptr, 10);

  if ((*errptr != '\0') || (errno == ERANGE))
    return FALSE;

  while (tv->tv_usec > 1000000)
    {
      tv->tv_sec++;
      tv->tv_usec -= 1000000;
    }

  return TRUE;
}


void      
nd_tpm_dialog_show(void)
{
  LND_Trace       *trace;
  GtkWidget      *w;
  GtkAdjustment  *adj;

  return_if_no_current_trace(trace);

  if (! trace->needs_nav)
    return;

  if (!nav_dialog)
    {
      nav_dialog = create_trace_navigation_dialog();

      ND_GTK_GET(w, nav_dialog, "nav_time_hscale");
      adj = gtk_range_get_adjustment(GTK_RANGE(w));
      gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			  GTK_SIGNAL_FUNC (tpm_time_scale_cb), NULL);

      ND_GTK_GET(w, nav_dialog, "nav_frac_hscale");
      adj = gtk_range_get_adjustment(GTK_RANGE(w));
      gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			  GTK_SIGNAL_FUNC (tpm_space_scale_cb), NULL);

      ND_GTK_GET(w, nav_dialog, "nav_frac_spinbutton");
      adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(w));
      gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			  GTK_SIGNAL_FUNC (tpm_space_spin_cb), NULL);
    }
  
  nd_tpm_dialog_set_mode(trace->area.mode);
  nd_tpm_dialog_sync();
  gtk_widget_show(nav_dialog);
}


void      
nd_tpm_dialog_hide(void)
{
  if (nav_dialog)
    gtk_widget_hide(nav_dialog);
}


void      
nd_tpm_dialog_apply(void)
{
  GtkWidget         *w;
  GtkAdjustment     *adj;
  LND_Trace         *trace;
  LND_TPM_NavMode    mode = nd_tpm_dialog_get_mode();
  struct bpf_timeval tv;
  pcapnav_result_t   result;
  double             fraction;
  off_t              offset;

  if (!nav_dialog)
    return;

  return_if_no_current_trace(trace);

  D_ENTER;

  switch(mode)
    {
    case LND_TPM_SPACE:

      ND_GTK_GET(w, nav_dialog, "nav_frac_spinbutton");
      result = libnd_tpm_goto_fraction(trace->tpm, gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w)));

      if (result != PCAPNAV_DEFINITELY && result != PCAPNAV_PERHAPS)
	D_RETURN;

      ND_GTK_GET(w, nav_dialog, "nav_time_hscale");
      adj = gtk_range_get_adjustment(GTK_RANGE(w));
      
      fraction = libnd_tpm_get_time_fraction(trace->tpm,
					     &trace->tpm->current->start_ts);
      adj->value = fraction;
      gtk_adjustment_value_changed(adj);
      break;

    case LND_TPM_TIME:

      if (!tpm_dialog_get_timeval(&tv))
	D_RETURN;

      result = libnd_tpm_goto_ts(trace->tpm, &tv);

      if (result != PCAPNAV_DEFINITELY && result != PCAPNAV_PERHAPS)
	D_RETURN;

      ND_GTK_GET(w, nav_dialog, "nav_frac_hscale");
      adj = gtk_range_get_adjustment(GTK_RANGE(w));
      
      offset = libnd_tpm_map_loc_to_offset(trace->tpm,
					   &trace->tpm->current->start);
      fraction = libnd_tpm_get_space_fraction(trace->tpm, offset);
      adj->value = fraction;
      gtk_adjustment_value_changed(adj);
      break;
      
    default:
      D_RETURN;
    }

  D_RETURN;
}


void      
nd_tpm_dialog_set_mode(LND_TPM_NavMode mode)
{
  GtkWidget *w;

  nav_mode = mode;
  ND_GTK_GET(w, nav_dialog, "nav_notebook");

  switch(mode)
    {
    case LND_TPM_SPACE:
      gtk_notebook_set_page(GTK_NOTEBOOK(w), 1);
      break;

    case LND_TPM_TIME:
      gtk_notebook_set_page(GTK_NOTEBOOK(w), 0);
      break;

    default:
      return;
    }
}


void 
nd_tpm_dialog_sync_time_scale(void)
{
  LND_Trace          *trace;
  GtkWidget          *w;
  GtkAdjustment      *adj;
  struct bpf_timeval  tv, delta, delta2;
  gfloat              f;

  if (!nav_dialog)
    return;

  return_if_no_current_trace(trace);
  
  ND_GTK_GET(w, nav_dialog, "nav_time_sec_entry");
  tv.tv_sec = strtol(gtk_entry_get_text(GTK_ENTRY(w)), (char **)NULL, 10);
  if (errno == ERANGE)
    return;

  ND_GTK_GET(w, nav_dialog, "nav_time_usec_entry");
  tv.tv_usec = strtol(gtk_entry_get_text(GTK_ENTRY(w)), (char **)NULL, 10);
  if (errno == ERANGE)
    return;
  
  pcapnav_timeval_sub(&tv, &trace->tpm->base->start_ts, &delta);
  if (delta.tv_sec == 0 && delta.tv_usec == 0)
    return;

  pcapnav_timeval_sub(&trace->tpm->base->end_ts, &trace->tpm->base->start_ts, &delta2);
  
  ND_GTK_GET(w, nav_dialog, "nav_time_hscale");
  adj = gtk_range_get_adjustment(GTK_RANGE(w));

  f =
    ((float) delta.tv_sec + (float) delta.tv_usec / 1000000) /
    ((float) delta2.tv_sec + (float) delta2.tv_usec / 1000000);
  
  if (0.0001 <= f && f <= 0.99999)
    {
      gtk_signal_handler_block_by_func(GTK_OBJECT(adj), tpm_time_scale_cb, NULL);
      gtk_adjustment_set_value(adj, f);
      gtk_signal_handler_unblock_by_func(GTK_OBJECT(adj), tpm_time_scale_cb, NULL);
    }
}


LND_TPM_NavMode   
nd_tpm_dialog_get_mode(void)
{
  GtkNotebook *w;

  if (!nav_dialog)
    return LND_TPM_ERROR;

  ND_GTK_GET(w, nav_dialog, "nav_notebook");
  
  if (gtk_notebook_get_current_page(w) == 0)
    return LND_TPM_TIME;

  return LND_TPM_SPACE;
}


void             
nd_tpm_dialog_prev_part(void)
{
  LND_Trace      *trace;

  if (!nav_dialog)
    return;

  return_if_no_current_trace(trace);

  libnd_tpm_load_prev_part(trace->tpm);
  nd_tpm_dialog_sync();  
}


void             
nd_tpm_dialog_next_part(void)
{
  LND_Trace      *trace;

  if (!nav_dialog)
    return;

  return_if_no_current_trace(trace);

  libnd_tpm_load_next_part(trace->tpm);
  nd_tpm_dialog_sync();  
}


void             
nd_tpm_dialog_sync(void)
{
  struct bpf_timeval  delta, delta2;
  LND_Trace          *trace;
  double              fraction;
  GtkWidget          *w;
  GtkAdjustment      *adj;
  char                s[MAXPATHLEN];
  off_t               offset;

  if (!nav_dialog)
    return;
  
  return_if_no_current_trace(trace);

  D_ENTER;
  
  nd_gui_update_view_indicator();

  ND_GTK_GET(w, nav_dialog, "nav_start_epoch_label");
  g_snprintf(s, MAXPATHLEN, "%lu.%lu",
	     trace->tpm->base->start_ts.tv_sec,
	     trace->tpm->base->start_ts.tv_usec);
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, nav_dialog, "nav_end_epoch_label");
  g_snprintf(s, MAXPATHLEN, "%lu.%lu",
	     (unsigned long) trace->tpm->base->end_ts.tv_sec,
	     (unsigned long) trace->tpm->base->end_ts.tv_usec);
  gtk_label_set_text(GTK_LABEL(w), s);

  pcapnav_timeval_sub(&trace->tpm->base->end_ts,
		      &trace->tpm->base->start_ts,
		      &delta);

  pcapnav_timeval_sub(&trace->tpm->current->pl->ph.ts,
		      &trace->tpm->base->start_ts,
		      &delta2);

  ND_GTK_GET(w, nav_dialog, "nav_span_epoch_label");
  g_snprintf(s, MAXPATHLEN, "%lu.%lu",
	     (unsigned long) delta.tv_sec,
	     (unsigned long) delta.tv_usec);
  gtk_label_set_text(GTK_LABEL(w), s);


  ND_GTK_GET(w, nav_dialog, "nav_start_time_label");
  libnd_misc_ctime(&trace->tpm->base->start_ts, s, MAXPATHLEN, TRUE, FALSE);
  s[strlen(s)-1] = '\0';
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, nav_dialog, "nav_end_time_label");
  libnd_misc_ctime(&trace->tpm->base->end_ts, s, MAXPATHLEN, TRUE, FALSE);
  s[strlen(s)-1] = '\0';
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, nav_dialog, "nav_span_time_label");
  g_snprintf(s, MAXPATHLEN, "%s", libnd_misc_timeval_to_string(&delta));
  gtk_label_set_text(GTK_LABEL(w), s);

  /* Obtain the current fraction, space-wise. */
  offset = libnd_tpm_map_loc_to_offset(trace->tpm,
				       &trace->tpm->current->start);
  fraction = libnd_tpm_get_space_fraction(trace->tpm, offset);
  
  ND_GTK_GET(w, nav_dialog, "nav_frac_hscale");
  adj = gtk_range_get_adjustment(GTK_RANGE(w));
  adj->value = fraction;
  gtk_adjustment_value_changed(adj);

  /* Obtain the current fraction, time-wise. */
  if (!trace->tpm->current->pl)
    D_RETURN;
  
  ND_GTK_GET(w, nav_dialog, "nav_time_sec_entry");
  g_snprintf(s, MAXPATHLEN, "%lu", (unsigned long) trace->tpm->current->pl->ph.ts.tv_sec);
  gtk_entry_set_text(GTK_ENTRY(w), s);

  ND_GTK_GET(w, nav_dialog, "nav_time_usec_entry");
  g_snprintf(s, MAXPATHLEN, "%lu", (unsigned long) trace->tpm->current->pl->ph.ts.tv_usec);
  gtk_entry_set_text(GTK_ENTRY(w), s);

  ND_GTK_GET(w, nav_dialog, "nav_time_label");
  libnd_misc_ctime(&trace->tpm->current->pl->ph.ts, s, MAXPATHLEN, TRUE, TRUE);
  gtk_label_set_text(GTK_LABEL(w), s);
  
  fraction = libnd_tpm_get_time_fraction(trace->tpm,
					 &trace->tpm->current->start_ts);
  
  ND_GTK_GET(w, nav_dialog, "nav_time_hscale");
  adj = gtk_range_get_adjustment(GTK_RANGE(w));
  adj->value = fraction;
  gtk_adjustment_value_changed(adj);

  D_RETURN;
}
