/*

Copyright (C) 2000 - 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <netdb.h>

#include <nd.h>
#include <nd_gui.h>
#include <nd_raw_protocol.h>

#include <nd_udp.h>
#include <nd_udp_callbacks.h>

#ifdef LINUX_HOST
struct ipovly
{
  u_char           ih_x1[9];
  u_char           ih_pr;
  u_int16_t        ih_len;
  struct in_addr   ih_src;
  struct in_addr   ih_dst;
};
#endif


static ND_ProtoField udp_fields[] = {
  { ND_VAL_FIELD, N_("Src. port (%u)"),    N_("Source port number"),      16, nd_udp_sport_cb  },
  { ND_VAL_FIELD, N_("Dst. port (%u)"),    N_("Destination port number"), 16, nd_udp_dport_cb  },
  { ND_VAL_FIELD, N_("Length (%u)"),       N_("UDP packet length"),       16, nd_udp_length_cb },
  { ND_VAL_FIELD, N_("Checksum (0x%.4x)"), N_("UDP checksum"),            16, nd_udp_csum_cb   },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData udp_menu_data[] = {
  { N_("Fix Checksums"),  N_("Fixes UDP checksums"), 0, nd_udp_cksum_fix_cb },
  { NULL, NULL, 0, NULL}
};


static LND_Protocol *udp;
static ND_Protocol  *udp_gui;


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("UDP Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the UDP protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION_MAJOR;
}


LND_Protocol *
init(void)
{
  if (! (udp = libnd_proto_registry_find(ND_PROTO_LAYER_TRANS, IPPROTO_UDP)))
    return NULL;
		
  udp_gui = nd_proto_new(udp);
	     		
  udp_gui->create_gui      = nd_udp_create_gui;
  udp_gui->set_gui         = nd_udp_set_gui;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  udp_gui->fields = udp_fields;
  udp_gui->header_width = 32;

  /* We provide a little menu that allows us to fix checksums. */
  udp_gui->menu = nd_gui_create_menu(udp_menu_data);
	     
  return udp;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_udp_create_gui(LND_Trace *trace, LND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_udp_set_gui(const LND_Packet *packet, LND_ProtoInfo *pinf)
{
  LND_ProtoData *pd;
  struct servent *serv;
  struct udphdr *udphdr;
  
  if (pinf->inst.nesting != 0)
    D(("WARNING -- UDP plugin doesn't support nested UDP\n"));

  udphdr = (struct udphdr*) libnd_packet_get_data(packet, udp, 0);

  nd_udp_set_gui_sport(pinf, udphdr);
  nd_udp_set_gui_dport(pinf, udphdr);
  nd_udp_set_gui_len(pinf, udphdr);
  nd_udp_set_gui_csum(pinf, udphdr, packet);

  /* Try to label the next tab appropriately if it's not a handled
   * protocol (e.g., "DNS" is nicer than saying "(remaining)").
   *
   * First check if there actually is uninterpreted data and
   * UDP coming before it, and if we can then find a string
   * representation of the port number, replace the label of
   * the raw data displayer with that service name.
   */
  if (! (pd = libnd_packet_get_last_nonraw(packet)) || !libnd_packet_get_trace(packet))
    return;
  
  if (pd->inst.proto != udp)
    return;

  serv = getservbyport(udphdr->uh_dport, "udp");
  if (!serv)
    serv = getservbyport(udphdr->uh_sport, "udp");
  
  if (serv)
    {
      ND_ProtoInfo *pinf_raw = nd_raw_proto_get_gui(libnd_packet_get_trace(packet));
      gtk_label_set_text(GTK_LABEL(pinf_raw->proto_label), serv->s_name);
    }
}

  
/* Misc helper stuff below --------------------------------------------- */

void
nd_udp_set_gui_sport(LND_ProtoInfo *pinf, struct udphdr *udphdr)
{
  nd_proto_field_set(pinf, &udp_fields[0], DATA_TO_PTR(ntohs(udphdr->uh_sport)));
}


void
nd_udp_set_gui_dport(LND_ProtoInfo *pinf, struct udphdr *udphdr)
{
  nd_proto_field_set(pinf, &udp_fields[1], DATA_TO_PTR(ntohs(udphdr->uh_dport)));
}


void
nd_udp_set_gui_len(LND_ProtoInfo *pinf, struct udphdr *udphdr)
{
  nd_proto_field_set(pinf, &udp_fields[2], DATA_TO_PTR(ntohs(udphdr->uh_ulen)));
}


void
nd_udp_set_gui_csum(LND_ProtoInfo *pinf, struct udphdr *udphdr, const LND_Packet *packet)
{
  nd_proto_field_set(pinf, &udp_fields[3], DATA_TO_PTR(ntohs(udphdr->uh_sum)));
  
  if (!libnd_udp_datagram_complete(packet, 0))
    {
      nd_proto_info_field_set_state(pinf,
				    &udp_fields[3],
				    ND_FIELD_STATE_UNKN);
      return;
    }
  
  if (!libnd_udp_csum_correct(packet, NULL))
    nd_proto_info_field_set_state(pinf,
				  &udp_fields[3],
				  ND_FIELD_STATE_ERROR);
  else
    nd_proto_info_field_set_state(pinf,
				  &udp_fields[3],
				  ND_FIELD_STATE_NORMAL);
}


LND_Protocol *
nd_udp_get(void)
{
  return udp;
}


ND_Protocol *
nd_udp_get_gui(void)
{
  return udp_gui;
}
