/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: input.c,v 1.38 2002/06/18 22:07:14 stamfest Exp $ */

#include <string.h>
#include <stdio.h>		/* perror() */
#include <stdlib.h>		/* free() */

#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#include <config.h>

#include "mainwin.h"
#include "common.h"
#include "util.h"
#include "errorchain.h"
#include "formfill.h"
#include "input.h"
#include "tinput.h"
#include "browse.h"
#include "encode.h"
#include "ldif.h" /* for b64_decode */
#include "syntax.h"
#include "i18n.h"

#include "../icons/line.xpm"
#include "../icons/textarea.xpm"

void refresh_inputform(GHashTable *browsehash);


void create_form_window(GHashTable *hash)
{
     GtkWidget *edit_window, *vbox;

     edit_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(edit_window), _("New entry"));
     gtk_widget_set_usize(edit_window, 500, 450);
     gtk_widget_show(edit_window);
     gtk_signal_connect_object(GTK_OBJECT(edit_window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) edit_window);

     gtk_signal_connect_object(GTK_OBJECT(edit_window), "destroy",
                               GTK_SIGNAL_FUNC(browsehash_free),
                               (gpointer) hash);

     vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox), 5);
     gtk_container_add(GTK_CONTAINER(edit_window), vbox);
     gtk_widget_show(vbox);

     g_hash_table_insert(hash, "parent window", edit_window);
     g_hash_table_insert(hash, "target_vbox", vbox);

}


void create_form_content(GHashTable *hash)
{
     gpointer edit;
     GdkPixmap *icon;
     GdkBitmap *icon_mask;
     GtkWidget *target_vbox, *vbox2, *hbox1, *hbox2;
     GtkWidget *button, *linebutton, *textareabutton;
     GtkWidget *scwin, *table;
     GtkWidget *pixmap;
     int detail_context;

     detail_context = error_new_context(_("Error getting entry"));
     target_vbox = g_hash_table_lookup(hash, "target_vbox");

     hbox1 = gtk_hbox_new(FALSE, 5);
     gtk_widget_show(hbox1);
     gtk_box_pack_start(GTK_BOX(target_vbox), hbox1, FALSE, FALSE, 0);

     /* line button */
     linebutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(linebutton), "transform", "make entry");
     GTK_WIDGET_UNSET_FLAGS(linebutton, GTK_CAN_FOCUS);
     icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(target_vbox)->window,
                                         &icon_mask,
                                         &target_vbox->style->white,
                                         line_xpm);
     pixmap = gtk_pixmap_new(icon, icon_mask);
     /* 
	Need to unref pixmap and bitmap to not leak memory. See:

	http://mail.gnome.org/archives/gtk-app-devel-list/2000-September/msg00129.html 
     */
     gdk_pixmap_unref(icon);
     gdk_bitmap_unref(icon_mask);

     gtk_widget_show(pixmap);
     gtk_container_add(GTK_CONTAINER(linebutton), pixmap);
     gtk_widget_show(linebutton);
     gtk_box_pack_start(GTK_BOX(hbox1), linebutton, FALSE, FALSE, 5);
     gtk_signal_connect(GTK_OBJECT(linebutton), "clicked",
                        GTK_SIGNAL_FUNC(change_displaytype),
                        (gpointer) hash);

     /* textarea button */
     textareabutton = gtk_button_new();
     gtk_object_set_data(GTK_OBJECT(textareabutton), "transform", "make text");
     GTK_WIDGET_UNSET_FLAGS(textareabutton, GTK_CAN_FOCUS);
     icon = gdk_pixmap_create_from_xpm_d(GTK_WIDGET(target_vbox)->window,
                                         &icon_mask,
                                         &target_vbox->style->white,
                                         textarea_xpm);
     pixmap = gtk_pixmap_new(icon, icon_mask);
     gdk_pixmap_unref(icon);
     gdk_bitmap_unref(icon_mask);

     gtk_widget_show(pixmap);
     gtk_container_add(GTK_CONTAINER(textareabutton), pixmap);
     gtk_widget_show(textareabutton);
     gtk_box_pack_start(GTK_BOX(hbox1), textareabutton, FALSE, FALSE, 0);
     gtk_signal_connect(GTK_OBJECT(textareabutton), "clicked",
                        GTK_SIGNAL_FUNC(change_displaytype),
                        (gpointer) hash);

     /* scrolled window with vbox2 inside */
     scwin = gtk_scrolled_window_new(NULL, NULL);
     gtk_container_border_width(GTK_CONTAINER(scwin), 5);
     gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
                                    GTK_POLICY_AUTOMATIC,
                                    GTK_POLICY_AUTOMATIC);
     gtk_widget_show(scwin);
     gtk_box_pack_start(GTK_BOX(target_vbox), scwin, TRUE, TRUE, 5);
     vbox2 = gtk_vbox_new(FALSE, 0);
     gtk_widget_show(vbox2);
     gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwin), vbox2);
     g_hash_table_insert(hash, "vbox_holding_table", vbox2);

     /* table inside vbox2, will self-expand */
     table = gtk_table_new(3, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(table), 5);
     gtk_table_set_row_spacings(GTK_TABLE(table), 1);
     gtk_table_set_col_spacings(GTK_TABLE(table), 10);
     gtk_widget_show(table);
     gtk_box_pack_start(GTK_BOX(vbox2), table, FALSE, TRUE, 0);
     g_hash_table_insert(hash, "table", table);

/*       hbox2 = gtk_hbox_new(TRUE, 0); */
     hbox2 = gtk_hbutton_box_new();
     gtk_container_set_border_width(GTK_CONTAINER(hbox2), 10);
     gtk_widget_show(hbox2);
     gtk_box_pack_end(GTK_BOX(target_vbox), hbox2, FALSE, TRUE, 0);

     edit = g_hash_table_lookup(hash, "edit");
     if(edit) {
	  button = gtk_button_new_with_label(_("Apply"));
          gtk_widget_show(button);
          gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(mod_entry_from_formlist),
                                    (gpointer) hash);
          GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
          GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);
          gtk_widget_grab_default(button);

          button = gtk_button_new_with_label(_("Refresh"));
/*  	  gtk_widget_set_sensitive(button, FALSE); */
/*            GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS); */
          gtk_widget_show(button);
          gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);

          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(refresh_inputform),
                                    (gpointer) hash);



	  if (g_hash_table_lookup(hash, "close window")) {
	       button = gtk_button_new_with_label(_("Close"));
	       gtk_widget_show(button);
	       gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
	       
	       gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
					 GTK_SIGNAL_FUNC(destroy_editwindow),
					 (gpointer) hash);
	  }


	  /* this will be used as the callback to set on "activate" on
	     inputfields (i.e. hitting return in an entry box). */
	  g_hash_table_insert(hash, "activate", mod_entry_from_formlist);

     }
     else {
          button = gtk_button_new_with_label(_("OK"));
          gtk_widget_show(button);
          gtk_box_pack_start(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(add_entry_from_formlist),
                                    (gpointer) hash);
          GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
          gtk_widget_grab_default(button);

          button = gtk_button_new_with_label(_("Cancel"));
          gtk_widget_show(button);
          gtk_box_pack_end(GTK_BOX(hbox2), button, FALSE, FALSE, 0);
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    GTK_SIGNAL_FUNC(destroy_editwindow),
                                    (gpointer) hash);

	  g_hash_table_insert(hash, "activate", add_entry_from_formlist);

     }

     statusbar_msg("");
     error_flush(detail_context);
}


void build_or_update_inputform(GHashTable *hash, gboolean build)
{
     GList *values, *formlist, *widgetList;
     GdkColor red = { 0, 0xffff, 0x0000, 0x0000 };
     GdkColor mustcol = { 0, 0x5c00, 0x3800, 0xffff };
     GdkColor delcol = { 0, 0xd600, 0xa000, 0x0000 };

     GtkStyle *not_in_schema, *must_in_schema, *del_schema;
     GtkWidget *target_table, *label, *inputbox, *arrowbutton, *vbox;
     GtkWidget *widget = NULL, *align;
     struct formfill *form, *focusform;
     void *activatefunc;
     int row, row_arrow, vcnt, currcnt;
     char *dn,  value[MAX_DN_LEN];
     char *decode_buf = NULL;
     int decode_buf_allocated = 0;
     GByteArray *gb; 

     gdk_colormap_alloc_color(gdk_colormap_get_system(), &red, FALSE, TRUE);
     not_in_schema = gtk_style_copy(gtk_widget_get_default_style());
     not_in_schema->fg[GTK_STATE_NORMAL] = red;

     gdk_colormap_alloc_color(gdk_colormap_get_system(), &mustcol, FALSE, TRUE);
     must_in_schema = gtk_style_copy(gtk_widget_get_default_style());
     must_in_schema->fg[GTK_STATE_NORMAL] = mustcol;

     gdk_colormap_alloc_color(gdk_colormap_get_system(), &delcol, FALSE, TRUE);
     del_schema = gtk_style_copy(gtk_widget_get_default_style());
     del_schema->fg[GTK_STATE_NORMAL] = delcol;

     /* mind form->num_inputfields and len(GList * values) may not
	be the same -- if adding a field with an arrowbutton */

     target_table = g_hash_table_lookup(hash, "table");
     formlist = g_hash_table_lookup(hash, "formlist");
     focusform = g_hash_table_lookup(hash, "focusform");
     activatefunc = g_hash_table_lookup(hash, "activate");

     if (build) {
	  /* DN input box */
	  label = gtk_label_new("dn");
	  gtk_misc_set_alignment(GTK_MISC(label), LABEL_JUSTIFICATION, .5);
	  gtk_widget_set_style(label, must_in_schema);
	  gtk_widget_show(label);
	  gtk_table_attach(GTK_TABLE(target_table), label, 0, 1, 0, 1,
			   GTK_FILL, 0, 0, 0);
	  
	  inputbox = gtk_entry_new();
	  gtk_widget_show(inputbox);
	  gtk_object_set_data(GTK_OBJECT(inputbox), "displaytype", GINT_TO_POINTER(DISPLAYTYPE_DN));
	  gtk_table_attach(GTK_TABLE(target_table), inputbox, 1, 2, 0, 1,
			   GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 0);
	  if(activatefunc)
	       gtk_signal_connect_object(GTK_OBJECT(inputbox), "activate",
					 GTK_SIGNAL_FUNC(activatefunc), (gpointer) hash);
	  dn = g_hash_table_lookup(hash, "dn");
	  if(dn)
	       gtk_entry_set_text(GTK_ENTRY(inputbox), 
				  decode_string(value, dn, strlen(dn)));
	  g_hash_table_insert(hash, "dn-widget", inputbox);
	  
	  /* set focus on DN for now, will get overridden by focusform in hash if present */
	  gtk_editable_set_position(GTK_EDITABLE(inputbox), 0);
	  gtk_widget_grab_focus(inputbox);
     }

     decode_buf = g_malloc(MAX_DN_LEN);
     decode_buf[0] = 0;
     decode_buf_allocated = MAX_DN_LEN;

     row = 1;
     while(formlist) {
	  form = (struct formfill *) formlist->data;

	  if (build || form->label == NULL) {
	       /* attribute name */
	       form->label = label = gtk_label_new(form->attrname);
	       gtk_misc_set_alignment(GTK_MISC(label), LABEL_JUSTIFICATION, .5);
	       gtk_widget_show(label);
	       gtk_table_attach(GTK_TABLE(target_table), label, 
				0, 1, row, row + 1,
				GTK_FILL, 0, 0, 0);

	       row_arrow = row;
	       
	       form->vbox = vbox = gtk_vbox_new(FALSE, 1);
	       gtk_widget_show(vbox);
	       
	       gtk_table_attach(GTK_TABLE(target_table), vbox, 
				1, 2, row, row + 1,
				GTK_FILL|GTK_EXPAND, 
				GTK_FILL|GTK_EXPAND, 0, 0);
	  }

	  gtk_widget_restore_default_style(form->label);
	  if(form->flags & FLAG_NOT_IN_SCHEMA)
	       gtk_widget_set_style(form->label, not_in_schema);
	  if(form->flags & FLAG_MUST_IN_SCHEMA)
	       gtk_widget_set_style(form->label, must_in_schema);
	  if(form->flags & FLAG_DEL_ME)
	       gtk_widget_set_style(form->label, del_schema);
	  
	  currcnt = form->widgetList ? g_list_length(form->widgetList) : 0;
	  vcnt = 0;
	  values = form->values;
	  widgetList = form->widgetList;

	  while(vcnt < form->num_inputfields) {
	       gb = NULL;
	       if (values) {
		    gb = (GByteArray*) values->data;
	       }

/*  if (gb) { char *z = g_malloc(gb->len+1); */
/*    strncpy(z, gb->data, gb->len); */
/*    z[gb->len] = 0; */
/*    printf("attr=%s data=%s\n", form->attrname, z); */
/*    g_free(z); */
/*  } */
	       if (form->dt_handler && form->dt_handler->get_widget) {
		    if (vcnt >= currcnt) {
			 widget = 
			      form->dt_handler->get_widget(form, gb,
							   activatefunc,
							   hash);
/*        printf("creating widget %08x\n", widget);	        */
			 gtk_widget_show(widget);
			 
			 gtk_object_set_data(GTK_OBJECT(widget), 
					     "formfill", form);
			 gtk_object_set_data(GTK_OBJECT(widget), "displaytype",
					     GINT_TO_POINTER(form->displaytype));
			 
			 gtk_box_pack_start(GTK_BOX(form->vbox), widget, TRUE, TRUE, 0);
			 if(form == focusform)
			      gtk_widget_grab_focus(widget);
			 
			 form->widgetList = g_list_append(form->widgetList, 
							  widget);
		    } else {
			 widget = widgetList ? widgetList->data : NULL;
/*        printf("setting data %08x\n", widget);	        */
/*  			 if (widget && form->dt_handler->set_data) { */
/*  			      form->dt_handler->set_data(form, gb, widget); */
/*  			 } */
		    }
	       }

	       if (values) values = values->next;
	       if (widgetList) widgetList = widgetList->next;

	       vcnt++;
	  }
	  
	  row_arrow = row++;

	  if ((build || form->morebutton == NULL) && 
	      ((form->flags & FLAG_SINGLE_VALUE) == 0)) {
	       align = gtk_alignment_new(0.5, 1, 0, 0);
	       gtk_widget_show(align);
	       
	       form->morebutton = arrowbutton = gq_new_arrowbutton(hash);
	       gtk_object_set_data(GTK_OBJECT(arrowbutton), "formfill", form);
	       gtk_container_add(GTK_CONTAINER(align), arrowbutton);
	       
	       gtk_table_attach(GTK_TABLE(target_table), align,
				2, 3, row_arrow, row_arrow + 1,
				0, GTK_FILL, 5, 0);
	  }
	  formlist = formlist->next;
     }
     g_free(decode_buf);

     gtk_style_unref(not_in_schema);
     gtk_style_unref(must_in_schema);
     gtk_style_unref(del_schema);

}

void build_inputform(GHashTable *hash)
{
     build_or_update_inputform(hash, TRUE);
}

void refresh_inputform(GHashTable *browsehash)
{
     GList *oldlist, *newlist, *children;
     char *dn;
     struct ldapserver *server;

     GtkWidget *target_vbox = 
	  (GtkWidget *) g_hash_table_lookup(browsehash, "target_vbox");
     
     while ((children = gtk_container_children(GTK_CONTAINER(target_vbox))) != NULL) {
	  gtk_container_remove(GTK_CONTAINER(target_vbox),
			       children->data);
     }

     dn = (char*) g_hash_table_lookup(browsehash, "dn");
     server = (struct ldapserver *) g_hash_table_lookup(browsehash, "server");

     oldlist = g_hash_table_lookup(browsehash, "oldlist");
     if (oldlist) {
	  free_formlist(oldlist);
	  g_hash_table_remove(browsehash, "oldlist");
     }

     oldlist = g_hash_table_lookup(browsehash, "formlist");
     if (oldlist) {
	  free_formlist(oldlist);
	  g_hash_table_remove(browsehash, "formlist");
     }

     oldlist = formlist_from_entry(server, dn, 0);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     oldlist = add_schema_attrs(server, oldlist);
#endif

     if(oldlist) {
	  newlist = dup_formlist(oldlist);
          g_hash_table_insert(browsehash, "formlist", newlist);
          g_hash_table_insert(browsehash, "oldlist", oldlist);

	  create_form_content(browsehash);
	  build_inputform(browsehash);
     }
}


void edit_entry(struct ldapserver *server, char *dn)
{
     GHashTable *formhash;
     GList *oldlist, *newlist;
     GtkWidget *edit_window, *vbox;

     edit_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
     gtk_window_set_title(GTK_WINDOW(edit_window), dn);
     gtk_widget_set_usize(edit_window, 500, 450);
     gtk_widget_show(edit_window);
     gtk_signal_connect_object(GTK_OBJECT(edit_window), "key_press_event",
                               GTK_SIGNAL_FUNC(close_on_esc),
                               (gpointer) edit_window);

     vbox = gtk_vbox_new(FALSE, 0);
     gtk_container_border_width(GTK_CONTAINER(vbox), 5);
     gtk_container_add(GTK_CONTAINER(edit_window), vbox);
     gtk_widget_show(vbox);

     formhash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(formhash, "parent window", edit_window);
     g_hash_table_insert(formhash, "target_vbox", vbox);

     gtk_signal_connect_object(GTK_OBJECT(edit_window), "destroy",
                               GTK_SIGNAL_FUNC(browsehash_free),
                               (gpointer) formhash);

     oldlist = formlist_from_entry(server, dn, 0);
#ifdef HAVE_LDAP_STR2OBJECTCLASS
     oldlist = add_schema_attrs(server, oldlist);
#endif

     if(oldlist) {
	  newlist = dup_formlist(oldlist);
	  g_hash_table_insert(formhash, "edit", GINT_TO_POINTER(1));
	  g_hash_table_insert(formhash, "close window", GINT_TO_POINTER(1));
	  g_hash_table_insert(formhash, "server", server);
          g_hash_table_insert(formhash, "dn", g_strdup(dn));
          g_hash_table_insert(formhash, "olddn", g_strdup(dn));
          g_hash_table_insert(formhash, "formlist", newlist);
          g_hash_table_insert(formhash, "oldlist", oldlist);

	  create_form_content(formhash);
	  build_inputform(formhash);
     }
}

void new_from_entry(struct ldapserver *server, char *dn)
{
     GHashTable *hash;
     GList *formlist;
     char *newdn;

     hash = g_hash_table_new(g_str_hash, g_str_equal);
     g_hash_table_insert(hash, "server", server);

#ifdef HAVE_LDAP_STR2OBJECTCLASS
     /* if schema stuff is available, construct formlist based on the objectclasses
	in the entry */
     formlist = formfill_from_entry_objectclass(server, dn);
#else
     /* no schema stuff: just use the current entry then, throw away all values
	except those in the objectclass attribute */
     formlist = formlist_from_entry(server, dn, 1);
#endif
     if(formlist) {
	  g_hash_table_insert(hash, "formlist", formlist);

#if 0
	  int i;
	  char **oldrdn;

	  /* don't need the RDN of the current entry */
	  newdn = g_malloc(strlen(dn));
	  newdn[0] = 0;
	  oldrdn = gq_ldap_explode_dn(dn, 0);
	  for(i = 1; oldrdn[i]; i++) {
	       strcat(newdn, ",");
	       strcat(newdn, oldrdn[i]);
	  }
	  gq_exploded_free(oldrdn);
#else 
	  newdn = g_malloc(strlen(dn) + 2);
	  strcpy(newdn, ","); /* Flawfinder: ignore */
	  strcat(newdn, dn);  /* Flawfinder: ignore */
#endif

	  g_hash_table_insert(hash, "dn", newdn);

	  create_form_window(hash);
	  create_form_content(hash);
	  build_inputform(hash);
     }
     else {
	  g_hash_table_destroy(hash);
     }

}


/*
 * update formlist from on-screen table
 */
void update_formlist(GHashTable *hash)
{
     GList *formlist, *fl, *children = NULL;
     GtkWidget *child;

     struct formfill *form;
     int displaytype;
     char *content, *old_dn;
     char *content_enc;

     formlist = (GList *) g_hash_table_lookup(hash, "formlist");
     if(formlist == NULL)
	  return;

     /* walk the formlist and check the inputList for each */
     /* clear any values in formlist, they'll be overwritten shortly
	with more authorative (newer) data anyway */
     
     for (fl = formlist ; fl ; fl = fl->next ) {
	  form = (struct formfill *) fl->data;
	  free_formfill_values(form);

	  displaytype = form->displaytype;
	  
	  for (children = form->widgetList ; children ; 
	       children = children->next) {
	       child = GTK_WIDGET(children->data);

	       if(form && displaytype) {
		    GByteArray *ndata = NULL;
		    content = NULL;
		    
		    if (form->dt_handler && form->dt_handler->get_data) {
			 ndata = form->dt_handler->get_data(form, child);
		    } else if (content && strlen(content) > 0) {
			 int l;
			 content_enc = encoded_string(content);
			 g_free(content);
			 
			 l = strlen(content_enc);
			 
			 ndata = g_byte_array_new();
			 g_byte_array_set_size(ndata, l + 1);
			 memcpy(ndata->data, content_enc, l);
			 free(content_enc);
			 
			 /* uuuhhh, so ugly... */
			 ndata->data[l] = 0;
			 
			 g_byte_array_set_size(ndata, l);
		    }
		    /* don't bother adding in empty fields */
		    if(ndata) {
/*  			 printf("%s: %s\n", form->attrname, ndata->data); */
			 form->values = g_list_append(form->values, ndata);
		    }
	       }
	  }
     }
     /* take care of the dn input widget */
     child = g_hash_table_lookup(hash, "dn-widget");
     if (child) {

	  content = gtk_editable_get_chars(GTK_EDITABLE(child), 0, -1);
	  content_enc = encoded_string(content);
	  g_free(content);
	  old_dn = g_hash_table_lookup(hash, "dn");
	  if(old_dn)
	       g_free(old_dn);
	  g_hash_table_insert(hash, "dn", g_strdup(content_enc));
/*  	  printf("dn: %s\n", content_enc); */
	  free(content_enc);
     }
}


void clear_table(GHashTable *hash)
{
     GtkWidget *vbox, *old_table, *new_table;
     GList *formlist;

     vbox = g_hash_table_lookup(hash, "vbox_holding_table");
     old_table = g_hash_table_lookup(hash, "table");
     formlist = g_hash_table_lookup(hash, "formlist");

     gtk_container_remove(GTK_CONTAINER(vbox), old_table);
/*       gtk_widget_destroy(old_table); */

     /* table inside vbox, will self-expand */
     new_table = gtk_table_new(3, 2, FALSE);
     gtk_container_border_width(GTK_CONTAINER(new_table), 5);
     gtk_table_set_row_spacings(GTK_TABLE(new_table), 1);
     gtk_table_set_col_spacings(GTK_TABLE(new_table), 10);
     gtk_widget_show(new_table);
     gtk_box_pack_start(GTK_BOX(vbox), new_table, FALSE, TRUE, 0);
     g_hash_table_insert(hash, "table", new_table);

     /* should use destroy signals on the widgets to remove widget
        references */

     for ( ; formlist ; formlist = formlist->next ) {
	  struct formfill *form = (struct formfill *) formlist->data;
	  form->label = form->vbox = form->morebutton = NULL;
	  g_list_free(form->widgetList);
	  form->widgetList = NULL;
     }

     
}


void add_entry_from_formlist(GHashTable *hash)
{
     LDAP *ld;
     LDAPMod **mods, *mod;
     GList *formlist;
     GList *tabs;
     struct ldapserver *server;
     struct formfill *form;
     int add_context, res, cmod;
     int i;
     char *dn;
     char *parentdn, **rdn;

     add_context = error_new_context(_("Error adding entry"));

     /* handle impossible errors first - BTW: no need for I18N here*/
     formlist = g_hash_table_lookup(hash, "formlist");
     if(!formlist) {
	  error_push(add_context, 
		     "Hmmm, no formlist to build entry! "
		     "Please notify bert@biot.com");
	  error_flush(add_context);
	  return;
     }
     server = g_hash_table_lookup(hash, "server");
     if(!server) {
	  error_push(add_context,
		     "Hmmm, no server! Please notify bert@biot.com");
	  error_flush(add_context);
	  return;
     }

     update_formlist(hash);

     dn = g_hash_table_lookup(hash, "dn");
     if(!dn || dn[0] == 0) {
	  error_push(add_context, _("You must enter a DN for a new entry."));
	  error_flush(add_context);
	  return;
     }

     set_busycursor();

     ld = open_connection(server);
     if(!ld) {
	  set_normalcursor();
	  error_flush(add_context);
	  return;
     }

     /* build LDAPMod from formlist */
     cmod = 0;
     mods = g_malloc(sizeof(void *) * (g_list_length(formlist) + 1));
     while(formlist) {
	  form = (struct formfill *) formlist->data;

	  if (g_list_length(form->values) > 0) {
	       mod = form->dt_handler->buildLDAPMod(form,
						    LDAP_MOD_ADD,
						    form->values);
	       mods[cmod++] = mod;
	  }

	  formlist = formlist->next;
     }
     mods[cmod] = NULL;

     res = ldap_add_s(ld, dn, mods);
     ldap_mods_free(mods, 1);

     if (res == LDAP_SERVER_DOWN) {
	  server->server_down++;
     }
     if(res != LDAP_SUCCESS) {
	  error_push(add_context, ldap_err2string(res));
	  error_flush(add_context);
	  set_normalcursor();
	  close_connection(server, FALSE);
	  return;
     }

     error_flush(add_context);

     /* Walk the list of browser tabs and refresh the parent
        node of the newly added entry to give visual feedback */

     /* construct parent DN */
     
     
     /* don't need the RDN of the current entry */
     parentdn = g_malloc(strlen(dn));
     parentdn[0] = 0;
     rdn = gq_ldap_explode_dn(dn, 0);

     /* FIXME: think about using openldap 2.1 specific stuff if
        available... */
     for(i = 1; rdn[i]; i++) {
	  if (parentdn[0]) strcat(parentdn, ","); /* Flawfinder: ignore */
	  strcat(parentdn, rdn[i]);		  /* Flawfinder: ignore */
     }
     gq_exploded_free(rdn);

     for (tabs = g_list_first(tablist) ; tabs ; tabs = g_list_next(tabs)) {
	  GHashTable *tabhash = (GHashTable *) tabs->data;
	  int mode = (int) g_hash_table_lookup(tabhash, "mode");

	  GtkCTreeNode *node;

	  if (mode & BROWSE_MODE) {
	       GtkCTree *ctree = 
		    (GtkCTree *) g_hash_table_lookup(tabhash, "ctreeroot");
	       if (ctree == NULL) continue;

	       node = tree_node_from_server_dn(ctree, server, parentdn);
	       if (node == NULL) continue;
	       
	       /* refresh the parent DN node... */
	       refresh_subtree(ctree, node);
	  }
     }     

     g_free(parentdn);
     destroy_editwindow(hash);

     set_normalcursor();

     close_connection(server, FALSE);
}

void mod_entry_from_formlist(GHashTable *hash)
{
     GList *newlist, *oldlist;
     GString *message;
     LDAPMod **mods;
     LDAP *ld;
     struct ldapserver *server;
     int mod_context, res;
     char *olddn, *dn;
     char *c;
     GtkCTreeNode *node = NULL;
     GtkCTree *ctreeroot = NULL;
     int do_modrdn = 0;

     update_formlist(hash);

     mod_context = error_new_context(_("Problem modifying entry"));

     olddn = g_hash_table_lookup(hash, "olddn");
     dn = g_hash_table_lookup(hash, "dn");

     /* prepare message with olddn - in case we change the DN */
     message = g_string_sized_new(256);
     g_string_sprintfa(message, _("modified %s"), c = decoded_string(olddn));
     if (c) free(c);

     /* obtain all needed stuff from the hash ASAP, as the hash might
	be destroyed during refresh_subtree (through reselection of a
	CTreeRow and the selection callback). */

     server = g_hash_table_lookup(hash, "server");
     oldlist = g_hash_table_lookup(hash, "oldlist");
     newlist = g_hash_table_lookup(hash, "formlist");
     ctreeroot = g_hash_table_lookup(hash, "ctreeroot");

     do_modrdn = strcasecmp(olddn, dn);

     if( (ld = open_connection(server)) == NULL) {
	  g_string_free(message, TRUE);
	  return;
     }

     if (do_modrdn) {
	  int rc;

#if HAVE_LDAP_CLIENT_CACHE
	  ldap_uncache_entry(ld, olddn);
#endif
	  
	  /* find node now, olddn will change during change_rdn */
	  if(g_hash_table_lookup(hash, "ctree_refresh")) {
/*  	       printf("refresh %s to become %s\n", olddn, dn); */

	       node = tree_node_from_server_dn(ctreeroot, 
					       server, olddn);
	  }	       

	  if((rc = change_rdn(hash, mod_context)) != 0) {
/*  	       printf("error %d\n", rc); */
	       error_flush(mod_context);
	       g_string_free(message, TRUE);
	       return;
	  }
	  statusbar_msg(message->str);
     }

     mods = formdiff_to_ldapmod(oldlist, newlist);
     if(mods != NULL && mods[0] != NULL) {
/*  	  close_connection(server, FALSE); */
/*  	  g_string_free(message, TRUE); */
/*  	  return; */
/*       } */

	  res = ldap_modify_s(ld, dn, mods);
	  if (res == LDAP_SERVER_DOWN) {
	       server->server_down++;
	  }
	  
	  if(res == LDAP_SUCCESS) {
	       if (oldlist) {
		    free_formlist(oldlist);
		    g_hash_table_remove(hash, "oldlist");
	       }
	       oldlist = dup_formlist(newlist);
	       g_hash_table_insert(hash, "oldlist", oldlist);
	  } else {
	       error_push(mod_context, ldap_err2string(res));
	  } 
	  ldap_mods_free(mods, 1);
     }

     /* free memory */

     statusbar_msg(message->str);
     if (g_hash_table_lookup(hash, "close window"))
	  destroy_editwindow(hash);
     
#if HAVE_LDAP_CLIENT_CACHE
     ldap_uncache_entry(ld, dn);
#endif

     error_flush(mod_context);

     close_connection(server, FALSE);
     g_string_free(message, TRUE);

     /* refresh visual if requested by browse mode */
     if (do_modrdn && node) {
	  refresh_subtree_new_dn(ctreeroot, node, dn, 0);
     }
}


int change_rdn(GHashTable *hash, int context)
{
     GString *message;
     LDAP *ld;
     struct ldapserver *server;
     int error, res, i;
     char *olddn, *dn;
     char **oldrdn, **rdn;
     char *c;

     server = g_hash_table_lookup(hash, "server");
     if( (ld = open_connection(server)) == NULL)
	  return(1);

     olddn = g_hash_table_lookup(hash, "olddn");
     dn = g_hash_table_lookup(hash, "dn");

     oldrdn = gq_ldap_explode_dn(olddn, 0);
     rdn = gq_ldap_explode_dn(dn, 0);

     message = g_string_sized_new(256);

     /* check if user really only attemps to change the RDN and not
	any other parts of the DN */
     error = 0;
     for(i = 1; rdn[i]; i++) {
	  if(oldrdn[i] == NULL || rdn[i] == NULL || 
	     (strcasecmp(oldrdn[i], rdn[i]) != 0)) {
	       g_string_sprintf(message, 
				_("You can only change the RDN of the DN (%s)"),
				c = decoded_string(oldrdn[0]));
	       if (c) free(c);
	       error_push(context, message->str);
	       error = 1;
	       break;
	  }
     }

     if(!error) {
	  g_string_sprintfa(message, _("modifying RDN to %s"), 
			    c = decoded_string(rdn[0]));
	  statusbar_msg(message->str);
	  if (c) free(c);

#if defined(HAVE_LDAP_RENAME)
	  res = ldap_rename_s(ld, olddn, rdn[0], NULL, 1, NULL, NULL);
#else
	  res = ldap_modrdn2_s(ld, olddn, rdn[0], 1);
#endif
	  if(res == LDAP_SUCCESS) {
	       /* get ready for subsequent DN changes */
	       g_free(olddn);
	       g_hash_table_insert(hash, "olddn", g_strdup(dn));
	  } else {
	       if (res == LDAP_SERVER_DOWN) {
		    server->server_down++;
	       }
#if defined(HAVE_LDAP_RENAME)
	       g_string_sprintf(message, "ldap_rename_s: %s",
				ldap_err2string(res));
#else
	       g_string_sprintf(message, "ldap_modrdn2_s: %s",
				ldap_err2string(res));
#endif
	       error_push(context, message->str);
	       error = 2;
	  }
     }

     gq_exploded_free(oldrdn);
     gq_exploded_free(rdn);
     
     g_string_free(message, TRUE);

     close_connection(server, FALSE);
     return(error);
}


/*
 * convert the differences between oldlist and newlist into
 * LDAPMod structures
 */
LDAPMod **formdiff_to_ldapmod(GList *oldlist, GList *newlist)
{
     GList *oldlist_tmp, *newlist_tmp, *values, *deleted_values, *added_values;
     LDAPMod **mods, *mod;
     struct formfill *oldform, *newform;
     int old_l, new_l, modcnt;

     oldlist_tmp = oldlist;
     newlist_tmp = newlist;

     old_l = g_list_length(oldlist);
     new_l = g_list_length(newlist);
     mods = malloc( sizeof(void *) * (( old_l > new_l ? old_l : new_l ) * 4) );
     if (mods == NULL) {
	  perror("formdiff_to_ldapmod");
	  exit(1);
     }
     mods[0] = NULL;
     modcnt = 0;

     /* pass 0: Actually delete those values marked with FLAG_DEL_ME. These
        will be picked up be the following passes */

     for ( ; newlist ; newlist = newlist->next ) {
	  newform = (struct formfill *) newlist->data;
	  if (newform->flags & FLAG_DEL_ME) {
	       free_formfill_values(newform);
	       newform->flags &= ~FLAG_DEL_ME;
	  }
     }

     newlist = newlist_tmp;

     /* pass 1: deleted attributes, and deleted/added values */
     while(oldlist) {
	  oldform = (struct formfill *) oldlist->data;
	  newform = lookup_attribute(newlist, oldform->attrname);
	  /* oldform->values can come up NULL if the attribute was in the form
	     only because add_attrs_by_oc() added it. Not a delete in this case... */
	  if(oldform->values != NULL && (newform == NULL || newform->values == NULL)) {
	       /* attribute deleted */
	       mod = malloc(sizeof(LDAPMod));
	       if (mod == NULL) {
		    perror("formdiff_to_ldapmod");
		    exit(2);
	       }
	       mod->mod_op = LDAP_MOD_DELETE;

	       mod->mod_type = g_malloc(strlen(oldform->attrname) + 20);
	       strcpy(mod->mod_type, oldform->attrname); /* Flawfinder: ignore */
	       
	       if (oldform->syntax && oldform->syntax->must_binary) {
		    strcat(mod->mod_type, ";binary"); /* Flawfinder: ignore */
	       }
	       mod->mod_values = NULL;
	       mods[modcnt++] = mod;
	  }
	  else {

	       /* pass 1.1: deleted values */
	       deleted_values = NULL;
	       values = oldform->values;
	       while(values) {
		    if(!find_value(newform->values, (GByteArray *) values->data)) {
			 deleted_values = g_list_append(deleted_values, values->data);
		    }
		    values = values->next;
	       }

	       /* pass 1.2: added values */
	       added_values = NULL;
	       values = newform->values;
	       while(values) {
		    if(!find_value(oldform->values, (GByteArray *) values->data))
			 added_values = g_list_append(added_values, values->data);
		    values = values->next;
	       }

	       if(deleted_values && added_values) {
		    /* values deleted and added -- likely a simple edit.
		       Optimize this into a MOD_REPLACE. This could be
		       more work for the server in case of a huge number
		       of values, but who knows...
		    */
		    mod = oldform->dt_handler->buildLDAPMod(oldform,
							    LDAP_MOD_REPLACE,
							    newform->values);
		    mods[modcnt++] = mod;
	       }
	       else {
		    if(deleted_values) {
			 mod = oldform->dt_handler->buildLDAPMod(oldform,
								 LDAP_MOD_DELETE,
								 deleted_values);
			 mods[modcnt++] = mod;
		    }
		    else if(added_values) {
			 mod = oldform->dt_handler->buildLDAPMod(oldform,
								 LDAP_MOD_ADD,
								 added_values);
			 mods[modcnt++] = mod;
		    }
	       }

	       g_list_free(deleted_values);
	       g_list_free(added_values);

	  }

	  oldlist = oldlist->next;
     }
     oldlist = oldlist_tmp;

     /* pass 2: added attributes */
     while(newlist) {
	  newform = (struct formfill *) newlist->data;
	  if(lookup_attribute(oldlist, newform->attrname) == NULL) {
	       /* new attribute was added */

	       if(newform->values) {
		    mod = newform->dt_handler->buildLDAPMod(newform,
							    LDAP_MOD_ADD,
							    newform->values);
		    
		    mods[modcnt++] = mod;
	       }
	  }

	  newlist = newlist->next;
     }

     mods[modcnt] = NULL;

     return(mods);
}

char **glist_to_mod_values(GList *values)
{
     int valcnt;
     char **od_values;

     valcnt = 0;
     od_values = g_malloc(sizeof(char *) * (g_list_length(values) + 1));
     while(values) {
	  if(values->data)
	       od_values[valcnt++] = g_strdup(values->data);
	  values = values->next;
     }
     od_values[valcnt] = NULL;

     return(od_values);
}

struct berval **glist_to_mod_bvalues(GList *values)
{
     int valcnt = 0;
     struct berval **od_bvalues;

     od_bvalues = g_malloc(sizeof(struct berval *) * (g_list_length(values) + 1));

     while(values) {
	  if(values->data) {
	       int l = strlen(values->data);
	       int approx = (l / 4) * 3 + 10;
	       GByteArray *gb = g_byte_array_new();
	       if (gb == NULL) break; /* FIXME */
	       g_byte_array_set_size(gb, approx);
	       b64_decode(gb, values->data, l);
	       
	       od_bvalues[valcnt] = (struct berval*) g_malloc(sizeof(struct berval));
	       od_bvalues[valcnt]->bv_len = gb->len;
	       od_bvalues[valcnt]->bv_val = gb->data;
	       
	       g_byte_array_free(gb, FALSE);
	       valcnt++;
	  }
	  values = values->next;
     }
     od_bvalues[valcnt] = NULL;

     return(od_bvalues);
}


int find_value(GList *list, GByteArray *value)
{
     GByteArray *gb;

     while(list) {
	  gb = (GByteArray *) list->data;
	  if( gb->len == value->len && 
	      memcmp(gb->data, value->data, gb->len) == 0)
	       return(1);

	  list = list->next;
     }

     return(0);
}


void destroy_editwindow(GHashTable *hash)
{
     GList *formlist;
     GtkWidget *window;
     char *dn;

     /* PSt: Note: Should be able to remove the code for formlist an dn and
	leave the job to browsehash_free */
     formlist = g_hash_table_lookup(hash, "formlist");
     if (formlist) {
	  free_formlist(formlist);
	  g_hash_table_remove(hash, "formlist");
     }

     dn = g_hash_table_lookup(hash, "dn");
     if(dn) {
	  g_free(dn);
	  g_hash_table_remove(hash, "dn");
     }


     window = g_hash_table_lookup(hash, "parent window");
     gtk_widget_destroy(window); /* OK, window is a toplevel widget */

     /* Now done in a destroy signal handler of the window, so it
        won't escape us (using browsehash_free)  */
/*       g_hash_table_destroy(hash); */

}


void add_row(GtkWidget *button, gpointer hash)
{
     struct formfill *form;

     update_formlist(hash);

     form = (struct formfill *) gtk_object_get_data(GTK_OBJECT(button), "formfill");
     if(!form)
	  return;

     form->num_inputfields++;

/*       clear_table(hash); */
     build_or_update_inputform(hash, FALSE);

}


GtkWidget *gq_new_arrowbutton(GHashTable *hash)
{
     GtkWidget *newabutton, *arrow;

     newabutton = gtk_button_new();
     GTK_WIDGET_UNSET_FLAGS(newabutton, GTK_CAN_FOCUS);
     gtk_signal_connect(GTK_OBJECT(newabutton), "clicked",
			GTK_SIGNAL_FUNC(add_row),
			(gpointer) hash);
     arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
     gtk_widget_show(arrow);
     gtk_container_add(GTK_CONTAINER(newabutton), arrow);
     gtk_widget_show(newabutton);

     return(newabutton);
}



GtkWidget *find_focusbox(GList *formlist)
{
     GList *widgets;

     if(!formlist)
	  return(NULL);

     for ( ; formlist ; formlist = formlist->next ) {
	  struct formfill *form = (struct formfill *) formlist->data;
	  if (form && form->widgetList) {
	       for (widgets = form->widgetList ; widgets ; 
		    widgets = widgets->next ) {
		    
		    if (GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(widgets->data))) {
			 return GTK_WIDGET(widgets->data);
		    }
	       }
	  }
     }

     return NULL;
}


/*
 * callback for entry or textbox buttons
 */
void change_displaytype(GtkWidget *button, gpointer hash)
{
     GtkWidget *focusbox;
     GtkTable *table;
     struct formfill *form, *focusform;
     char *transformtype;
     GList *formlist;

     update_formlist(hash);

     table = (GtkTable *) g_hash_table_lookup(hash, "table");
     formlist = (GList *) g_hash_table_lookup(hash, "formlist");

     focusbox = find_focusbox(formlist);
     if(focusbox == NULL)
	  /* nothing focused */
	  return;

     focusform = (struct formfill *) gtk_object_get_data(GTK_OBJECT(focusbox),
							 "formfill");
     if(focusform == NULL)
	  /* field can't be resized anyway */
	  return;
     g_hash_table_insert(hash, "focusform", focusform);

     transformtype = gtk_object_get_data(GTK_OBJECT(button), "transform");

     form = (struct formfill *) gtk_object_get_data(GTK_OBJECT(focusbox),
						    "formfill");
     if(!form)
	  return;

     if(!strcmp(transformtype, "make entry")) {
	  if(GTK_IS_ENTRY(focusbox))
	       return;

	  form->displaytype = DISPLAYTYPE_ENTRY;
	  form->dt_handler = get_dt_handler(form->displaytype);
     }
     else if(!strcmp(transformtype, "make text")) {
	  if(GTK_IS_TEXT(focusbox))
	       return;

	  form->displaytype = DISPLAYTYPE_TEXT;
	  form->dt_handler = get_dt_handler(form->displaytype);
     }

     /* redraw */
     clear_table(hash);
     build_inputform(hash);

}

/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
