/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: formfill.c,v 1.27 2002/06/18 22:07:14 stamfest Exp $ */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <glib.h>

#include <lber.h>
#include <ldap.h>

#include "common.h"
#include "util.h"
#include "formfill.h"
#include "ldif.h"
#include "syntax.h"
#include "schema.h"
#include "encode.h"
#include "i18n.h"

struct formfill *new_formfill(void)
{
     struct formfill *form;

     if( (form = (struct formfill *) g_new0(struct formfill, 1)) == NULL)
	  return(NULL);

     form->attrname[0] = '\0';
     form->num_inputfields = 1;
     form->displaytype = DISPLAYTYPE_ENTRY;
     form->dt_handler = get_dt_handler(form->displaytype);
     form->flags = 0;
     form->values = NULL;

     form->label = form->vbox = form->morebutton = NULL;
     form->widgetList = NULL;

     return(form);
}


void free_formlist(GList *formlist)
{
     GList *n, *f;
     
     for (f = formlist ; f ; f = n) {
	  n = f->next;
	  free_formfill( (struct formfill *) f->data);
     }
     g_list_free(formlist);
}


void free_formfill_values(struct formfill *form) {
     if (form) {
	  GList *vals = form->values;
	  
	  if (vals) {
	       while(vals) {
		    g_byte_array_free((GByteArray*)vals->data, TRUE);
		    vals->data = NULL;
		    vals = vals->next;
	       }
	       g_list_free(form->values);
	       form->values = NULL;
	  }
     }
}

void free_formfill(struct formfill *form)
{
     free_formfill_values(form);
     if (form->widgetList) g_list_free(form->widgetList);
     g_free(form);
}


/*
 * add form to formlist, checking first if the attribute type isn't
 * already in the list
 */
GList *formlist_append(GList *formlist, struct formfill *form)
{
     GList *fl;
     struct formfill *tmp;

     fl = formlist;
     while(fl) {

	  tmp = (struct formfill *) fl->data;
	  if(!strcmp(tmp->attrname, form->attrname)) {
	       /* already have this attribute */
	       free_formfill(form);
	       return(formlist);
	  }

	  fl = fl->next;
     }

     formlist = g_list_append(formlist, form);

     return(formlist);
}

char *attr_strip(const char *attr)
{
     char *c = g_strdup(attr);
     char *d = strchr(c, ';');

     if (d) {
	  *d = 0;
     }
     return c;
}

GList *formlist_from_entry(struct ldapserver *server, char *dn, int ocvalues_only)
{
     GList *formlist;
     GString *message;
     LDAP *ld;
     LDAPMessage *res, *entry;
     BerElement *ber;
     struct formfill *form;
     int msg, i;
     char *attr; /* , **vals; */
     struct berval **bervals;
     char *c = NULL;

     formlist = NULL;

     set_busycursor();

     message = g_string_sized_new(128);
     g_string_sprintf(message, _("fetching %s from %s"), 
		      c = decoded_string(dn), server->name);
     if (c) free(c);
     statusbar_msg(message->str);
     g_string_free(message, TRUE);

     if( (ld = open_connection(server)) == NULL) {
          set_normalcursor();
          return(NULL);
     }

     msg = ldap_search_s(ld, dn, LDAP_SCOPE_BASE, "(objectclass=*)", 
			 NULL, 0, &res);

     if (msg == LDAP_SERVER_DOWN) {
	  server->server_down++;
     }

     if(msg != LDAP_SUCCESS) {
	  statusbar_msg(ldap_err2string(msg));
	  set_normalcursor();
	  close_connection(server, FALSE);
	  return(NULL);
     }

     if (res) {
	  entry = ldap_first_entry(ld, res);
	  if(entry) {
	       char *c;
	       for(attr = ldap_first_attribute(ld, entry, &ber); attr ;
		   attr = ldap_next_attribute(ld, entry, ber)) {
		    if( (form = new_formfill()) == NULL) {
			 ldap_memfree(attr);
			 break;
		    }
		    c = attr_strip(attr);
		    strncpy(form->attrname, c, MAX_ATTR_LEN);
		    if (c) g_free(c);

		    if(ocvalues_only == 0 ||
		       !strcasecmp(attr, "objectClass")) {
			 bervals = ldap_get_values_len(ld, res, attr);
			 if(bervals) {
			      for(i = 0; bervals[i]; i++) {
				   GByteArray *gb = g_byte_array_new();
				   g_byte_array_append(gb, 
						       bervals[i]->bv_val,
						       bervals[i]->bv_len);
				   form->values = g_list_append(form->values,
								gb);
			      }

			      form->num_inputfields = i;
			      ldap_value_free_len(bervals);
			 }
		    }

		    set_displaytype(server, form);
		    formlist = g_list_append(formlist, form);
		    ldap_memfree(attr);
	       }
#ifndef HAVE_OPENLDAP12
	       /* this bombs out with openldap 1.2.x libs... */
	       /* PSt: is this still true??? - introduced a configure check */
	       if(ber) {
		    ber_free(ber, 0);
	       }
#endif
	  }
	  ldap_msgfree(res);
     }

     close_connection(server, FALSE);
     set_normalcursor();

     return(formlist);
}


GList *dup_formlist(GList *oldlist)
{
     GList *newlist, *oldval, *newval;
     struct formfill *oldform, *newform;

     newlist = NULL;

     while(oldlist) {
	  oldform = (struct formfill *) oldlist->data;
	  if( (newform = new_formfill()) == NULL)
	       break;
	  /* no problem with NUL bytes here - new_formfill returns a
             completly NUL inited object */
	  strncpy(newform->attrname, oldform->attrname,
		  sizeof(newform->attrname));
	  newform->num_inputfields = oldform->num_inputfields;
	  newform->displaytype = oldform->displaytype;
	  newform->dt_handler = oldform->dt_handler;
	  newform->flags = oldform->flags;
	  oldval = oldform->values;
	  newval = NULL;
	  while(oldval) {
	       if(oldval->data) {
		    GByteArray *old = (GByteArray*) (oldval->data);
		    GByteArray *gb = g_byte_array_new();
		    g_byte_array_append(gb, old->data, old->len);

		    newval = g_list_append(newval, gb);
	       }

	       oldval = oldval->next;
	  }
	  newform->values = newval;
	  newlist = g_list_append(newlist, newform);

	  oldlist = oldlist->next;
     }

     return(newlist);
}


void dump_hash(GHashTable *hash)
{
     GList *formlist;
     char *dn;

     dn = g_hash_table_lookup(hash, "dn");
     if(dn)
	  printf("dn: %s\n", dn);

     formlist = g_hash_table_lookup(hash, "formlist");
     if(!formlist)
	  return;

     dump_formlist(formlist);

}


void dump_formlist(GList *formlist)
{
     GList *values;
     struct formfill *form;

     while(formlist) {
	  form = (struct formfill *) formlist->data;
	  printf("%s\n", form->attrname);
	  values = form->values;

	  while(values) {
	       if(values->data) {
		    GByteArray *d = (GByteArray*) values->data;
/*  		    GByteArray *gb = form->dt_handler->encode(d->data, d->len); */
		    printf("\t");
		    fwrite(d->data, d->len, 1, stdout);
		    printf("\n");

/*  		    g_byte_array_free(gb, TRUE); */
	       }
	       values = values->next;
	  }

	  formlist = formlist->next;
     }

}


struct formfill *lookup_attribute(GList *formlist, char *attr)
{
     struct formfill *form;

     while(formlist) {
	  form = (struct formfill *) formlist->data;
	  if(form->attrname && !strcasecmp(form->attrname, attr))
	       return(form);

	  formlist = formlist->next;
     }

     return(NULL);
}

/* Returns an existing formfill object (if any) for attribute attr
   from the passed-in list.  The server_schema gets consulted to also
   check for aliases. The canonical attribute type may be retrieved
   when passing a non NULL value for attrtype */
struct formfill *lookup_attribute_using_schema(GList *formlist, char *attr,
					       struct server_schema *schema,
					       LDAPAttributeType **attrtype)
{
     struct formfill *form;

#ifdef HAVE_LDAP_STR2OBJECTCLASS
     char **n;
     LDAPAttributeType *trythese = NULL;

     if (attrtype) *attrtype = NULL;

     if (schema) {
	  trythese = find_canonical_at_by_at(schema, attr);
	  if (attrtype) *attrtype = trythese;
     }
#endif /* HAVE_LDAP_STR2OBJECTCLASS */

     for( ; formlist ; formlist = formlist->next ) {
	  form = (struct formfill *) formlist->data;

#ifdef HAVE_LDAP_STR2OBJECTCLASS
	  if (trythese && form->attrname) {
	       for (n = trythese->at_names ; n && *n ; n++) {
		    if(!strcasecmp(form->attrname, *n)) return(form);
	       }
	  } else {
	       if(form->attrname && !strcasecmp(form->attrname, attr))
		    return(form);
	  }
#else /* HAVE_LDAP_STR2OBJECTCLASS */
	  if(form->attrname && !strcasecmp(form->attrname, attr))
	       return(form);
#endif /* HAVE_LDAP_STR2OBJECTCLASS */
     }

     return(NULL);
}

/*
 * sure it is nice to use serverschema for this
 *
 */
int find_displaytype(struct ldapserver *server, struct formfill *form)
{
     return get_display_type_of_attr(server, form->attrname);
}

/*
 * now we use the serverschema for this
 *
 * hint, hint
 *
 */
void set_displaytype(struct ldapserver *server, struct formfill *form)
{
     form->syntax = get_syntax_handler_of_attr(server, form->attrname, NULL);
     form->displaytype = find_displaytype(server, form);
     form->dt_handler = get_dt_handler(form->displaytype);
}


/****************** unused functions *********************/



/*
 * arrow next to inputfield hit: add empty field below this one
 */
void add_field_to_formfill(GList *formlist, char *attr)
{
     struct formfill *form;

     while(formlist) {

	  form = (struct formfill *) formlist->data;
	  if(!strcmp(form->attrname, attr)) {
	       form->num_inputfields++;
	       break;
	  }

	  formlist = formlist->next;
     }

}



/* 
   Local Variables:
   c-basic-offset: 5
   End:
 */
