/*-
 * Copyright (c) 2003 MAEKAWA Masahide <maekawa@cvsync.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>

#include <stdlib.h>

#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <limits.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>

#include "cvsync.h"
#include "logmsg.h"
#include "network.h"
#include "pid.h"

#include "defs.h"

boolean
daemonize(const char *pidfile, const char *euid, const char *egid)
{
	const char *devnull = CVSYNC_PATH_DEV_NULL;
	struct group *grp;
	struct passwd *pwd;
	struct sigaction act;
	gid_t gid = (gid_t)-1;
	uid_t uid = (uid_t)-1;
	char *resolved_pidfile;
	int fd;

	switch (fork()) {
	case -1:
		logmsg_err("daemonize: %s", strerror(errno));
		return (FALSE);
	case 0:
		break;
	default:
		_exit(EXIT_SUCCESS);
		/* NOTREACHED */
	}

	if (setsid() == -1) {
		logmsg_err("daemonize: %s", strerror(errno));
		return (FALSE);
	}
	if (sigaction(SIGHUP, NULL, &act) == -1) {
		logmsg_err("daemonize: %s", strerror(errno));
		return (FALSE);
	}
	act.sa_handler = cvsync_signal;
	if (sigaction(SIGHUP, &act, NULL) == -1) {
		logmsg_err("daemonize: %s", strerror(errno));
		return (FALSE);
	}

	switch (fork()) {
	case -1:
		logmsg_err("daemonize: %s", strerror(errno));
		return (FALSE);
	case 0:
		break;
	default:
		_exit(EXIT_SUCCESS);
		/* NOTREACHED */
	}

	if ((resolved_pidfile = pid_create(pidfile)) == NULL)
		return (FALSE);

	if (chdir("/") == -1) {
		logmsg_err("daemonize: %s", strerror(errno));
		pid_remove();
		return (FALSE);
	}
	umask(S_IWGRP|S_IWOTH);

	if (devnull != NULL) {
		if ((fd = open(devnull, O_RDWR, 0)) == -1) {
			logmsg_err("daemonize: %s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if (dup2(fd, STDIN_FILENO) == -1) {
			logmsg_err("daemonize: %s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if (dup2(fd, STDOUT_FILENO) == -1) {
			logmsg_err("daemonize: %s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if (dup2(fd, STDERR_FILENO) == -1) {
			logmsg_err("daemonize: %s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if (close(fd) == -1) {
			logmsg_err("daemonize: %s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
	} else {
		close(STDIN_FILENO);
		close(STDOUT_FILENO);
		close(STDERR_FILENO);
	}

	if (egid != NULL) {
		if ((grp = getgrnam(egid)) == NULL) {
			logmsg_err("GID %s: %s", egid, strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if (getgid() != grp->gr_gid)
			gid = grp->gr_gid;
	}
	if (euid != NULL) {
		if ((pwd = getpwnam(euid)) == NULL) {
			logmsg_err("UID %s: %s", euid, strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if (getuid() != pwd->pw_uid)
			uid = pwd->pw_uid;
	}
	if ((gid != (gid_t)-1) || (uid != (uid_t)-1)) {
		if (chown(resolved_pidfile, uid, gid) == -1) {
			logmsg_err("%s: %s", resolved_pidfile,
				   strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if ((gid != (gid_t)-1) && (setegid(gid) == -1)) {
			logmsg_err("%s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
		if ((uid != (uid_t)-1) && (seteuid(uid) == -1)) {
			logmsg_err("%s", strerror(errno));
			pid_remove();
			return (FALSE);
		}
	}

	return (TRUE);
}
