/*-
 * Copyright (c) 2003 MAEKAWA Masahide <maekawa@cvsync.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>

#include <errno.h>
#include <limits.h>
#include <signal.h>
#include <string.h>

#include "cvsync.h"
#include "logmsg.h"

#include "defs.h"

void signal_handler(int);

static struct sigaction _sig_ign;
static int cvsup2cvsync_interrupted = 0;

boolean
signal_init(void)
{
	struct sigaction act;

	_sig_ign.sa_handler = SIG_IGN;
	if (sigemptyset(&_sig_ign.sa_mask) == -1) {
		logmsg_err("sigemptyset: %s", strerror(errno));
		return (FALSE);
	}
	_sig_ign.sa_flags = 0;

	if (sigaction(SIGINT, NULL, &act) == -1) {
		logmsg_err("sigaction: %s", strerror(errno));
		return (FALSE);
	}
	act.sa_handler = signal_handler;
	act.sa_flags &= ~SA_RESTART;
	if (sigaction(SIGINT, &act, NULL) == -1) {
		logmsg_err("sigaction: %s", strerror(errno));
		return (FALSE);
	}

	if (sigaction(SIGTERM, NULL, &act) == -1) {
		logmsg_err("sigaction: %s", strerror(errno));
		return (FALSE);
	}
	act.sa_handler = signal_handler;
	act.sa_flags &= ~SA_RESTART;
	if (sigaction(SIGTERM, &act, NULL) == -1) {
		logmsg_err("sigaction: %s", strerror(errno));
		return (FALSE);
	}

	return (TRUE);
}

void
signal_handler(int sig)
{
	sigaction(SIGINT, &_sig_ign, NULL);
	sigaction(SIGTERM, &_sig_ign, NULL);

	switch (sig) {
	case SIGINT:
	case SIGTERM:
		cvsup2cvsync_interrupted = TRUE;
		break;
	default:
		break;
	}
}

boolean
cvsync_isinterrupted(void)
{
	return (cvsup2cvsync_interrupted);
}

boolean
cvsync_isterminated(void)
{
	return (FALSE);
}
