/*-
 * Copyright (c) 2003 MAEKAWA Masahide <maekawa@cvsync.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <errno.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>

#include "logmsg.h"
#include "network.h"

static struct timeval __timeout;
static fd_set __readfds;
static int *__fds, __nfds = 0, __fdmax = 0;

boolean
sock_init(int *socks)
{
	__fds = socks;

	for (__nfds = 0 ; socks[__nfds] != -1 ; __nfds++) {
		if (socks[__nfds] >= FD_SETSIZE)
			return (FALSE);
		if (__fdmax < socks[__nfds])
			__fdmax = socks[__nfds];
	}
	__fdmax++;

	__timeout.tv_sec = 1;
	__timeout.tv_usec = 0;

	return (TRUE);
}

void
sock_destroy(void)
{
	/* Nothing to do. */
}

boolean
sock_select(void)
{
	int i;

	FD_ZERO(&__readfds);

	for (i = 0 ; i < __nfds ; i++)
		FD_SET(__fds[i], &__readfds);

	if (select(__fdmax, &__readfds, NULL, NULL, &__timeout) == -1)
		return (FALSE);

	return (TRUE);
}

int
sock_accept(void)
{
	int sock = -1, i;

	for (i = 0 ; i < __nfds ; i++) {
		if (!FD_ISSET(__fds[i], &__readfds))
			continue;

		if ((sock = accept(__fds[i], NULL, NULL)) == -1) {
			if (errno == EINTR) {
				logmsg_intr();
				return (-1);
			}
			if ((errno != EAGAIN) && (errno != EWOULDBLOCK)) {
				logmsg_err("%s", strerror(errno));
				return (-1);
			}
			continue;
		}

		break;
	}

	return (sock);
}

boolean
sock_wait_send(int sock)
{
	fd_set writefds;

	FD_ZERO(&writefds);
	FD_SET(sock, &writefds);
	if (select(sock + 1, NULL, &writefds, NULL, NULL) == -1) {
		logmsg_err("Socket Error: select: %s", strerror(errno));
		return (FALSE);
	}
	if (!FD_ISSET(sock, &writefds)) {
		logmsg_err("Socket Error: select");
		return (FALSE);
	}

	return (TRUE);
}

boolean
sock_wait_recv(int sock)
{
	fd_set readfds;

	FD_ZERO(&readfds);
	FD_SET(sock, &readfds);
	if (select(sock + 1, &readfds, NULL, NULL, NULL) == -1) {
		logmsg_err("Socket Error: select: %s", strerror(errno));
		return (FALSE);
	}
	if (!FD_ISSET(sock, &readfds)) {
		logmsg_err("Socket Error: select");
		return (FALSE);
	}

	return (TRUE);
}
