/*-
 * Copyright (c) 2003 MAEKAWA Masahide <maekawa@cvsync.org>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/socket.h>

#include <stdlib.h>

#include <errno.h>
#include <poll.h>
#include <string.h>

#include "logmsg.h"
#include "network.h"

static struct pollfd *__fds = NULL;
static nfds_t __nfds = 0;

boolean
sock_init(int *socks)
{
	nfds_t i;

	for (i = 0 ; socks[i] != -1 ; i++)
		__nfds++;
	if (__nfds == 0)
		return (FALSE);
	if ((__fds = malloc(__nfds * sizeof(*__fds))) == NULL) {
		logmsg_err("%s", strerror(errno));
		return (FALSE);
	}
	for (i = 0 ; i < __nfds ; i++) {
		__fds[i].fd = socks[i];
		__fds[i].events = POLLIN;
	}

	return (TRUE);
}

void
sock_destroy(void)
{
	free(__fds);
}

boolean
sock_select(void)
{
	nfds_t i;

	for (i = 0 ; i < __nfds ; i++)
		__fds[i].revents = 0;

	if (poll(__fds, __nfds, 1000 /* 1sec */) == -1)
		return (FALSE);

	return (TRUE);
}

int
sock_accept(void)
{
	nfds_t i;
	int sock = -1;

	for (i = 0 ; i < __nfds ; i++) {
		if (!(__fds[i].revents & POLLIN))
			continue;

		if ((sock = accept(__fds[i].fd, NULL, NULL)) == -1) {
			if (errno == EINTR) {
				logmsg_intr();
				return (-1);
			}
			if ((errno != EAGAIN) && (errno != EWOULDBLOCK)) {
				logmsg_err("%s", strerror(errno));
				return (-1);
			}
			continue;
		}

		break;
	}

	return (sock);
}

boolean
sock_wait_send(int sock)
{
	struct pollfd fds[1];

	fds[0].fd = sock;
	fds[0].events = POLLOUT;
	fds[0].revents = 0;

	if (poll(fds, 1, -1) == -1) {
		logmsg_err("Socket Error: poll: %s", strerror(errno));
		return (FALSE);
	}
	if (!(fds[0].revents & POLLOUT)) {
		logmsg_err("Socket Error: poll");
		return (FALSE);
	}

	return (TRUE);
}

boolean
sock_wait_recv(int sock)
{
	struct pollfd fds[1];

	fds[0].fd = sock;
	fds[0].events = POLLIN;
	fds[0].revents = 0;

	if (poll(fds, 1, -1) == -1) {
		logmsg_err("Socket Error: poll: %s", strerror(errno));
		return (FALSE);
	}
	if (!(fds[0].revents & POLLIN)) {
		logmsg_err("Socket Error: poll");
		return (FALSE);
	}

	return (TRUE);
}
