/* Copyright (c) 2000  Kevin Sullivan <nite@gis.net>
 *
 * Please refer to the COPYRIGHT file for more information.
 */

/* This file contains functions that are used to handle socket events.
   The scheduler for these events (main loop) is in scheck.c. */

#ifndef MCURSES
  #include <ncurses.h>
#endif
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <stdlib.h>
#include <ctype.h>

#include "alias.h"
#include "codes.h"
#include "colors.h"
#include "defines.h"
#include "event.h"
#include "scheck.h"
#include "nap.h"
#include "sscr.h"
#include "cmds.h"
#include "timer.h"
#include "winio.h"

#ifdef MCURSES
  #include "wind.h"
#endif

extern cmds_t *cmdl;
extern int curx;
extern WINDOW *winput;
extern file_t *up, *down;
extern info_t info; 
extern int upsocks, downsocks;

int noprint=0;

/* not used */
void initssock(int port)
{
  struct sockaddr_in frm;
  int s, on = 1;
  
  s = socket(AF_INET, SOCK_DGRAM, 0);
  
  frm.sin_addr.s_addr = INADDR_ANY;
  frm.sin_port = htons(port);
  frm.sin_family = AF_INET;
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  
  bind(s, (struct sockaddr *)&frm, sizeof(frm));
  
  addsock(s, "ssock", S_R, inssock);
}

/* not used */
int inssock(WINDOW *win, sock_t *m)
{
  char buf[1024];  /* note: don't need malloc */
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  
  memset(buf, 0, 1024);
  
  recvfrom(m->fd, buf, 1024, 0, (struct sockaddr *)&frm, &frmlen);
  if (!strncmp(buf, "VERSION", strlen("VERSION")))
  {
    if (strcmp(strchr(buf, ' ')+1, VERSION))
    {
      wp(win, "%s* Attention: Version %s of nap is out!%s\n", BRIGHT(BLUE), strchr(buf, ' ')+1, WHITE);
      wp(win, "%s* Attention: Please visit http://www.gis.net/~nite/ to upgrade immediately%s\n", BRIGHT(BLUE), WHITE);
      drw(win);
    }
  }
  else if (!strncmp(buf, "MSG", strlen("MSG")))
  {
    wp(win, "%s* %s%s\n", BRIGHT(BLUE), strchr(buf, ' ')+1, WHITE);
    drw(win);
  }
  
  return(1);
}

/* "ipc" stands for "inter process communication". This is simply a
 * pipeline, created at the beginning of main(). It is used by child
 * processes to send output to the main screen. For instance, when
 * doing time-consuming things such as rebuilding the library, we fork
 * a child, but the child must then be able to send output to the main
 * screen. Using wp() directly won't work, both because the output
 * would be asynchronous, and because the child can't update the
 * parents' data structures.
 *
 * The reading end of the connection is here, and simply prints any
 * received data to the main screen. The sending happens in the many
 * calls in cmds.c of the form
 *
 *   ssock(ipcs[1], "* Successfully rebuilt your library\n"); 
 */

int inipc(WINDOW *win, sock_t *m)
{
  int s = m->fd;
  char *buf;
  
  rsock(s, &buf);
  if (!buf)
    return(1);
  
  /* addscroll(win, buf);
     dscr(win); */
  wp(win, buf);  /* use wp so the output goes to the log file also */
  drw(win);
  
  free(buf);
  
  return(1);
}

int inserv(WINDOW *win, sock_t *m)
{
  int s = m->fd, j, n, r;
  phead_t *pk;
  char *data;
  sock_t *t;
  cmds_t *cur;
  
  r = recvpack(s, &data, &pk);
  
  if (r == -1)
  {
    wp(win, "%s* Error reading from socket: %s%s\n", RED, sys_errlist[errno], WHITE);
    drw(win);
    return(0);
  }
  else if (r == -2)
    return(1);

  for (j=0;data[j];j++)
    if (data[j] == 10)
    {
      strncpy(data+j, data+j+1, strlen(data+j+1));
      data[strlen(data)-1] = 0;
    }
    
  n = parsein(s, pk->op, data, win);
  noprint = 0;
  if (n == -1)
    return(0);
  else if (n == -2)
    return(-1);
  
  free(pk);
  free(data);
  
  dstatus();

  t = findsock("input");
  if (!t)
    return(1);
  if (!cmdl)
  {
    cmdl = (cmds_t *)malloc(sizeof(cmds_t));
    cur = cmdl;
    cur->next = NULL;
    cur->prev = NULL;
    memset(cur->cmd, 0, sizeof(cur->cmd));
    curx = 0;
    t->d = (void *)cur;
  }

  cur = (cmds_t *)t->d;
  indraw(cur->cmd, curx, winput);
  drw(winput);
  
  return(1);
}

int gconn(WINDOW *win, sock_t *m)
{
  int s = m->fd, r;
  struct sockaddr_in frm;
  int frmlen = sizeof(frm);
  
  r = accept(s, (struct sockaddr *)&frm, &frmlen);
  if (r == -1)
    return(1);
  
  addsock(r, "conn", S_R, lsock);
  ssock(r, "1");
  
  return(1);
}

int lsock(WINDOW *win, sock_t *m)
{
  char buf[5];
  int r;

  memset(buf, 0, sizeof(buf));
  r = recv(m->fd, &buf, 4, MSG_PEEK);

  if (r==-1)
  {
    wp(win, ""RED"* Bogus connection from remote client (%s): %s"WHITE"\n", getpeerip(m->fd), sys_errlist[errno]);
    drw(win);
    return(0);
  }
  if (!strncmp(buf, "GET", 3))
  {
    recv(m->fd, &buf, 3, 0);
    m->func = doget;
  }
  else if (!strncmp(buf, "SEND", 4))
  {
    recv(m->fd, &buf, 4, 0);
    m->func = dosend;
  }
  else
  {
    char *data=NULL;

    /* see what the peer was saying */
    r = rsock(m->fd, &data);

    wp(win, ""RED"* Bogus connection from remote client (%s) [%s]"WHITE"\n", getpeerip(m->fd), data ? quote(data) : "");
    drw(win);

    ssock(m->fd, "INVALID REQUEST\n");

    free(data);
    return(0);
  }
    
  return(1);
}

/* remote client connected to our data port and issued a GET command */
int doget(WINDOW *win, sock_t *m)
{
  char *buf, *nm, *fn;
  unsigned long sz;
  file_t *cur;
  struct stat st;
  sock_t *sv;
  char **tok;
  int i, cnt;
  
  if (rsock(m->fd, &buf) == -1) {
    return(0);
  }    

  tok = form_tokso(buf, &cnt); 
  /* we should be able to use form_tokso() here instead of the more general
   * form_toks() */
  
  /* the client should have sent a string of the form
   * <remotenick> "<filename>" <offset> */

  if (cnt < 3)
  {
    wp(win, ""RED"* Bogus GET command from remote client (%s) [GET%s]"WHITE"\n", getpeerip(m->fd), quote(buf));
    drw(win);
    ssock(m->fd, "INVALID REQUEST\n");
    free(buf);
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(0);
  }
  
  nm = strdup(tok[0]);
  fn = strdup(tok[1]);
  sz = atof(tok[2]);
  free(buf);
  
  for (i=0;i<cnt;i++)
    free(tok[i]);
  free(tok);
  
  cur = ffile(up, nm, fn);

  if (!cur)
  {
    wp(win, "%s* %s requested upload of \"%s\" but it isn't in the upload list! (GET)%s\n", RED, nm, fn, WHITE);
    drw(win);
    ssock(m->fd, "INVALID REQUEST\n");
    /* close(m->fd); */
    free(nm);
    free(fn);
    return(0);
  }
  
  if (cur->st)
  {
    wp(win, "%s* %s requested upload of \"%s\" but it is already uploading! (GET)%s\n", RED, nm, fn, WHITE);
    drw(win);
    ssock(m->fd, "INVALID REQUEST\n");
    /* close(m->fd); */
    free(nm);
    free(fn);
    return(0);
  }
  
  if (!cur->f)
  {
    wp(win, "%s* \"%s\" not found (requested by %s)%s\n", RED, \
        fn, nm, WHITE);
    drw(win);
    ssock(m->fd, "FILE NOT FOUND\n");
    /* close(m->fd); */
    dupload(cur);
    free(nm);
    free(fn);
    return(0);
  }
  
  free(nm);
  free(fn);

  fseek(cur->f, sz, SEEK_SET);
  
  m->d = (void *)cur;
  m->func = sfile;
  m->t = S_W;
  free(m->nm);
  m->nm = NULL;
  msprintf(&m->nm, "u %i", gnum(1));
  
  sv = findsock("server");
  if (sv)
    sendpack(sv->fd, F_UP, NULL);
  upsocks++;
  
  fstat(fileno(cur->f), &st);
  cur->csz = 0;
  cur->sz = st.st_size-sz;
  cur->bsz = sz;
  cur->g = 1;
  cur->sk = m;
  deltimer(cur->timer);
  
  /* send the file size */
  ssock(m->fd, "%lu", st.st_size);
  
  wp(win, "* Sending file \"%s\" to %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

/* we requested a file from a firewalled client. The remote client has
 * connected to us and is about to start uploading the file */
int dosend(WINDOW *win, sock_t *m)
{
  char *buf, *nm, *fn, *t;
  unsigned long sz;
  file_t *cur;
  struct stat st;
  sock_t *sv;
  char **tok;
  int i, cnt;
  
  if (rsock(m->fd, &buf) == -1)
  {
    /* close(m->fd); */
    return(0);
  }

  tok = form_tokso(buf, &cnt);
  
  /* the remote client should have sent a string of the form 
   * <remotenick> "<filename>" <size> */

  if (cnt < 3)
  {
    wp(win, ""RED"* Bogus SEND command from remote client (%s) [SEND%s]"WHITE"\n", getpeerip(m->fd), quote(buf));
    /* maybe print IP address as well? */
    drw(win);
    ssock(m->fd, "INVALID REQUEST\n");
    free(buf);
    /* close(m->fd); */
    for (i=0;i<cnt;i++)
      free(tok[i]);
    free(tok);
    return(0);
  }
  
  nm = strdup(tok[0]);
  fn = strdup(tok[1]);
  sz = atof(tok[2]);
  /* sz is unsigned long, why not use atol()? Is it because atol()
   * returns a (signed) long int, and we want that one extra bit of 
   * number space that an unsigned long int gives?? */

  free(buf);
  
  for (i=0;i<cnt;i++)
    free(tok[i]);
  free(tok);
  
  /* can remote client send "INVALID REQUEST" after a "SEND" ? */
  if (!strcasecmp(nm, "FILE")) /* "FILE NOT SHARED" or "FILE NOT FOUND" */
  {
    wp(win, "%s* Error downloading from firewalled remote client " \
        "[FILE NOT SHARED]%s\n", RED, WHITE);
    /* maybe print IP address and/or try to identify the nick by searching
     * through download list? */
    drw(win);
    /* the download this error message corresponds to will be removed from
     * the download list when the item's timer times out */
    free(nm);
    free(fn);
    /* close(m->fd); */
    return(0);
  }
  
  if (strchr(fn, '/'))
    t = strrchr(fn, '/')+1;
  else if (strchr(fn, '\\'))
    t = strrchr(fn, '\\')+1;
  else
    t = fn;
  
  if (!sz)
  {
    rsock(m->fd, &buf);
      
    wp(win, "* Error downloading \"%s\" from %s (returned a size of 0) " \
        "[%s]\n", t, nm, buf?quote(buf):"");
    drw(win);
    cur = ffile(down, nm, t);
    if (cur)
      ddownload(win, cur);
    free(fn);
    free(nm);
    /* close(m->fd); */
    return(0);
  }
  
  cur = ffile(down, nm, t);

  if (!cur)
  {
    wp(win, "%s* Error: %s tried to send (push) \"%s\" but the file is not "\
        "in your download list!%s\n", RED, nm, t, WHITE);
    drw(win);
    /* close(m->fd); */
    free(nm);
    free(fn);
    return(0);
  }
  free(nm);
  free(fn);
  
  m->d = (void *)cur;
  m->func = gfile;
  /* rename connection (i.e. sock_t *m) to "d #" (used to be "conn") */
  free(m->nm);
  m->nm = NULL;
  msprintf(&m->nm, "d %i", gnum(0));

  sv = findsock("server");
  if (sv)
    sendpack(sv->fd, F_DOWN, NULL);
  downsocks++;
  
  fstat(fileno(cur->f), &st);
  cur->bsz = st.st_size;
  cur->csz = 0;
  cur->sz = sz - st.st_size;
  cur->g = 1;
  cur->sk = m;
  deltimer(cur->timer);
  
  /* send offset */
  ssock(m->fd, "%lu", cur->bsz);
  
  wp(win, "* Getting file \"%s\" from %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

/* try to open a port that is specified by the string
   dataport. Dataport must either be a port number such as "6699" or a
   range such as "6699-6799". Return port number used if successful,
   or -1 if failure (with errno set). If port 0 is requested, return 0
   without opening anything - this probably was configured by the user
   if we're behind a firewall. */

int initfserv(char *dataport)
{
  struct sockaddr_in me;
  int s, on = 1;
  int port, first, last;
  char *p;

  if (!dataport)
    return(0);
  
  /* parse dataport */
  first = last = strtol(dataport, &p, 10);
  while (p && isspace(*p)) {
    p++;
  }
  if (p && *p=='-') { /* range given? parse second part */
    last = strtol(p+1, &p, 10);
  }
  if (!p || *p!='\0') { /* parse error */
    errno = EINVAL;
    return -1;
  }

  s = socket(AF_INET, SOCK_STREAM, 0);
  if (s == -1)
    return(-1);
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  
  for (port = first; port <= last; port++) {
    me.sin_addr.s_addr = INADDR_ANY;
    me.sin_port = htons(port);
    me.sin_family = AF_INET;
    
    if (bind(s, (struct sockaddr *)&me, sizeof(me)) != -1) { 
      /* success */
      listen(s, 5);
      addsock(s, "fserv", S_R, gconn);
      return(port);
    }
  }
  return(-1);  /* overall failure */
}

void closefserv()
{
  sock_t *sk;
  
  sk = findsock("fserv");
  if (!sk)
    return;
  
  delsock(sk->fd);
}

/* We are about to upload a file to a remote client. We had to initiate
 * the connection because we are firewalled and cannot accept incoming
 * connections. */
int initsend(WINDOW *win, sock_t *m)
{
  int s = m->fd, r;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;
  struct stat st;
  char c;
  
  /* read the initial "1" from remote client */
  if (recv(s, &c, 1, 0) <= 0)
  {
    wp(win, "%s* Error sending file to %s: connection refused by remote client%s\n", RED, cur->nm, WHITE);
    drw(win);
    dupload(cur);
    /* close(s); */
    return(0);
  }

  if (!cur->f)
    return(0);
  
  fstat(fileno(cur->f), &st);
  
  r = ssock(s, "SEND");
  
  /* send "mynick filename filesize" */
  r = ssock(s, "%s \"%s\" %lu", info.user, cur->rfn, st.st_size);
  
  if (r <= 0)
  {
    wp(win, "%s* %s is misconfigured%s\n", RED, cur->nm, WHITE);
    drw(win);
    sk = findsock("server");
    if (sk)
      sendpack(sk->fd, F_MISCONFIGURE, "%s", cur->nm);
    dupload(cur);
    /* close(m->fd); */
    return(0);
  }
  
  m->func = ssize;
  m->t = S_R;
  cur->sz = st.st_size;
  cur->csz = 0;
  
  return(1);
}

int ssize(WINDOW *win, sock_t *m)
{
  char buf[32]; /* note: don't need malloc. */
  /* Use of "sizeof(buf)" below used to be erroneous, as buf was a pointer */
  int n=1, s = m->fd;
  file_t *cur = (file_t *)m->d;
  struct timeval tv;
  fd_set fs;
  
  memset(buf, 0, 32);
  
  recv(s, buf, sizeof(buf), 0);
  
  while (n)
  {
    FD_ZERO(&fs);
    FD_SET(s, &fs);
  
    tv.tv_usec = 50;
    tv.tv_sec = 0;
  
    n = select((s+1), &fs, NULL, NULL, &tv);
  
    if (FD_ISSET(s, &fs))
    {
      if (recv(s, buf+strlen(buf), sizeof(buf), 0) <= 0)
      {
        wp(win, "%s* Error sending file to %s%s\n", RED, cur->nm, WHITE);
        drw(win);
        dupload(cur);
        /* close(m->fd); */
        return(0);
      }
    }
  }
  
  fseek(cur->f, strtoul(buf, (char **)NULL, 10), SEEK_SET);
  
  cur->bsz = strtoul(buf, (char **)NULL, 10);
  cur->sz -= cur->bsz;
  cur->g = 1;
  deltimer(cur->timer);
  
  m->func = sfile;
  m->t = S_W;
  
  wp(win, "* Sending file \"%s\" to %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

/* we have connected to a remote client and we are about to issue a request
 * for a file */
int initget(WINDOW *win, sock_t *m)
{
  int s = m->fd, r;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;
  struct stat st;
  char ch;
  
  /* remote client should send a '1' (ASCII character 49) */
  /* (this used to be in gsize, presumably erroneously so). */
  if (recv(s, &ch, 1, 0) <= 0)
  {
    wp(win, "%s* Error getting file (%s) from %s: connection refused by remote client%s\n", RED, cur->fn, cur->nm, WHITE);
    drw(win);
    ddownload(win, cur);
    /* close(m->fd); */
    return(0);
  }
  
  if (!cur)
    return(0);
  
  if (!cur->f)
    return(0);
  
  fstat(fileno(cur->f), &st);
  
  /* note local file was already opened for append.  st.st_size is the
     file size; this is also the location of the first character to be
     sent by the remote host, in case of a resume. */

  r = ssock(s, "GET");
  r = ssock(s, "%s \"%s\" %lu", info.user, cur->rfn, st.st_size);
  
  if (r <= 0)
  {
    if (r<0)
      wp(win, "%s* Error getting file (%s) from %s, remote client is misconfigured: %s%s\n", RED, cur->fn, cur->nm, sys_errlist[errno], WHITE);
    else
      wp(win, "%s* Error getting file (%s) from %s, remote client is misconfigured%s\n", RED, cur->fn, cur->nm, WHITE);
    drw(win);
    sk = findsock("server");
    if (sk)
      sendpack(sk->fd, F_MISCONFIGURE, "%s", cur->nm);
    ddownload(win, cur);
    /* close(m->fd); */
    return(0);
  }
  
  m->func = gsize;
  m->t = S_R;
  cur->bsz = st.st_size;
  cur->csz = 0;
  
  return(1);
}

/* continue the GET started by initget, now receive remote host's answer */

int gsize(WINDOW *win, sock_t *m)
{
  char buf[32];
  int i, s = m->fd;
  file_t *cur = (file_t *)m->d;
  sock_t *sk;

  /* read file size. drscholl's napster spec says:
   * "keep reading until you hit a character that is not a digit" */
  for (i=0; i<31;i++) /* but don't overflow buf[] ! */
  {
    if (recv(s, &buf[i], 1, MSG_PEEK) <= 0 && !i)
    {
      wp(win, "%s* Error getting (%s) from %s: remote client is misconfigured%s\n", RED, cur->fn, cur->nm, WHITE);
      drw(win);
      sk = findsock("server");
      if (sk)
        sendpack(sk->fd, F_MISCONFIGURE, "%s", cur->nm);
      ddownload(win, cur);
      /* close(m->fd); */
      return(0);
    }
    /* abort only if first character is not a digit.
     * this catches messages such as "INVALID REQUEST" or "FILE NOT SHARED"
     * but does not barf on ID3 tags that follow the file size.
     * (Before, an error was signaled if an alphabetic character 
     * followed 0 or more numeric characters. This caused a problem with
     * mp3 files that start with ID3 tags.) -NBL */
    if (!isdigit(buf[i]) && (i == 0))
    {
      recv(s, buf, 32, 0);
      buf[31]=0;          /* make sure it's 0 terminated */
      wp(win, "* Error getting \"%s\" from %s [%s]\n", cur->fn, cur->nm, quote(buf));
      drw(win);
      ddownload(win, cur);
      /* close(m->fd); */
      return(0);
    }
    if (!isdigit(buf[i]))
      break;
    recv(s, &buf[i], 1, 0);
  }
  /* get rid of the non-digit that was just read
   * (or if i reached 31, make sure that buf[] is '\0'-terminated) */
  buf[i] = '\0'; 

  if (nvar("debug") == 2)
  {
    wp(win, ""DARK GREEN"<-- [from %d=%s] <%s>"WHITE"\n", s, m->nm, quote(buf));
    drw(win);
  }
  
  cur->sz = atol(buf)-cur->bsz;
  cur->g = 1;
  deltimer(cur->timer);
  
  m->func = gfile;
  
  wp(win, "* Getting file \"%s\" from %s (%lu bytes)\n", cur->fn, cur->nm, cur->sz);
  drw(win);
  
  cur->st = time(NULL);
  
  return(1);
}

int sfile(WINDOW *win, sock_t *m)
{
  char buf[2049], *st, *end; /* note: don't need malloc for buf */
  file_t *cur = (file_t *)m->d;
  FILE *f;
  int n;
  time_t t;
  sock_t *sk;

  memset(buf, 0, 2049);
  /* is this mix of low-level and high-level i/o safe? Why use FILE*
   object if using low-level i/o? -PS */

  /* I once got a segfault on the following line, and it turned out
     (FILE *)(cur->f) pointed into unreadable memory - probably this
     file had been closed before we got here. Need to find and fix
     this bug. -PS */

  n = read(fileno(cur->f), buf, 2048); 
  if (n <= 0)
  {
    t = time(NULL);
    wp(win, "* Finished sending \"%s\" to %s (%lu of %lu bytes) in %lu\n", cur->fn, cur->nm, cur->csz, cur->sz, time(NULL)-cur->st);
    drw(win);
    if (info.logfile) {
      f = fopen(info.logfile, "a");
      if (f) {
	st = strdup(ctime(&cur->st));
	st[strlen(st)-1] = 0;
	end = strdup(ctime(&t));
	end[strlen(end)-1] = 0;
	fprintf(f, "S %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
	fclose(f);
	free(st);
	free(end);
      }
    }
    dupload(cur);
    /* close(m->fd); */
    return(0);
  }
  
  if (send(m->fd, buf, n, 0) <= 0)
  {
    t = time(NULL);
    wp(win, "* Transfer interrupted while sending \"%s\" to %s (%lu of %lu bytes): %s\n", cur->fn, cur->nm, cur->csz, cur->sz, sys_errlist[errno]);
    drw(win);
    
    if (errno==EPIPE && nvar("tracksigpipes")==1) {
      wp(win, "### SIGPIPE: will finger user %s to collect data on which clients have this problem.\n", cur->nm);
      drw(win);
      sk = findsock("server");
      if (sk)
	sendpack(sk->fd, F_WHOIS, "%s", cur->nm);
    }

    if (info.logfile) {
      f = fopen(info.logfile, "a");
      if (f) {
	st = strdup(ctime(&cur->st));
	st[strlen(st)-1] = 0;
	end = strdup(ctime(&t));
	end[strlen(end)-1] = 0;
	fprintf(f, "SI %s %s \"%s\" %s\n", st, cur->nm, cur->fn, end);
	fclose(f);
	free(st);
	free(end);
      }
    }
    dupload(cur);
    /* close(m->fd); */
    return(0);
  }
  
  cur->csz += n;
  
  return(1);
}

/* receive a raw file on socket, after initget and gsize did the
   initial communication. */
int gfile(WINDOW *win, sock_t *m)
{
  char buf[2049], *st1, *end; /* note: don't need malloc for buf */
  file_t *cur = (file_t *)m->d;
  FILE *f;
  int n;
  time_t t;

  memset(buf, 0, 2049);
  if ((n = recv(m->fd, buf, 2048, 0)) <= 0)
  {
    t = time(NULL);

    if (cur->csz == cur->sz) { /* complete download */
      wp(win, "* Finished transfer of \"%s\" from %s (%lu of %lu bytes in %lu seconds)\n", cur->fn, cur->nm, cur->csz, cur->sz, t-cur->st);
      drw(win);
    } else {                   /* imcomplete download */
      wp(win, "* Transfer of \"%s\" from %s interrupted (%lu of %lu bytes in %lu seconds)\n", cur->fn, cur->nm, cur->csz, cur->sz, t-cur->st);
      drw(win);
    }
    
    if (info.logfile) {
      f = fopen(info.logfile, "a");
      if (f) {
	st1 = strdup(ctime(&cur->st));
	st1[strlen(st1)-1] = 0;
	end = strdup(ctime(&t));
	end[strlen(end)-1] = 0;
	if (cur->csz != cur->sz)
	  fprintf(f, "RI %s %s \"%s\" %s\n", st1, cur->nm, cur->fn, end);
	else
	  fprintf(f, "R %s %s \"%s\" %s\n", st1, cur->nm, cur->fn, end);
	fflush(f);
	free(st1);
	free(end);
      }
    }

    /* note: further handling of the file (closing the file, moving it
       to the download folder if appropriate, adding MD5 information
       for incompletes, etc, will happen in ddownload(), because that
       is more general and also catches downloads that were deleted by
       the user. */
    ddownload(win, cur);

    /* close(m->fd); */
    return(0);
  }
  write(fileno(cur->f), buf, n);
  
  cur->csz += n;
  
  return(1);
}
