/* valaconditionalexpression.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaconditionalexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>




struct _ValaConditionalExpressionPrivate {
	ValaExpression* _condition;
	ValaExpression* _true_expression;
	ValaExpression* _false_expression;
};

#define VALA_CONDITIONAL_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CONDITIONAL_EXPRESSION, ValaConditionalExpressionPrivate))
enum  {
	VALA_CONDITIONAL_EXPRESSION_DUMMY_PROPERTY
};
static void vala_conditional_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_conditional_expression_real_is_pure (ValaExpression* base);
static gpointer vala_conditional_expression_parent_class = NULL;
static void vala_conditional_expression_finalize (ValaCodeNode* obj);



/**
 * Creates a new conditional expression.
 *
 * @param cond       a condition
 * @param true_expr  expression to be evaluated if condition is true
 * @param false_expr expression to be evaluated if condition is false
 * @return           newly created conditional expression
 */
ValaConditionalExpression* vala_conditional_expression_construct (GType object_type, ValaExpression* cond, ValaExpression* true_expr, ValaExpression* false_expr, ValaSourceReference* source) {
	ValaConditionalExpression* self;
	g_return_val_if_fail (cond != NULL, NULL);
	g_return_val_if_fail (true_expr != NULL, NULL);
	g_return_val_if_fail (false_expr != NULL, NULL);
	g_return_val_if_fail (source != NULL, NULL);
	self = ((ValaConditionalExpression*) (g_type_create_instance (object_type)));
	vala_conditional_expression_set_condition (self, cond);
	vala_conditional_expression_set_true_expression (self, true_expr);
	vala_conditional_expression_set_false_expression (self, false_expr);
	vala_code_node_set_source_reference (((ValaCodeNode*) (self)), source);
	return self;
}


ValaConditionalExpression* vala_conditional_expression_new (ValaExpression* cond, ValaExpression* true_expr, ValaExpression* false_expr, ValaSourceReference* source) {
	return vala_conditional_expression_construct (VALA_TYPE_CONDITIONAL_EXPRESSION, cond, true_expr, false_expr, source);
}


static void vala_conditional_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaConditionalExpression * self;
	self = ((ValaConditionalExpression*) (base));
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept (((ValaCodeNode*) (self->priv->_condition)), visitor);
	vala_code_node_accept (((ValaCodeNode*) (self->priv->_true_expression)), visitor);
	vala_code_node_accept (((ValaCodeNode*) (self->priv->_false_expression)), visitor);
	vala_code_visitor_visit_conditional_expression (visitor, self);
	vala_code_visitor_visit_expression (visitor, ((ValaExpression*) (self)));
}


static gboolean vala_conditional_expression_real_is_pure (ValaExpression* base) {
	ValaConditionalExpression * self;
	self = ((ValaConditionalExpression*) (base));
	return vala_expression_is_pure (self->priv->_condition) && vala_expression_is_pure (self->priv->_true_expression) && vala_expression_is_pure (self->priv->_false_expression);
}


ValaExpression* vala_conditional_expression_get_condition (ValaConditionalExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_condition;
}


void vala_conditional_expression_set_condition (ValaConditionalExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_condition = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_condition == NULL ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL))), _tmp2);
}


ValaExpression* vala_conditional_expression_get_true_expression (ValaConditionalExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_true_expression;
}


void vala_conditional_expression_set_true_expression (ValaConditionalExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_true_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_true_expression == NULL ? NULL : (self->priv->_true_expression = (vala_code_node_unref (self->priv->_true_expression), NULL))), _tmp2);
}


ValaExpression* vala_conditional_expression_get_false_expression (ValaConditionalExpression* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_false_expression;
}


void vala_conditional_expression_set_false_expression (ValaConditionalExpression* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_false_expression = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : vala_code_node_ref (_tmp1))), (self->priv->_false_expression == NULL ? NULL : (self->priv->_false_expression = (vala_code_node_unref (self->priv->_false_expression), NULL))), _tmp2);
}


static void vala_conditional_expression_class_init (ValaConditionalExpressionClass * klass) {
	vala_conditional_expression_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_conditional_expression_finalize;
	g_type_class_add_private (klass, sizeof (ValaConditionalExpressionPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_conditional_expression_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_conditional_expression_real_is_pure;
}


static void vala_conditional_expression_instance_init (ValaConditionalExpression * self) {
	self->priv = VALA_CONDITIONAL_EXPRESSION_GET_PRIVATE (self);
}


static void vala_conditional_expression_finalize (ValaCodeNode* obj) {
	ValaConditionalExpression * self;
	self = VALA_CONDITIONAL_EXPRESSION (obj);
	(self->priv->_condition == NULL ? NULL : (self->priv->_condition = (vala_code_node_unref (self->priv->_condition), NULL)));
	(self->priv->_true_expression == NULL ? NULL : (self->priv->_true_expression = (vala_code_node_unref (self->priv->_true_expression), NULL)));
	(self->priv->_false_expression == NULL ? NULL : (self->priv->_false_expression = (vala_code_node_unref (self->priv->_false_expression), NULL)));
	VALA_CODE_NODE_CLASS (vala_conditional_expression_parent_class)->finalize (obj);
}


GType vala_conditional_expression_get_type (void) {
	static GType vala_conditional_expression_type_id = 0;
	if (vala_conditional_expression_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaConditionalExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_conditional_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaConditionalExpression), 0, (GInstanceInitFunc) vala_conditional_expression_instance_init, NULL };
		vala_conditional_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaConditionalExpression", &g_define_type_info, 0);
	}
	return vala_conditional_expression_type_id;
}




