<?php
// $Horde: horde/lib/Prefs/ldap.php,v 1.14.2.3 2001/12/19 12:41:42 jan Exp $

require_once 'PEAR.php';

Horde::functionCheck('ldap_connect', true,
    'Prefs_ldap: Required LDAP functions were not found.');

/**
 * Preferences storage implementation for PHP's LDAP extention.
 *
 * Required values for $params:
 *      'hostspec'      The hostname of the LDAP server.
 *      'basedn'        The base DN for the LDAP server.
 *      'uid'           The username search key.
 *      'username'      The user as which to bind for write operations.
 *      'password'      'username's password for bind authentication.
 *      'always_bind'   Always bind to the LDAP server (store and retrieve).
 *
 * @author  Jon Parise <jon@horde.org>
 * @version $Revision: 1.14.2.3 $
 * @since   Horde 1.3
 * @package horde.prefs
 */
class Prefs_ldap extends Prefs {

    /** Hash containing connection parameters.
        @var array $params */
    var $params = array();

    /** Handle for the current LDAP connection.
        @var int $connection */
    var $connection;

    /** Boolean indicating whether or not we're connected to the LDAP server.
        @var boolean $connected */
    var $_connected = false;

    /** String holding the user's DN.
        @var string $dn */
    var $dn = '';

    /** Boolean indicating whether $connection has bound to the LDAP
        server as an authenticated user (non-anonymous).
        @var boolean $anonymous */
    var $anonymous = true;
    
    
    /**
     * Constructs a new LDAP preferences object.
     *
     * @param string $user      The user who owns these preferences.
     * @param string $password  The password associated with $user.
     * @param string $scope     The current application scope.
     * @param array  $params    A hash containing connection parameters.
     * @param boolean $caching  (optional) Should caching be used?
     */
    function Prefs_ldap($user, $password, $scope = '', $params = array(),
        $caching = false)
    {
        $this->user = $user;
        $this->scope = $scope;
        $this->params = $params;
        $this->caching = $caching;

        /*
         * If $params['username'] is empty, set it to the name of the
         * current user.  Also, the $params['password'] to the current
         * user's password.
         *
         * Note: This assumes the user is allowed to modify their own LDAP
         *       entry.
         */
        if (empty($this->params['username'])) {
            $this->params['username'] = $user;
            $this->params['password'] = $password;
        }

        /* If the 'always_bind' parameter isn't specified, default to false. */
        if (empty($this->params['always_bind'])) {
            $this->params['always_bind'] = false;
        }
    }

    /**
     * Opens a connection to the LDAP server.
     *
     * @param boolean $bind  Boolean indicating whether or not to bind to the
     *                       LDAP server using an authenticated connected.  
     *
     * @return mixed         True on success or a PEAR_Error object on failure.
     */
    function _connect($bind = false)
    {
        if (!is_array($this->params)) {
            Horde::fatal(new PEAR_Error(_("No configuration information specified for LDAP Preferences.")), __FILE__, __LINE__);
        }
        if (!isset($this->params['hostspec'])) {
            Horde::fatal(new PEAR_Error(_("Required 'hostspec' not specified in preferences configuration.")), __FILE__, __LINE__);
        }

        /* Connect to the LDAP server anonymously. */
        $conn = ldap_connect($this->params['hostspec']);
        if (!$conn) {
            Horde::fatal(new PEAR_Error(sprintf(_("Failed to open an LDAP connection to %s."), $this->params['hostspec'])), __FILE__, __LINE__);
        }

        /* Search for the user's full DN. */
        $search = ldap_search($conn, $this->params['basedn'],
                              $this->params['uid'] . '=' . $this->user,
                              array($this->params['uid']));
        $result = ldap_get_entries($conn, $search);
        if (is_array($result)) {
            $this->dn = $result[0]['dn'];
        } else {
            Horde::fatal(new PEAR_Error(sprintf(_("No entry found for %s."), $this->params['uid'])), __FILE__, __LINE__);
        }

        /* Store the connection handle at the instance level. */
        $this->connection = $conn;
        $this->_connected = true;
        $this->anonymous = true;

        /* Bind to the LDAP server, if requested. */
        if ($bind) {
            $this->_bind();
        }

        return true;
    }

    /**
     * Bind to the LDAP server using authentication.
     *
     * @return mixed       True on success or a PEAR_Error object on failure.
     */
    function _bind()
    {
        /*
         * If we're not already connected, return an error.  Invoking
         * connect() again could lead to circular function calls.
         */
        if (!$this->_connected) {
            Horde::fatal(new PEAR_Error(_("Attempt to bind without a valid connection.")), __FILE__, __LINE__);
        }

        /* Make sure the connection values for ldap_bind() were provided. */
        if (!isset($this->params['basedn'])) {
            Horde::fatal(new PEAR_Error(_("Required 'basedn' not specified in preferences configuration.")), __FILE__, __LINE__);
        }
        if (!isset($this->params['uid'])) {
            Horde::fatal(new PEAR_Error(_("Required 'uid' not specified in preferences configuration.")), __FILE__, __LINE__);
        }
        if (!isset($this->params['username'])) {
            Horde::fatal(new PEAR_Error(_("Required 'username' not specified in preferences configuration.")), __FILE__, __LINE__);
        }
        if (!isset($this->params['password'])) {
            Horde::fatal(new PEAR_Error(_("Required 'password' not specified in preferences configuration.")), __FILE__, __LINE__);
        }

        /* Search for the full DN and bind to the LDAP server as that user. */
        $search = ldap_search($this->connection, $this->params['basedn'],
            $this->params['uid'] . '=' . $this->params['username'],
            array($this->params['uid']));
        $result = ldap_get_entries($this->connection, $search);
        if (is_array($result)) {
            $this->dn = $result[0]['dn'];
            $bind = @ldap_bind($this->connection, $this->dn, $this->params['password']);
            if (!$bind) {
                Horde::fatal(new PEAR_Error(sprintf(_("Bind to server %s, dn %s as %s failed: [%d] %s"),
                                                    $this->params['hostspec'],
                                                    $this->dn,
                                                    $this->params['uid'],
                                                    ldap_errno($this->connection),
                                                    ldap_error($this->connection))), __FILE__, __LINE__);
            }
        } else {
            Horde::fatal(new PEAR_Error(sprintf(_("Attempt to bind failed: no entry found for %s."), $this->params['uid'])), __FILE__, __LINE__);
        }

        $this->anonymous = false;

        return true;
    }

    /**
     * Disconnect from the LDAP server and clean up the connection.
     *
     * @return boolean     true on success, false on failure.
     */
    function _disconnect()
    {
        $this->dn = '';
        $this->_connected = false;

        return ldap_close($this->connection);
    }

    /**
     * Retrieves the requested set of preferences from the user's LDAP
     * entry.
     *
     * @param optional array $prefs  An array listing the preferences to
     *                     retrieve. If not specified, retrieve all of the
     *                     preferences listed in the $prefs hash.
     *
     * @return mixed       True on success or a PEAR_Error object on failure.
     */
    function retrieve($prefs = array())
    {
        /*
         * If a list of preferences to retrieve hasn't been provided in
         * $prefs, assume all preferences are desired.
         */
        if (count($prefs) == 0) {
            $prefs = Prefs::listAll();
        }
        if (!is_array($prefs) || (count($prefs) == 0)) {
            return (new PEAR_Error(_("No preferences are available.")));
        }

        /* Attempt to pull the values from the session cache first. */
        if ($this->cacheLookup($prefs)) {
            return true;
        }

        /* If we're not already connected, invoke the connect() method. */
        if (!$this->_connected) {
            $this->_connect($this->params['always_bind']);
        }

        $search = ldap_search($this->connection, $this->params['basedn'],
                              $this->params['uid'] . '=' . $this->user, $prefs);
        if ($search) {
            $result = ldap_get_entries($this->connection, $search);
        } else {
            Horde::fatal(new PEAR_Error(_("Failed to connect to LDAP preferences server.")), __FILE__, __LINE__);
        }

        if (isset($result) && is_array($result)) {
            /*
             * Set the requested values in the $this->prefs hash based on
             * the contents of the LDAP result.
             *
             * Note that Prefs::setValue() can't be used here because of the
             * check for the "changeable" bit.  We want to override that
             * check when populating the $this->prefs hash from the LDAP
             * server.
             */
            foreach ($prefs as $pref) {
                if (isset($result[0][$pref][0])) {
                    $this->prefs[$pref]['val'] = $result[0][$pref][0];
                }
            }
        } else {
            Horde::logMessage(_("No preferences were retrieved."), __FILE__, __LINE__);
            return;
        }

        /* Update the session cache. */
        $this->cacheUpdate();

        return true;
    }

    /**
     * Stores preferences to LDAP server.
     *
     * @param array $prefs (optional) An array listing the preferences to be
     *                     stored.  If not specified, store all of the
     *                     preferences listed in the $prefs hash.
     *
     * @return mixed       True on success or a PEAR_Error object on failure.
     */
    function store($prefs = array())
    {
        /*
         * If a list of preferences to store hasn't been provided in
         * $prefs, assume all preferences are desired.
         */
        if (count($prefs) == 0) {
            $prefs = Prefs::listAll();
        }
        if (!is_array($prefs)) {
            return (new PEAR_Error(_("No preferences are available.")));
        }

        /* Check for any "dirty" preferences. */
        $dirty_prefs = array();
        foreach ($prefs as $pref) {
            if (Prefs::isDirty($pref)) {
                $dirty_prefs[] = $pref;
            }
        }

        /*
         * If no "dirty" preferences were found, there's no need to update
         * the LDAP server.  Exit successfully.
         */
        if (count($dirty_prefs) == 0) {
            return true;
        }

        /* If we're not already connected, invoke the connect(). */
        if (!$this->_connected) {
            $this->_connect(true);
        } else {
            /* Bind if the LDAP server if we haven't already. */
            if ($this->anonymous) {
                $this->_bind();
            }
        }

        /*
         * Build a hash of the preferences and their values that need to be
         * stored in the LDAP server.
         */
        $new_values = array();
        foreach($dirty_prefs as $pref) {
            $new_values[$pref] = Prefs::getValue($pref);
        }

        /* Sent the hash of new values off to the LDAP server. */
        if (!ldap_modify($this->connection, $this->dn, $new_values)) {
            Horde::fatal(_("Unable to modified preferences."), __FILE__, __LINE__);
        }

        /* Attemp to cache the preferences in the session. */
        $this->cacheUpdate();

        return true;
    }


    /**
     * Perform cleanup operations.
     *
     * @param boolean  $all    (optional) Cleanup all Horde preferences.
     */
    function cleanup($all = false)
    {
        /* Close the LDAP connection. */
        if (isset($this->connection)) {
            ldap_close($this->connection);
        }

        parent::cleanup($all);
    }

}
?>
