<?php
// $Horde: horde/lib/Identity.php,v 1.4.2.3 2001/11/16 00:46:54 chuck Exp $

/**
 * This class provides an interface to all identities a user might have.
 * Its methods take care of any restrictions configured site specific in
 * prefs.php and conf.php.
 * 
 * @author  Jan Schneider <jan@ammma.de>
 * @version $Revision: 1.4.2.3 $
 * @since   Horde 1.3.5
 * @package horde.identity
 */
class Identity {
    
    /**
     * Array containing all the user's identities.
     * @var array $identities
     */
    var $identities = array();
    
    /**
     * A pointer to the user's standard identity.
     * This one is used by the methods returning values
     * if no other one is specified.
     * @var integer $default
     */
    var $default = 0;
    
    /**
     * Array containing all of the properties in this identity,
     * excluding the id.
     * @var array $properties
     */
    var $properties = array();
    
    /**
     * Reads all the user's identities from the prefs object or builds
     * a new identity from the standard values given in prefs.php.
     */
    function Identity()
    {
        global $prefs;
        
        $this->default = $prefs->getValue('default_identity');
        $this->identities = @unserialize($prefs->getValue('identities'));
    
        if (!is_array($this->identities) || (count($this->identities) <= 0)) {
            $identity = array('id' => _("Default Identity"));
            foreach ($this->properties as $key) {
                $identity[$key] = $prefs->getValue($key);
            }
            
            $this->identities[] = $identity;
            $this->save();
        }
    }
    
    /**
     * Saves all identities in the prefs backend.
     */
    function save()
    {
        global $prefs;
        
        $prefs->setValue('identities', serialize($this->identities));
        $prefs->setValue('default_identity', $this->default);
        $prefs->store();
    }
    
    /**
     * Adds a new empty identity to the array of identities.
     * @return integer      The pointer to the created identity
     */
    function add()
    {
        $this->identities[] = array();
        return count($this->identities) - 1;
    }
    
    /**
     * Remove an identity from the array of identities
     * @param integer $identity The pointer to the identity to be removed
     * @return array            The removed identity
     */
    function delete($identity)
    {
        $deleted = array_splice($this->identities, $identity, 1);
        $this->save();
        return $deleted;
    }
    
    /**
     * Returns a pointer to the current default identity.
     * @return integer      The pointer to the current default identity
     */
    function getDefault()
    {
        return $this->default;
    }
    
    /**
     * Sets the current default identity.
     * If the identity doesn't exist, the old default identity
     * stays the same.
     * @param integer $identity     The pointer to the new default identity
     * @return boolean              True on success, false on failure
     */
    function setDefault($identity)
    {
        if (isset($this->identities[$identity])) {
            $this->default = $identity;
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * Returns a property from one of the identities. If this value doesn't exist
     * or is locked, the property is retrieved from the prefs backend.
     * @param string $key           The property to retrieve.
     * @param integer $identity     (optional) The identity to retrieve the property from
     * @return mixed                The value of the property.
     */
    function getValue($key, $identity = null)
    {
        global $prefs;
        
        if (!isset($identity) || !isset($this->identities[$identity])) {
            $identity = $this->default;
        }
        
        if (!isset($this->identities[$identity][$key]) || $prefs->isLocked($key)) {
            $val = $prefs->getValue($key);
        } else {
            $val = $this->identities[$identity][$key];
        }
        
        return $val;
    }
    
    /**
     * Returns an array with the specified property from all existing
     * identities.
     * @param string $key       The property to retrieve.
     * @return array            The array with the values from all identities
     */
    function getAll($key)
    {
        
        foreach ($this->identities as $identity => $null) {
            $list[$identity] = $this->getValue($key, $identity);
        }
        return $list;
    }
    
    /**
     * Sets a property with a specified value.
     * @param string $key       The property to set
     * @param mixed $val        The value to which the property should be set
     * @param integer $identity (optional) The identity to set the property in
     * @return boolean          True on success, false on failure (property was locked)
     */
    function setValue($key, $val, $identity = null)
    {
        global $prefs;
        
        if (!isset($identity)) {
            $identity = $this->default;
        }
        
        if (!$prefs->isLocked($key)) {
            $this->identities[$identity][$key] = $val;
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * Returns true if all properties are locked and therefore nothing
     * in the identities can be changed.
     * @return boolean          True if all properties are locked, false otherwise
     */
    function isLocked()
    {
        global $prefs;
        
        foreach ($this->properties as $key) {
            if (!$prefs->isLocked($key)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Returns true if the given address belongs to one of the identities.\
     *
     * @param string $key          The identity key to search.
     * @param string $value        The value to search for in $key.
     *
     * @return boolean             True if the $value was found in $key.
     */
    function hasValue($key, $valueA)
    {
        $list = $this->getAll($key);
        foreach ($list as $valueB) {
            if (strpos(strtolower($valueA), strtolower($valueB)) !== false) {
                return true;
            }
        }
        return false;
    }
    
}
?>
