<?php
// $Horde: horde/lib/Help.php,v 1.26 2001/08/24 02:29:40 jon Exp $

if (!function_exists('xml_parser_create')) {
    die('The XML functions are not available.  Rebuild PHP with --with-xml.');
}

/* Help system constants */
/** @constnt HELP_SOURCE_RAW Raw help in the string. */
define('HELP_SOURCE_RAW', 0);

/** @constant HELP_SOURCE_FILE Help text is in a file. */
define('HELP_SOURCE_FILE', 1);

/** @constant HELP_SOURCE_DB Help comes from a database. */
define('HELP_SOURCE_DB', 2);

/**
 * The Help:: class provides an interface to the online help subsystem.
 *
 * @author  Jon Parise <jon@horde.org>
 * @version $Revision: 1.26 $
 * @since   Horde 1.3
 * @package horde
 */
class Help {

    /**
     * Handle for the XML parser object.
     * @var object $parser
     */
    var $parser = 0;

    /**
     * String buffer to hold the XML help source.
     * @var string $buffer
     */
    var $buffer = '';

    /**
     * String containing the ID of the requested help entry.
     * @var string $req_entry
     */
    var $req_entry = '';

    /**
     * String containing the ID of the current help entry.
     * @var string $cur_entry
     */
    var $cur_entry = '';

    /**
     * String containing the formatted output.
     * @var string $output
     */
    var $output = '';

    /**
     * Boolean indicating whether we're inside a <help> block.
     * @var boolean $in_help
     */
    var $in_help = false;

    /**
     * Boolean indicating whether we're inside the requested block.
     * @var boolean $in_block
     */
    var $in_block = false;

    /**
     * Boolean indicating whether we're inside a <title> block.
     * @var boolean $in_title
     */
    var $in_title = false;

    /**
     * Hash containing an index of all of the help entries.
     * @var array $entries
     */
    var $entries = array();

    /**
     * Hash of user-defined function handlers for the XML elements.
     * @var array $handlers
     */
    var $handlers = array(
        'HELP' =>       'helpHandler',
        'ENTRY' =>      'entryHandler',
        'TITLE' =>      'titleHandler',
        'HEADING' =>    'headingHandler',
        'PARA' =>       'paraHandler',
        'REF' =>        'refHandler'
    );

    /**
     * Constructor
     * 
     * @param int $source   The source of the XML help data, based on the
     *                      HELP_SOURCE_* constants.
     * @param optional string $arg  Source-dependent argument for this
     *                              Help instance.
     */
    function Help($source, $arg = '')
    {
        /* Populate $this->buffer based on $source. */
        switch ($source) {
        case HELP_SOURCE_RAW:
            $this->buffer = $arg;
            break;
        case HELP_SOURCE_FILE:
            if (@file_exists($arg) && ($fp = @fopen($arg, 'r'))) {
                $this->buffer = fread($fp, filesize($arg));
                fclose($fp);
            } else {
                $this->buffer = '';
            }
            break;
        default:
            $this->buffer = '';
            break;
        } // switch
    }

    /**
     * Initialzes the XML parser.
     *
     * @return  boolean     Returns true on success, false on failure.
     */
    function init()
    {
        /* Create a new parser and set its default properties. */
        $this->parser = xml_parser_create();
        xml_set_object($this->parser, $this);
        xml_parser_set_option($this->parser, XML_OPTION_CASE_FOLDING, true);
        xml_set_element_handler($this->parser, 'startElement', 'endElement');
        xml_set_character_data_handler($this->parser, 'defaultHandler');

        return ($this->parser != 0);
    }

    /**
     * Cleans up the Help class resources.
     *
     * @return  boolean     Returns true on success, false on failure.
     */
    function cleanup()
    {
        $this->buffer = '';
        return xml_parser_free($this->parser);
    }

    /**
     * Looks up the requested entry in the XML help buffer.
     *
     * @param string $entry        String containing the entry ID.
     */
    function lookup($entry)
    {
        $this->output = '';
        $this->req_entry = strtoupper($entry);
        if (!$this->parser) $this->init();
        xml_parse($this->parser, $this->buffer, true);
    }

    /**
     * Returns a hash of all of the topics in this help buffer.
     *
     * @return  array       Hash of all of the topics in this buffer.
     */
    function topics()
    {
        if (!$this->parser) $this->init();
        xml_parse($this->parser, $this->buffer, true);
        return $this->entries;
    }

    /**
     * Display the contents of the formatted output buffer.
     *
     * @return  string      Formatted output text.
     */
    function display()
    {
        echo $this->output;
    }

    /**
     * User-defined function callback for start elements.
     *
     * @param object $parser        Handle to the parser instance.
     * @param string $name          The name of this XML element.
     * @param array  $attrs         List of this element's attributes.
     */
    function startElement($parser, $name, $attrs)
    {
        /* Call the assigned handler for this element, if one is available. */
        if (in_array($name, array_keys($this->handlers))) {
            call_user_func(array(&$this, $this->handlers[$name]), true, $attrs);
        }
    }

    /**
     * User-defined function callback for end elements.
     *
     * @param object $parser        Handle to the parser instance.
     * @param string $name          The name of this XML element.
     */
    function endElement($parser, $name)
    {
        /* Call the assigned handler for this element, if one is available. */
        if (in_array($name, array_keys($this->handlers))) {
            call_user_func(array(&$this, $this->handlers[$name]), false);
        }
    }

    /**
     * User-defined function callback for character data.
     *
     * @param object $parser        Handle to the parser instance.
     * @param string $data          String of character data.
     */
    function defaultHandler($parser, $data)
    {
        if ($this->in_title) {
            $this->entries[$this->cur_entry] .= $data;
        }
        if ($this->in_help && $this->in_block) {
            $this->output .= htmlspecialchars(trim($data));
        }
    }

    /**
     * XML element handler for the <help> tag.
     *
     * @param boolean $startTag     Boolean indicating whether this instance is
     *                              a start tag.
     * @param optional array $attrs Additional element attributes.
     */
    function helpHandler($startTag, $attrs = array())
    {
        $this->in_help = ($startTag) ? true : false;
    }

    /**
     * XML element handler for the <entry> tag.
     *
     * @param boolean $startTag     Boolean indicating whether this instance is
     *                              a start tag.
     * @param optional array $attrs Additional element attributes.
     */
    function entryHandler($startTag, $attrs = array())
    {
        if (!$startTag) {
            $this->in_block = false;
        } else {
            $id = strtoupper($attrs['ID']);
            $this->cur_entry = $id;
            $this->entries[$id] = '';
            $this->in_block = ($id == $this->req_entry);
        }
    }

    /**
     * XML element handler for the <title> tag.
     *
     * @param boolean $startTag     Boolean indicating whether this instance is
     *                              a start tag.
     * @param optional array $attrs Additional element attributes.
     */
    function titleHandler($startTag, $attrs = array())
    {
        $this->in_title = $startTag;
        if ($this->in_help && $this->in_block) {
            $this->output .= ($startTag) ? '<h3>' : '</h3>';
        }
    }

    /**
     * XML element handler for the <heading> tag.
     *
     * @param boolean $startTag     Boolean indicating whether this instance is
     *                              a start tag.
     * @param optional array $attrs Additional element attributes.
     */
    function headingHandler($startTag, $attrs = array())
    {
        if ($this->in_help && $this->in_block) {
            $this->output .= ($startTag) ? '<h4>' : '</h4>';
        }
    }

    /**
     * XML element handler for the <para> tag.
     *
     * @param boolean $startTag     Boolean indicating whether this instance is
     *                              a start tag.
     * @param optional array $attrs Additional element attributes.
     */
    function paraHandler($startTag, $attrs = array())
    {
        if ($this->in_help && $this->in_block) {
            $this->output .= ($startTag) ? '<p>' : '</p>';
        }
    }

    /**
     * XML element handler for the <ref> tag.
     *
     * @param boolean $startTag     Boolean indicating whether this instance is
     *                              a start tag.
     * @param optional array $attrs Additional element attributes.
     */
    function refHandler($startTag, $attrs = array())
    {
        global $module, $PHP_SELF;

        if ($this->in_help && $this->in_block) {
            if ($startTag) {
                $this->output .= '<a href="' . $PHP_SELF . '?show=entry';
                $this->output .= '&module=' . basename($module);
                $this->output .= '&topic=' . $attrs['ENTRY'] . '">';
            } else {
                $this->output .= '</a>';
            }
        }
    }
    
    /**
     * Includes the JavaScript necessary to create a new pop-up help window.
     */
    function javascript()
    {
        global $registry;

        include_once $registry->getTemplatePath('horde') . '/javascript/open_help_win.js';
    }

    /**
     * Generates the HTML link that will pop up a help window for the
     * requested topic.
     *
     * @param string $module The name of the current Horde module.
     * @param string $topic  The help topic to be displayed.
     *
     * @return  string       The HTML to create the help link.
     */
    function link($module, $topic)
    {
        global $registry;

        $html = '<a onmouseout="status=\'\';" onmouseover="status=\'';
        $html .= _("Help"). '\'; return true;" ';
        $html .= 'href="" onclick="open_help_win(\'' . $module . '\', \'';
        $html .= $topic . '\'); return false;" tabindex="-1"><img border="0" src="';
        $html .= $registry->getGraphicsPath("horde") . '/help.gif" width="12" ';
        $html .= 'height="17" ';
        $html .= ' alt="' . _("Help") . '" /></a>';

        return $html;
    }

    /**
     * Generates the URL that will pop up a help window for the list
     * of topics.
     *
     * @param string $module The name of the current Horde module.
     *
     * @return string        The HTML to create the help link.
     */
    function listLink($module)
    {
        return "javascript:open_help_win('$module');";
    }
}
?>
