<?php
/**
 * $Horde: horde/lib/Category.php,v 1.8.2.3 2002/01/02 17:05:38 jan Exp $
 * Copyright 1999-2002 Original Author <shuther@bigfoot.com>
 * See the enclosed file COPYING for license information (LGPL).  If you 
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 */

require_once HORDE_BASE . '/lib/Serialize.php';

/**
 * Required values for $params:
 * groupid: define each group of categories we want to build
 *
 * optional call_fill_mode This is the name of a function, similar
 *                         to CategoryObject::getData($mode), do exactly
 *                         the same job, to add some operations.
 *                         prototype array function call_fill_mode($mode, $key)
 */

// Kind of format
/** @constant CATEGORY_FORMAT_LIST List every category in an array [ parent => child ]*/
define('CATEGORY_FORMAT_LIST', 1);

/** @constant CATEGORY_FORMAT_TREE List every category in an array,
    similar to PEAR/html/menu.php */
define('CATEGORY_FORMAT_TREE', 2);

/** @constant CATEGORY_FORMAT_CHIPAR List every category in an array child-parent, similar to the interal format of $categories */
define('CATEGORY_FORMAT_CHIPAR', 3);

/** @constant CATEGORY_FORMAT_FETCH Similar to CATEGORY_FORMAT_CHIPAR, except that we manage an index 0 in an internal array, comes from driver pear/sql */
define('CATEGORY_FORMAT_FETCH', 4);

/** @constant CATEGORY_FORMAT_FLIST Get a full list - an array of keys */
define('CATEGORY_FORMAT_FLIST', 5);

/** @constant CATEGORY_FORMAT_3D Use a specific format, comes from the project olbookmarks - sourceforge / libdrawtree.php
    $data[0][0]['name']="Root";            $data[0][0]['dad']="0.0";
    $data[1][1]['name']="dir1";        $data[1][1]['dad']="0.0";
    $data[2][2]['name']="subdir1";  $data[2][2]['dad']="1.1";
    $data[3][3]['name']="data1"; $data[3][3]['dad']="2.2";
    $data[3][4]['name']="data2"; $data[3][4]['dad']="2.2";
    $data[3][5]['name']="data3"; $data[3][5]['dad']="2.2";
    $data[2][6]['name']="subdir2";  $data[2][6]['dad']="1.1";
    $data[1][7]['name']="dir2";        $data[1][7]['dad']="0.0";
    $data[2][8]['name']="subdir3";  $data[2][8]['dad']="1.7";
    $data[2][9]['name']="subdir4";  $data[2][9]['dad']="1.7";
*/
define('CATEGORY_FORMAT_3D', 6);
define('CATEGORY_FORMAT_3D_NAME', 'name');
define('CATEGORY_FORMAT_3D_DAD', 'dad');
define('CATEGORY_FORMAT_3D_ROOT', 'root');

/** @constant CATEGORY_FORMAT_3D_OPT similar to CATEGORY_FORMAT_3D
@see CATEGORY_FORMAT_3D

We will have something similar to this:
    $data[0]['Root']='Root';
        data[1]['dir1']='Root';
            $data[2]['subdir1']='dir1';
                $data[3]['data1']='dir1';
                $data[3]['data2']='dir1';
                $data[3]['data3']='dir1';
            $data[2]['subdir2']='dir1';
        $data[1]['dir2']='Root';
            $data[2]['subdir3']='dir2';
            $data[2]['subdir4']='dir2';
*/
define('CATEGORY_FORMAT_3D_OPT', 7);

/** @constant CATEGORY_FORMAT_OPT_SPC specific format, optimize to save in a DB.
Such a hierarchy:
A
  B
  C
     D
  E
F

Will be transformed like this:
;A
 ;B
 ;C
  ;D
 ;E
;F

Note the SPACES and the carriage return!! Also, the order is very important at this stage!
The spaces means the depth.

*/
define('CATEGORY_FORMAT_OPT_SPC', 8);

/** @constant CATEGORY_FORMAT_OPT_SPC_SEPARATOR separator between the depth and
    the name of the category
    @note: some characters may not work properly, since we use a regular expression!*/
define('CATEGORY_FORMAT_OPT_SPC_SEPARATOR', ';');

/** @constant CATEGORY_FORMAT_OPT_SPC_CRLF Code used to separate the different lines, migth be a CRLF or something else
 */
define('CATEGORY_FORMAT_OPT_SPC_CRLF', "\n");

/** @constant CATEGORY_FORMAT_OPT_SPC_DEPTH which character defines the depth 
    @note: this paramater is never used, just the length of the 1st part is rigth now used
*/
define('CATEGORY_FORMAT_OPT_SPC_DEPTH', ' ');

/** @constant CATEGORY_FORMAT_OPT_NB
    @see CATEGORY_FORMAT_OPT_SPC
    Same thing except that the number of spaces are replaced with a number from 0 to ... to design the spaces
*/
define('CATEGORY_FORMAT_OPT_NB', 9);


// Format used to serialize
define('CATEGORY_SERIALIZE_FORMAT', SERIALIZEUNIT_BASIC);

/**
 * The Category:: class provides a common abstracted interface into the
 * various backends for the Horde system.
 * A category is just a title that is saved in the page for the null driver
 * or can be saved in a database to be accessed from everywhere. Every category
 * must have a different name (for a same groupid). A category may have
 * different parent categories.
 *
 * @todo it migth be very interesting to transform the code, and using
 * XML/DOM instead, we could imagine it as a new driver, since we would require
 * a PHP extension for that
 * another approach is to use something similar to JTree (form Java)/and a
 *  node notion
 * some constants such as CATEGORY_FORMAT_3D_NAME should be inserted in $params
 *
 * @note: -1 is nknows as the root, but it is a STRING, it is important because
 * database methods in PHP are working only with string, so we avoid confusion.
 *
 * @author  Stephane Huther <shuther@bigfoot.com>
 * @author  Chuck Hagenbuch <chuck@horde.org>
 * @version $Revision: 1.8.2.3 $
 * @since   Horde 2.1
 * @package horde.category
 */
class Category {

    /**
     * Array of all categories: indexed by name = parent
     * right now, the format is array(name => array(parents)), but it could change
     * to an XML/DOM organization
     */
    var $categories = array();

    /**
     * Array of extended status for categories: indexed by name= serialized status
     */
    var $data = array();

    /**
     * Internal format to store the categories - ,must not be changed
     * @access read only
     */
    var $internalFormat = CATEGORY_FORMAT_CHIPAR;

    /**
     * Say if the data are cached in the memory
     * just a flag
     * not already used
     */
    var $cache = true;

    /**
     * Hash containing connection parameters.
     * @var array $params
     */
    var $params = array();
    
    /**
     * Constructor
     * @param array  $params A hash containing any additional
     *                       configuration or connection parameters a subclass
     *                       might need.
     *                       here, we need  'groupid' = a constant that defines
     *                       in each group we will work
     */
    function Category($params)
    {
        $this->params = $params;
    }

    /**
     * Attempts to return a concrete Category instance based on $driver.
     * 
     * @param string $driver The type of concrete Category subclass to return.
     *                       This is based on the storage driver ($driver). The
     *                       code is dynamically included.
     * @param array  $params A hash containing any additional
     *                       configuration or connection parameters a subclass
     *                       might need.
     *                       here, we need 'groupid' = a string that defines
     *                       top-level categories of categories.
     *
     * @return object Category   The newly created concrete Category instance,
     *                       or false on an error.
     */
    function &factory($driver, $params)
    {
        $driver = strtolower($driver);

        if (empty($driver) || (strcmp($driver, 'none') == 0)) {
            return new Category($params);
        }
        include_once dirname(__FILE__) . '/Category/' . $driver . '.php';
        $class = 'Category_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }
    
    /**
     * Attempts to return a reference to a concrete Category instance based on
     * $driver. It will only create a new instance if no Category instance
     * with the same parameters currently exists.
     *
     * This should be used if multiple permissions sources (and, thus,
     * multiple Category instances) are required.
     *
     * This method must be invoked as: $var = &Category::singleton()
     *
     * @param string $driver The type of concrete Category subclass to return.
     *                       This is based on the storage driver ($driver). The
     *                       code is dynamically included.
     * @param array  $params (optional) A hash containing any additional
     *                       configuration or connection parameters a subclass
     *                       might need.
     *
     * @return object Category  The concrete Category reference, or false on an
     *                        error.
     */
    function &singleton($driver, $params = array())
    {
        static $instances;
        if (!isset($instances)) $instances = array();

        $signature = md5(strtolower($driver) . '][' . implode('][', $params));
        if (!isset($instances[$signature])) {
            $instances[$signature] = &Category::factory($driver, $params);
        }

        return $instances[$signature];
    }

    /**
     * Add a category
     * 
     * Note: there is no check against circular reference.
     *
     * @param mixed $name    The name of the category.
     *                       If it is a string, just the name, if it is a
     *                       sub-class of CategoryObject, we get the real
     *                       information from this object (getData)
     * @param optional string $parent   the name of the parent category
     *
     * @access protected
     */
    function addCategory($name, $parent = '-1', $extended = null)
    {
        if (is_subclass_of($name, 'CategoryObject')) {
            $name = SerializeUnit::serializeUnit($name->getData(), CATEGORY_SERIALIZE_FORMAT);
        }
        
        // Important to avoid issues with the array - confusion of type
        settype($name, 'string');
        settype($parent, 'string');

        if ($this->exists($name, $parent)) {
            return new PEAR_Error('Already exists');
        } elseif ($parent != '-1' && !isset($this->categories[$parent])) {
            return new PEAR_Error('Add failed');
        } elseif (!isset($this->categories[$name])) {
            $this->categories[$name] = array();
        }

        $this->categories[$name][$parent] = true;
        
        if (is_subclass_of($this, 'CategoryObject')) {
            $this->data[$name] = SerializeUnit::serializeUnit($this->getData(), CATEGORY_SERIALIZE_FORMAT);
        }
        
        return true;
    }

    /**
     * Remove a category
     * 
     * @param string $category          The category to remove.
     *
     * @param optional string $parent   The name of the parent category to remove $name from.
     *                                  If default, we removed it from every category.
     *                                    0 means every branch
     *                                    -1 means the root
     *                                    Other means just that sub-category
     * @param optional boolean force [default = false] Force to remove
     *                         every child NOT YET IMPLEMENTED
     *
     * @note, the extended status is not removed!
     */
    function removeCategory($category, $parent = '0', $force = false)
    {
        if ($force) {
            return new PEAR_Error('Not supported');
        }
        
        if (is_subclass_of($category, 'CategoryObject')) {
            $name = $category->getName();
        } else {
            $name = $category;
        }
        
        // Important to avoid issues with the array - confusion of type
        settype($name, 'string');
        settype($parent, 'string');
        
        if ($this->exists($name, $parent) != true) {
            return new PEAR_Error('Does not exist');
        }
        
        switch ($parent) {
        case '0':
            unset($this->categories[$name]);
            unset($this->data[$name]);
            break;
        case '-1':
            if (!isset($this->categories[$name][$parent])) {
                return new PEAR_Error('Does not exist');
            }
            unset($this->categories[$name][$parent]);
            break;
        default:
            if (!isset($this->categories[$name][$parent])) {
                return new PEAR_Error('Does not exist');
            }
            unset($this->categories[$name][$parent]);
        }
        
        if (isset($this->categories[$name]) &&
            count($this->categories[$name]) == 0) {
            unset($this->categories[$name]);
            unset($this->data[$name]);
        }
        
        return true;
    }

    /**
     * Move a category from one parent to a new one.
     * 
     * Note: there is no check against circular references.
     * @param string $name       The name of the category.
     * @param string $old_parent The name of the old parent.
     * @param string $new_parent The name of the new parent.
     */
    function moveCategory($name, $old_parent, $new_parent)
    {
        if (is_subclass_of($name, 'CategoryObject')) {
            $name = SerializeUnit::serializeUnit($name->getData(), CATEGORY_SERIALIZE_FORMAT);
        }

        settype($name, 'string');
        settype($old_parent, 'string');
        settype($new_parent, 'string');

        if ($this->exists($name, $old_parent) != true) {
            return new PEAR_Error('Does not exist');
        }
        if ($this->exists($new_parent) != true) {
            return new PEAR_Error('Does not exist');
        }

        unset($this->categories[$name][$old_parent]);
        $this->categories[$name][$new_parent] = true;

        return true;
    }

    /**
     * Update the data in a category. Does not change the category's
     * parent, just name and/or serialized data.
     *
     * @param string $old_category  The old category data.
     * @param string $new_category  The new category data.
     */
    function updateCategory($old_category, $new_category)
    {
        if (is_subclass_of($old_category, 'CategoryObject')) {
            $old_name = $old_category->getName();
        } else {
            $old_name = $old_category;
        }
        
        if (is_subclass_of($new_category, 'CategoryObject')) {
            $new_name = $new_category->getName();
            $new_data = $new_category->getData();
        } else {
            $new_name = $new_category;
            $new_data = null;
        }

        if (!$this->exists($old_name)) {
            return new PEAR_Error('Does not exist');
        }
        if ($this->exists($new_name)) {
            return new PEAR_Error('Duplicate name');
        }
        
        $this->categories[$new_name] = $this->categories[$old_name];
        $this->data[$new_name] = $new_data;
        unset($this->categories[$old_name]);
        unset($this->data[$old_name]);
        
        return true;
    }

    /**
     * Export a list of categories
     * 
     * @param integer $format       Format of the export
     * @param string optional $parent The name of the parent from 
     *                                where we export.
     *
     * @return mixed - usually an array
     * 
     */
    function &export($format, $parent = '-1')
    {
        $out = array();
        
        switch ($format) {
        case CATEGORY_FORMAT_LIST:
            $this->extractAllLevelList($out, $parent);
            break;

        case CATEGORY_FORMAT_TREE:
            $this->extractAllLevelTree($out, $parent);
            break;

        case CATEGORY_FORMAT_FLIST:
            $out2 = $this->export(CATEGORY_FORMAT_LIST, $parent);
            if (empty($out2)) {
                $out[$parent] = true;
            } else {
                foreach ($out2 as $key=>$val) {
                    $out[$key]=true;
                    foreach ($val as $valtrue => $useless) {
                        $out[$valtrue]=true;
                    }
                }
            }
            break;

        case CATEGORY_FORMAT_3D:
            $out2=$this->export(CATEGORY_FORMAT_TREE, $parent);
            $id=0;
            $this->map3d($out, $out2, 0, $id, 0);
            break;

        case CATEGORY_FORMAT_OPT_SPC:
            $out2=$this->export(CATEGORY_FORMAT_3D, $parent);
            $id=0;
            foreach ($out2 as $key => $val) {
                foreach($val as $id_nuse => $nelem) {
                    $out[$id]=str_pad('', CATEGORY_FORMAT_OPT_SPC_DEPTH) . CATEGORY_FORMAT_OPT_SPC_SEPARATOR . $nelem[CATEGORY_FORMAT_3D_DAD];
                    $id++;
                }
            }
            $c='';
            for($i=0; $i<$id; $i++)
            {
                $c.=$out[$i].CATEGORY_FORMAT_OPT_SPC_CRLF;
            }
            $out=$c;
            break;
            
        default:
            return new PEAR_Error('Not supported');
        }

        return $out;
    }

    /**
     * Used by the export function to handle CATEGORY_FORMAT_3D.
     * @param array $out Array that will contain the result
     * @param array $arr Array from export(CATEGORY_FORMAT_LIST)
     * @param integer $depth of the child
     * @param integer $id kind of auto increment value
     * @param integer $dadId $id of the parent, the depth will be $depth-1
     * @access private
     * @see export()
     */
    function map3d(&$out, $arr, $depth, &$id, $dadId)
    {
        foreach ($arr as $key=>$val){
            if (0==$depth)
                $dadDepth=0;
            else
                $dadDepth=$depth-1;

            if ('-1'==$key)
                $key=CATEGORY_FORMAT_3D_ROOT;

            $out[$depth][$id][CATEGORY_FORMAT_3D_DAD]=$dadDepth .'.' .$dadId;
            $a=&$this->unserializeObject($key);
            if (is_string($a)) {
                if (isset($this->params['call_fill_mode']))
                    $out[$depth][$id] = call_user_func( $this->params['call_fill_mode'], CATEGORY_FORMAT_3D, $key);
                else
                    $out[$depth][$id][CATEGORY_FORMAT_3D_NAME]=$key;
            } else {
                $out[$depth][$id]=$a->getData(CATEGORY_FORMAT_3D);
            }

            $id=$id+1;
            if (is_array($val)) {
                $this->map3d($out, $val, $depth+1, $id, $id-1);
            }
        }
    }

    /**
     * Import a list of categories
     * 
     * @param integer $format  Format of the import
     * @param array $data      The name of the parent from 
     *                         where we import.
     * @param boolean $sync    If true, force to use the method add to add
     *                         an element. Useful if you use a driver, to know if
     *                         they synschornize at each line, or if they have an
     *                         internal way to do it just after as once
     *
     */
    function import($format, $data, $sync = false)
    {
        switch ($format) {
        case CATEGORY_FORMAT_CHIPAR:
            if ($sync) {
                return new PEAR_Error('Not supported');
            }
            $this->categories = array_merge_recursive($this->categories, $data);
            break;
            
        case CATEGORY_FORMAT_OPT_SPC:
            $line = explode(CATEGORY_FORMAT_OPT_SPC_CRLF, $data);
            $parent = array(-1);
            
            for ($i = 0; $i < count($line); $i++) {
                eregi("([ ]*)" . CATEGORY_FORMAT_OPT_SPC_SEPARATOR . "(.*)", $line[$i], $r);
                $a = strlen($r[1]) - count($parent) + 1;
                if ($a < 0) { // we have to compute a new parent
                    for ($j = 0; $j < -$a; $j++) {
                        array_pop($parent);
                    }
                }

                $acparent = $parent[count($parent) -1];
                if ($sync) {
                    $this->addCategory($r[2], $acparent);
                } else {
                    $this->categories[$acparent][$r[2]] = true;
                }
                array_push($parent, $r[2]);
            }
            break;

        case CATEGORY_FORMAT_FETCH:
            foreach ($data as $key1 => $val) {
                $data2 = array();
                foreach($val as $key => $val2) {
                    $data2[$val2] = true;
                }
                $data[$key1] = $data2;
            }
            
            $this->import(CATEGORY_FORMAT_CHIPAR, $data, $sync);
            break;
            
        default:
            return new PEAR_Error('Not supported');
        }

        return true;
    }

    /**
     * Give the number of children a category has. We are talking about the
     * direct childs
     * 
     * @param string optional $parent The name of the parent from 
     *                                where we begin.
     *
     * @return integer
     * @todo could be easily optimized ;-)
     */
    function getNumberChilds($parent = '-1')
    {
        $out = $this->extract1Level($parent);
        return (count($out[$parent]));
    }

    /**
     * Extract one level of categories, based on a parent, get the childs
     * format parent - name
     * We can see this function as a way to get a collection of node's children
     * 
     * @param string optional $parent The name of the parent from 
     *                                where we begin.
     *
     * @return array
     */
    function extract1Level($parent = '-1')
    {
        $out = array();
        foreach ($this->categories as $name => $qparent) {
            foreach ($qparent as $vparent => $notuse) {
                if ($vparent == $parent) {
                    if (!isset($out[$parent])) $out[$parent] = array();
                    $out[$parent][$name] = true;
                }
            }
        }
        return $out;
    }


    /**
     * Extract all level of categories, based on a parent
     * Tree format
     * 
     * @param array $out    Contain the result
     * @param string optional $parent The name of the parent from 
     *                                where we begin.
     * @param integer optional $maxlevel The number of level of depth to check it
     *
     * Note, if nothing is returned that means there is no child, but
     * don't forget to add the parent if you make some operations!
     */
    function extractAllLevelTree(&$out, $parent='-1', $level=-1)
    {

        if ($level == 0) {
            return false;
        }

        $k = $this->extract1Level($parent);
        if (!isset($k[$parent])) {
            return false;
        }

        $k = $k[$parent];
        foreach ($k as $name=>$v) {
            if (!isset($out[$parent]) || !is_array($out[$parent])) $out[$parent]=array();
            $out[$parent][$name] = true;
            $this->extractAllLevelTree($out[$parent], $name, $level - 1);
        }
    }

    /**
     * Extract all level of categories, based on a parent
     * List format: array(parent => array(child => true))
     * 
     * @param string optional $parent The name of the parent from 
     *                                where we begin.
     * @param integer optional $maxlevel The number of levels of depth to check it
     * @param array $out    Contain the result
     *
     * Note, if nothing is returned that means there is no child, but
     * don't forget to add the parent if you make some operations!
     */
    function extractAllLevelList(&$out, $parent = '-1', $level = -1)
    {
        if ($level == 0) {
            return false;
        }

        $k = $this->extract1Level($parent);
        if (!isset($k[$parent])) {
            return false;
        }

        $k = $k[$parent];

        foreach ($k as $name => $v) {
            if (!isset($out[$parent])) {
                $out[$parent] = array();
            }
            if (!isset($out[$parent][$name])) {
                $out[$parent][$name] = true;
                $this->extractAllLevelList($out, $name, $level - 1);
            }
        }
    }

    /**
     * Get a list of parents, based on a child - just one level
     * 
     * @param string $child The name of the child
     * @param optional string $parent The name of the parent from where
     *                           we want to check.
     *
     * @return array 
     */
    function getParents1($child, $parentfrom='0')
    {
        if ($this->exists($child, $parentfrom) != true) {
            return new PEAR_Error('Does not exist');
        }

        if (is_subclass_of($child, 'CategoryObject')) {
            $child = SerializeUnit::serializeUnit ($child->getData(), CATEGORY_SERIALIZE_FORMAT);
        }

        return $this->categories[$child];
    }

    /**
     * Get a list of parents, based on a child - every levels
     * 
     * @param string $child The name of the child
     * @param optional string $parent The name of the parent from where
     *                           we want to check.
     * @return array [child] [parent] with a tree format
     * @see Category::explodeArray to get it in list format
     */
    function getParents($child, $parentfrom='0')
    {

        $ret=$this->getParents1($child, $parentfrom);
        if (!is_array($ret))
            return new PEAR_Error('Parents not found');

        foreach ($ret as $parent=>$trueval) {
            if ($parent!='-1') {
                $ret[$parent]=$this->getParents($parent);
            }
        }

        return $ret;
    }

    /**
     * Explode an array returned by getParents to a list
     * 
     * @param array $ret must be initialized to an array - here will be
     *                   the result
     * @param array $ar array from getParents
     *
     * @return array list format
     * @see Category::getParents
     */
    function explodeArray(&$ret, $ar)
    {
        foreach ($ar as $key => $val) {
            array_push($ret, $key);
            if (is_array($val)) {
                $this->explodeArray($ret, $val);
            }
        }
    }

    /**
     * Say if a category exists or not.
     * The category -1 is always true
     * 
     * @param string $category The name of the category
     * @param optional string $parent The name of the parent from where
     *                                we want to check.
     *                                0 means every parent
     * 
     *
     * @return mixed (true means yes, otherwise, kind of error)
     * so if result>=0 everything is all rigth
     */
    function exists($category, $parent = '0')
    {
        settype($category, 'string');
        settype($parent, 'string');
        
        if (is_subclass_of($category, 'CategoryObject')) {
            $category = SerializeUnit::serializeUnit($category->getData(), CATEGORY_SERIALIZE_FORMAT);
        }
        
        if ($category == '-1') {
            return true;
        }
        
        if (!isset($this->categories[$category])) {
            return false;
        }
        
        if ($parent == '0') {
            return true;
        }
        
        if (!isset($this->categories[$category][$parent])) {
            return false;
        }
        
        return true;
    }

    /**
     * get the extended status of a name
     * @param string $name
     * @param string $nameobject Name of the class that we will build on.
     * @return object an instance of $nameobject with the good $name unserialized
     */
    function &getData($name, $nameobject = null)
    {
        if (!isset($this->categories[$name])) {
            return new PEAR_Error('Does not exist');
        }
        if (!isset($this->data[$name])) {
            return new PEAR_Error('Does not exist');
        }

        return $this->unserializeObject($this->data[$name], $nameobject);
    }

    /**
     * Get a name, and adapt it if required to an object, or let it as a string
     * @param string $name Comes from a category, normal string or serialized one
     * @param string $nameobject Name of the class that we will build on.
     * @return object an instance of $nameobject with the good $name unserialized
     * @note: this method must be called by another class to unserialize the data,
     *        if there is a doubt, if the data was not serialized, you will get the
     *        same string
     */
    function &unserializeObject($name, $nameobject = 'CategoryObject')
    {
        $ret = new $nameobject();
        $unser = SerializeUnit::unSerializeUnit($name, CATEGORY_SERIALIZE_FORMAT);
        if (false == $unser) {
            return $name;
        }

        $ret->mergeData($unser); //We don't specify the format, to be sure that even a basic string will be returned
        return $ret;
    }

}

/**
 * Class that can be extended to save further information in a category.
 */
class CategoryObject {

    /**
     * Key-value hash that will be serialized.
     * @see getData()
     */
    var $data = array();

    /**
     * Get a pointer/accessor to the array that we will save
     * needed because PHP is not an object language
     * @return array reference to the internal array to serialize
     */
    function &getData()
    {
        return $this->data;
    }

    /**
     * Merge the data of an array with the one already in the class
     * @param array $arr
     */
    function mergeData(&$arr)
    {
        $this->data = array_merge_recursive($this->getData(), $arr);
    }

}
?>
