/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>

#include <libnd.h>
#include <protocols/tcp/libnd_tcp.h>
#include <protocols/udp/libnd_udp.h>

#include <libnd_demux.h>
#include <libnd_demux_progress.h>

#ifndef RLIMIT_OFILE
#define RLIMIT_OFILE RLIMIT_NOFILE
#endif

#define LND_DEMUX_QUEUE_MAX 100000

static LND_Protocol *ip;
static LND_Protocol *tcp;
static LND_Protocol *udp;
static mode_t        mode_640 = S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP;

struct demux_packet_queue
{
  LND_Packet *q;
  LND_Packet *q_end;

  int q_count;
};

struct demux_dumper_stats
{
  int         count;
};

/* For debugging purposes when running out of fds
static guint
demux_get_fdcount(void)
{
  char procdir[MAXPATHLEN];
  struct dirent *dirent;
  DIR *dir;
  guint fd_count = 0;

  g_snprintf(procdir, MAXPATHLEN, "/proc/%u/fd", getpid());
  
  if (! (dir = opendir(procdir)))
    {
      printf("Couldn't open %s\n", procdir);
      return 0;
    }
  
  while ( (dirent = readdir(dir)))
    fd_count++;
  
  closedir(dir);
  
  return fd_count - 2;
} */


static gboolean
demux_mkdirs(const char *path)
{
  gchar **dirs = g_strsplit(path, G_DIR_SEPARATOR_S, -1);
  GString *dir = g_string_new("");
  char buf[256];
  int i;
  
  for (i = 0; dirs[i]; i++)
    {
      g_snprintf(buf, 256, "%s/", dirs[i]);      
      dir = g_string_append(dir, buf);
      
      if (mkdir(dir->str, mode_640) < 0 && errno != EEXIST)
	goto error_return;
    }
  
  return TRUE;
  
 error_return:
  printf("Could not create directory %s: %s\n", path, strerror(errno));
  
  for (i = 0; dirs[i]; i++)
    g_free(dirs[i]);
  g_free(dirs);
  g_string_free(dir, TRUE);

  return FALSE;
}


static char *
demux_get_dir_host_pairs(LND_Demux *dm, LND_ConnID *conn)
{
  char dir[MAXPATHLEN];
  char src[32], dst[32];
  struct in_addr ip_src, ip_dst;

  if (!conn)
    {
      D(("Invalid input\n"));
      return NULL;
    }

  libnd_conn_get_src(conn, &ip_src, NULL);
  libnd_conn_get_dst(conn, &ip_dst, NULL);

  g_snprintf(src, 32, "%s", inet_ntoa(ip_src));
  g_snprintf(dst, 32, "%s", inet_ntoa(ip_dst));

  g_snprintf(dir, MAXPATHLEN, "%s/%s%u/%s%s/%s%s",
	     dm->output_dir, (dm->do_mark ? "p":""), conn->proto,
	     (dm->do_mark ? "S":""), src, (dm->do_mark ? "D":""), dst);
  return g_strdup(dir);
}

static char *
demux_get_dir_dest_ports(LND_Demux *dm, LND_ConnID *conn)
{
  char dir[MAXPATHLEN];

  if (!conn)
    {
      D(("Invalid input\n"));
      return NULL;
    }

  g_snprintf(dir, MAXPATHLEN, "%s/%s%u", dm->output_dir,
	     (dm->do_mark ? "p":""), conn->proto);
  return g_strdup(dir);
}

static char *
demux_get_dir_dest_hostports(LND_Demux *dm, LND_ConnID *conn)
{
  char dir[MAXPATHLEN];
  struct in_addr ip_dst;
  u_char *bytes = (u_char*) &ip_dst.s_addr;

  if (!conn)
    {      
      printf("Invalid input\n");
      return NULL;
    }

  libnd_conn_get_dst(conn, &ip_dst, NULL);
  
  g_snprintf(dir, MAXPATHLEN, "%s/%s%u/%u/%u/%u/%u", dm->output_dir,
	     (dm->do_mark ? "p":""), conn->proto,
	     bytes[0], bytes[1], bytes[2], bytes[3]);
  return g_strdup(dir);
}

static char *
demux_get_dir_name(LND_Demux *dm, LND_ConnID *conn)
{
  switch (dm->mode)
    {
    case LND_DEMUX_MODE_DEST_PORTS:
      return demux_get_dir_dest_ports(dm, conn);

    case LND_DEMUX_MODE_DEST_HOSTPORTS:
      return demux_get_dir_dest_hostports(dm, conn);
      
    case LND_DEMUX_MODE_HOST_PAIRS:
    default:
      return demux_get_dir_host_pairs(dm, conn);
    }
}


static const char *
demux_get_conn_dumper_name(LND_Demux *dm, LND_ConnID *conn)
{
  static char name[MAXPATHLEN];
  guint16 sport, dport;

  switch (dm->mode)
    {
    case LND_DEMUX_MODE_DEST_PORTS:
    case LND_DEMUX_MODE_DEST_HOSTPORTS:
      libnd_conn_get_dst(conn, NULL, &dport);
      
      g_snprintf(name, MAXPATHLEN, "%s%05u.trace",
		 (dm->do_mark ? "d":""), ntohs(dport));
      break;
      
    case LND_DEMUX_MODE_HOST_PAIRS:
    default:
      switch (conn->proto)
	{
	case IPPROTO_TCP:
	case IPPROTO_UDP:
	  libnd_conn_get_src(conn, NULL, &sport);
	  libnd_conn_get_dst(conn, NULL, &dport);
	  
	  g_snprintf(name, MAXPATHLEN, "%lu.%lu-%s%u-%s%u.trace",
		     conn->start_ts.tv_sec, conn->start_ts.tv_usec,
		     (dm->do_mark ? "s":""), ntohs(sport),
		     (dm->do_mark ? "d":""), ntohs(dport));
	  break;
	  
	default:
	  g_snprintf(name, MAXPATHLEN, "%lu.%lu.trace",
		     conn->start_ts.tv_sec, conn->start_ts.tv_usec);
	}
    }

  return name;
}


static gboolean
demux_conn_queue_add_packet(LND_Demux *dm, LND_ConnID *conn)
{
  struct demux_packet_queue *queue;
  LND_Packet *p_copy;

  if (libnd_conn_data_get(conn, "ignore"))
    return FALSE;

  if (! (queue = libnd_conn_data_get(conn, "queue")))
    {
      if (! (queue = g_new0(struct demux_packet_queue, 1)))
	{
	  D(("Out of memory.\n"));
	  return FALSE;
	}

      libnd_conn_data_set(conn, "queue", queue);
    }

  if (! (p_copy = libnd_packet_duplicate(dm->current)))
    {
      D(("Out of memory.\n"));
      return FALSE;
    }
  
  if (! queue->q_end)
    {
      queue->q = p_copy;
      queue->q_end = p_copy;
    }
  else
    {
      queue->q_end->next = p_copy;
      p_copy->prev = queue->q_end;
      queue->q_end = p_copy;
    }

  queue->q_count++;

  return TRUE;
}

static void
demux_conn_queue_delete(struct demux_packet_queue *queue)
{
  LND_Packet *packet, *p_next;
  
  if ( (packet = queue->q))
    {
      while (packet)
	{
	  p_next = packet->next;
	  libnd_packet_free(packet);
	  packet = p_next;
	}
    }
  
  g_free(queue);
}

static void
demux_conn_queue_flush(LND_Demux *dm, LND_ConnID *conn, struct demux_packet_queue *queue)
{
  char dumper_name[MAXPATHLEN];
  pcap_dumper_t *dumper;
  char *dir;
  LND_Packet *packet, *p_next;

  if (! (packet = queue->q))
    goto cleanup_return;

  dir = libnd_conn_data_get(conn, "dir");
  D_ASSERT_PTR(dir);
  
  if (! demux_mkdirs(dir))
    {
      D(("Could not create directory %s\n", dir));
      goto cleanup_return;
    }

  g_snprintf(dumper_name, MAXPATHLEN, "%s/%s",
	     dir, demux_get_conn_dumper_name(dm, conn));
  
  /* Unless using per-IP/port-pair granularity, now increment output dumper's
   * flow count and mark connection as ignored in case we've exceeded limit.
   */
  if (dm->mode != LND_DEMUX_MODE_HOST_PAIRS &&
      ! libnd_conn_data_get(conn, "counted"))
    {
      struct demux_dumper_stats *stats = NULL;
      
      if (! (stats = g_hash_table_lookup(dm->dumper_usage, dumper_name)))
	{
	  stats = g_new0(struct demux_dumper_stats, 1);
	  g_hash_table_insert(dm->dumper_usage, g_strdup(dumper_name), stats);
	}
      
      libnd_conn_data_set(conn, "counted", (void*) 1);

      if (dm->max_flows > 0 && stats->count >= dm->max_flows)
	{
	  libnd_conn_data_set(conn, "ignore", (void*) 1);
	  goto cleanup_return;
	}

      ++stats->count;
    }

  if (libnd_conn_data_get(conn, "ignore"))
    goto cleanup_return;
  
  if (! (dumper = pcapnav_dump_open(dm->pcap, dumper_name,
				    PCAPNAV_DUMP_APPEND_FAST)))
    {
      D(("Cannot open dump: %s\n", pcap_geterr(dm->pcap)));
      goto cleanup_return;
    }
  
  while (packet)
    {
      pcap_dump((u_char *) dumper, &packet->ph, packet->data);
      p_next = packet->next;
      libnd_packet_free(packet);
      packet = p_next;
    }
  
  queue->q = NULL;
  pcap_dump_close(dumper);  

 cleanup_return:
  demux_conn_queue_delete(queue);
}


static gboolean
demux_conn_cleanup_cb(LND_ConnID *conn, LND_Demux *dm)
{
  struct demux_packet_queue *queue;
  char *dir;

  /* Flush packets unless we want content and the connection
   * is not yet marked as having content.
   */
  if (!dm->need_content || libnd_conn_data_get(conn, "content"))
    {
      if ( (queue = libnd_conn_data_remove(conn, "queue")))
	demux_conn_queue_flush(dm, conn, queue);
    }

  /* Check if connection is now dead, and if so,
   * remove it from the connection table.
   */
  if (libnd_conn_is_dead(conn, dm->current))
    {
      LND_ConnID *conn_result;
      
      /* If the queue is still associated to the connection,
       * we can now delete it. It will only still be associated
       * if we required content and the connection didn't deliver any.
       */
      if ( (queue = libnd_conn_data_remove(conn, "queue")))
	demux_conn_queue_delete(queue);
      
      if ( (dir = libnd_conn_data_remove(conn, "dir")))
	g_free(dir);
      
      conn_result = libnd_conn_table_remove(dm->conns, conn);
     
      /* There is a possibilty of more than one connection in 
       * the table matching this connection -- to prevent segfaults
       * make sure we only nuke the one we mean.
       */
      if (conn_result == conn)
	libnd_conn_free(conn);
    }
  
  return TRUE;
}

struct age_info {
  int         i;
  LND_Demux  *dm;
};

static gboolean
demux_show_state(LND_ConnID *conn, struct age_info *info)
{
  struct bpf_timeval diff_tv;

  if (++(info->i) == 10)
    return FALSE;

  pcapnav_timeval_sub(&info->dm->current->ph.ts, &conn->latest_ts, &diff_tv);
  
  if (conn->proto == IPPROTO_TCP)
    {
      printf("%i: proto %i, age %u.%u, state %i\n", info->i, conn->proto,
	     (int) diff_tv.tv_sec, (int) diff_tv.tv_usec,
	     libnd_tcpconn_state_get((LND_TCPConn *) conn));
    }
  else
    {
      printf("%i: proto %i, age %u.%u\n", info->i, conn->proto,
	     (int) diff_tv.tv_sec, (int) diff_tv.tv_usec);;
    }

  return TRUE;
}

static void
demux_conn_table_cleanup(LND_Demux *dm)
{
  struct age_info info;

  if (! dm)
    return;

  libnd_conn_table_set_policy(dm->conns, LND_CONN_TABLE_INCLUDE_DEAD);
  libnd_conn_table_foreach(dm->conns,
			   (LND_ConnFunc) demux_conn_cleanup_cb, dm);

  info.i = 0;
  info.dm = dm;

  /*
  libnd_conn_table_foreach_oldest(dm->conns,
				  (LND_ConnFunc) demux_show_state, &info);
  */
  libnd_conn_table_set_policy(dm->conns, LND_CONN_TABLE_IGNORE_DEAD);
}


static void
demux_dump_non_ip(LND_Demux *dm, LND_Packet *packet)
{
  if (!dm || !dm->pcap || !dm->others_file)
    {
      D(("Invalid input.\n"));
      return;
    }
  
  if (! dm->others_dumper)
    {
      if (! (dm->others_dumper = pcapnav_dump_open(dm->pcap, dm->others_file,
						   PCAPNAV_DUMP_APPEND_FAST)))
	{
	  D(("Cannot create output for non-IP traffic: %s\n", pcap_geterr(dm->pcap)));
	  return;
	}
    }
  
  pcap_dump((u_char *) dm->others_dumper, &packet->ph, packet->data);
}


static gboolean
dm_demux_cb(LND_TraceSet *set, LND_Trace *trace, LND_Demux *dm)
{
  LND_PacketIterator  pit;
  LND_Packet *packet;
  LND_ConnID *conn;
  char *dir;
  
  libnd_demux_prog_set_file(trace->filename);
  dm->pcap = pcapnav_pcap(trace->tpm->base->pcn);
  
  /* Iterate over entire individual trace, read-only. */
  for (libnd_pit_init_mode(&pit, trace, LND_PACKET_IT_AREA_R);
       libnd_pit_get(&pit);
       libnd_pit_next(&pit))
    {
      packet = libnd_pit_get(&pit);
      dm->current = packet;

      libnd_demux_prog_set_time(&packet->ph.ts);

      /* First check if we have connection state for this packet.
       * If not, create and register.
       */
      if (! (conn = libnd_conn_table_lookup(dm->conns, packet)))
	{
	  if (! (conn = libnd_conn_new(packet)))
	    {
	      demux_dump_non_ip(dm, packet);
	      continue;
	    }

	  libnd_conn_table_add(dm->conns, conn);

	  if (! (dir = demux_get_dir_name(dm, conn)))
	    {
	      printf("Error creating output directories for %p %p\n", dm, conn);
	      dm->current = NULL;
	      return FALSE;
	    }

	  libnd_conn_data_set(conn, "dir", dir);
	}
      
      /* Now, add the packet to the connection's output queue. Only if
       * that succeeded, bump up the iterations counter, to make sure
       * we don't attempt to flush even though we haven't actually
       * collected enough packets.
       */
      if (demux_conn_queue_add_packet(dm, conn))
	dm->iter++;

      /* Update the state of the current connection: */
      libnd_conn_update(conn, packet);

      /* If we require content, do check now */
      if (dm->need_content && conn->content_tx > 0)
	libnd_conn_data_set(conn, "content", (void*) 1);
      
      /* Once we hit the output threshold, go through all connections
       * in the table and write packets to their corresponding dumpers.
       * At the same time, remove dead connections from the table.
       */
      if (dm->iter >= dm->iter_lim)
	{
	  demux_conn_table_cleanup(dm);
	  dm->iter = 0;
	}
    }

  /* Flush one more time at the end of the iteration to make sure
   * we write out everything.
   */
  dm->current = NULL;
  demux_conn_table_cleanup(dm);

  return TRUE;
  TOUCH(set);
}




LND_Demux *
libnd_demux_new(void)
{
  LND_Demux *dm;

  if (! (dm = g_new0(LND_Demux, 1)))
    return NULL;
  
  if (! (dm->traces = libnd_traceset_new()))
    {
      D(("out of memory"));
      g_free(dm);
      return NULL;
    }

  dm->mode = LND_DEMUX_MODE_HOST_PAIRS;
  dm->iter = 0;
  dm->iter_lim = LND_DEMUX_QUEUE_MAX;

  dm->conns = libnd_conn_table_new(LND_CONN_TABLE_IGNORE_DEAD);
  dm->output_dir = strdup(".");

  dm->max_flows = 0;
  dm->dumper_usage = g_hash_table_new(g_str_hash, g_str_equal);
  
  return dm;
}


static gboolean
demux_free_cb(LND_ConnID *conn, void *user_data)
{
  char *dir = libnd_conn_data_remove(conn, "dir");

  if (dir)
    g_free(dir);

  return TRUE;
  TOUCH(user_data);
}

static gboolean
demux_clear_dumper_usage_cb(gpointer	key,
			    gpointer	value,
			    gpointer	user_data)
{
  g_free(key);   /* char* name of dumper file */
  g_free(value); /* struct demux_dumper_stats */
  
  return TRUE;
  TOUCH(value);
  TOUCH(user_data);
}

static void
demux_report_cb(char *dumper, struct demux_dumper_stats *stats, LND_Demux *dm)
{
  printf("%8u  %s\n", stats->count, dumper);

  return;
  TOUCH(dm);
}

void
libnd_demux_report(LND_Demux *dm)
{
  if (! dm->dumper_usage)
    return;
  
  g_hash_table_foreach(dm->dumper_usage, (GHFunc) demux_report_cb, dm);
}


void
libnd_demux_free(LND_Demux *dm)
{
  if (! dm)
    return;

  fflush(stdout);

  libnd_traceset_free(dm->traces);

  libnd_conn_table_set_policy(dm->conns, LND_CONN_TABLE_INCLUDE_DEAD);
  libnd_conn_table_foreach(dm->conns, (LND_ConnFunc) demux_conn_cleanup_cb, dm);
  libnd_conn_table_foreach(dm->conns, demux_free_cb, NULL);
  libnd_conn_table_free(dm->conns);

  g_free(dm->output_dir);
  g_free(dm->others_file);
  
  if (dm->dumper_usage)
    {
      g_hash_table_foreach_remove(dm->dumper_usage,
				  demux_clear_dumper_usage_cb, NULL);
      g_hash_table_destroy(dm->dumper_usage);
    }
  
  if (dm->others_dumper)
    pcap_dump_close(dm->others_dumper);
}


gboolean
libnd_demux_set_output_dir(LND_Demux *dm, const char *output_dir)
{
  char others_file[MAXPATHLEN];

  if (! dm || !output_dir || !*output_dir)
    return FALSE;

  g_free(dm->output_dir);
  dm->output_dir = g_strdup(output_dir);

  while (dm->output_dir[strlen(dm->output_dir)-1] == G_DIR_SEPARATOR)
    dm->output_dir[strlen(dm->output_dir)-1] = '\0';

  g_free(dm->others_file);
  g_snprintf(others_file, MAXPATHLEN, "%s/non_ip.trace", dm->output_dir);
  dm->others_file = g_strdup(others_file);
    
  return TRUE;
}

void
libnd_demux(LND_Demux *dm)
{
  if (!dm)
    return;

  /* Iterate over all input traces and demux flows. */
  libnd_traceset_foreach(dm->traces, (LND_TraceSetCB) dm_demux_cb, dm);
}


const char *
name(void)
{
  return "Demux";
}

const char *
author(void)
{
  return "Christian Kreibich <christian.kreibich-AT-cl.cam.ac.uk>";
}

const char *
version(void)
{
  return VERSION;
}

gboolean
init(void)
{
  if (! libnd_plugin_find("Conntrack"))
    {
      D(("Connection tracker plugin not found\n"));
      return FALSE;
    }
  
  if (! libnd_plugin_find("Trace-Set"))
    {
      D(("Trace-Set plugin not found\n"));
      return FALSE;
    }

  if (! (ip = libnd_proto_registry_find(LND_PROTO_LAYER_NET, 0x0800)))
    {
      D(("IP protocol plugin not found.\n"));
      return FALSE;
    }

  if (! (tcp = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
					 IPPROTO_TCP)))
    {
      D(("TCP protocol plugin not found.\n"));
      return FALSE;
    }

  if (! (udp = libnd_proto_registry_find(LND_PROTO_LAYER_TRANS,
					 IPPROTO_UDP)))
    {
      D(("UDP protocol plugin not found.\n"));
      return FALSE;
    }
  
  return TRUE;
}


static void
demux_help(void)
{
  printf("Flow Demultiplexer plugin\n"
	 "USAGE: lndtool -r demux [--host-pairs|--dest-ports|--dest-hostports]\n"
	 "               [--help|-h] [--debug|-d] [--output-dir|-o DIR] [--progress|-p]\n"
	 "               [--max-flows|-m NUM] [--names-file|-f FILE] <TRACE1> [<TRACE2> ...]\n"
	 "\n"
	 "  --help, -h, -?           This message.\n"
	 "  --host-pairs             Per host-pair output granularity (default).\n"
	 "  --dest-ports             Per dest-port output granularity.\n"
	 "  --dest-hostports         Per dest-host+port output granularity.\n"
	 "  --mark|-M                In generated output files (+ dirs), prefix IPs, protocol\n"
	 "                           numbers, and port numbers with strings, for easier grepping.\n"
	 "\n"
	 "                             - src IP:   'S'    - dst IP:   'D'\n"
	 "                             - src port: 's'    - dst port: 'd'\n"
	 "                             - protocol number: 'p'\n"
	 "\n"
	 "  --output-dir, -o DIR     Output directory in which to dump output. Default is\n"
	 "                           current directory. Directory is created if not existent.\n"
	 "  --progress|-p            Displays a progress indicator at the shell.\n"
	 "  --max-flows|-m NUM       Limit the maximum number of flows written to a single file.\n"
	 "  --content|-c             Only demux flows that contain app-layer content.\n"
	 "  --report|-r              Report counts of flows per output trace to stdout.\n"
	 "  --names-file|-f FILE     Take filenames of traces from flatfile FILE.\n");
}

gboolean
run(LND_Trace *trace, LND_PluginArgs *args)
{
  int i, counter = 0;
  LND_Demux *dm;

  if (!args)
    return FALSE;

  if (args->argc == 0)
    {
      printf("Please provide one or more traces to scan.\n");
      demux_help();
      return FALSE;
    }
  
  dm = libnd_demux_new();
  
  for (i = 0; i < args->argc; i++)
    {
      if (!strcmp(args->argv[i], "-h")     ||
	  !strcmp(args->argv[i], "--help") ||
	  !strcmp(args->argv[i], "-?"))
	{
	  demux_help();
	  return FALSE;
	}
      else if (!strcmp(args->argv[i], "--host-pairs"))
	{
	  dm->mode = LND_DEMUX_MODE_HOST_PAIRS;
	}
      else if (!strcmp(args->argv[i], "--dest-ports"))
	{
	  dm->mode = LND_DEMUX_MODE_DEST_PORTS;
	}
      else if (!strcmp(args->argv[i], "--dest-hostports"))
	{
	  dm->mode = LND_DEMUX_MODE_DEST_HOSTPORTS;
	}
      else if (!strcmp(args->argv[i], "-M") ||
	       !strcmp(args->argv[i], "--mark"))
	{
	  dm->do_mark = TRUE;
	}
      else if (!strcmp(args->argv[i], "-o") ||
	       !strcmp(args->argv[i], "--output-dir"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass an output directory to the --output-dir|-o option.\n");
	      return FALSE;
	    }

	  if (! libnd_demux_set_output_dir(dm, args->argv[i]))
	    {
	      printf("Cannot use %s as output directory: %s.\n", args->argv[i], strerror(errno));
	      return FALSE;
	    }
	}
      else if (!strcmp(args->argv[i], "-f") ||
	       !strcmp(args->argv[i], "--names-file"))
	{
	  if (++i == args->argc)
	    {
	      printf("You need to pass the name of a file that contains\n"
		     "a list of trace files, one filename per line, when\n"
		     "using the --names-file|-f option.\n");
	      return FALSE;
	    }
	  
	  if (! libnd_traceset_add_trace_name_list(dm->traces, args->argv[i]))
	    printf("Could not add all traces provided in '%s'\n", args->argv[i]);
	  else
	    counter++;
	}
      else if (!strcmp(args->argv[i], "-p") ||
	       !strcmp(args->argv[i], "--progress"))
	{
	  libnd_demux_prog_observer_add(dm);
	}
      else if (!strcmp(args->argv[i], "-r") ||
	       !strcmp(args->argv[i], "--report"))
	{
	  dm->do_report = TRUE;
	}
      else if (!strcmp(args->argv[i], "-m") ||
	       !strcmp(args->argv[i], "--max-flows"))
	{
	  int max_flows;

	  if (++i == args->argc)
	    {
	      printf("You need to pass the maximum number of flows you want\n"
		     "per output file to the the --max-flows|-m option.\n");
	      return FALSE;
	    }

	  max_flows = atoi(args->argv[i]);
	  if (max_flows > 0)
	    dm->max_flows = max_flows;
	}
      else if (!strcmp(args->argv[i], "-c") ||
	       !strcmp(args->argv[i], "--content"))
	{
	  dm->need_content = TRUE;
	}
      else if (args->argv[i][0] != '-') /* Often these are just mistyped args */
	{
	  if (! libnd_traceset_add_trace_name(dm->traces, args->argv[i]))
	    printf("Trace file %s not added.\n", args->argv[i]);
	  else
	    counter++;
	}
      else
	printf("Warning: option %s unknown.\n", args->argv[i]);
    }

  if (counter > 0)
    libnd_demux(dm);

  if (dm->do_report)
    libnd_demux_report(dm);

  libnd_demux_free(dm);
  
  return TRUE;
  TOUCH(trace);
}
