/***************************************************************************
                          unit.h  -  description
                             -------------------
    begin                : Fri Jan 19 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef __UNIT_H
#define __UNIT_H

/* weather types */
enum {
    FAIR = 0,
    CLOUDS,
    RAIN,
    SNOW
};

/* IMPORTANT -- unit_class 0 is always infantry */
enum { INFANTRY = 0 };

/* unit class struct */
typedef struct {
    char *entry_name;
    char *name;
#ifdef WITH_SOUND
    Sound_Chunk *move_sound;
#endif
} Unit_Class;

/* contains the unit definitions (target types/unit classes)*/
typedef struct {
    int type_count;
    char **type_names;
    // these target types are flying; the attack value of a ground value for this type is influenced by weather //
    int *air_type;
    int class_count;
    Unit_Class *classes;
} Unit_Def;

/* unit flags */
enum {
    SWIMMING       = ( 1L << 1 ),
    DIVING         = ( 1L << 2 ),
    FLYING         = ( 1L << 3 ),
    PARACHUTE      = ( 1L << 4 ),
    WHEELED        = ( 1L << 5 ),
    TRACKED        = ( 1L << 6 ),
    TRANSPORTER    = ( 1L << 7 ),
    RECON          = ( 1L << 8 ),
    ARTILLERY      = ( 1L << 9 ),
    INTERCEPTOR    = ( 1L << 10 ),
    AIR_DEFENCE    = ( 1L << 11 ),
    BRIDGE_ENG     = ( 1L << 12)
};

/* embark types */
enum {
    NO_EMBARK = 0,
    GROUND_EMBARK,
    SEA_EMBARK,
    AIR_EMBARK
};

/* picture types */
enum {
    SINGLE = 0,
    ALL_DIRS
};

/* looking directions */
enum {
    RIGHT = 0,
    LEFT,
    UP = 0,
    RIGHT_UP,
    RIGHT_DOWN,
    DOWN,
    LEFT_DOWN,
    LEFT_UP
};

/* experience stuff */
enum {
    MAX_EXP = 100,
    MAX_EXP_LEVEL = 5,
    EXP_PER_LEVEL = MAX_EXP / MAX_EXP_LEVEL
};

/* unit life bar stuff */
/* there aren't used colors but bitmaps with small colored tiles */
enum {
    BAR_WIDTH = 31,
    BAR_HEIGHT = 4,
    BAR_TILE_WIDTH = 3,
    BAR_TILE_HEIGHT = 4
};

/* things a unit can supply */
enum {
    AMMO = 1,
    FUEL = 2,
    ANYTHING
};

/* type of attack -- for can_attack_unit */
enum {
    ATTACK = 0,
    DEFEND
};

/* life types for switch unit/tran life */
enum {
    UNIT_LIFE,
    TRAN_LIFE
};

enum { TARGET_TYPE_LIMIT = 10 };
/* entry in the unit library */
typedef struct {
    char *entry_name; /* identification of this entry */
    char *cap; /* caption */
    int nation; /* nation of origin */
    int class; /* unit class */
    int type; /* target type */
    int ini; /* inititative */
    int mov; /* movement */
    int spot; /* spotting */
    int range; /* attack range */
    int attack_count; /* number of attacks per turn */
    int attack[TARGET_TYPE_LIMIT]; /* attack values */
    Unit_Def *unit_def; // SHIT that's what I call a useless hack! but hell it's the easiest way //
    int def; /* defence value */
    int ammo;
    int fuel;
    SDL_Surface *pic;
    SDL_Surface *small_pic; /* half the size; used to display air and ground unit at one tile */
    int pic_type; /* either single or all_dirs */
    int width, height; /* picture size */
    int flags;
#ifdef WITH_SOUND
    /* sound stuff */
    int own_sound; /* special sounds loaded for this unit type; must be freed by this struct */
    Sound_Chunk *move_sound; /* pointer to the unit class default sound if own_sound is not set */
#endif
} Unit_Lib_Entry;

enum { MAX_DAMAGE = 10 };

/* unit */
typedef struct {
    char name[128]; /* special unit name */
    /* properties are not a pointer as these values may change when mergeing units */
    Unit_Lib_Entry prop; /* properties */
    Unit_Lib_Entry tran_prop; /* transporters properties */
    int tran_prop_set; /* if the transporter properties were set this flag is set, too */
    Unit_Lib_Entry *sel_prop; /* selected props: either prop or tran_prop */
    Unit_Lib_Entry cur_prop; /* current values */
    int player_id; /* which player does this unit belong to ? */
    int x, y; /* map position */
    int exp; /* experience */
    int exp_level; /* exp level computed from exp */
    int delay; /* delay in turns */
    int embark; /* embark type */
    int entr; /* current entrenchment */
    int dir; /* looking direction */
    int pic_offset; /* unit picture offset */
    int kill_next_turn; /* mark for destruction - last chance to refuel! */
    int killed; /* destroy is destroyed */
    int damage; /* damage the unit had to take */
    int supply_level; /* in percent; military targets are centers of supply */
    int old_fuel; /* used to backup unit's fuel when embarked in sea/air transporter */
    int ground_tran; /* has ground transporter ? */
    int no_action_yet; /* no action taken yet? important for embark/supply */
    int damage_bar_width; /* current life bar width in map->life_icons */
    int damage_bar_offset; /* offset in map->damage_icons */
    int add_fuel_cost; /* additional fuel cost for flying units; substracted at begin of turn */
    int rugged_def; /* dogged defence? */
    int rugged_def_chance; /* chance for dogged defence on D20 */
    int sup; // unit suppression //
} Unit;

/* damage predicition if these two units will fight */
enum { SUPPORT_LIMIT = 10 };
typedef struct {
    Unit *att, *def;
    int att_damage, def_damage; /* relative damage in percent */
    int x, y; /* unit needs to move here to attack unit; if x = y = -1 no movement needed */
    Unit *support[SUPPORT_LIMIT]; /* these units will give supporting fire; this is accomplished
    with half attack value but att can't defend against these attacks */
    int support_count;
} Damage_Pred;

/* open unit file and return file handle (adds the correct path)*/
FILE* open_unit_file( char *file_name );

/* prepares the dynamic list which will be used as unit library */
void init_unit_lib( Dyn_List *list );

/* read all units from file to Unit_Lib_Entry structs and add them to the given dynamic list */
void read_units_from_file( char *file_name, Dyn_List *list, Unit_Def *def, Nation **nations, int nation_number );

/* deletes a unit struct completely. compatible with dyn list callback because void* is given as
argument */
void delete_unit_lib_entry( void* poi );

/* find unit lib entry by searching for entry_name */
Unit_Lib_Entry* find_unit_lib_entry( Dyn_List *list, char *entry_name );

/* read unit definitions (target types/ unit_classes) */
Unit_Def* read_unit_def( char *file_name );

/* delete unit definitions (target types/ unit_classes) (always returns 0)*/
Unit_Def* delete_unit_def( Unit_Def *unit_def );

/* create and add unit */
Unit* create_unit( int delay, int player_id, Unit_Lib_Entry *unit_lib_entry, char *unit_name, int dir, int x, int y, int entr, int exp_level, Unit_Lib_Entry *tran );

/* delete unit */
void delete_unit( void *poi );

/* adjust picture offset */
inline void adjust_unit_pic_offset( Unit *unit );

/* check and set direction in which unit looks when moving to passed position */
void check_unit_dir( Unit *unit, int x, int y );

/* update current values by checking experience and entrenchement */
/* this does not include fuel, ammo, attack_number ! */
void update_cur_unit_prop( Unit *unit );

/* adds perc percentage fuel to current fuel and returns the value of fuel added */
int give_fuel( Unit *unit, int perc );

/* adds perc percentage ammo to current ammo and returns the value of ammo added */
int give_ammo( Unit *unit, int perc );

/* return if unit uses fuel or is embarked and transporter uses fuel */
inline int use_fuel( Unit *unit );

/* mount unit on ground transporter */
void mount_unit( Unit *unit );

/* unmount unit from ground transporter */
void unmount_unit( Unit *unit );

/* check if unit can supply */
int unit_can_supply( Unit *unit, int type );

/* supply unit */
int supply_unit( Unit *unit );

/* add to units experience and compute exp_level (0-5) -- return 1 if level has changed */
int gain_exp( Unit *unit, int exp );

/* can unit attack other unit? check range, ammo, flying etc */
/* type is either ATTACK or DEFEND */
int can_attack_unit( Unit *att, Unit *def, int type );

/* return absolute damage def takes when attacked by att;
type specifies if this is a prediction(no random values) or real combat (random values) */
enum {
    DAMAGE_PRED = 0,
    REAL_COMBAT
};
int get_damage( Unit *att, Unit *def, int type );

/* execute combat between two units both attack each other if possible, take damages and
gain experience; also checks for rugged defence;
returns real damage take by both units */
enum {
    NORMAL = 0,
    AMBUSH,
    DEF_FIRE
};
void combat( Unit *att, Unit *def, int type, int *att_damage, int *def_damage );

/*
====================================================================
Check if these two units are allowed to merge with each other.
====================================================================
*/
int units_may_merge( Unit *unit1, Unit *unit2 );
/*
====================================================================
Melt these two units: unit1 is the new unit and unit2 should be
removed from map and memory after this function was called.
====================================================================
*/
void merge_units( Unit *unit1, Unit *unit2 );

#endif
