/*
 * h323ep.h
 *
 * H.323 protocol handler
 *
 * Open H323 Library
 *
 * Copyright (c) 1998-2001 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Portions of this code were written with the assisance of funding from
 * Vovida Networks, Inc. http://www.vovida.com.
 *
 * Contributor(s): ______________________________________.
 *
 * $Log: h323ep.h,v $
 * Revision 1.19  2002/04/10 06:48:47  robertj
 * Added functions to set port member variables.
 *
 * Revision 1.18  2002/03/14 03:49:38  dereks
 * Fix minor documentation error.
 *
 * Revision 1.17  2002/02/04 07:17:52  robertj
 * Added H.450.2 Consultation Transfer, thanks Norwood Systems.
 *
 * Revision 1.16  2002/01/24 06:29:02  robertj
 * Added option to disable H.245 negotiation in SETUP pdu, this required
 *   API change so have a bit mask instead of a series of booleans.
 *
 * Revision 1.15  2002/01/17 07:04:58  robertj
 * Added support for RFC2833 embedded DTMF in the RTP stream.
 *
 * Revision 1.14  2002/01/13 23:59:43  robertj
 * Added CallTransfer timeouts to endpoint, hanks Ben Madsen of Norwood Systems.
 *
 * Revision 1.13  2002/01/08 04:45:35  robertj
 * Added MakeCallLocked() so can start a call with the H323Connection instance
 *   initally locked so can do things to it before the call really starts.
 *
 * Revision 1.12  2001/12/22 03:20:44  robertj
 * Added create protocol function to H323Connection.
 *
 * Revision 1.11  2001/12/13 10:55:30  robertj
 * Added gatekeeper access token OID specification for auto inclusion of
 *   access tokens frm ACF to SETUP pdu.
 *
 * Revision 1.10  2001/11/09 05:39:54  craigs
 * Added initial T.38 support thanks to Adam Lazur
 *
 * Revision 1.9  2001/11/01 00:27:33  robertj
 * Added default Fast Start disabled and H.245 tunneling disable flags
 *   to the endpoint instance.
 *
 * Revision 1.8  2001/09/11 01:24:36  robertj
 * Added conditional compilation to remove video and/or audio codecs.
 *
 * Revision 1.7  2001/09/11 00:21:21  robertj
 * Fixed missing stack sizes in endpoint for cleaner thread and jitter thread.
 *
 * Revision 1.6  2001/08/24 14:03:26  rogerh
 * Fix some spelling mistakes
 *
 * Revision 1.5  2001/08/16 07:49:16  robertj
 * Changed the H.450 support to be more extensible. Protocol handlers
 *   are now in separate classes instead of all in H323Connection.
 *
 * Revision 1.4  2001/08/10 11:03:49  robertj
 * Major changes to H.235 support in RAS to support server.
 *
 * Revision 1.3  2001/08/08 23:54:11  robertj
 * Fixed problem with setting gk password before have a gk variable.
 *
 * Revision 1.2  2001/08/06 03:15:17  robertj
 * Improved access to H.235 secure RAS functionality.
 *
 * Revision 1.1  2001/08/06 03:08:11  robertj
 * Fission of h323.h to h323ep.h & h323con.h, h323.h now just includes files.
 *
 */

#ifndef __H323EP_H
#define __H323EP_H


#include "h323con.h"


/* The following classes have forward references to avoid including the VERY
   large header files for H225 and H245. If an application requires access
   to the protocol classes they can include them, but for simple usage their
   inclusion can be avoided.
 */
class H225_EndpointType;
class H225_VendorIdentifier;
class H225_H221NonStandard;

class H323SignalPDU;
class H323ConnectionsCleaner;

class OpalT120Protocol;
class OpalT38Protocol;


#ifdef __GNUC__
#pragma interface
#endif


///////////////////////////////////////////////////////////////////////////////

/**This class manages the H323 endpoint.
   An endpoint may have zero or more listeners to create incoming connections
   or zero or more outgoing conenctions initiated via the MakeCall() function.
   Once a conection exists it is managed by this class instance.

   The main thing this class embodies is the capabilities of the application,
   that is the codecs and protocols it is capable of.

   An application may create a descendent off this class and overide the
   CreateConnection() function, if they require a descendent of H323Connection
   to be created. This would be quite likely in most applications.
 */
class H323EndPoint : public PObject
{
  PCLASSINFO(H323EndPoint, PObject);

  public:
  /**@name Construction */
  //@{
    /**Create a new endpoint.
     */
    H323EndPoint();

    /**Destroy endpoint.
     */
    ~H323EndPoint();

    /**Set the endpoint information in H225 PDU's.
      */
    virtual void SetEndpointTypeInfo(
      H225_EndpointType & info
    ) const;

    /**Set the vendor information in H225 PDU's.
      */
    virtual void SetVendorIdentifierInfo(
      H225_VendorIdentifier & info
    ) const;

    /**Set the H221NonStandard information in H225 PDU's.
      */
    virtual void SetH221NonStandardInfo(
      H225_H221NonStandard & info
    ) const;
  //@}


  /**@name Capabilities */
  //@{
    /**Add a codec to the capabilities table. This will assure that the
       assignedCapabilityNumber field in the codec is unique for all codecs
       installed on this endpoint.

       If the specific instnace of the capability is already in the table, it
       is not added again. Ther can be multiple instances of the same
       capability class however.
     */
    void AddCapability(
      H323Capability * capability   /// New codec specification
    );

    /**Set the capability descriptor lists. This is three tier set of
       codecs. The top most level is a list of particular capabilities. Each
       of these consists of a list of alternatives that can operate
       simultaneously. The lowest level is a list of codecs that cannot
       operate together. See H323 section 6.2.8.1 and H245 section 7.2 for
       details.

       If descriptorNum is P_MAX_INDEX, the the next available index in the
       array of descriptors is used. Similarly if simultaneous is P_MAX_INDEX
       the the next available SimultaneousCapabilitySet is used. The return
       value is the index used for the new entry. Note if both are P_MAX_INDEX
       then the return value is the descriptor index as the simultaneous index
       must be zero.

       Note that the capability specified here is automatically added to the
       capability table using the AddCapability() function. A specific
       instance of a capability is only ever added once, so multiple
       SetCapability() calls with the same H323Capability pointer will only
       add that capability once.
     */
    PINDEX SetCapability(
      PINDEX descriptorNum, /// The member of the capabilityDescriptor to add
      PINDEX simultaneous,  /// The member of the SimultaneousCapabilitySet to add
      H323Capability * cap  /// New capability specification
    );

    /**Add all matching capabilities in list.
       All capabilities that match the specified name are added. See the
       capabilities code for details on the matching algorithm.
      */
    PINDEX AddAllCapabilities(
      PINDEX descriptorNum, /// The member of the capabilityDescriptor to add
      PINDEX simultaneous,  /// The member of the SimultaneousCapabilitySet to add
      const PString & name  /// New capabilities name, if using "known" one.
    );

    /**Add all user input capabilities to this endpoints capability table.
      */
    void AddAllUserInputCapabilities(
      PINDEX descriptorNum, /// The member of the capabilityDescriptor to add
      PINDEX simultaneous   /// The member of the SimultaneousCapabilitySet to add
    );

    /**Remove capabilites in table.
      */
    void RemoveCapabilities(
      const PStringArray & codecNames
    );

    /**Reorder capabilites in table.
      */
    void ReorderCapabilities(
      const PStringArray & preferenceOrder
    );

    /**Find a capability that has been registered.
     */
    H323Capability * FindCapability(
      const H245_Capability & cap  /// H245 capability table entry
    ) const;

    /**Find a capability that has been registered.
     */
    H323Capability * FindCapability(
      const H245_DataType & dataType  /// H245 data type of codec
    ) const;

    /**Find a capability that has been registered.
     */
    H323Capability * FindCapability(
      H323Capability::MainTypes mainType,   /// Main type of codec
      unsigned subType                      /// Subtype of codec
    ) const;
  //@}

  /**@name Gatekeeper management */
  //@{
    /**Select and register with an explicit gatekeeper.
       This will use the specified transport and a string giving a transport
       dependent address to locate a specific gatekeeper. The endpoint will
       register with that gatekeeper and, if successful, set it as the current
       gatekeeper used by this endpoint.

       Note the transport being passed in will be deleted by this function or
       the H323Gatekeeper object it becomes associated with. Also if transport
       is NULL then a H323TransportUDP is created.
     */
    BOOL SetGatekeeper(
      const PString & address,          /// Address of gatekeeper to use.
      H323Transport * transport = NULL  /// Transport over which to talk to gatekeeper.
    );

    /**Select and register with an explicit gatekeeper and zone.
       This will use the specified transport and a string giving a transport
       dependent address to locate a specific gatekeeper. The endpoint will
       register with that gatekeeper and, if successful, set it as the current
       gatekeeper used by this endpoint.

       The gatekeeper identifier is set to the spplied parameter to allow the
       gatekeeper to either allocate a zone or sub-zone, or refuse to register
       if the zones do not match.

       Note the transport being passed in will be deleted by this function or
       the H323Gatekeeper object it becomes associated with. Also if transport
       is NULL then a H323TransportUDP is created.
     */
    BOOL SetGatekeeperZone(
      const PString & address,          /// Address of gatekeeper to use.
      const PString & identifier,       /// Identifier of gatekeeper to use.
      H323Transport * transport = NULL  /// Transport over which to talk to gatekeeper.
    );

    /**Locate and select gatekeeper.
       This function will use the automatic gatekeeper discovery methods to
       locate the gatekeeper on the particular transport that has the specified
       gatekeeper identifier name. This is often the "Zone" for the gatekeeper.

       Note the transport being passed in will be deleted becomes owned by the
       H323Gatekeeper created by this function and will be deleted by it. Also
       if transport is NULL then a H323TransportUDP is created.
     */
    BOOL LocateGatekeeper(
      const PString & identifier,       /// Identifier of gatekeeper to locate.
      H323Transport * transport = NULL  /// Transport over which to talk to gatekeeper.
    );

    /**Discover and select gatekeeper.
       This function will use the automatic gatekeeper discovery methods to
       locate the first gatekeeper on a particular transport.

       Note the transport being passed in will be deleted becomes owned by the
       H323Gatekeeper created by this function and will be deleted by it. Also
       if transport is NULL then a H323TransportUDP is created.
     */
    BOOL DiscoverGatekeeper(
      H323Transport * transport = NULL  /// Transport over which to talk to gatekeeper.
    );

    /**Create a gatekeeper.
       This allows the application writer to have the gatekeeper as a
       descendent of the H323Gatekeeper in order to add functionality to the
       base capabilities in the library.

       The default creates an instance of the H323Gatekeeper class.
     */
    virtual H323Gatekeeper * CreateGatekeeper(
      H323Transport * transport  /// Transport over which gatekeepers communicates.
    );

    /**Get the gatekeeper we are registered with.
     */
    H323Gatekeeper * GetGatekeeper() const { return gatekeeper; }

    /**Unregister and delete the gatekeeper we are registered with.
       The return value indicates FALSE if there was an error during the
       unregistration. However the gatekeeper is still removed and its
       instance deleted regardless of this error.
     */
    BOOL RemoveGatekeeper(
      int reason = -1    /// Reason for gatekeeper removal
    );

    /**Set the H.235 password for the gatekeeper.
      */
    void SetGatekeeperPassword(
      const PString & password
    );
  //@}

  /**@name Connection management */
  //@{
    /**Add a listener to the endpoint.
       This allows for the automatic creating of incoming call connections. An
       application should use OnConnectionEstablished() to monitor when calls
       have arrived and been successfully negotiated.
      */
    BOOL StartListener(
      H323Listener * listener /// Transport dependent listener.
    );

    /**Add a listener to the endpoint.
       This allows for the automatic creating of incoming call connections. An
       application should use OnConnectionEstablished() to monitor when calls
       have arrived and been successfully negotiated.
      */
    BOOL StartListener(
      const H323TransportAddress & iface /// Address of interface to listen on.
    );

    /**Remove a listener from the endpoint.
       If the listener parameter is NULL then all listeners are removed.
      */
    BOOL RemoveListener(
      H323Listener * listener /// Transport dependent listener.
    );

    /**Make a call to a remote party. An appropriate transport is determined
       from the remoteParty parameter. The general form for this parameter is
       [alias@][transport$]host[:port] where the default alias is the same as
       the host, the default transport is "ip" and the default port is 1720.

       This function returns almost immediately with the call occurring in a
       new background thread. Note that the call could be created and cleared
       ie OnConnectionCleared is called BEFORE this function returns. It is
       guaranteed that the token variable is set before OnConnectionCleared
       called.

       Note, the returned pointer to the connection is not locked and may be
       deleted at any time. This is extremely unlikely immediately after the
       function is called, but you should not keep this pointer beyond that
       brief time. The the FindConnectionWithLock() function for future
       references to the connection object. It is recommended that
       MakeCallLocked() be usedin instead.
     */
    H323Connection * MakeCall(
      const PString & remoteParty,  /// Remote party to call
      PString & token,              /// String to receive token for connection
      void * userData = NULL        /// user data to pass to CreateConnection
    );

    /**Make a call to a remote party using the specified transport.
       The remoteParty may be a hostname, alias or other user name that is to
       be passed to the transport, if present.

       If the transport parameter is NULL the transport is determined from the
       remoteParty description.

       This function returns almost immediately with the call occurring in a
       new background thread. Note that the call could be created and cleared
       ie OnConnectionCleared is called BEFORE this function returns. It is
       guaranteed that the token variable is set before OnConnectionCleared
       called.

       Note, the returned pointer to the connection is not locked and may be
       deleted at any time. This is extremely unlikely immediately after the
       function is called, but you should not keep this pointer beyond that
       brief time. The the FindConnectionWithLock() function for future
       references to the connection object. It is recommended that
       MakeCallLocked() be usedin instead.
     */
    H323Connection * MakeCall(
      const PString & remoteParty,  /// Remote party to call
      H323Transport * transport,    /// Transport to use for call.
      PString & token,              /// String to receive token for connection
      void * userData = NULL        /// user data to pass to CreateConnection
    );

    /**Make a call to a remote party using the specified transport.
       The remoteParty may be a hostname, alias or other user name that is to
       be passed to the transport, if present.

       If the transport parameter is NULL the transport is determined from the
       remoteParty description.

       This function returns almost immediately with the call occurring in a
       new background thread. However the call will not progress very far 
     */
    H323Connection * MakeCallLocked(
      const PString & remoteParty,     /// Remote party to call
      PString & token,                 /// String to receive token for connection
      void * userData = NULL,          /// user data to pass to CreateConnection
      H323Transport * transport = NULL /// Transport to use for call.
    );

    /**Setup the transfer of an existing call (connection) to a new remote party
       using H.450.2.  This sends a Call Transfer Setup Invoke message from the
       B-Party (transferred endpoint) to the C-Party (transferred-to endpoint).

       If the transport parameter is NULL the transport is determined from the
       remoteParty description. The general form for this parameter is
       [alias@][transport$]host[:port] where the default alias is the same as
       the host, the default transport is "ip" and the default port is 1720.

       This function returns almost immediately with the transfer occurring in a
       new background thread.

       Note, the returned pointer to the connection is not locked and may be
       deleted at any time. This is extremely unlikely immediately after the
       function is called, but you should not keep this pointer beyond that
       brief time. The the FindConnectionWithLock() function for future
       references to the connection object.

       This function is declared virtual to allow an application to override
       the function and get the new call token of the forwarded call.
     */
    virtual H323Connection * SetupTransfer(
      const PString & token,        /// Existing connection to be transferred
      const PString & callIdentity, /// Call identity of the secondary call (if it exists)
      const PString & remoteParty,  /// Remote party to transfer the existing call to
      PString & newToken,           /// String to receive token for the new connection
      void * userData = NULL        /// user data to pass to CreateConnection
    );

    /**Initiate the transfer of an existing call (connection) to a new remote
       party using H.450.2.  This sends a Call Transfer Initiate Invoke
       message from the A-Party (transferring endpoint) to the B-Party
       (transferred endpoint).
     */
    void TransferCall(
      const PString & token,        /// Existing connection to be transferred
      const PString & remoteParty,  /// Remote party to transfer the existing call to
      const PString & callIdentity = PString()
                                    /// Call Identity of secondary call if present
    );

    /**Transfer the call through consultation so the remote party in the
       primary call is connected to the called party in the second call
       using H.450.2.  This sends a Call Transfer Identify Invoke message
       from the A-Party (transferring endpoint) to the C-Party
       (transferred-to endpoint).
     */
    void ConsultationTransfer(
      const PString & primaryCallToken,   /// Token of primary call
      const PString & secondaryCallToken  /// Token of secondary call
    );

    /**Place the call on hold, suspending all media channels (H.450.4)
    * NOTE: Only Local Hold is implemented so far. 
    */
    void HoldCall(
      const PString & token,        /// Existing connection to be transferred
      BOOL localHold   /// true for Local Hold, false for Remote Hold
    );

    /**Parse a party address into alias and transport components.
       An appropriate transport is determined from the remoteParty parameter.
       The general form for this parameter is [alias@][transport$]host[:port]
       where the default alias is the same as the host, the default transport
       is "ip" and the default port is 1720.
      */
    void ParsePartyName(
      const PString & party,          /// Party name string.
      PString & alias,                /// Parsed alias name
      H323TransportAddress & address  /// Parsed transport address
    ) const;

    /**Clear a current connection.
       This hangs up the connection to a remote endpoint. Note that this function
       is asynchronous
      */
    virtual BOOL ClearCall(
      const PString & token,  /// Token for identifying connection
      H323Connection::CallEndReason reason =
                  H323Connection::EndedByLocalUser /// Reason for call clearing
    );

    /**Clear a current connection.
       This hangs up the connection to a remote endpoint. Note that these functions
       are synchronous
      */
    virtual BOOL ClearCallSynchronous(
      const PString & token,            /// Token for identifying connection
      H323Connection::CallEndReason reason =
                        H323Connection::EndedByLocalUser /// Reason for call clearing
    );
    virtual BOOL ClearCallSynchronous(
      const PString & token,                /// Token for identifying connection
      H323Connection::CallEndReason reason, /// Reason for call clearing
      PSyncPoint * sync
    );

    /**Clear all current connections.
       This hangs up all the connections to remote endpoints. The wait
       parameter is used to wait for all the calls to be cleared and their
       memory usage cleaned up before returning. This is typically used in
       the destructor for your descendant of H323EndPoint.
      */
    virtual void ClearAllCalls(
      H323Connection::CallEndReason reason =
                  H323Connection::EndedByLocalUser, /// Reason for call clearing
      BOOL wait = TRUE   /// Flag for wait for calls to e cleared.
    );

    /**Determine if a connection is active.
      */
    virtual BOOL HasConnection(
      const PString & token   /// Token for identifying connection
    );

    /**Find a connection that uses the specified token.
       This searches the endpoint for the connection that contains the token
       as provided by functions such as MakeCall() (as built by the
       BuildConnectionToken() function). if not found it will then search for
       the string representation of the CallIdentifier for the connection, and
       finally try for the string representation of the ConferenceIdentifier.

       Note the caller of this function MUSt call the H323Connection::Unlock()
       function if this function returns a non-NULL pointer. If it does not
       then a deadlock can occur.
      */
    H323Connection * FindConnectionWithLock(
      const PString & token     /// Token to identify connection
    );

    /**Call back for incoming call.
       This function is called from the OnReceivedSignalSetup() function
       before it sends the Alerting PDU. It gives an opportunity for an
       application to alter the reply before transmission to the other
       endpoint.

       If FALSE is returned the connection is aborted and a Release Complete
       PDU is sent.

       The default behaviour simply returns TRUE.
     */
    virtual BOOL OnIncomingCall(
      H323Connection & connection,    /// Connection that was established
      const H323SignalPDU & setupPDU,   /// Received setup PDU
      H323SignalPDU & alertingPDU       /// Alerting PDU to send
    );

    /**Handle a connection transfer.
       This gives the application an opportunity to abort the transfer.
       The default behaviour just returns TRUE.
      */
    BOOL OnCallTransferInitiate(
      H323Connection & connection,    /// Connection to transfer
      const PString & remoteParty     /// Party transferring to.
    );

    /**Handle a transfer via consultation.
       This gives the transferred-to user an opportunity to abort the transfer.
       The default behaviour just returns TRUE.
      */
    virtual BOOL OnCallTransferIdentify(
      H323Connection & connection    /// Connection to transfer
    );

    /**Call back for answering an incoming call.
       This function is called from the OnReceivedSignalSetup() function
       before it sends the Connect PDU. It gives an opportunity for an
       application to alter the reply before transmission to the other
       endpoint.

       It also gives an application time to wait for some event before
       signalling to the endpoint that the connection is to proceed. For
       example the user pressing an "Answer call" button.

       If AnswerCallDenied is returned the connection is aborted and a Release
       Complete PDU is sent. If AnswerCallNow is returned then the H.323
       protocol proceeds. Finally if AnswerCallPending is returned then the
       protocol negotiations are paused until the AnsweringCall() function is
       called.

       The default behaviour simply returns AnswerNow.
     */
    virtual H323Connection::AnswerCallResponse OnAnswerCall(
      H323Connection & connection,    /// Connection that was established
      const PString & callerName,       /// Name of caller
      const H323SignalPDU & setupPDU,   /// Received setup PDU
      H323SignalPDU & connectPDU        /// Connect PDU to send. 
    );

    /**Call back for remote party being alerted.
       This function is called from the SendSignalSetup() function after it
       receives the optional Alerting PDU from the remote endpoint. That is
       when the remote "phone" is "ringing".

       If FALSE is returned the connection is aborted and a Release Complete
       PDU is sent.

       The default behaviour simply returns TRUE.
     */
    virtual BOOL OnAlerting(
      H323Connection & connection,    /// Connection that was established
      const H323SignalPDU & alertingPDU,  /// Received Alerting PDU
      const PString & user                /// Username of remote endpoint
    );

    /**A call back function whenever connection indicates it should be
       forwarded.
       Return TRUE if the call is being redirected.

       The default behaviour simply returns FALSE.
      */
    virtual BOOL OnConnectionForwarded(
      H323Connection & connection,    /// Connection to be forwarded
      const PString & forwardParty,   /// Remote party to forward to
      const H323SignalPDU & pdu       /// Full PDU initiating forwarding
    );

    /**A call back function whenever a connection is established.
       This indicates that a connection to a remote endpoint was established
       with a control channel and zero or more logical channels.

       The default behaviour does nothing.
      */
    virtual void OnConnectionEstablished(
      H323Connection & connection,    /// Connection that was established
      const PString & token           /// Token for identifying connection
    );

    /**Determine if a connection is established.
      */
    virtual BOOL IsConnectionEstablished(
      const PString & token   /// Token for identifying connection
    );

    /**A call back function whenever a connection is broken.
       This indicates that a connection to a remote endpoint is no longer
       available.

       The default behaviour does nothing.
      */
    virtual void OnConnectionCleared(
      H323Connection & connection,    /// Connection that was established
      const PString & token           /// Token for identifying connection
    );

    /**Build a unique token for the connection.
       This identifies the call using the Q931 transport host name and the
       Q931 call reference number.
      */
    static PString BuildConnectionToken(
      const H323Transport & transport,  /// Transport for connection
      unsigned callReference,           /// Call reference of Q.931 link
      BOOL fromRemote                   /// Call reference is from remote endpoint
    );

    /**Handle a new incoming connection.
       This will examine the setup PDU and either attach the signalling
       transport to an existing connection that has the same Q.931 call
       reference, or creates a new connection using CreateConnection().
      */
    H323Connection * OnIncomingConnection(
      H323Transport * transport,  /// Transport for connection
      H323SignalPDU & setupPDU    /// Setup PDU
    );

    /**Create a connection that uses the specified call reference.
      */
    virtual H323Connection * CreateConnection(
      unsigned callReference,     /// Call reference to use
      void * userData,            /// user data to pass to CreateConnection
      H323Transport * transport,  /// Transport for connection
      H323SignalPDU * setupPDU    /// Setup PDU, NULL if outgoing call
    );
    virtual H323Connection * CreateConnection(
      unsigned callReference,   /// Call reference to use
      void * userData           /// user data to pass to CreateConnection
    );
    virtual H323Connection * CreateConnection(
      unsigned callReference    /// Call reference to use
    );

    /**Clean up connections.
       This function is called from a background thread and checks for closed
       connections to clean up.

       This would not normally be called by an application.
      */
    void CleanUpConnections();
  //@}


  /**@name Logical Channels management */
  //@{
    /**Call back for opening a logical channel.

       The default behaviour simply returns TRUE.
      */
    virtual BOOL OnStartLogicalChannel(
      H323Connection & connection,    /// Connection for the channel
      H323Channel & channel           /// Channel being started
    );

    /**Call back for closed a logical channel.

       The default behaviour does nothing.
      */
    virtual void OnClosedLogicalChannel(
      H323Connection & connection,    /// Connection for the channel
      const H323Channel & channel     /// Channel being started
    );

    /**Open a channel for use by an audio codec.
       The H323AudioCodec class will use this function to open the channel to
       read/write PCM data.

       The default function creates a PSoundChannel using the member variables
       soundChannelPlayDevice or soundChannelRecordDevice.
      */
    virtual BOOL OpenAudioChannel(
      H323Connection & connection,  /// Connection for the channel
      BOOL isEncoding,              /// Direction of data flow
      unsigned bufferSize,          /// Size of each sound buffer
      H323AudioCodec & codec        /// codec that is doing the opening
    );

#ifndef NO_H323_VIDEO
    /**Open a channel for use by an video codec.
       The H323VideoCodec class will use this function to open the channel to
       read/write image data (which is one frame in a video stream - YUV411 format).

       The default function creates a PVideoChannel using the member variables.
      */
    virtual BOOL OpenVideoChannel(
      H323Connection & connection,  /// Connection for the channel
      BOOL isEncoding,              /// Direction of data flow
      H323VideoCodec & codec        /// codec doing the opening
    );
#endif // NO_H323_VIDEO

    /**Callback from the RTP session for statistics monitoring.
       This is called every so many packets on the transmitter and receiver
       threads of the RTP session indicating that the statistics have been
       updated.

       The default behaviour does nothing.
      */
    virtual void OnRTPStatistics(
      const H323Connection & connection,  /// Connection for the channel
      const RTP_Session & session         /// Session with statistics
    ) const;
  //@}

  /**@name Indications */
  //@{
    /**Call back for remote enpoint has sent user input as a string.

       The default behaviour does nothing.
      */
    virtual void OnUserInputString(
      H323Connection & connection,  /// Connection for the input
      const PString & value         /// String value of indication
    );

    /**Call back for remote enpoint has sent user input.

       The default behaviour calls H323Connection::OnUserInputTone().
      */
    virtual void OnUserInputTone(
      H323Connection & connection,  /// Connection for the input
      char tone,                    /// DTMF tone code
      unsigned duration,            /// Duration of tone in milliseconds
      unsigned logicalChannel,      /// Logical channel number for RTP sync.
      unsigned rtpTimestamp         /// RTP timestamp in logical channel sync.
    );
  //@}

  /**@name Other services */
  //@{
    /**Create an instance of the T.120 protocol handler.
       This is called when the OpenLogicalChannel subsystem requires that
       a T.120 channel be established.

       The default behavour returns NULL.
      */
    virtual OpalT120Protocol * CreateT120ProtocolHandler(
      const H323Connection & connection  /// Connection for which T.120 handler created
    ) const;

    /**Create an instance of the T.38 protocol handler.
       This is called when the OpenLogicalChannel subsystem requires that
       a T.38 fax channel be established.

       The default behavour returns NULL.
      */
    virtual OpalT38Protocol * CreateT38ProtocolHandler(
      const H323Connection & connection  /// Connection for which T.38 handler created
    ) const;
  //@}

  /**@name Member variable access */
  //@{
    /**Set the user name to be used for the local end of any connections. This
       defaults to the logged in user as obtained from the
       PProcess::GetUserName() function.

       Note that this name is technically the first alias for the endpoint.
       Additional aliases may be added by the use of the AddAliasName()
       function, however that list will be cleared when this function is used.
     */
    void SetLocalUserName(
      const PString & name  /// Local name of endpoint (prime alias)
    );

    /**Get the user name to be used for the local end of any connections. This
       defaults to the logged in user as obtained from the
       PProcess::GetUserName() function.
     */
    const PString & GetLocalUserName() const { return localAliasNames[0]; }

    /**Add an alias name to be used for the local end of any connections. If
       the alias name already exists in the list then is is not added again.

       The list defaults to the value set in the SetLocalUserName() function.
       Note that calling SetLocalUserName() will clear the alias list.
     */
    BOOL AddAliasName(
      const PString & name  /// New alias name to add
    );

    /**Remove an alias name used for the local end of any connections. 
       defaults to an empty list.
     */
    BOOL RemoveAliasName(
      const PString & name  /// New alias namer to add
    );

    /**Get the user name to be used for the local end of any connections. This
       defaults to the logged in user as obtained from the
       PProcess::GetUserName() function.
     */
    const PStringList & GetAliasNames() const { return localAliasNames; }

    /**Get the default fast start mode.
      */
    BOOL IsFastStartDisabled() const
      { return disableFastStart; }

    /**Set the default fast start mode.
      */
    void DisableFastStart(
      BOOL mode /// New default mode
    ) { disableFastStart = mode; } 

    /**Get the default H.245 tunneling mode.
      */
    BOOL IsH245TunnelingDisabled() const
      { return disableH245Tunneling; }

    /**Set the default H.245 tunneling mode.
      */
    void DisableH245Tunneling(
      BOOL mode /// New default mode
    ) { disableH245Tunneling = mode; } 

    /**Get the default H.245 tunneling mode.
      */
    BOOL IsH245inSetupDisabled() const
      { return disableH245inSetup; }

    /**Set the default H.245 tunneling mode.
      */
    void DisableH245inSetup(
      BOOL mode /// New default mode
    ) { disableH245inSetup = mode; } 

    /**Set the name for the sound channel to be used for output.
       If the name is not suitable for use with the PSoundChannel class then
       the function will return FALSE and not change the device.

       This defaults to the value of the PSoundChannel::GetDefaultDevice()
       function.
     */
    virtual BOOL SetSoundChannelPlayDevice(const PString & name);

    /**Get the name for the sound channel to be used for output.
       This defaults to the value of the PSoundChannel::GetDefaultDevice()
       function.
     */
    const PString & GetSoundChannelPlayDevice() const { return soundChannelPlayDevice; }

    /**Set the name for the sound channel to be used for input.
       If the name is not suitable for use with the PSoundChannel class then
       the function will return FALSE and not change the device.

       This defaults to the value of the PSoundChannel::GetDefaultDevice()
       function.
     */
    virtual BOOL SetSoundChannelRecordDevice(const PString & name);

    /**Get the name for the sound channel to be used for input.
       This defaults to the value of the PSoundChannel::GetDefaultDevice()
       function.
     */
    const PString & GetSoundChannelRecordDevice() const { return soundChannelRecordDevice; }

    /**Get default the sound channel buffer depth.
      */
    unsigned GetSoundChannelBufferDepth() const { return soundChannelBuffers; }

    /**Set the default sound channel buffer depth.
      */
    void SetSoundChannelBufferDepth(
      unsigned depth    // New depth
    );

    /**Get the default silence detection mode.
      */
    H323AudioCodec::SilenceDetectionMode GetSilenceDetectionMode() const
      { return defaultSilenceDetection; }

    /**Set the default silence detection mode.
      */
    void SetSilenceDetectionMode(
      H323AudioCodec::SilenceDetectionMode mode /// New default mode
    ) { defaultSilenceDetection = mode; } 

    /**Get the default mode for sending User Input Indications.
      */
    H323Connection::SendUserInputModes GetSendUserInputModes() const { return defaultSendUserInputMode; }

    /**Set the default mode for sending User Input Indications.
      */
    void SetSendUserInputModes(H323Connection::SendUserInputModes mode) { defaultSendUserInputMode = mode; }

    /**See if should auto-start receive video channels on connection.
     */
    BOOL CanAutoStartReceiveVideo() const { return autoStartReceiveVideo; }

    /**See if should auto-start transmit video channels on connection.
     */
    BOOL CanAutoStartTransmitVideo() const { return autoStartTransmitVideo; }

    /**See if should auto-start receive fax channels on connection.
     */
    BOOL CanAutoStartReceiveFax() const { return autoStartReceiveFax; }

    /**See if should auto-start transmit fax channels on connection.
     */
    BOOL CanAutoStartTransmitFax() const { return autoStartTransmitFax; }

    /**Get the set of listeners (incoming call transports) for this endpoint.
     */
    const H323ListenerList & GetListeners() const { return listeners; }

    /**Get the current capability table for this endpoint.
     */
    const H323Capabilities & GetCapabilities() const { return capabilities; }

    /**Endpoint types.
     */
    enum TerminalTypes {
      e_TerminalOnly = 50,
      e_TerminalAndMC = 70,
      e_GatewayOnly = 60,
      e_GatewayAndMC = 80,
      e_GatewayAndMCWithDataMP = 90,
      e_GatewayAndMCWithAudioMP = 100,
      e_GatewayAndMCWithAVMP = 110,
      e_GatekeeperOnly = 120,
      e_GatekeeperWithDataMP = 130,
      e_GatekeeperWithAudioMP = 140,
      e_GatekeeperWithAVMP = 150,
      e_MCUOnly = 160,
      e_MCUWithDataMP = 170,
      e_MCUWithAudioMP = 180,
      e_MCUWithAVMP = 190
    };

    /**Get the endpoint terminal type.
     */
    TerminalTypes GetTerminalType() const { return terminalType; }

    /**Determine if endpoint is terminal type.
     */
    BOOL IsTerminal() const;

    /**Determine if endpoint is gateway type.
     */
    BOOL IsGateway() const;

    /**Determine if endpoint is gatekeeper type.
     */
    BOOL IsGatekeeper() const;

    /**Determine if endpoint is gatekeeper type.
     */
    BOOL IsMCU() const;

    /**Get the default maximum audio delay jitter parameter.
     */
    WORD GetMaxAudioDelayJitter() const { return maxAudioDelayJitter; }

    /**Set the default maximum audio delay jitter parameter.
     */
    void SetMaxAudioDelayJitter(unsigned jitter) { maxAudioDelayJitter = (WORD)jitter; }

    /**Get the initial bandwidth parameter.
     */
    unsigned GetInitialBandwidth() const { return initialBandwidth; }

    /**Get the initial bandwidth parameter.
     */
    void SetInitialBandwidth(unsigned bandwidth) { initialBandwidth = bandwidth; }

    /**Get the TCP port number base for H.245 channels
     */
    WORD GetTCPPortBase() const { return tcpPortBase; }

    /**Get the TCP port number base for H.245 channels.
     */
    WORD GetTCPPortMax() const { return tcpPortMax; }

    /**Set the TCP port number base and max for H.245 channels.
     */
    void SetTCPPorts(unsigned tcpBase, unsigned tcpMax);

    /**Provide TCP address translation hook
     */
    virtual void TranslateTCPAddress(PIPSocket::Address & /*localAddr*/, const PIPSocket::Address & /*remoteAddr */) { }

    /**Get the UDP port number base for RTP channels.
     */
    WORD GetRtpIpPortBase() const { return rtpIpPortBase; }

    /**Get the max UDP port number for RTP channels.
     */
    WORD GetRtpIpPortMax() const { return rtpIpPortMax; }

    /**Set the UDP port number base and max for RTP channels.
     */
    void SetRtpIpPorts(unsigned udpBase, unsigned udpMax);

    /**Get the IP Type Of Service byte for RTP channels.
     */
    BYTE GetRtpIpTypeofService() const { return rtpIpTypeofService; }

    /**Set the IP Type Of Service byte for RTP channels.
     */
    void SetRtpIpTypeofService(unsigned tos) { rtpIpTypeofService = (BYTE)tos; }

    /**Get the default timeout for calling another endpoint.
     */
    PTimeInterval GetSignallingChannelCallTimeout() const { return signallingChannelCallTimeout; }

    /**Get the default timeout for master slave negotiations.
     */
    PTimeInterval GetMasterSlaveDeterminationTimeout() const { return masterSlaveDeterminationTimeout; }

    /**Get the default retries for H245 master slave negotiations.
     */
    unsigned GetMasterSlaveDeterminationRetries() const { return masterSlaveDeterminationRetries; }

    /**Get the default timeout for H245 capability exchange negotiations.
     */
    PTimeInterval GetCapabilityExchangeTimeout() const { return capabilityExchangeTimeout; }

    /**Get the default timeout for H245 logical channel negotiations.
     */
    PTimeInterval GetLogicalChannelTimeout() const { return logicalChannelTimeout; }

    /**Get the default timeout for H245 request mode negotiations.
     */
    PTimeInterval GetRequestModeTimeout() const { return logicalChannelTimeout; }

    /**Get the default timeout for H245 round trip delay negotiations.
     */
    PTimeInterval GetRoundTripDelayTimeout() const { return roundTripDelayTimeout; }

    /**Get the default rate H245 round trip delay is calculated by connection.
     */
    PTimeInterval GetRoundTripDelayRate() const { return roundTripDelayRate; }

    /**Get the default rate H245 round trip delay is calculated by connection.
     */
    BOOL ShouldClearCallOnRoundTripFail() const { return clearCallOnRoundTripFail; }

    /**Get the default timeout for Call Transfer Timer CT-T1.
     */
    PTimeInterval GetCallTransferT1() const { return callTransferT1; }

    /**Get the default timeout for Call Transfer Timer CT-T2.
     */
    PTimeInterval GetCallTransferT2() const { return callTransferT2; }

    /**Get the default timeout for Call Transfer Timer CT-T3.
     */
    PTimeInterval GetCallTransferT3() const { return callTransferT3; }

    /**Get the default timeout for Call Transfer Timer CT-T4.
     */
    PTimeInterval GetCallTransferT4() const { return callTransferT4; }

    /**Get the default timeout for GatekeeperRequest and Gatekeeper discovery.
     */
    PTimeInterval GetGatekeeperRequestTimeout() const { return gatekeeperRequestTimeout; }

    /**Get the default retries for GatekeeperRequest and Gatekeeper discovery.
     */
    unsigned GetGatekeeperRequestRetries() const { return gatekeeperRequestRetries; }

    /**Get the default timeout for RAS protocol transactions.
     */
    PTimeInterval GetRasRequestTimeout() const { return rasRequestTimeout; }

    /**Get the default retries for RAS protocol transations.
     */
    unsigned GetRasRequestRetries() const { return rasRequestRetries; }

    /**Get the default time for gatekeeper to reregister.
       A value of zero disables the keep alive facility.
     */
    PTimeInterval GetGatekeeperTimeToLive() const { return registrationTimeToLive; }

    /**Get the iNow Gatekeeper Access Token OID.
     */
    const PString GetGkAccessTokenOID() const { return gkAccessTokenOID; }

    /**Get the default stack size of cleaner threads.
     */
    PINDEX GetCleanerThreadStackSize() const { return cleanerThreadStackSize; }

    /**Get the default stack size of listener threads.
     */
    PINDEX GetListenerThreadStackSize() const { return listenerThreadStackSize; }

    /**Get the default stack size of signalling channel threads.
     */
    PINDEX GetSignallingThreadStackSize() const { return signallingThreadStackSize; }

    /**Get the default stack size of control channel threads.
     */
    PINDEX GetControlThreadStackSize() const { return controlThreadStackSize; }

    /**Get the default stack size of logical channel threads.
     */
    PINDEX GetChannelThreadStackSize() const { return logicalThreadStackSize; }

    /**Get the default stack size of RAS channel threads.
     */
    PINDEX GetRasThreadStackSize() const { return rasThreadStackSize; }

    /**Get the default stack size of jitter buffer threads.
     */
    PINDEX GetJitterThreadStackSize() const { return jitterThreadStackSize; }

    /**Get the dictionary of <callIdentities, connections>
     */
    H323CallIdentityDict& GetCallIdentityDictionary() { return secondaryConenctionsActive; }
  //@}


  protected:
    H323Gatekeeper * InternalCreateGatekeeper(H323Transport * transport);
    BOOL InternalRegisterGatekeeper(H323Gatekeeper * gk, BOOL discovered);
    H323Connection * FindConnectionWithoutLocks(const PString & token);
    H323Connection * InternalMakeCall(
      const PString & existingToken,/// Existing connection to be transferred
      const PString & callIdentity, /// Call identity of the secondary call (if it exists)
      const PString & remoteParty,  /// Remote party to call
      H323Transport * transport,    /// Transport to use for call.
      PString & token,              /// String to receive token for connection
      void * userData = NULL        /// user data to pass to CreateConnection
    );


    // Configuration variables, commonly changed
    PStringList localAliasNames;
    PString     soundChannelPlayDevice;
    PString     soundChannelRecordDevice;
    PString     videoChannelPlayDevice;
    PString     videoChannelRecordDevice;
    BOOL        autoStartReceiveVideo;
    BOOL        autoStartTransmitVideo;
    BOOL        autoStartReceiveFax;
    BOOL        autoStartTransmitFax;
    BOOL        disableFastStart;
    BOOL        disableH245Tunneling;
    BOOL        disableH245inSetup;
    H323AudioCodec::SilenceDetectionMode defaultSilenceDetection;
    H323Connection::SendUserInputModes defaultSendUserInputMode;

    BYTE          t35CountryCode;
    BYTE          t35Extension;
    WORD          manufacturerCode;
    TerminalTypes terminalType;

    WORD     maxAudioDelayJitter;
    unsigned initialBandwidth;  // in 100s of bits/sev
    BOOL     clearCallOnRoundTripFail;

    // Some more configuration variables, rarely changed.
    unsigned      soundChannelBuffers;
    WORD          rtpIpPortBase, rtpIpPortMax;
    WORD          tcpPortBase, tcpPortMax;
    BYTE          rtpIpTypeofService;
    PTimeInterval signallingChannelCallTimeout;
    PTimeInterval masterSlaveDeterminationTimeout;
    unsigned      masterSlaveDeterminationRetries;
    PTimeInterval capabilityExchangeTimeout;
    PTimeInterval logicalChannelTimeout;
    PTimeInterval requestModeTimeout;
    PTimeInterval roundTripDelayTimeout;
    PTimeInterval roundTripDelayRate;
    PTimeInterval gatekeeperRequestTimeout;
    unsigned      gatekeeperRequestRetries;
    PTimeInterval rasRequestTimeout;
    unsigned      rasRequestRetries;
    PTimeInterval registrationTimeToLive;
    PString       gkAccessTokenOID;

    /* Protect against absence of a response to the ctIdentify reqest
       (Transferring Endpoint - Call Transfer with a secondary Call) */
    PTimeInterval callTransferT1;
    /* Protect against failure of completion of the call transfer operation
       involving a secondary Call (Transferred-to Endpoint) */
    PTimeInterval callTransferT2;
    /* Protect against failure of the Transferred Endpoint not responding
       within sufficient time to the ctInitiate APDU (Transferring Endpoint) */
    PTimeInterval callTransferT3;
    /* May optionally operate - protects against absence of a response to the
       ctSetup request (Transferred Endpoint) */
    PTimeInterval callTransferT4;

    PINDEX cleanerThreadStackSize;
    PINDEX listenerThreadStackSize;
    PINDEX signallingThreadStackSize;
    PINDEX controlThreadStackSize;
    PINDEX logicalThreadStackSize;
    PINDEX rasThreadStackSize;
    PINDEX jitterThreadStackSize;

    // Dynamic variables
    H323ListenerList listeners;
    H323Capabilities capabilities;
    H323Gatekeeper * gatekeeper;
    PString          gatekeeperPassword;

    H323ConnectionDict       connectionsActive;
    H323CallIdentityDict     secondaryConenctionsActive;
    PMutex                   connectionsMutex;
    PStringSet               connectionsToBeCleaned;
    H323ConnectionsCleaner * connectionsCleaner;
    PSyncPoint               connectionsAreCleaned;
};


#endif // __H323_H


/////////////////////////////////////////////////////////////////////////////
