#ifndef DRS_H
#define DRS_H

/*
 * Interface definition for the directory discriptors (drs).
 */

typedef struct drs* drs_t;

extern drs_t
drs_new(void);

/* Delete a drs instance. */
extern void
drs_delete(drs_t drs);

/* Return the maximum number of entries supported by ISO 9660. */
extern unsigned long
drs_get_directory_entries_max();

/* The "current" element is the directory entry used for getting and
 * setting values.  You can use this method to get which entry is
 * current. */
extern size_t
drs_get_current(drs_t drs);

/* The "current" element is the directory entry used for getting and
 * setting values.  You can use this method to set which entry is
 * current.  The value returned is 1 on success and 0 on error.  You
 * can use this to iterate over the entries. */
extern int
drs_set_current(drs_t drs, size_t current);

/* dss_push_back() adds a directory entry to the end of the list and
 * changes the "current" entry to point to the new entry.  The entire
 * block will be initialized with all zeros.  If you need different
 * values, you should populate the entry by calling the drs_set-*()
 * methods below. */
extern void
drs_push_back(drs_t drs);

/* Print ALL drs entries to a "f". */
extern void
drs_dump_to_file(drs_t drs, FILE* f);

/* Return a printable string that lists ALL the contents of the
 * "current" drs directory entry.  The calling function needs to
 * provide a buffer to hold the string.  The buffer must be at least
 * as large as the value returned by
 * drs_get_current_as_string_min_buf_size().  If you want to print all
 * drs entry to a FILE, use drs_dump_to_file() above. */
extern void
drs_get_current_as_string(drs_t drs, char* buf);

extern size_t
drs_get_current_as_string_min_buf_size(void);

/*
 * Getters.
 */

extern unsigned char
drs_get_length_of_directory_record(drs_t drs);

extern unsigned char
drs_get_length_of_extended_attribute_record(drs_t drs);

extern unsigned long int
drs_get_location_of_extent(drs_t drs);

extern unsigned long int
drs_get_data_length(drs_t drs);

extern unsigned char
drs_get_years_since_1900(drs_t drs);

extern unsigned char
drs_get_month(drs_t drs);

extern unsigned char
drs_get_day(drs_t drs);

extern unsigned char
drs_get_hour(drs_t drs);

extern unsigned char
drs_get_minute(drs_t drs);

extern unsigned char
drs_get_second(drs_t drs);

extern unsigned char
drs_get_offset_from_GMT(drs_t drs);

extern unsigned char
drs_get_file_flags(drs_t drs);

extern unsigned char
drs_get_file_unit_size(drs_t drs);

extern unsigned char
drs_get_interleave_gap_size(drs_t drs);

extern unsigned long int
drs_get_volume_sequence_number(drs_t drs);

extern unsigned char
drs_get_length_of_file_identifier(drs_t drs);

/* WARNING: This method does not necessarily return a properly
 * null-terminated c-style string.  You MUST call
 * drs_get_length_of_file_identifier() in order to know the length of
 * the file_idnetifier.  The reason for this is that ISO 9660 denotes
 * the "this" and "parent" directories usin binary 0x00 and 0x01. */
extern const char*
drs_get_file_identifier(drs_t drs);

/* Returns the length of the padding field.  It will either return
 * zero or one. */
extern int
drs_get_length_of_padding_field(drs_t drs);


/*
 * Setters.
 */

/*
 * NOTE: There are no drs_set_length_of_directory_record() and
 * drs_set_length_of_file_identifier() methods.  These values are
 * derived from other values.
 */

extern void
drs_set_length_of_extended_attribute_record(drs_t drs, 
                   unsigned char length_of_extended_attribute_record);

extern void
drs_set_location_of_extent(drs_t drs, unsigned long int location_of_extent);

extern void
drs_set_data_length(drs_t drs, unsigned long int data_length);

extern void
drs_set_years_since_1900(drs_t drs, unsigned char years_since_1900);

extern void
drs_set_month(drs_t drs, unsigned char month);

extern void
drs_set_day(drs_t drs, unsigned char day);

extern void
drs_set_hour(drs_t drs, unsigned char hour);

extern void
drs_set_minute(drs_t drs, unsigned char minute);

extern void
drs_set_second(drs_t drs, unsigned char second);

extern void
drs_set_offset_from_GMT(drs_t drs, unsigned char offset_from_GMT);

extern void
drs_set_file_flags(drs_t drs, unsigned char file_flags);

extern void
drs_set_file_unit_size(drs_t drs, unsigned char file_unit_size);

extern void
drs_set_interleave_gap_size(drs_t drs, unsigned char interleave_gap_size);

extern void
drs_set_volume_sequence_number(drs_t drs,
                               unsigned long int volume_sequence_number);

extern void
drs_set_file_identifier(drs_t drs, const char* file_identifier);


#endif  /*  DRS_H  */
