/* ppmtoxpm.c - read a portable pixmap and produce a (version 3) X11 pixmap
**
** Copyright (C) 1990 by Mark W. Snitily
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** This tool was developed for Schlumberger Technologies, ATE Division, and
** with their permission is being made available to the public with the above
** copyright notice and permission notice.
**
** Upgraded to XPM2 by
**   Paul Breslaw, Mecasoft SA, Zurich, Switzerland (paul@mecazh.uu.ch)
**   Thu Nov  8 16:01:17 1990
**
** Upgraded to XPM version 3 by
**   Arnaud Le Hors (lehors@mirsa.inria.fr)
**   Tue Apr 9 1991
**
** Rainer Sinkwitz sinkwitz@ifi.unizh.ch - 21 Nov 91:
**  - Bug fix, should should malloc space for rgbn[j].name+1 in line 441
**    caused segmentation faults
**    
**  - lowercase conversion of RGB names def'ed out,
**    considered harmful.
**
** Michael Pall (pall@rz.uni-karlsruhe.de) - 29 Nov 93:
**  - Use the algorithm from xpm-lib for pixel encoding
**    (base 93 not base 28 -> saves a lot of space for colorful xpms)
*/

#define _BSD_SOURCE   /* Make sure strdup() is in string.h */
#define _XOPEN_SOURCE 500  /* Make sure strdup() is in string.h */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "ppm.h"
#include "shhopt.h"
#include "nstring.h"
#include "mallocvar.h"

/* Max number of entries we will put in the XPM color map 
   Don't forget the one entry for transparency.

   We don't use this anymore.  Ppmtoxpm now has no arbitrary limit on
   the number of colors.

   We're assuming this isn't in fact an XPM format requirement, because
   we've seen it work with 257, and 257 seems to be common, because it's
   the classic 256 colors, plus transparency.  The value was 256 for
   ages before we added transparency capability to this program in May
   2001.  At that time, we started failing with 256 color images.
   Some limit was also necessary before then because ppm_computecolorhash()
   required us to specify a maximum number of colors.  It doesn't anymore.

   If we find out that some XPM processing programs choke on more than
   256 colors, we'll have to readdress this issue.  - Bryan.  2001.05.13.
*/
#define MAXCOLORS    256

#define MAXPRINTABLE 92         /* number of printable ascii chars
                                 * minus \ and " for string compat
                                 * and ? to avoid ANSI trigraphs. */

static const char * const printable =
" .XoO+@#$%&*=-;:>,<1234567890qwertyuipasdfghjklzxcvbnmMNBVCZ\
ASDFGHJKLPIUYTREWQ!~^/()_`'][{}|";


struct cmdline_info {
    /* All the information the user supplied in the command line,
       in a form easy for the program to use.
    */
    const char *input_filespec;  /* Filespecs of input files */
    const char *name;
    const char *rgb;
    const char *alpha_filename;
    unsigned int hexonly;
};



static void
parseCommandLine(int argc, char ** argv,
                 struct cmdline_info * const cmdlineP) {
/*----------------------------------------------------------------------------
   Note that the file spec array we return is stored in the storage that
   was passed to us as the argv array.
-----------------------------------------------------------------------------*/
    optEntry *option_def = malloc(100*sizeof(optStruct));
        /* Instructions to OptParseOptions3 on how to parse our options.
         */
    optStruct3 opt;

    unsigned int option_def_index;
    const char * nameOpt;
    unsigned int nameSpec;

    option_def_index = 0;   /* incremented by OPTENTRY */
    OPTENT3(0,   "alphamask",   OPT_STRING, &cmdlineP->alpha_filename, 
            NULL, 0);
    OPTENT3(0,   "name",        OPT_STRING, &nameOpt,                   
            &nameSpec, 0);
    OPTENT3(0,   "rgb",         OPT_STRING, &cmdlineP->rgb,            
            NULL, 0);
    OPTENT3(0,   "hexonly",     OPT_FLAG, NULL,
            &cmdlineP->hexonly, 0);

    /* Set the defaults */
    cmdlineP->alpha_filename = NULL;  /* no transparency */
    cmdlineP->rgb = NULL;      /* no rgb file specified */

    opt.opt_table = option_def;
    opt.short_allowed = FALSE;  /* We have no short (old-fashioned) options */
    opt.allowNegNum = FALSE;  /* We may have parms that are negative numbers */

    optParseOptions3(&argc, argv, opt, sizeof(opt), 0);
        /* Uses and sets argc, argv, and some of *cmdlineP and others. */

    if (argc-1 == 0) 
        cmdlineP->input_filespec = "-";
    else if (argc-1 != 1)
        pm_error("Program takes zero or one argument (filename).  You "
                 "specified %d", argc-1);
    else
        cmdlineP->input_filespec = argv[1];

    /* If output filename not specified, use input filename as default. */
    if (nameSpec)
        cmdlineP->name = nameOpt;
    else if (strcmp(cmdlineP->input_filespec, "-") == 0)
        cmdlineP->name = "noname";
    else {
        static char name[80+1];
        char *cp;

        STRSCPY(name, cmdlineP->input_filespec);
        cp = strchr(name, '.');
        if (cp)
            *cp = '\0';     /* remove extension */
        cmdlineP->name = name;
    }
}


typedef struct {            /* rgb values and ascii names (from
                             * rgb text file) */
    int r, g, b;            /* rgb values, range of 0 -> 65535 */
    char *name;             /* color mnemonic of rgb value */
} rgb_names;

typedef struct {            
    /* Information for an XPM color map entry */
    char *cixel;    
       /* XPM color number, as might appear in the XPM raster */
    const char *rgbname;  
       /* color name, e.g. "blue" or "#01FF23" */
} cixel_map;



static char *
genNumstr(int const input, int const digits) {
/*---------------------------------------------------------------------------
   Given a number and a base (MAXPRINTABLE), this routine prints the
   number into a malloc'ed string and returns it.  The length of the
   string is specified by "digits".  The ascii characters of the
   printed number range from printable[0] to printable[MAXPRINTABLE].
   The string is printable[0] filled, (e.g. if printable[0]==0,
   printable[1]==1, MAXPRINTABLE==2, digits==5, i=3, routine would
   return the malloc'ed string "00011").
---------------------------------------------------------------------------*/
    char *str, *p;
    int d;
    int i;

    /* Allocate memory for printed number.  Abort if error. */
    if (!(str = (char *) malloc(digits + 1)))
        pm_error("out of memory");

    i = input;
    /* Generate characters starting with least significant digit. */
    p = str + digits;
    *p-- = '\0';            /* nul terminate string */
    while (p >= str) {
        d = i % MAXPRINTABLE;
        i /= MAXPRINTABLE;
        *p-- = printable[d];
    }

    if (i != 0)
        pm_error("Overflow converting %d to %d digits in base %d",
                 input, digits, MAXPRINTABLE);

    return str;
}



static void
genCmap(colorhist_vector const chv, 
        int              const ncolors, 
        pixval           const maxval, 
        colorhash_table  const colornameHash,
        const char *           colornames[],
        cixel_map **     const cmapP, 
        int              const charsPerPixel) {
/*----------------------------------------------------------------------------
   Generate the XPM color map in cixel_map format (which is just a step
   away from the actual text that needs to be written the XPM file).  The
   color map is defined by 'chv', which contains 'ncolors' colors which
   have maxval 'maxval'.

   Output is in newly malloc'ed storage, with its address returned as
   *cmapP.

   This map includes an entry for transparency, whether the raster uses
   it or not.  Its index is ncolors (so cmap's dimension is ncolors +1).

   In the map, identify colors by the names give by 'colornameHash' and
   colornames[].  'colornameHash' maps a color in 'pixel' form to an
   index into colornames[]; colornames[] contains the text to identify the
   color in the XPM format.  The colors in 'colornameHash' have maxval 255.
   If a color is not in 'colornameHash', use hexadecimal notation in the
   output colormap.

   But if 'colornameHash' is null, don't use color names at all.  Just use
   hexadecimal notation.
-----------------------------------------------------------------------------*/
    int cmapIndex;
    int mval;
    cixel_map * cmap;  /* malloc'ed */
    
    MALLOCARRAY(cmap, ncolors + 1);
    if (cmapP == NULL)
        pm_error("Out of memory allocating %d bytes for a color map.",
                 sizeof(cixel_map) * (ncolors+1));
    /*
     * Determine how many hex digits we'll be normalizing to if the rgb
     * value doesn't match a color mnemonic. 
     */
    mval = (unsigned int) maxval;
    if (mval <= 0x000F)
        mval = 0x000F;
    else if (mval <= 0x00FF)
        mval = 0x00FF;
    else if (mval <= 0x0FFF)
        mval = 0x0FFF;
    else
        mval = 0xFFFF;
    
    /*
     * Generate the character-pixel string and the rgb name for each
     * colormap entry. 
     */
    for (cmapIndex = 0; cmapIndex < ncolors; ++cmapIndex) {
        pixel const color = chv[cmapIndex].color;

        pixel color255;
            /* The color, scaled to maxval 255 */
        const char * colorname;  /* malloc'ed */
        /*
         * The character-pixel string is simply a printed number in base
         * MAXPRINTABLE where the digits of the number range from
         * printable[0] .. printable[MAXPRINTABLE-1] and the printed length
         * of the number is 'charsPerPixel'. 
         */
        cmap[cmapIndex].cixel = genNumstr(cmapIndex, charsPerPixel);
        
        PPM_DEPTH(color255, color, maxval, 255);

        if (colornameHash == NULL)
            colorname = NULL;
        else {
            int colornameIndex;
            colornameIndex = ppm_lookupcolor(colornameHash, &color255);
            if (colornameIndex >= 0)
                colorname = strdup(colornames[colornameIndex]);
            else
                colorname = NULL;
        }
        if (colorname)
            cmap[cmapIndex].rgbname = colorname;
        else {
            /* Color has no name; represent it in hexadecimal */

            pixel scaledColor;
            const char * hexString;  /* malloc'ed */

            PPM_DEPTH(scaledColor, color, maxval, mval);

            asprintfN(&hexString, mval == 0x000F ? "#%X%X%X" :
                      mval == 0x00FF ? "#%02X%02X%02X" :
                      mval == 0x0FFF ? "#%03X%03X%03X" :
                      "#%04X%04X%04X", 
                      PPM_GETR(scaledColor),
                      PPM_GETG(scaledColor),
                      PPM_GETB(scaledColor)
                );

            if (hexString == NULL)
                pm_error("Unable to allocate storage for hex string");
            cmap[cmapIndex].rgbname = hexString;
        }
    }

    /* Add the special transparency entry to the colormap */
    cmap[ncolors].rgbname = strdup("None");
    cmap[ncolors].cixel = genNumstr(ncolors, charsPerPixel);

    *cmapP = cmap;
}



static void
destroyCmap(cixel_map * const cmap, int const ncolors) {

    int i;
    /* Free the real color entries */
    for (i = 0; i < ncolors; i++) {
        strfree(cmap[i].rgbname);
        free(cmap[i].cixel);
    }
    strfree(cmap[ncolors].rgbname);  /* The transparency entry */
    free(cmap[ncolors].cixel);  /* The transparency entry */

    free(cmap);
}



static void
writeXpmFile(FILE * const outfile, pixel ** const pixels, 
             gray ** const alpha, pixval const alphamaxval,
             const char name[], 
             int const cols, int const rows, int const colormapsize,
             int charsPerPixel, 
             cixel_map cmap[],
             colorhash_table const cht) {
/*----------------------------------------------------------------------------
   Write the whole XPM file to the open stream 'outfile'.

   colormapsize is the total number of entries in the colormap -- one for
   each color, plus one for transparency.
-----------------------------------------------------------------------------*/
    /* First the header */
    printf("/* XPM */\n");
    fprintf(outfile, "static char *%s[] = {\n", name);
    fprintf(outfile, "/* width height ncolors chars_per_pixel */\n");
    fprintf(outfile, "\"%d %d %d %d\",\n", cols, rows, 
            colormapsize, charsPerPixel);

    {
        int i;
        /* Write out the colormap (part of header) */
        fprintf(outfile, "/* colors */\n");
        for (i = 0; i < colormapsize; i++) { 
            fprintf(outfile, "\"%s c %s\",\n", cmap[i].cixel, cmap[i].rgbname);
        }
    }
    {
        int row;

        /* And now the raster */
        fprintf(outfile, "/* pixels */\n");
        for (row = 0; row < rows; row++) {
            int col;
            fprintf(outfile, "\"");
            for (col = 0; col < cols; col++) {
                if (alpha && alpha[row][col] <= alphamaxval/2)
                    /* It's a transparent pixel */
                    fprintf(outfile, "%s", cmap[colormapsize-1].cixel);
                else 
                    fprintf(outfile, "%s", 
                            cmap[ppm_lookupcolor(cht, 
                                                 &pixels[row][col])].cixel);
            }
            fprintf(outfile, "\"%s\n", (row == (rows - 1) ? "" : ","));
        }
    }
    /* And close up */
    fprintf(outfile, "};\n");
}



static void
readAlpha(const char filespec[], gray *** const alphaP,
          int const cols, int const rows, pixval * const alphamaxvalP) {

    FILE * alpha_file;
    int alphacols, alpharows;
        
    alpha_file = pm_openr(filespec);
    *alphaP = pgm_readpgm(alpha_file, &alphacols, &alpharows, alphamaxvalP);
    pm_close(alpha_file);
    
    if (cols != alphacols || rows != alpharows)
        pm_error("Alpha mask is not the same dimensions as the "
                 "image.  Image is %d by %d, while mask is %d x %d.",
                 cols, rows, alphacols, alpharows);
}
    

static void
computeColormap(pixel ** const pixels, int const cols, int const rows,
                colorhist_vector * const chvP, 
                colorhash_table * const chtP,
                int * const ncolorsP) {
/*----------------------------------------------------------------------------
   Compute the color map for the image 'pixels', which is 'cols' by 'rows',
   in Netpbm data structures (a colorhist_vector for index-to-color lookups
   and a colorhash_table for color-to-index lookups).
-----------------------------------------------------------------------------*/
    colorhash_table hist_cht;

    pm_message("(Computing colormap...");
    hist_cht = ppm_computecolorhash(pixels, cols, rows, 0, ncolorsP);
    /* Old versions of the library don't have the special 0 value for 
       maxcolors -- they take it literally.
    */
    if (hist_cht == NULL)
        pm_error("You need a newer version of the Netpbm library 'libppm' "
                 "to run this program.");
    pm_message("...Done.  %d colors found.)\n", *ncolorsP);

    *chvP = ppm_colorhashtocolorhist(hist_cht, *ncolorsP);
    ppm_freecolorhash(hist_cht);
    /* Despite the name, the following generates an index on cht */
    *chtP = ppm_colorhisttocolorhash(*chvP, *ncolorsP);
}



int
main(int argc, char *argv[]) {

    FILE *ifp;
    int rows, cols, ncolors;
    pixval maxval, alphamaxval;
    colorhash_table cht;
    colorhist_vector chv;

    colorhash_table colornameHash;
        /* Hash table to map colors to their names */
    const char ** colornames;
        /* Table of color names; 'colornameHash' yields an index into this
           array.
        */

    pixel **pixels;
    gray **alpha;

    /* Used for rgb value -> character-pixel string mapping */
    cixel_map *cmap;  /* malloc'ed */
    int charsPerPixel;  

    struct cmdline_info cmdline;

    ppm_init(&argc, argv);

    parseCommandLine(argc, argv, &cmdline);

    ifp = pm_openr(cmdline.input_filespec);
    pixels = ppm_readppm(ifp, &cols, &rows, &maxval);
    pm_close(ifp);

    if (cmdline.alpha_filename) 
        readAlpha(cmdline.alpha_filename, &alpha, cols, rows, &alphamaxval);
    else
        alpha = NULL;

    computeColormap(pixels, cols, rows, &chv, &cht, &ncolors);

    if (cmdline.hexonly)
        colornameHash = NULL;
    else
        ppm_readcolornamefile(cmdline.rgb, FALSE, &colornameHash, &colornames);

    { 
        /* Compute characters per pixel in the XPM file */
        int j;
        for (charsPerPixel = 0, j = ncolors-1; j > 0; ++charsPerPixel)
            j /= MAXPRINTABLE;
    }
    /* Now generate the character-pixel colormap table. */
    genCmap(chv, ncolors, maxval, colornameHash, colornames,
            &cmap, charsPerPixel);

    writeXpmFile(stdout, pixels, alpha, alphamaxval,
                 cmdline.name, cols, rows, ncolors + 1, 
                 charsPerPixel, cmap, cht);
    
    if (colornameHash) {
        ppm_freecolorhash(colornameHash);
        ppm_freecolornames(colornames);
    }
    destroyCmap(cmap, ncolors);
    ppm_freearray(pixels, rows);
    if (alpha) pgm_freearray(alpha, rows);

    return 0;
}
