/*
 * poor man's snprintf() - for systems which don't have their own
 *
 * This version of snprintf() currently supports only %s, %c, %d, %u, %ld, %lu.
 * It also behaves differently from standard snprintf() in that if the
 * output is truncated, it will return -1. Otherwise, it will return the
 * number of printed chars.
 *
 * Copyright (c) Cornelius Krasel 2000.
 * Modified by Matthias Andree <matthias.andree@gmx.de>.
 * Modifications (C)opyright 2002.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "leafnode.h"
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

int
vsnprintf(char *str, size_t n, const char *format, va_list ap)
{
    const char *p;
    char *q;
    int flag = 0;
    int lflag = 0;		/* checking for longs */
    size_t i = 1;		/* because the terminating \0 also counts */
    size_t len;
    char buf[30];		/* buffer for converting longs and ints */

    int d;
    long l;
    char *s;
    char c;

    p = format;
    q = str;
    while (p && *p && (i < n)) {
	if ((*p == '%') && !flag) {
	    lflag = 0;
	    flag = 1;
	    p++;
	} else if (flag) {
	    switch (*p) {
	    case 's':{
		    s = va_arg(ap, char *);
		    len = strlen(s);
		    if (len > (n - i))
			len = n - i;
		    *q = '\0';
		    strncat(q, s, len);
		    p++;
		    q += len;
		    i += len;
		    flag = 0;
		    break;
		}
	    case 'u':{
		    if (lflag) {
			l = va_arg(ap, unsigned long);
			sprintf(buf, "%lu", l);
		    } else {
			d = va_arg(ap, unsigned int);
			sprintf(buf, "%u", d);
		    }
		    len = strlen(buf);
		    if (len > (n - i))
			len = n - i;
		    *q = '\0';
		    strncat(q, buf, len);
		    q += len;
		    i += len;
		    lflag = 0;
		    flag = 0;
		    p++;
		    break;
		}
	    case 'd':{
		    if (lflag) {
			l = va_arg(ap, long);
			sprintf(buf, "%ld", l);
		    } else {
			d = va_arg(ap, int);
			sprintf(buf, "%d", d);
		    }
		    len = strlen(buf);
		    if (len > (n - i))
			len = n - i;
		    *q = '\0';
		    strncat(q, buf, len);
		    q += len;
		    i += len;
		    lflag = 0;
		    flag = 0;
		    p++;
		    break;
		}
	    case 'l':{
		    /* next argument will be long */
		    lflag = 1;
		    p++;
		    break;
		}
	    case 'c':{
		    c = va_arg(ap, char);
		    lflag = 0;
		    flag = 0;
		    i++;
		    *q++ = c;
		    p++;
		    break;
		}
	    case '%':{
		    lflag = 0;
		    flag = 0;
		    i++;
		    *q++ = *p++;
		    break;
		}
	    }
	} else {
	    lflag = 0;
	    i++;
	    *q++ = *p++;
	}
    }
    va_end(ap);
    *q = '\0';
    if (i > n)
	return -1;
    else
	return (i - 1);
}

int
snprintf(char *str, size_t n, const char *format, ...)
{
    int r;
    va_list ap;

    va_start(ap, format);
    r = vsnprintf(str, n, format, ap);
    va_end(ap);
    return r;
}
