/* Copyright (C) 2006 P.L. Lucas
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA. 
 */

#include "table.h"
#include <QTextStream>
#include <QRegExp>
#include <QMenu>
#include <QClipboard>
#include <QMessageBox>

Table::Table(QWidget * parent):BaseWidget(parent)
{
	widget_type=TABLE;

	table_form=new TableForm();
	table_form->setupUi(this);
	setWindowIcon( QIcon( QString( ICON_PATH )+"/table.png" ) );
	table_form->reloadButton->setIcon( QIcon( QString( ICON_PATH )+"/reload.png" ) );
	table_form->reloadButton->setToolTip(tr("<b>Reload matrix.</b><br> Some operations change matrix, use reload to view changes."));
	connect(table_form->table_widget,SIGNAL(cellChanged( int, int)),this,SLOT(cellChanged( int, int)));
	connect(table_form->rows_lineedit,SIGNAL( returnPressed ()),this,SLOT(rows_changed()));
	connect(table_form->rows_lineedit,SIGNAL( clicked() ),this,SLOT( windowActivated() ));
	
	connect(table_form->cols_lineedit,SIGNAL( returnPressed ()),this,SLOT(cols_changed()));
	connect(table_form->cols_lineedit,SIGNAL( clicked() ),this,SLOT( windowActivated() ));
	
	connect(table_form->changeOrderButton,SIGNAL( clicked() ),this,SLOT( rows_changed() ));
	connect(table_form->changeOrderButton,SIGNAL( clicked() ),this,SLOT( cols_changed() ));
	connect(table_form->changeOrderButton,SIGNAL( clicked() ),this,SLOT( windowActivated() ));
	
	connect(table_form->reloadButton,SIGNAL( clicked() ),this,SLOT( windowActivated() ));
	
	build_menu();
	setContextMenuPolicy ( Qt::DefaultContextMenu );
}

void Table::setOctaveConnection(OctaveConnection *octave_connection)
{
	this->octave_connection=octave_connection;
	connect(octave_connection,SIGNAL(line_ready(QString)),this,SLOT(line_ready(QString)));
}

void Table::setMatrix(QString matrix)
{
	this->matrix=matrix;
}

void Table::windowActivated()
{
	windowActivated(this);
}

void Table::windowActivated(QWidget *w)
{
	if(w!=this) return;
	QString command;
	/*
	command+="function qtoctave_show_matrix(m);";
	command+="fprintf(stderr,\"~~matrix:"+matrix+" #%d %d\\n\",rows(m),columns(m));";
	command+="for i=1:rows(m);for j=1:columns(m);";
	command+="if(imag(m(i,j))==0);";
	command+="fprintf(stderr,\"~~matrix:"+matrix+" %d %d %g\\n\",i,j,m(i,j));";
	command+="elseif(imag(m(i,j))>0);";
	command+="fprintf(stderr,\"~~matrix:"+matrix+" %d %d %g+%gi\\n\",i,j,m(i,j),imag(m(i,j)));";
	command+="else;";
	command+="fprintf(stderr,\"~~matrix:"+matrix+" %d %d %g%gi\\n\",i,j,m(i,j),imag(m(i,j)));";
	command+="endif;";
	command+="endfor;endfor;endfunction;qtoctave_show_matrix("+matrix+")";
	*/
	command+="function qtoctave_show_matrix(m);";
	command+="fprintf(stderr,\"~~matrix:"+matrix+" #%d %d\\n\",rows(m),columns(m));";
	command+="for i=1:rows(m);";
		command+="fprintf(stderr,\"~~matrix:"+matrix+" %d\",i);";
		command+="for j=1:columns(m);";
			command+="if(imag(m(i,j))==0);";
				command+="fprintf(stderr,\" %g\",m(i,j));";
			command+="elseif(imag(m(i,j))>0);";
				command+="fprintf(stderr,\" %g+%gi\",m(i,j),imag(m(i,j)));";
			command+="else;";
				command+="fprintf(stderr,\" %g%gi\",m(i,j),imag(m(i,j)));";
			command+="endif;";
		command+="endfor;";
		command+="fprintf(stderr,\"\\n\");";
	command+="endfor;endfunction;";
	command+="eval(\"qtoctave_show_matrix("+matrix+")\", \"qtoctave_show_matrix("+matrix+"=[0])\")";
	octave_connection->command_enter(command,false);
}

void Table::line_ready(QString line)
{
	if( ! line.startsWith("~~matrix:"+matrix) ) return;
	line=line.right(line.length() - (10+matrix.length()) );
	//printf("line:>%s<\n",line.ascii());
	if(line.indexOf('#')>-1)
	{	//La l�ea contiene el nmero de filas y columnas
		int _rows, _columns;
		sscanf(line.toAscii().data(),"#%d %d",&_rows,&_columns);
		table_form->table_widget->setColumnCount(_columns);
		table_form->table_widget->setRowCount(_rows);
		table_form->rows_lineedit->setText(line.setNum(_rows));
		table_form->cols_lineedit->setText(line.setNum(_columns));
		return;
	}
	else
	{
		//La l�ea contiene los datos de la celda
		int row=0, col;
		/*
		char data[256];
		sscanf(line.toAscii().data(),"%d %d %[0-9+iej .-]s",&row,&col,data);
		//printf("(%d,%d) %f",row,col,data);
		
		disconnect(table_form->table_widget,SIGNAL(cellChanged( int, int)),this,SLOT(cellChanged( int, int)));
		QTableWidgetItem *cell= new QTableWidgetItem(data);
		table_form->table_widget->setItem(row-1,col-1,cell);
		connect(table_form->table_widget,SIGNAL(cellChanged( int, int)),this,SLOT(cellChanged( int, int)));
		*/
		QRegExp rx("([0-9ieEj\\.\\-\\+]+)");
		QString value;
		int pos = 0;
		if ((pos = rx.indexIn(line, pos)) != -1) 
		{
			value=rx.cap(0);
			pos += rx.matchedLength();
			row=value.toInt()-1;
		}
		col=0;
		disconnect(table_form->table_widget,SIGNAL(cellChanged( int, int)),this,SLOT(cellChanged( int, int)));
		while ((pos = rx.indexIn(line, pos)) != -1) {
			value=rx.cap(0);
			pos += rx.matchedLength();
			QTableWidgetItem *cell= new QTableWidgetItem(value);
			table_form->table_widget->setItem(row,col,cell);
			col++;
		}
		connect(table_form->table_widget,SIGNAL(cellChanged( int, int)),this,SLOT(cellChanged( int, int)));
	}
}

void Table::cellChanged ( int row, int col )
{
	QString command(matrix+"("), aux;
	command+=aux.setNum(row+1)+",";
	QTableWidgetItem *item=table_form->table_widget->item(row,col);
	command+=aux.setNum(col+1)+")="+item->text()+";";
	octave_connection->command_enter(command);
}

void Table::change_rows()
{
	int rows=table_form->rows_lineedit->text().toInt();
	int old_rows=table_form->table_widget->rowCount();
	int old_cols=table_form->table_widget->columnCount();
	
	if(rows<old_rows)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "=" << matrix << "(1:" << rows 
					<< ",1:" << old_cols << ");";
		octave_connection->command_enter(command);
	}
	else if(rows>old_rows)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "(" << rows 
					<< "," << old_cols << ")=0;";
		octave_connection->command_enter(command);
	}
}

void Table::rows_changed()
{
	change_rows();
	windowActivated(this);
}

void Table::cols_changed()
{
	change_cols();
	windowActivated(this);
}

void Table::change_cols()
{
	int cols=table_form->cols_lineedit->text().toInt();
	int old_rows=table_form->table_widget->rowCount();
	int old_cols=table_form->table_widget->columnCount();
	
	if(cols<old_cols)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "=" << matrix << "(1:" << old_rows 
					<< ",1:" << cols << ");";
		octave_connection->command_enter(command);
	}
	else if(cols>old_cols)
	{
		QString command;
		QTextStream(&command) 	<< matrix << "(" << old_rows 
					<< "," << cols << ")=0;";
		octave_connection->command_enter(command);
	}
}

void Table::order_changed()
{
	change_rows();
	change_cols();
	windowActivated(this);
}

QString Table::getMatrix()
{
	return matrix;
}

void Table::build_menu()
{
	menu=new QMenu("Table menu:", this);
	QAction *copyAction=new QAction("Copy", menu);
	menu->addAction(copyAction);
	connect(copyAction, SIGNAL(triggered()), this, SLOT(copy_cb()) );
	QAction *copyMatrixAction=new QAction("Copy as Octave matrix", menu);
	menu->addAction(copyMatrixAction);
	connect(copyMatrixAction, SIGNAL(triggered()), this, SLOT(copy_matrix_cb()) );
	QAction *pasteAction=new QAction("Paste", menu);
	menu->addAction(pasteAction);
	connect(pasteAction, SIGNAL(triggered()), this, SLOT(paste_cb()) );
	QAction *deleteRowsAction=new QAction("Delete rows", menu);
	menu->addAction(deleteRowsAction);
	connect(deleteRowsAction, SIGNAL(triggered()), this, SLOT(delete_rows_cb()) );
	QAction *deleteColumnsAction=new QAction("Delete columns", menu);
	menu->addAction(deleteColumnsAction);
	connect(deleteColumnsAction, SIGNAL(triggered()), this, SLOT(delete_columns_cb()) );
	QAction *insertColumnRightAction=new QAction("Insert column (right)", menu);
	menu->addAction(insertColumnRightAction);
	connect(insertColumnRightAction, SIGNAL(triggered()), this, SLOT(insert_column_right_cb()) );
	QAction *insertColumnLeftAction=new QAction("Insert column (left)", menu);
	menu->addAction(insertColumnLeftAction);
	connect(insertColumnLeftAction, SIGNAL(triggered()), this, SLOT(insert_column_left_cb()) );
	QAction *insertRowUpAction=new QAction("Insert row (up)", menu);
	menu->addAction(insertRowUpAction);
	connect(insertRowUpAction, SIGNAL(triggered()), this, SLOT(insert_row_up_cb()) );
	QAction *insertRowDownAction=new QAction("Insert row (down)", menu);
	menu->addAction(insertRowDownAction);
	connect(insertRowDownAction, SIGNAL(triggered()), this, SLOT(insert_row_down_cb()) );

	/* Gráficas */
	menu->addSeparator();

	QMenu *plotMenu = new QMenu("Plot", menu);
	menu->addMenu(plotMenu);
	QMenu *plot2dMenu = new QMenu("2D", plotMenu);
	plotMenu->addMenu(plot2dMenu);
	QMenu *plot3dMenu = new QMenu("3D", plotMenu);
	plotMenu->addMenu(plot3dMenu);

	QAction *plotAction = new QAction("Plot", plot2dMenu);
	plot2dMenu->addAction(plotAction);
	connect(plotAction, SIGNAL(triggered()), this, SLOT(plotPlot()));

	QAction *polarAction = new QAction("Polar", plot2dMenu);
	plot2dMenu->addAction(polarAction);
	connect(polarAction, SIGNAL(triggered()), this, SLOT(plotPolar()));

	QAction *logxyAction = new QAction("Log scale for the x and y axis", plot2dMenu);
	plot2dMenu->addAction(logxyAction);
	connect(logxyAction, SIGNAL(triggered()), this, SLOT(plotLogXandY()));

	QAction *logyAction = new QAction("Log scale for the y axis", plot2dMenu);
	plot2dMenu->addAction(logyAction);
	connect(logyAction, SIGNAL(triggered()), this, SLOT(plotLogY()));

	QAction *logxAction = new QAction("Log scale for the x axis", plot2dMenu);
	plot2dMenu->addAction(logxAction);
	connect(logxAction, SIGNAL(triggered()), this, SLOT(plotLogX()));

	QAction *barAction = new QAction("Bar graph", plot2dMenu);
	plot2dMenu->addAction(barAction);
	connect(barAction, SIGNAL(triggered()), this, SLOT(plotBar()));

	QAction *plot3dAction = new QAction("Plot of three-dimensional surface", plot3dMenu);
	plot3dMenu->addAction(plot3dAction);
	QAction *contourAction = new QAction("Contour plot of three-dimensional surface",
					     plot3dMenu);
	plot3dMenu->addAction(contourAction);
}

void Table::insert_row_up_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int row=table_form->table_widget->currentRow()+1;
	
	command << matrix << "=[" << matrix << "(1:" << row-1 << ",:); zeros(1,columns(" << matrix << ")); "
			<< matrix << "(" << row << ":rows(" << matrix << "),:)]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::insert_row_down_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int row=table_form->table_widget->currentRow()+1;
	
	command << matrix << "=[" << matrix << "(1:" << row << ",:); zeros(1,columns(" << matrix << ")); "
			<< matrix << "(" << row+1 << ":rows(" << matrix << "),:)]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::insert_column_left_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int col=table_form->table_widget->currentColumn()+1;
	
	command << matrix << "=[" << matrix << "(:,1:" << col-1 << "), zeros(rows(" << matrix << "),1), "
			<< matrix << "(:," << col << ":columns(" << matrix << "))]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::insert_column_right_cb()
{
	QString _command;
	QTextStream command(&_command);
	
	int col=table_form->table_widget->currentColumn()+1;
	
	command << matrix << "=[" << matrix << "(:,1:" << col << "), zeros(rows(" << matrix << "),1), "
			<< matrix << "(:," << col+1 << ":columns(" << matrix << "))]";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::delete_columns_cb()
{
	int result=QMessageBox::question (this, "Warning", "Comlumns will be deleted. Continue?", QMessageBox::Ok, QMessageBox::Cancel);
	
	if(result!=QMessageBox::Ok) return;
	
	
	QString _command;
	QTextStream command(&_command);
	
	command << matrix << "=[";
	QList<QTableWidgetSelectionRange> ranges=table_form->table_widget->selectedRanges ();
	int column=0;
	int start, end;
	bool ok=false;
	start=-1;
	end=-1;
	while(column < table_form->table_widget->columnCount ())
	{
		ok=true;
		
		for(int i=0;i<ranges.size();i++)
		{
			if(ranges.at(i).leftColumn  ()<= column && column<=ranges.at(i).rightColumn () )
			{
				ok=false;
				break;
			}
		}
		if(ok && start<0) start=column;
		if(!ok)
		{
			if(start>=0)
			{
				end=column;
				command << matrix << "(1:rows(" << matrix << "),"<< start+1 << ":" << end <<"), ";
				start=-1;
			}
		}
		column++;
	}
	if(ok && start>=0)
	{
		end=column;
		command << matrix << "(1:rows(" << matrix << "),"<< start+1 << ":" << end <<") ";
	}
	command << "];";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}


void Table::delete_rows_cb()
{
	int result=QMessageBox::question (this, "Warning", "Rows will be deleted. Continue?", QMessageBox::Ok, QMessageBox::Cancel);
	
	if(result!=QMessageBox::Ok) return;
	
	
	QString _command;
	QTextStream command(&_command);
	
	command << matrix << "=[";
	QList<QTableWidgetSelectionRange> ranges=table_form->table_widget->selectedRanges ();
	int row=0;
	int start, end;
	bool ok=false;
	start=-1;
	end=-1;
	while(row < table_form->table_widget->rowCount ())
	{
		ok=true;
		
		for(int i=0;i<ranges.size();i++)
		{
			if(ranges.at(i).topRow ()<= row && row<=ranges.at(i).bottomRow () )
			{
				ok=false;
				break;
			}
		}
		if(ok && start<0) start=row;
		if(!ok)
		{
			if(start>=0)
			{
				end=row;
				command << matrix << "(" << start+1 << ":" << end << ",1:columns(" << matrix << "));\n";
				start=-1;
			}
		}
		row++;
	}
	if(ok && start>=0)
	{
		end=row;
		command << matrix << "(" << start+1 << ":" << end << ",1:columns(" << matrix << "))\n";
	}
	command << "];";
	
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::paste_cb()
{
	QClipboard *clipboard = QApplication::clipboard();
	QString text=clipboard->text();
	
	QString _command;
	QTextStream command(&_command);
	
	int row=table_form->table_widget->currentRow()+1, col=table_form->table_widget->currentColumn()+1;
	
	QRegExp rx("([0-9ieEj\\.\\-\\+]+|\\n)");
	QString value;
	int pos = 0;
	while ((pos = rx.indexIn(text, pos)) != -1) {
		value=rx.cap(0);
		pos += rx.matchedLength();
		if(value=="\n") {row++;col=table_form->table_widget->currentColumn()+1;}
		else
		{
			command << matrix << "(" << row << "," << col << ")=" << value << ";\n";
			col++;
		}
	}
	octave_connection->command_enter(_command);
	
	windowActivated();
}

void Table::copy_matrix_cb()
{
	QString str("[");
	QList<QTableWidgetSelectionRange> ranges=table_form->table_widget->selectedRanges ();
	for(int i=0;i<ranges.size();i++)
	{
		for(int row=ranges.at(i).topRow (); row<=ranges.at(i).bottomRow (); row++)
		{
			int col=ranges.at(i).leftColumn ();
			str+=table_form->table_widget->item(row,col)->text();
			col++;
			for(; col<=ranges.at(i).rightColumn (); col++)
			{
				str+=","+table_form->table_widget->item(row,col)->text();
			}
			str+=";\n";
		}
	}
	str+="]";
	QClipboard *clipboard = QApplication::clipboard();
	clipboard->setText(str);
}

void Table::copy_cb()
{
	QString str;
	QList<QTableWidgetSelectionRange> ranges=table_form->table_widget->selectedRanges ();
	for(int i=0;i<ranges.size();i++)
	{
		for(int row=ranges.at(i).topRow (); row<=ranges.at(i).bottomRow (); row++)
		{
			for(int col=ranges.at(i).leftColumn (); col<=ranges.at(i).rightColumn (); col++)
			{
				str+=" "+table_form->table_widget->item(row,col)->text();
			}
			str+="\n";
		}
	}
	QClipboard *clipboard = QApplication::clipboard();
	clipboard->setText(str);
}

void Table::contextMenuEvent ( QContextMenuEvent * event )
{
	QPoint p(event->globalX(),event->globalY());
	menu->popup(p);
}

void Table::plot(TablePlot::Type type)
{
  TablePlot *dialog = new TablePlot(this, table_form->table_widget,
				    type);
  
  if(dialog->exec() == QDialog::Accepted)
  {
    try
    {
      this->octave_connection->command_enter(dialog->command());
    }catch(const char *str)
    {
      //QMessageBox errorMsg(QMessageBox::Warning, "Error", str, QMessageBox::Ok, this);
      //errorMsg.exec();
      
      QMessageBox::warning (NULL, "Error", str);
    }
  }

  delete dialog;
}

void Table::plotPlot()
{
  plot(TablePlot::PLOT);
}

void Table::plotPolar()
{
  plot(TablePlot::POLAR);
}

void Table::plotLogXandY()
{
  plot(TablePlot::LOGLOG);
}

void Table::plotLogY()
{
  plot(TablePlot::SEMILOGY);
}

void Table::plotLogX()
{
  plot(TablePlot::SEMILOGX);
}

void Table::plotBar()
{
  plot(TablePlot::BAR);
}
