/*

Copyright (C) 2000 - 2003 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <nd.h>
#include <nd_gui.h>
#include <nd_dialog.h>

#include <nd_ip.h>
#include <nd_ipfrag.h>
#include "support.h"

void   
nd_ip_v_cb(LND_Packet  *packet,
	   guchar      *header,
	   guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  cb_data = nd_proto_cb_data_new(nd_ip_get(), data, 0, FALSE);
  
  nd_dialog_number(_("Enter 4-bit IP version number:"),
		   ND_BASE_DEC,
		   iphdr->ip_v, 15,
		   nd_proto_iterate_4bit_hi_cb,
		   NULL,
		   packet,
		   cb_data);
}


void
nd_ip_hl_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  cb_data = nd_proto_cb_data_new(nd_ip_get(), data, 0, FALSE);
  
  nd_dialog_number(_("Enter header size in 32-bit words:"),
		   ND_BASE_DEC,
		   iphdr->ip_hl, 15,
		   nd_proto_iterate_4bit_lo_cb,
		   NULL,
		   packet,
		   cb_data);
}


void   
nd_ip_ecn_value_cb(LND_Packet  *packet,
		   guchar      *header,
		   int          value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      iphdr->ip_tos &= 0x3F;
      iphdr->ip_tos |= (value << 6);
      
      libnd_packet_modified(libnd_pit_get(&pit));
    }
}


void   
nd_ip_ecn_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  static GtkWidget *menu = NULL;

  if (!menu)
    menu = nd_gui_create_menu(ip_menu_ecn_data);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 0);
      
  return;
  TOUCH(header);
  TOUCH(packet);
  TOUCH(data);
}


void   
nd_ip_tos_value_cb(LND_Packet  *packet,
		   guchar      *header,
		   int          value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      iphdr->ip_tos = value;
      libnd_packet_modified(libnd_pit_get(&pit));
    }
}


void   
nd_ip_tos_custom_cb(LND_Packet  *packet,
		    guchar      *header,
		    int          value)
{
  struct ip             *iphdr;
  
  iphdr = (struct ip*) header;
  nd_dialog_number(_("Enter TOS value:"),
		   ND_BASE_HEX,
		   iphdr->ip_tos,
		   255,
		   (ND_NumberCallback) nd_ip_tos_value_cb,
		   NULL,
		   packet, header);
  return;
  TOUCH(value);
}


void   
nd_ip_tos_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  static GtkWidget *menu = NULL;

  if (!menu)
    menu = nd_gui_create_menu(ip_menu_tos_data);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 0);
      
  return;
  TOUCH(header);
  TOUCH(packet);  
  TOUCH(data);
}


void   
nd_ip_len_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  cb_data = nd_proto_cb_data_new(nd_ip_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter IP packet size in bytes:"),
		   ND_BASE_DEC,
		   ntohs(iphdr->ip_len), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL,
		   packet, cb_data);
}


void   
nd_ip_id_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  cb_data = nd_proto_cb_data_new(nd_ip_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter IP packet ID:"),
		   ND_BASE_DEC,
		   ntohs(iphdr->ip_id), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL,
		   packet, cb_data);
}


void   
nd_ip_rf_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      if (! (ntohs(iphdr->ip_off) & IP_RF))
	iphdr->ip_off = htons(ntohs(iphdr->ip_off) | IP_RF);
      else
	iphdr->ip_off = htons(ntohs(iphdr->ip_off) & ~IP_RF);

      libnd_packet_modified(libnd_pit_get(&pit));
    }

  return;
  TOUCH(data);
}


void   
nd_ip_df_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      if (!(ntohs(iphdr->ip_off) & IP_DF))
	iphdr->ip_off = htons(ntohs(iphdr->ip_off) ^ IP_DF);
      else
	iphdr->ip_off = htons(ntohs(iphdr->ip_off) & ~IP_DF);

      libnd_packet_modified(libnd_pit_get(&pit));
    }

  return;
  TOUCH(data);
}


void   
nd_ip_mf_cb(LND_Packet  *packet,
	    guchar      *header,
	    guchar      *data)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      if (!(ntohs(iphdr->ip_off) & IP_MF))
	iphdr->ip_off = htons(ntohs(iphdr->ip_off) ^ IP_MF);
      else
	iphdr->ip_off = htons(ntohs(iphdr->ip_off) & ~IP_MF);

      libnd_packet_modified(libnd_pit_get(&pit));
    }

  return;
  TOUCH(data);
}


static void
ip_off_ok_cb(LND_Packet *packet, void *user_data, guint value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), (guchar*) user_data);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      iphdr->ip_off = htons(ntohs(iphdr->ip_off) & ~IP_OFFMASK);
      iphdr->ip_off = htons(ntohs(iphdr->ip_off) | (value & IP_OFFMASK));
      libnd_packet_modified(libnd_pit_get(&pit));
    }
}


void   
nd_ip_off_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  nd_dialog_number(_("Enter fragment offset in 8-byte units:"),
		   ND_BASE_DEC,
		   ntohs(iphdr->ip_off) & IP_OFFMASK, (1<<13) - 1,
		   ip_off_ok_cb,
		   NULL,
		   packet, data);
}


void   
nd_ip_ttl_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  cb_data = nd_proto_cb_data_new(nd_ip_get(), data, data-header, FALSE);
  
  nd_dialog_number(_("Enter time-to-live value:"),
		   ND_BASE_DEC,
		   iphdr->ip_ttl, 255,
		   nd_proto_iterate_8bit_cb,
		   NULL,
		   packet,
		   cb_data);
}


void   
nd_ip_p_value_cb(LND_Packet  *packet,
		 guchar      *header,
		 int          value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      iphdr->ip_p = (guchar) value;
      libnd_packet_update(libnd_pit_get(&pit), nd_ip_get(), nesting);
      libnd_packet_modified(libnd_pit_get(&pit));
    }
}


void   
nd_ip_p_custom_cb(LND_Packet  *packet,
		  guchar      *header,
		  int          value)
{
  struct ip             *iphdr;
  
  iphdr = (struct ip*) header;
  nd_dialog_number(_("Enter Protocol ID:"),
		   ND_BASE_DEC,
		   iphdr->ip_p,
		   255,
		   (ND_NumberCallback) nd_ip_p_value_cb,
		   NULL,
		   packet,
		   header);
  return;
  TOUCH(value);
}


void   
nd_ip_p_cb(LND_Packet  *packet,
	   guchar      *header,
	   guchar      *data)
{
  static GtkWidget *menu = NULL;

  if (!menu)
    menu = nd_gui_create_menu(ip_menu_p_data);
  
  gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 0);
  
  return;
  TOUCH(header);
  TOUCH(packet);  
  TOUCH(data);
}


void   
nd_ip_sum_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  ND_ProtoCallbackData *cb_data;
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  cb_data = nd_proto_cb_data_new(nd_ip_get(), data, data-header, FALSE);

  nd_dialog_number(_("Enter IP checksum:"),
		   ND_BASE_HEX,
		   ntohs(iphdr->ip_sum), 65535,
		   nd_proto_iterate_16bit_cb,
		   NULL,
		   packet,
		   cb_data);
}


void 
ip_src_ok_cb(guchar *address,
	     int address_len,
	     LND_Packet *packet,
	     void *user_data)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), (guchar *) user_data);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      memcpy(&iphdr->ip_src, address, sizeof(guint32));
      libnd_packet_modified(libnd_pit_get(&pit));
    }

  return;
  TOUCH(address_len);
}


void   
nd_ip_src_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  nd_dialog_ip(_("Enter IP source address:"),
	       (guchar*) &(iphdr->ip_src),
	       ip_src_ok_cb,
	       NULL,
	       packet,
	       data);
}


void 
ip_dst_ok_cb(guchar     *address,
	     int         address_len,
	     LND_Packet *packet,
	     void       *user_data)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), (guchar*) user_data);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      memcpy(&iphdr->ip_dst, address, sizeof(guint32));
      libnd_packet_modified(libnd_pit_get(&pit));
    }

  return;
  TOUCH(address_len);
}


void   
nd_ip_dst_cb(LND_Packet  *packet,
	     guchar      *header,
	     guchar      *data)
{
  struct ip            *iphdr;

  iphdr = (struct ip *) header;
  nd_dialog_ip(_("Enter IP destination address:"),
	       (guchar*) &(iphdr->ip_dst),
	       ip_dst_ok_cb,
	       NULL,
	       packet, data);
}


void   
nd_ip_cksum_fix_cb(LND_Packet  *packet,
		   guchar      *header,
		   int          value)
{
  LND_Trace            *trace;
  LND_PacketIterator    pit;
  int                   nesting;
  struct ip            *iphdr;
  guint16               correct_sum;

  if (! (trace = libnd_packet_get_trace(packet)))
    return;

  nesting = libnd_packet_get_proto_nesting(packet, nd_ip_get(), header);
  if (nesting < 0)
    return;

  for (libnd_pit_init(&pit, trace); libnd_pit_get(&pit); libnd_pit_next(&pit))
    {
      iphdr = (struct ip *) libnd_packet_get_data(libnd_pit_get(&pit), nd_ip_get(), nesting);
      if (!iphdr)
	continue;
      
      if (!libnd_ip_csum_correct(iphdr, &correct_sum))
	{
	  iphdr->ip_sum = correct_sum;
	  libnd_packet_modified(libnd_pit_get(&pit));
	}
    }

  return;
  TOUCH(value);
}


void   
nd_ip_fragment_cb(LND_Packet  *packet,
		  guchar      *header,
		  int          value)
{
  nd_ip_frag_show_dialog(packet);

  return;
  TOUCH(header);
  TOUCH(value);
}


void   
nd_ip_reassemble_cb(LND_Packet  *packet,
		    guchar      *header,
		    int          value)
{
  nd_ip_frag_reassemble(packet);

  return;
  TOUCH(header);
  TOUCH(value);
}

     


void 
ip_opt_addr_ok_cb(guchar     *address,
		  int         address_len,
		  LND_Packet *packet,
		  void       *user_data)
{
  guchar               *data = (guchar*) user_data;
  
  memcpy(data, address, sizeof(guchar) * 4);
  libnd_packet_modified(packet);

  return;
  TOUCH(address_len);
}


void   
nd_ip_opt_addr_cb(LND_Packet  *packet,
		  guchar      *header,
		  guchar      *data)
{
  nd_dialog_ip(_("Enter IP address:"),
	       (guchar*) data,
	       ip_opt_addr_ok_cb,
	       NULL,
	       packet, data);

  return;
  TOUCH(header);
}
