/*  This file is part of the KDE project
    Copyright (C) 2000 Simon Hausmann <hausmann@kde.org>
    Copyright (C) 2003 Luciano Montanaro <mikelima@cirulla.net>
    Copyright (C) 2001 Dirk Mueller <mueller@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "kconfig.h"
#include "kstandarddirs.h"
#include "kdebug.h"

#include <qfile.h>
#include <qtextstream.h>

#include <assert.h>

KConfigBackend::~KConfigBackend()
{
}

KConfigFileBackend::KConfigFileBackend( const QString &fileName )
    : m_fileName( fileName ), m_current( m_data.end() )
{
}

KConfigFileBackend::~KConfigFileBackend()
{
}

void KConfigFileBackend::load()
{
    m_data.clear();
    m_current = m_data.end();

    if ( m_fileName.isEmpty() )
        return;

    QStringList files = KGlobal::dirs()->findAllResources( "config",
                                                           m_fileName );
    QStringList::ConstIterator it = files.fromLast();
    QStringList::ConstIterator end = files.end();
    for (; it != end; --it )
    {
        //kdDebug() << "parsing " << *it << endl;
        parse( *it );
    }
}

void KConfigFileBackend::save()
{
    if ( m_fileName.isEmpty() )
        return;

    QString localFileName = m_fileName;
    if ( localFileName[0] != '/' )
        localFileName.prepend( KGlobal::dirs()->saveLocation( "config" ) );

    //kdDebug() << "writing to " << localFileName << endl;

    QFile f( localFileName );
    if ( !f.open( IO_WriteOnly ) )
        return;

    QString s;
    GroupMap::ConstIterator gIt = m_data.begin();
    GroupMap::ConstIterator gEnd = m_data.end();
    for (; gIt != gEnd; ++gIt )
    {
        EntryMap m = gIt.data();

        if ( m.isEmpty() ) // don't write empty groups
            continue;

        s += QString::fromLatin1("[") + gIt.key() + QString::fromLatin1("]\n");

        EntryMap::ConstIterator eIt = m.begin();
        EntryMap::ConstIterator eEnd = m.end();
        for (; eIt != eEnd; ++eIt )
            s += eIt.key() + "=" + eIt.data() + "\n";
    }
    
    QCString save = s.utf8();
    f.writeBlock(save.data(), save.length());
    f.close();
}

QString KConfigFileBackend::group()
{
    if ( m_current == m_data.end() )
        return QString::null;
    return m_current.key();
}

void KConfigFileBackend::setGroup( const QString &group )
{
    QString grp = group;
    if ( grp.isEmpty() )
        grp = "<default>";

    m_current = m_data.find( grp );
    if ( m_current == m_data.end() )
        m_current = m_data.insert( grp, EntryMap() );
}

QStringList KConfigFileBackend::groupList() const
{
    QStringList res;
    GroupMap::ConstIterator it = m_data.begin();
    GroupMap::ConstIterator end = m_data.end();
    for (; it != end; ++it )
        res << it.key();

    return res;
}

bool KConfigFileBackend::hasGroup( const QString &group ) const
{
    return m_data.contains( group );
}

bool KConfigFileBackend::hasKey( const QString &key )
{
    if ( m_current == m_data.end() )
        return false;

    return m_current.data().contains( key );
}

QMap<QString, QString> KConfigFileBackend::entryMap( const QString &group )
{
    QString currentKey = m_current.key();

    GroupMap::ConstIterator grp = m_data.find( group );
    if ( grp == m_data.end() )
    {
        grp = m_data.insert( group, EntryMap() );
        m_current = m_data.find( currentKey );
    }

    return grp.data();
}

bool KConfigFileBackend::lookup( const QString &key, QString &value )
{
    if ( m_current == m_data.end() )
        return false;

    EntryMap::ConstIterator it = m_current.data().find( key );

    if ( it == m_current.data().end() )
        return false;

    value = *it;
    return true;
}

void KConfigFileBackend::put( const QString &key, const QString &value )
{
    if ( m_current != m_data.end() )
        m_current.data().replace( key, value );
    else
        kdDebug() << "writeEntry without previous setGroup!" << endl;
}

bool KConfigFileBackend::deleteGroup( const QString &group )
{
    GroupMap::Iterator groupIt = m_data.find( group );

    if ( m_current == groupIt )
        m_current = m_data.end();

    m_data.remove( groupIt );
    return true;
}

void KConfigFileBackend::deleteEntry( const QString &entry )
{
    if ( m_current == m_data.end() )
        return;

    m_current.data().remove( entry );
}

void KConfigFileBackend::parse( const QString &fileName )
{
    if ( fileName.isEmpty() )
        return;

    QFile f( fileName );
    if ( !f.open( IO_ReadOnly ) )
        return;

    while ( !f.atEnd() )
    {
        char buf[200];
        f.readLine(buf, sizeof(buf));

        QString stripped = QString::fromUtf8(buf).stripWhiteSpace();

        // comment? skip
        if ( stripped[ 0 ] == '#' )
            continue;

        // group?
        if ( stripped[ 0 ] == '[' )
        {
            // find end of group
            int pos = stripped.find( ']' );

            if ( pos == -1 )
                continue;

            QString group = stripped.mid( 1, pos - 1 );
            m_current = m_data.find( group );

            if ( m_current == m_data.end() )
                m_current = m_data.insert( group, EntryMap() );
        }
        else if ( stripped.find( '=' ) != -1 )
        {
            uint pos = (uint)stripped.find( '=' );

            if ( m_current == m_data.end() )
            {
                // default group
                m_current = m_data.find( "<default>" );
                if ( m_current == m_data.end() )
                    m_current = m_data.insert( "<default>", EntryMap() );
            }

            if ( pos+1 == stripped.length() ) // need empty value
                m_current.data()[ stripped.left( pos ) ] = "";
            else
                m_current.data()[ stripped.left( pos ) ] = stripped.mid( pos+1 );
        }
    }
}

KConfig::KConfig( const QString &fileName, bool readOnly, bool, const char * )
    : m_backend( new KConfigFileBackend( fileName ) ),
      m_readOnly( readOnly )
{
    reparseConfiguration();
}

KConfig::KConfig( KConfigBackend *backEnd, bool readOnly )
    : m_backend( backEnd ), m_readOnly( readOnly )
{
    assert( m_backend );
}

KConfig::~KConfig()
{
    sync();
    delete m_backend;
}

void KConfig::writeEntry( const QString &key, const QString &value )
{
    m_backend->put( key, value );
}

void KConfig::writeEntry( const QString &key, int value )
{
    writeEntry( key, QString::number( value ) );
}

void KConfig::writeEntry( const QString &key, unsigned int value )
{
    writeEntry( key, QString::number( value ) );
}

void KConfig::writeEntry( const QString &key, bool value )
{
    writeEntry( key, value ? QString::fromLatin1( "true" ) :
                             QString::fromLatin1( "false" ) );
}

void KConfig::writeEntry( const QString &key, const QStringList &value, QChar sep )
{
    QString val;

    QStringList::ConstIterator it = value.begin();
    QStringList::ConstIterator end = value.end();
    while ( it != end )
    {
        val += *it;
        ++it;
        if ( it != end )
            val += sep;
    }

    writeEntry( key, val );
}

void KConfig::writeEntry( const QString &key, const QDateTime &dateTime )
{
    QStringList list;

    QTime time = dateTime.time();
    QDate date = dateTime.date();

    list << QString::number( date.year() );
    list << QString::number( date.month() );
    list << QString::number( date.day() );

    list << QString::number( time.hour() );
    list << QString::number( time.minute() );
    list << QString::number( time.second() );

    writeEntry( key, list, ',' );
}

QString KConfig::readEntry( const QString &key, const QString &defaultValue ) const
{
    QString value;
    bool gotKey = m_backend->lookup( key, value );

#if defined(CONFIG_DEBUG)
    kdDebug() << "Reading Entry " << group() << "/" << key << endl;
#endif

    if ( !gotKey )
        return defaultValue;

    return value;
}

int KConfig::readNumEntry( const QString &key, int defaultValue ) const
{
    QString val = readEntry( key );

    if ( val.isEmpty() )
        return defaultValue;

    return val.toInt();
}

uint KConfig::readUnsignedNumEntry( const QString &key, int defaultValue ) const
{
    QString val = readEntry( key );

    if ( val.isEmpty() )
        return defaultValue;

    return val.toUInt();
}

QStringList KConfig::readListEntry( const QString &key, QChar sep ) const
{
    QString val = readEntry( key );

    if ( val.isEmpty() )
        return QStringList();

    return QStringList::split( sep, val );
}

bool KConfig::readBoolEntry( const QString &key, bool defaultValue ) const
{
    QString val = readEntry( key );

    if ( val.isEmpty() )
        return defaultValue;

    val = val.lower();

    if ( val == "true" )
        return true;

    if ( val == "on" )
        return true;

    if ( val == "off" )
        return false;

    if ( val == "false" )
        return false;

    return static_cast<bool>( val.toInt() );
}

QColor KConfig::readColorEntry( const QString &key, const QColor *defaultVal ) const
{
    QString val = readEntry( key );

    QColor res = *defaultVal;

    if ( !val.isEmpty() && val.at( 0 ) == '#' )
        res.setNamedColor( val );

    return res;
}

QDateTime KConfig::readDateTimeEntry( const QString &key )
{
    QDateTime res = QDateTime::currentDateTime();

    if( !hasKey( key ) )
        return res;

    QStringList list = readListEntry( key );

    if ( list.count() != 6 )
        return res;

    QTime time;
    QDate date;

    date.setYMD( list[ 0 ].toInt(),
                 list[ 1 ].toInt(),
                 list[ 2 ].toInt() );
    time.setHMS( list[ 3 ].toInt(),
                 list[ 4 ].toInt(),
                 list[ 5 ].toInt() );

    res.setTime( time );
    res.setDate( date );
    return res;
}

QPoint KConfig::readPointEntry( 
        const QString &key, const QPoint& defaultvalue) const
{
    QString aValue = readEntry(key);

    if (!aValue.isEmpty()) {
        int x, y;

        if (sscanf(aValue.utf8(), "%d,%d", &x, &y) == 4) { 
	    return QPoint(x, y); 
	} 
    } 
    return defaultvalue;
}

QRect KConfig::readRectEntry( 
        const QString &key, const QRect& defaultvalue) const
{
    QString aValue = readEntry(key);

    if (!aValue.isEmpty()) {
        int left, top, width, height;

        if (sscanf(aValue.utf8(), "%d,%d,%d,%d", 
                    &left, &top, &width, &height) == 4) { 
	    return QRect(left, top, width, height); 
	} 
    } 
    return defaultvalue;
}

/*
 * vim:et:sw=4
 *
 */
