/*************************************************************************

    Midway Wolf-unit system

    driver by Ernesto Corvi, Aaron Giles
    based on Y/Z-unit driver by Alex Pasadyn, Zsolt Vasvari, Kurt Mahan

    Games supported:
        * Mortal Kombat 3
        * Ultimate Mortal Kombat 3
        * NBA Hangtime
        * NBA Maximum Hangtime
        * 2 On 2 Open Ice Challenge
        * WWF Wrestlemania
        * Rampage World Tour

    Known bugs:
        * WWF has an unimplemented and not Y2K compatible real-time clock

***************************************************************************

MK3/ Ultimate Mortal Kombat 3 / Williams-Midway Wolf hardware
Midway, 1995

PCB Layout
----------

5770-14086-05
|---------------------------------------------------------------------------------------------------|
| 42C8128 42C8128 42C8128 42C8128                       *U102       *U103       *U104       *U105   |
| 42C8128 42C8128 42C8128 42C8128                                                                   |
|                                                       *U106       *U107       *U108       *U109   |
|                                                                                                   |
|                |---------|                             U110        U111        U112        U113   |
| |-------|      |5410-    |                                                                        |
| |5410-  |      |12862-   |                            *U114       *U115       *U116       *U117   |
| |14120-0|      |00       |          |------|                                                      |
| |-------|      |WILLIAMS |          |ALTERA|           U118        U119        U120        U121   |
| 84256  84256   |---------|          |FPGA3 |                                                      |
|                                     |------|           U122        U123        U124        U125   |
|                                        RESET_SW                                                   |
|                                                        U126        U127        U128        U129   |
| 8MHz      |------|                                                                                |
|           |ALTERA|              BATTERY                U130        U131        U132        U133   |
||------|   |FPGA2 |                                                                                |
||ALTERA|   |      |      42S4260                                                                   |
||FPGA1 |   |------|                            10MHz                                               |
||      |50MHz                               GAL       |-----|               ROM.U5         *U9     |
||------|                  ROM.U54                     |ADSP-|                                      |
|              LH5268                                  |2105 |               ROM.U4         *U8     |
|   |------|                                           |-----|                                      |
|   |TMS   |               ROM.U63      PIC16C57.U64                         ROM.U3         *U7     |
|   |34010 |                                             65764                                      |
|   |      | LED LED                                     65764               ROM.U2         *U6     |
|   |------|                                 P6          65764                                      |
|                                                                                                   |
|                                                                                                   |
|    TL084                                                    MAX693               DSW2    DSW1     |
|                AD1851                                       ULN2064                               |
|TDA7240                                                         LED                                |
|            |----|         JAMMA           |----|             P9    P10     P11      P12           |
|------------|    |-------------------------|    |--------------------------------------------------|
Notes:
      *            - Empty DIP32 socket (for UMK3)
      P9/10/11/12  - connector for extra controls
      P6           - connector possibly tied to the PIC via some logic?
      GAL          - labelled 'MKIII-U57 A-20093 (C)1995 MIDWAY MFG. CO.'
      PIC.U64      - labelled 'MK3 ULTIMATE U64'
      ALTERA FPGA1 - labelled 'MKIII-U45 A-20096 (C)1995 MIDWAY MFG. CO.'
      ALTERA FPGA2 - labelled 'MKIII-U47 A-20249 (C)1995 MIDWAY MFG. CO.'
      ALTERA FPGA3 - labelled 'MKIII-U35 A-20095 (C)1995 MIDWAY MFG. CO.'
      TMS34010     - input clock (pin5) 50.000MHz
      ADSP-2105    - clock input 10.000MHz
      PIC16C57     - clock input 4.000MHz [8/2]
      TDA7240      - Sound Amplifier IC
      RAM          - 42C8128 (x8), 42S4260 (x1), 84256 (x2), LH5268 (x1), 65764 (x3)
      VSync        - 54.5Hz (measured on pin 31 of TMS34010)
      HSync        - 15.81kHz (measured on pin 30 of TMS34010)

      All ROMs from U102 to U133 are labelled 'MORTAL KOMBAT III ULTIMATE (C) MIDWAY MFG U' + U# as 3 digits + 'VIDEO IMAGE'
      U3 labelled 'MORTAL KOMBAT 3 U3 MUSIC/SPCH'
      U2 labelled 'MORTAL KOMBAT 3 U2 ULTIMATE'

**************************************************************************/


#include "driver.h"
#include "cpu/tms34010/tms34010.h"
#include "cpu/adsp2100/adsp2100.h"
#include "sndhrdw/dcs.h"
#include "midwunit.h"



/*************************************
 *
 *  Memory maps
 *
 *************************************/

static ADDRESS_MAP_START( main_map, ADDRESS_SPACE_PROGRAM, 16 )
	AM_RANGE(0x00000000, 0x003fffff) AM_READWRITE(midtunit_vram_r, midtunit_vram_w)
	AM_RANGE(0x01000000, 0x013fffff) AM_RAM
	AM_RANGE(0x01400000, 0x0145ffff) AM_READWRITE(midwunit_cmos_r, midwunit_cmos_w) AM_BASE(&generic_nvram16) AM_SIZE(&generic_nvram_size)
	AM_RANGE(0x01480000, 0x014fffff) AM_WRITE(midwunit_cmos_enable_w)
	AM_RANGE(0x01600000, 0x0160001f) AM_READWRITE(midwunit_security_r, midwunit_security_w)
	AM_RANGE(0x01680000, 0x0168001f) AM_READWRITE(midwunit_sound_r, midwunit_sound_w)
	AM_RANGE(0x01800000, 0x0187ffff) AM_READWRITE(midwunit_io_r, midwunit_io_w)
	AM_RANGE(0x01880000, 0x018fffff) AM_READWRITE(MRA16_RAM, midtunit_paletteram_w) AM_BASE(&paletteram16)
	AM_RANGE(0x01a00000, 0x01a000ff) AM_MIRROR(0x00080000) AM_READWRITE(midtunit_dma_r, midtunit_dma_w)
	AM_RANGE(0x01b00000, 0x01b0001f) AM_READWRITE(midwunit_control_r, midwunit_control_w)
	AM_RANGE(0x02000000, 0x06ffffff) AM_READ(midwunit_gfxrom_r) AM_BASE((UINT16 **)&midwunit_decode_memory)
	AM_RANGE(0xc0000000, 0xc00001ff) AM_READWRITE(tms34010_io_register_r, tms34010_io_register_w)
	AM_RANGE(0xff800000, 0xffffffff) AM_ROM AM_REGION(REGION_USER1, 0)
ADDRESS_MAP_END



/*************************************
 *
 *  Input ports
 *
 *************************************/

INPUT_PORTS_START( mk3 )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(2)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON6 ) PORT_PLAYER(2)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0001, 0x0001, "Test Switch" )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_DIPNAME( 0x0002, 0x0000, "Counters" )
	PORT_DIPSETTING(      0x0002, "One" )
	PORT_DIPSETTING(      0x0000, "Two" )
	PORT_DIPNAME( 0x007c, 0x007c, DEF_STR( Coinage ))
	PORT_DIPSETTING(      0x007c, "USA-1" )
	PORT_DIPSETTING(      0x003c, "USA-2" )
	PORT_DIPSETTING(      0x005c, "USA-3" )
	PORT_DIPSETTING(      0x001c, "USA-4" )
	PORT_DIPSETTING(      0x006c, "USA-ECA" )
	PORT_DIPSETTING(      0x000c, "USA-Free Play" )
	PORT_DIPSETTING(      0x0074, "German-1" )
	PORT_DIPSETTING(      0x0034, "German-2" )
	PORT_DIPSETTING(      0x0054, "German-3" )
	PORT_DIPSETTING(      0x0014, "German-4" )
	PORT_DIPSETTING(      0x0064, "German-5" )
	PORT_DIPSETTING(      0x0024, "German-ECA" )
	PORT_DIPSETTING(      0x0004, "German-Free Play" )
	PORT_DIPSETTING(      0x0078, "French-1" )
	PORT_DIPSETTING(      0x0038, "French-2" )
	PORT_DIPSETTING(      0x0058, "French-3" )
	PORT_DIPSETTING(      0x0018, "French-4" )
	PORT_DIPSETTING(      0x0068, "French-ECA" )
	PORT_DIPSETTING(      0x0008, "French-Free Play" )
	PORT_DIPNAME( 0x0080, 0x0000, "Coinage Source" )
	PORT_DIPSETTING(      0x0080, "Dipswitch" )
	PORT_DIPSETTING(      0x0000, "CMOS" )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0200, 0x0000, "Powerup Test" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0200, DEF_STR( On ))
	PORT_DIPNAME( 0x0400, 0x0400, "Bill Validator" )
	PORT_DIPSETTING(      0x0400, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x1000, 0x1000, "Attract Sound" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x1000, DEF_STR( On ))
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x4000, 0x4000, "Blood" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x4000, DEF_STR( On ))
	PORT_DIPNAME( 0x8000, 0x8000, "Violence" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x8000, DEF_STR( On ))

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_TILT ) /* Slam Switch */
	PORT_BIT(0x0010, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME(DEF_STR( Test )) PORT_CODE(KEYCODE_F2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_VOLUME_DOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_VOLUME_UP )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNUSED ) /* interrupt */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( openice )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(4)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(4)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(4)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0001, 0x0000, "Coinage Source" )
	PORT_DIPSETTING(      0x0001, "Dipswitch" )
	PORT_DIPSETTING(      0x0000, "CMOS" )
	PORT_DIPNAME( 0x003e, 0x003e, DEF_STR( Coinage ))
	PORT_DIPSETTING(      0x003e, "USA-1" )
	PORT_DIPSETTING(      0x003c, "USA-2" )
	PORT_DIPSETTING(      0x003a, "USA-3" )
	PORT_DIPSETTING(      0x0038, "USA-4" )
	PORT_DIPSETTING(      0x0034, "USA-9" )
	PORT_DIPSETTING(      0x0032, "USA-10" )
	PORT_DIPSETTING(      0x0036, "USA-ECA" )
	PORT_DIPSETTING(      0x0030, "USA-Free Play" )
	PORT_DIPSETTING(      0x002e, "German-1" )
	PORT_DIPSETTING(      0x002c, "German-2" )
	PORT_DIPSETTING(      0x002a, "German-3" )
	PORT_DIPSETTING(      0x0028, "German-4" )
	PORT_DIPSETTING(      0x0024, "German-5" )
	PORT_DIPSETTING(      0x0026, "German-ECA" )
	PORT_DIPSETTING(      0x0020, "German-Free Play" )
	PORT_DIPSETTING(      0x001e, "French-1" )
	PORT_DIPSETTING(      0x001c, "French-2" )
	PORT_DIPSETTING(      0x001a, "French-3" )
	PORT_DIPSETTING(      0x0018, "French-4" )
	PORT_DIPSETTING(      0x0014, "French-11" )
	PORT_DIPSETTING(      0x0012, "French-12" )
	PORT_DIPSETTING(      0x0016, "French-ECA" )
	PORT_DIPSETTING(      0x0010, "French-Free Play" )
	PORT_DIPNAME( 0x0040, 0x0000, "Counters" )
	PORT_DIPSETTING(      0x0040, "One" )
	PORT_DIPSETTING(      0x0000, "Two" )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0100, 0x0100, "Bill Validator" )
	PORT_DIPSETTING(      0x0100, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_DIPNAME( 0x0200, 0x0200, "Attract Sound" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0200, DEF_STR( On ))
	PORT_DIPNAME( 0x0400, 0x0000, "Powerup Test" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0400, DEF_STR( On ))
	PORT_DIPNAME( 0x0800, 0x0800, "Head Size" )
	PORT_DIPSETTING(      0x0800, DEF_STR( Normal ) )
	PORT_DIPSETTING(      0x0000, "Large" )
	PORT_DIPNAME( 0x1000, 0x0000, DEF_STR( Cabinet ))
	PORT_DIPSETTING(      0x0000, "2-player" )
	PORT_DIPSETTING(      0x1000, "4-player" )
	PORT_BIT( 0x6000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x8000, 0x8000, "Test Switch" )
	PORT_DIPSETTING(      0x8000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_TILT ) /* Slam Switch */
	PORT_BIT(0x0010, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME(DEF_STR( Test )) PORT_CODE(KEYCODE_F2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_VOLUME_DOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_VOLUME_UP )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNUSED ) /* interlock */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( nbahangt )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(4)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(4)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(4)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(4)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0001, 0x0001, "Test Switch" )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_DIPNAME( 0x0002, 0x0000, "Powerup Test" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0002, DEF_STR( On ))
	PORT_BIT( 0x003c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0040, 0x0040, "Bill Validator" )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_DIPNAME( 0x0080, 0x0000, DEF_STR( Cabinet ))
	PORT_DIPSETTING(      0x0000, "2-player" )
	PORT_DIPSETTING(      0x0080, "4-player" )
	PORT_DIPNAME( 0x0300, 0x0300, "Counters" )
	PORT_DIPSETTING(      0x0300, "One, 1/1" )
	PORT_DIPSETTING(      0x0200, "One, Totalizing" )
	PORT_DIPSETTING(      0x0100, "Two, 1/1" )
	PORT_DIPNAME( 0x0c00, 0x0c00, "Country" )
	PORT_DIPSETTING(      0x0c00, DEF_STR( USA ) )
	PORT_DIPSETTING(      0x0800, DEF_STR( French ) )
	PORT_DIPSETTING(      0x0400, DEF_STR( German ) )
/*  PORT_DIPSETTING(      0x0000, DEF_STR( Unused )) */
	PORT_DIPNAME( 0x7000, 0x7000, DEF_STR( Coinage ))
	PORT_DIPSETTING(      0x7000, "1" )
	PORT_DIPSETTING(      0x3000, "2" )
	PORT_DIPSETTING(      0x5000, "3" )
	PORT_DIPSETTING(      0x1000, "4" )
	PORT_DIPSETTING(      0x6000, "ECA" )
/*  PORT_DIPSETTING(      0x4000, DEF_STR( Unused )) */
/*  PORT_DIPSETTING(      0x2000, DEF_STR( Unused )) */
	PORT_DIPSETTING(      0x0000, DEF_STR( Free_Play ))
	PORT_DIPNAME( 0x8000, 0x0000, "Coinage Source" )
	PORT_DIPSETTING(      0x8000, "Dipswitch" )
	PORT_DIPSETTING(      0x0000, "CMOS" )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_TILT ) /* Slam Switch */
	PORT_BIT(0x0010, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME(DEF_STR( Test )) PORT_CODE(KEYCODE_F2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_VOLUME_DOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_VOLUME_UP )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNUSED ) /* interlock */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( rmpgwt )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(3)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(3)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(3)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(3)
	PORT_BIT( 0xff80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0001, 0x0000, "Coinage Source" )
	PORT_DIPSETTING(      0x0001, "Dipswitch" )
	PORT_DIPSETTING(      0x0000, "CMOS" )
	PORT_DIPNAME( 0x003e, 0x003e, DEF_STR( Coinage ))
	PORT_DIPSETTING(      0x003e, "USA-1" )
	PORT_DIPSETTING(      0x003c, "USA-2" )
	PORT_DIPSETTING(      0x003a, "USA-3" )
	PORT_DIPSETTING(      0x0038, "USA-4" )
	PORT_DIPSETTING(      0x0034, "USA-9" )
	PORT_DIPSETTING(      0x0032, "USA-10" )
	PORT_DIPSETTING(      0x0036, "USA-ECA" )
	PORT_DIPSETTING(      0x0030, "USA-Free Play" )
	PORT_DIPSETTING(      0x002e, "German-1" )
	PORT_DIPSETTING(      0x002c, "German-2" )
	PORT_DIPSETTING(      0x002a, "German-3" )
	PORT_DIPSETTING(      0x0028, "German-4" )
	PORT_DIPSETTING(      0x0024, "German-5" )
	PORT_DIPSETTING(      0x0026, "German-ECA" )
	PORT_DIPSETTING(      0x0020, "German-Free Play" )
	PORT_DIPSETTING(      0x001e, "French-1" )
	PORT_DIPSETTING(      0x001c, "French-2" )
	PORT_DIPSETTING(      0x001a, "French-3" )
	PORT_DIPSETTING(      0x0018, "French-4" )
	PORT_DIPSETTING(      0x0014, "French-11" )
	PORT_DIPSETTING(      0x0012, "French-12" )
	PORT_DIPSETTING(      0x0016, "French-ECA" )
	PORT_DIPSETTING(      0x0010, "French-Free Play" )
	PORT_DIPNAME( 0x0040, 0x0000, "Counters" )
	PORT_DIPSETTING(      0x0040, "One" )
	PORT_DIPSETTING(      0x0000, "Two" )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0100, 0x0100, "Bill Validator" )
	PORT_DIPSETTING(      0x0100, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0400, 0x0000, "Powerup Test" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0400, DEF_STR( On ))
	PORT_BIT( 0x7800, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x8000, 0x8000, "Test Switch" )
	PORT_DIPSETTING(      0x8000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_TILT ) /* Slam Switch */
	PORT_BIT(0x0010, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME(DEF_STR( Test )) PORT_CODE(KEYCODE_F2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_VOLUME_DOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_VOLUME_UP )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNUSED ) /* interlock */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( wwfmania )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(1)
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(1)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(1)
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(1)
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT ) PORT_8WAY PORT_PLAYER(2)
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 ) PORT_PLAYER(2)
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 ) PORT_PLAYER(2)
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 ) PORT_PLAYER(2)
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(1)
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(1)
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 ) PORT_PLAYER(2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON5 ) PORT_PLAYER(2)
	PORT_BIT( 0x00c0, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0001, 0x0001, "Test Switch" )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_DIPNAME( 0x0002, 0x0000, "Powerup Test" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0002, DEF_STR( On ))
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0008, 0x0008, "Realtime Clock" )
	PORT_DIPSETTING(      0x0008, DEF_STR( No ))
	PORT_DIPSETTING(      0x0000, DEF_STR( Yes ))
	PORT_BIT( 0x0030, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0040, 0x0040, "Bill Validator" )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ))
	PORT_DIPSETTING(      0x0000, DEF_STR( On ))
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0300, 0x0300, "Counters" )
	PORT_DIPSETTING(      0x0300, "One, 1/1" )
	PORT_DIPSETTING(      0x0200, "One, Totalizing" )
	PORT_DIPSETTING(      0x0100, "Two, 1/1" )
	PORT_DIPNAME( 0x0c00, 0x0c00, "Country" )
	PORT_DIPSETTING(      0x0c00, DEF_STR( USA ) )
	PORT_DIPSETTING(      0x0800, DEF_STR( French ) )
	PORT_DIPSETTING(      0x0400, DEF_STR( German ) )
/*  PORT_DIPSETTING(      0x0000, DEF_STR( Unused )) */
	PORT_DIPNAME( 0x7000, 0x7000, DEF_STR( Coinage ))
	PORT_DIPSETTING(      0x7000, "1" )
	PORT_DIPSETTING(      0x3000, "2" )
	PORT_DIPSETTING(      0x5000, "3" )
	PORT_DIPSETTING(      0x1000, "4" )
	PORT_DIPSETTING(      0x6000, "ECA" )
/*  PORT_DIPSETTING(      0x4000, DEF_STR( Unused )) */
/*  PORT_DIPSETTING(      0x2000, DEF_STR( Unused )) */
	PORT_DIPSETTING(      0x0000, DEF_STR( Free_Play ))
	PORT_DIPNAME( 0x8000, 0x0000, "Coinage Source" )
	PORT_DIPSETTING(      0x8000, "Dipswitch" )
	PORT_DIPSETTING(      0x0000, "CMOS" )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_TILT ) /* Slam Switch */
	PORT_BIT(0x0010, IP_ACTIVE_LOW, IPT_SERVICE ) PORT_NAME(DEF_STR( Test )) PORT_CODE(KEYCODE_F2)
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_VOLUME_DOWN )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_VOLUME_UP )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_UNUSED ) /* interrupt */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END



/*************************************
 *
 *  34010 configuration
 *
 *************************************/

static struct tms34010_config tms_config =
{
	0,								/* halt on reset */
	NULL,							/* generate interrupt */
	midtunit_to_shiftreg,			/* write to shiftreg function */
	midtunit_from_shiftreg,			/* read from shiftreg function */
	0,								/* display address changed */
	0								/* display interrupt callback */
};



/*************************************
 *
 *  Machine drivers
 *
 *************************************/

/*
    all games use identical visible areas and VBLANK timing
    based on these video params:

              VERTICAL                   HORIZONTAL
    mk3:      0014-0112 / 0120 (254)     0065-001F5 / 01F9 (400)
    umk3:     0014-0112 / 0120 (254)     0065-001F5 / 01F9 (400)
    wwfmania: 0014-0112 / 0120 (254)     0065-001F5 / 01F9 (400)
    openice:  0014-0112 / 0120 (254)     0065-001F5 / 01F9 (400)
    nbahangt: 0014-0112 / 0120 (254)     0065-001F5 / 01F9 (400)
    nbamht:   0014-0112 / 0120 (254)     0065-001F5 / 01F9 (400)
*/

static MACHINE_DRIVER_START( wunit )

	MDRV_CPU_ADD(TMS34010, 50000000/TMS34010_CLOCK_DIVIDER)
	MDRV_CPU_CONFIG(tms_config)
	MDRV_CPU_PROGRAM_MAP(main_map,0)

	MDRV_FRAMES_PER_SECOND(MKLA5_FPS)
	MDRV_VBLANK_DURATION((1000000 * (288 - 254)) / (MKLA5_FPS * 288))
	MDRV_MACHINE_INIT(midwunit)
	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(400, 256)
	MDRV_VISIBLE_AREA(0, 399, 0, 253)
	MDRV_PALETTE_LENGTH(32768)

	MDRV_VIDEO_START(midwunit)
	MDRV_VIDEO_UPDATE(midtunit)

	/* sound hardware */
	MDRV_IMPORT_FROM(dcs_audio_8k)
MACHINE_DRIVER_END



/*************************************
 *
 *  ROM definitions
 *
 *************************************/

ROM_START( mk3 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "umk3-u2.bin",  0x000000, 0x100000, CRC(3838cfe5) SHA1(e3d2901f3bae1362742fc6ee0aa31c9f63b4dfa3) )
	ROM_LOAD16_BYTE( "umk3-u3.bin",  0x200000, 0x100000, CRC(856fe411) SHA1(6165ebecfce7500e948d84492ffa19eed7f47091) )
	ROM_LOAD16_BYTE( "umk3-u4.bin",  0x400000, 0x100000, CRC(428a406f) SHA1(e70ec83cd054de0da1e178720ed0035b8887f797) )
	ROM_LOAD16_BYTE( "umk3-u5.bin",  0x600000, 0x100000, CRC(3b98a09f) SHA1(edf1d02a56dcf3349e6b4bb4097acfe7592305f4) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "mk321u54.bin",  0x00000, 0x80000, CRC(9e344401) SHA1(5760b355f0a5c27c9746f33abfdedf4302f1af38) )
	ROM_LOAD16_BYTE( "mk321u63.bin",  0x00001, 0x80000, CRC(64d34776) SHA1(d8f09e1e946dc13fec5e9f83fdaf61d4076ba9ea) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "umk-u133.bin",  0x0000000, 0x100000, CRC(79b94667) SHA1(31bba640c351fdccc6685cadb74dd79a3f910ce8) )
	ROM_LOAD( "umk-u132.bin",  0x0100000, 0x100000, CRC(13e95228) SHA1(405b05f5a5a55667c2be17d4b399129bdacefd90) )
	ROM_LOAD( "umk-u131.bin",  0x0200000, 0x100000, CRC(41001e30) SHA1(2cec91116771951c0380cec5debf4cbb40c14c61) )
	ROM_LOAD( "umk-u130.bin",  0x0300000, 0x100000, CRC(49379dd7) SHA1(e6dfab4e23d9cc38ae56c1bbf10ccd160e8fad5e) )

	ROM_LOAD( "umk-u129.bin",  0x0400000, 0x100000, CRC(a8b41803) SHA1(9697e35e8bb51d6d36b1d7ae47377b446e57682f) )
	ROM_LOAD( "umk-u128.bin",  0x0500000, 0x100000, CRC(b410d72f) SHA1(ac5c1c6f744186540f4ab100d9bd4ce6007e600b) )
	ROM_LOAD( "umk-u127.bin",  0x0600000, 0x100000, CRC(bd985be7) SHA1(f5183abea2e5eb2c2c8cefa72c9ed321679f5128) )
	ROM_LOAD( "umk-u126.bin",  0x0700000, 0x100000, CRC(e7c32cf4) SHA1(94ea7b2eed7dae66f5dd676c20d6b360140e3e0e) )

	ROM_LOAD( "umk-u125.bin",  0x0800000, 0x100000, CRC(9a52227e) SHA1(0474a14fa8dbfea0b0889c1d1756b86391683558) )
	ROM_LOAD( "umk-u124.bin",  0x0900000, 0x100000, CRC(5c750ebc) SHA1(45d68af1a56994376e086d840502453c8d6be700) )
	ROM_LOAD( "umk-u123.bin",  0x0a00000, 0x100000, CRC(f0ab88a8) SHA1(cdc9dc12e162255845c6627b1e35182b7e8502d0) )
	ROM_LOAD( "umk-u122.bin",  0x0b00000, 0x100000, CRC(9b87cdac) SHA1(a5f8db559293978f23e6f105543d8b2e170a2e0d) )

	ROM_LOAD( "mk3-u121.bin",  0x0c00000, 0x100000, CRC(b6c6296a) SHA1(7b92a92d65493bb201daf5ece6f00140f017ac51) )
	ROM_LOAD( "mk3-u120.bin",  0x0d00000, 0x100000, CRC(8d1ccc3b) SHA1(35d91c00113718a08a9d56eb04366f8cf4069ba6) )
	ROM_LOAD( "mk3-u119.bin",  0x0e00000, 0x100000, CRC(63215b59) SHA1(709bce15fba1520bcba40f0a5cb614542f1b460f) )
	ROM_LOAD( "mk3-u118.bin",  0x0f00000, 0x100000, CRC(8b681e34) SHA1(524104ba2eb2deb3cfae9760e6b2125ce6b6633e) )

	ROM_LOAD( "mk3-u117.bin",  0x1000000, 0x080000, CRC(1ab20377) SHA1(0c0d14464d8b23a60e0693669af2ddb82655eff8) )
	ROM_LOAD( "mk3-u116.bin",  0x1100000, 0x080000, CRC(ba246ad0) SHA1(ec6bdd4b9cd3007bb66bb8de36d148abb30e7f11) )
	ROM_LOAD( "mk3-u115.bin",  0x1200000, 0x080000, CRC(3ee8b124) SHA1(1523d51e36d3c336b134a562da36a29ba137c0f6) )
	ROM_LOAD( "mk3-u114.bin",  0x1300000, 0x080000, CRC(a8d99922) SHA1(04ff8d85448701070672f44dbf5bcfd744f1bc8a) )
ROM_END


ROM_START( mk3r20 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "umk3-u2.bin",  0x000000, 0x100000, CRC(3838cfe5) SHA1(e3d2901f3bae1362742fc6ee0aa31c9f63b4dfa3) )
	ROM_LOAD16_BYTE( "umk3-u3.bin",  0x200000, 0x100000, CRC(856fe411) SHA1(6165ebecfce7500e948d84492ffa19eed7f47091) )
	ROM_LOAD16_BYTE( "umk3-u4.bin",  0x400000, 0x100000, CRC(428a406f) SHA1(e70ec83cd054de0da1e178720ed0035b8887f797) )
	ROM_LOAD16_BYTE( "umk3-u5.bin",  0x600000, 0x100000, CRC(3b98a09f) SHA1(edf1d02a56dcf3349e6b4bb4097acfe7592305f4) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "mk320u54.bin",  0x00000, 0x80000, CRC(453da302) SHA1(d9a4814e7abb49ac0eb306ad05adcceac68df6a5) )
	ROM_LOAD16_BYTE( "mk320u63.bin",  0x00001, 0x80000, CRC(f8dc0600) SHA1(6eb689d92619c751252155b40af119ad47e94cfa) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "umk-u133.bin",  0x0000000, 0x100000, CRC(79b94667) SHA1(31bba640c351fdccc6685cadb74dd79a3f910ce8) )
	ROM_LOAD( "umk-u132.bin",  0x0100000, 0x100000, CRC(13e95228) SHA1(405b05f5a5a55667c2be17d4b399129bdacefd90) )
	ROM_LOAD( "umk-u131.bin",  0x0200000, 0x100000, CRC(41001e30) SHA1(2cec91116771951c0380cec5debf4cbb40c14c61) )
	ROM_LOAD( "umk-u130.bin",  0x0300000, 0x100000, CRC(49379dd7) SHA1(e6dfab4e23d9cc38ae56c1bbf10ccd160e8fad5e) )

	ROM_LOAD( "umk-u129.bin",  0x0400000, 0x100000, CRC(a8b41803) SHA1(9697e35e8bb51d6d36b1d7ae47377b446e57682f) )
	ROM_LOAD( "umk-u128.bin",  0x0500000, 0x100000, CRC(b410d72f) SHA1(ac5c1c6f744186540f4ab100d9bd4ce6007e600b) )
	ROM_LOAD( "umk-u127.bin",  0x0600000, 0x100000, CRC(bd985be7) SHA1(f5183abea2e5eb2c2c8cefa72c9ed321679f5128) )
	ROM_LOAD( "umk-u126.bin",  0x0700000, 0x100000, CRC(e7c32cf4) SHA1(94ea7b2eed7dae66f5dd676c20d6b360140e3e0e) )

	ROM_LOAD( "umk-u125.bin",  0x0800000, 0x100000, CRC(9a52227e) SHA1(0474a14fa8dbfea0b0889c1d1756b86391683558) )
	ROM_LOAD( "umk-u124.bin",  0x0900000, 0x100000, CRC(5c750ebc) SHA1(45d68af1a56994376e086d840502453c8d6be700) )
	ROM_LOAD( "umk-u123.bin",  0x0a00000, 0x100000, CRC(f0ab88a8) SHA1(cdc9dc12e162255845c6627b1e35182b7e8502d0) )
	ROM_LOAD( "umk-u122.bin",  0x0b00000, 0x100000, CRC(9b87cdac) SHA1(a5f8db559293978f23e6f105543d8b2e170a2e0d) )

	ROM_LOAD( "mk3-u121.bin",  0x0c00000, 0x100000, CRC(b6c6296a) SHA1(7b92a92d65493bb201daf5ece6f00140f017ac51) )
	ROM_LOAD( "mk3-u120.bin",  0x0d00000, 0x100000, CRC(8d1ccc3b) SHA1(35d91c00113718a08a9d56eb04366f8cf4069ba6) )
	ROM_LOAD( "mk3-u119.bin",  0x0e00000, 0x100000, CRC(63215b59) SHA1(709bce15fba1520bcba40f0a5cb614542f1b460f) )
	ROM_LOAD( "mk3-u118.bin",  0x0f00000, 0x100000, CRC(8b681e34) SHA1(524104ba2eb2deb3cfae9760e6b2125ce6b6633e) )

	ROM_LOAD( "mk3-u117.bin",  0x1000000, 0x080000, CRC(1ab20377) SHA1(0c0d14464d8b23a60e0693669af2ddb82655eff8) )
	ROM_LOAD( "mk3-u116.bin",  0x1100000, 0x080000, CRC(ba246ad0) SHA1(ec6bdd4b9cd3007bb66bb8de36d148abb30e7f11) )
	ROM_LOAD( "mk3-u115.bin",  0x1200000, 0x080000, CRC(3ee8b124) SHA1(1523d51e36d3c336b134a562da36a29ba137c0f6) )
	ROM_LOAD( "mk3-u114.bin",  0x1300000, 0x080000, CRC(a8d99922) SHA1(04ff8d85448701070672f44dbf5bcfd744f1bc8a) )
ROM_END


ROM_START( mk3r10 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "umk3-u2.bin",  0x000000, 0x100000, CRC(3838cfe5) SHA1(e3d2901f3bae1362742fc6ee0aa31c9f63b4dfa3) )
	ROM_LOAD16_BYTE( "umk3-u3.bin",  0x200000, 0x100000, CRC(856fe411) SHA1(6165ebecfce7500e948d84492ffa19eed7f47091) )
	ROM_LOAD16_BYTE( "umk3-u4.bin",  0x400000, 0x100000, CRC(428a406f) SHA1(e70ec83cd054de0da1e178720ed0035b8887f797) )
	ROM_LOAD16_BYTE( "umk3-u5.bin",  0x600000, 0x100000, CRC(3b98a09f) SHA1(edf1d02a56dcf3349e6b4bb4097acfe7592305f4) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "mk310u54.bin",  0x00000, 0x80000, CRC(41829228) SHA1(5686b50a08b528d41b28ef578cfb171da9905c45) )
	ROM_LOAD16_BYTE( "mk310u63.bin",  0x00001, 0x80000, CRC(b074e1e8) SHA1(fe1a6f622614b1ebd8edc3edeec442d39ba2924c) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "umk-u133.bin",  0x0000000, 0x100000, CRC(79b94667) SHA1(31bba640c351fdccc6685cadb74dd79a3f910ce8) )
	ROM_LOAD( "umk-u132.bin",  0x0100000, 0x100000, CRC(13e95228) SHA1(405b05f5a5a55667c2be17d4b399129bdacefd90) )
	ROM_LOAD( "umk-u131.bin",  0x0200000, 0x100000, CRC(41001e30) SHA1(2cec91116771951c0380cec5debf4cbb40c14c61) )
	ROM_LOAD( "umk-u130.bin",  0x0300000, 0x100000, CRC(49379dd7) SHA1(e6dfab4e23d9cc38ae56c1bbf10ccd160e8fad5e) )

	ROM_LOAD( "umk-u129.bin",  0x0400000, 0x100000, CRC(a8b41803) SHA1(9697e35e8bb51d6d36b1d7ae47377b446e57682f) )
	ROM_LOAD( "umk-u128.bin",  0x0500000, 0x100000, CRC(b410d72f) SHA1(ac5c1c6f744186540f4ab100d9bd4ce6007e600b) )
	ROM_LOAD( "umk-u127.bin",  0x0600000, 0x100000, CRC(bd985be7) SHA1(f5183abea2e5eb2c2c8cefa72c9ed321679f5128) )
	ROM_LOAD( "umk-u126.bin",  0x0700000, 0x100000, CRC(e7c32cf4) SHA1(94ea7b2eed7dae66f5dd676c20d6b360140e3e0e) )

	ROM_LOAD( "umk-u125.bin",  0x0800000, 0x100000, CRC(9a52227e) SHA1(0474a14fa8dbfea0b0889c1d1756b86391683558) )
	ROM_LOAD( "umk-u124.bin",  0x0900000, 0x100000, CRC(5c750ebc) SHA1(45d68af1a56994376e086d840502453c8d6be700) )
	ROM_LOAD( "umk-u123.bin",  0x0a00000, 0x100000, CRC(f0ab88a8) SHA1(cdc9dc12e162255845c6627b1e35182b7e8502d0) )
	ROM_LOAD( "umk-u122.bin",  0x0b00000, 0x100000, CRC(9b87cdac) SHA1(a5f8db559293978f23e6f105543d8b2e170a2e0d) )

	ROM_LOAD( "mk3-u121.bin",  0x0c00000, 0x100000, CRC(b6c6296a) SHA1(7b92a92d65493bb201daf5ece6f00140f017ac51) )
	ROM_LOAD( "mk3-u120.bin",  0x0d00000, 0x100000, CRC(8d1ccc3b) SHA1(35d91c00113718a08a9d56eb04366f8cf4069ba6) )
	ROM_LOAD( "mk3-u119.bin",  0x0e00000, 0x100000, CRC(63215b59) SHA1(709bce15fba1520bcba40f0a5cb614542f1b460f) )
	ROM_LOAD( "mk3-u118.bin",  0x0f00000, 0x100000, CRC(8b681e34) SHA1(524104ba2eb2deb3cfae9760e6b2125ce6b6633e) )

	ROM_LOAD( "mk3-u117.bin",  0x1000000, 0x080000, CRC(1ab20377) SHA1(0c0d14464d8b23a60e0693669af2ddb82655eff8) )
	ROM_LOAD( "mk3-u116.bin",  0x1100000, 0x080000, CRC(ba246ad0) SHA1(ec6bdd4b9cd3007bb66bb8de36d148abb30e7f11) )
	ROM_LOAD( "mk3-u115.bin",  0x1200000, 0x080000, CRC(3ee8b124) SHA1(1523d51e36d3c336b134a562da36a29ba137c0f6) )
	ROM_LOAD( "mk3-u114.bin",  0x1300000, 0x080000, CRC(a8d99922) SHA1(04ff8d85448701070672f44dbf5bcfd744f1bc8a) )
ROM_END


ROM_START( umk3 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "umk3-u2.bin",  0x000000, 0x100000, CRC(3838cfe5) SHA1(e3d2901f3bae1362742fc6ee0aa31c9f63b4dfa3) )
	ROM_LOAD16_BYTE( "umk3-u3.bin",  0x200000, 0x100000, CRC(856fe411) SHA1(6165ebecfce7500e948d84492ffa19eed7f47091) )
	ROM_LOAD16_BYTE( "umk3-u4.bin",  0x400000, 0x100000, CRC(428a406f) SHA1(e70ec83cd054de0da1e178720ed0035b8887f797) )
	ROM_LOAD16_BYTE( "umk3-u5.bin",  0x600000, 0x100000, CRC(3b98a09f) SHA1(edf1d02a56dcf3349e6b4bb4097acfe7592305f4) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "um312u54.bin",  0x00000, 0x80000, CRC(712b4db6) SHA1(7015a55f3d745c6aeb8630903e2d5cd9554b2766) )
	ROM_LOAD16_BYTE( "um312u63.bin",  0x00001, 0x80000, CRC(6d301faf) SHA1(18a8e29cc3e8ce5cc0e10f8386d43e7f44fd7b75) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "umk-u133.bin",  0x0000000, 0x100000, CRC(79b94667) SHA1(31bba640c351fdccc6685cadb74dd79a3f910ce8) )
	ROM_LOAD( "umk-u132.bin",  0x0100000, 0x100000, CRC(13e95228) SHA1(405b05f5a5a55667c2be17d4b399129bdacefd90) )
	ROM_LOAD( "umk-u131.bin",  0x0200000, 0x100000, CRC(41001e30) SHA1(2cec91116771951c0380cec5debf4cbb40c14c61) )
	ROM_LOAD( "umk-u130.bin",  0x0300000, 0x100000, CRC(49379dd7) SHA1(e6dfab4e23d9cc38ae56c1bbf10ccd160e8fad5e) )

	ROM_LOAD( "umk-u129.bin",  0x0400000, 0x100000, CRC(a8b41803) SHA1(9697e35e8bb51d6d36b1d7ae47377b446e57682f) )
	ROM_LOAD( "umk-u128.bin",  0x0500000, 0x100000, CRC(b410d72f) SHA1(ac5c1c6f744186540f4ab100d9bd4ce6007e600b) )
	ROM_LOAD( "umk-u127.bin",  0x0600000, 0x100000, CRC(bd985be7) SHA1(f5183abea2e5eb2c2c8cefa72c9ed321679f5128) )
	ROM_LOAD( "umk-u126.bin",  0x0700000, 0x100000, CRC(e7c32cf4) SHA1(94ea7b2eed7dae66f5dd676c20d6b360140e3e0e) )

	ROM_LOAD( "umk-u125.bin",  0x0800000, 0x100000, CRC(9a52227e) SHA1(0474a14fa8dbfea0b0889c1d1756b86391683558) )
	ROM_LOAD( "umk-u124.bin",  0x0900000, 0x100000, CRC(5c750ebc) SHA1(45d68af1a56994376e086d840502453c8d6be700) )
	ROM_LOAD( "umk-u123.bin",  0x0a00000, 0x100000, CRC(f0ab88a8) SHA1(cdc9dc12e162255845c6627b1e35182b7e8502d0) )
	ROM_LOAD( "umk-u122.bin",  0x0b00000, 0x100000, CRC(9b87cdac) SHA1(a5f8db559293978f23e6f105543d8b2e170a2e0d) )

	ROM_LOAD( "umk-u121.bin",  0x0c00000, 0x100000, CRC(cc4b95db) SHA1(3d53180eec649e9616c4b87db55573f12d9bfee3) )
	ROM_LOAD( "umk-u120.bin",  0x0d00000, 0x100000, CRC(1c8144cd) SHA1(77cdc1eaf630ccb7233f5532f8b08191d00f0816) )
	ROM_LOAD( "umk-u119.bin",  0x0e00000, 0x100000, CRC(5f10c543) SHA1(24dc83b7aa531ebd399258ffa7b2e028f1c4a28e) )
	ROM_LOAD( "umk-u118.bin",  0x0f00000, 0x100000, CRC(de0c4488) SHA1(227cab34798c440b2a45223567113df5f17d913f) )

	ROM_LOAD( "umk-u113.bin",  0x1400000, 0x100000, CRC(99d74a1e) SHA1(ed3068afa98287ea290d1f537f5009d3b6d683da) )
	ROM_LOAD( "umk-u112.bin",  0x1500000, 0x100000, CRC(b5a46488) SHA1(dbf22e55d200eb9ff550f48b223cf0c6114a9357) )
	ROM_LOAD( "umk-u111.bin",  0x1600000, 0x100000, CRC(a87523c8) SHA1(e70b7599fef82001f762fc2c48f7b85474431ccc) )
	ROM_LOAD( "umk-u110.bin",  0x1700000, 0x100000, CRC(0038f205) SHA1(059c1c71a2d92ee6db36c09831d213a48a7e81d0) )
ROM_END


ROM_START( umk3r11 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "umk3-u2.bin",  0x000000, 0x100000, CRC(3838cfe5) SHA1(e3d2901f3bae1362742fc6ee0aa31c9f63b4dfa3) )
	ROM_LOAD16_BYTE( "umk3-u3.bin",  0x200000, 0x100000, CRC(856fe411) SHA1(6165ebecfce7500e948d84492ffa19eed7f47091) )
	ROM_LOAD16_BYTE( "umk3-u4.bin",  0x400000, 0x100000, CRC(428a406f) SHA1(e70ec83cd054de0da1e178720ed0035b8887f797) )
	ROM_LOAD16_BYTE( "umk3-u5.bin",  0x600000, 0x100000, CRC(3b98a09f) SHA1(edf1d02a56dcf3349e6b4bb4097acfe7592305f4) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "um311u54.bin",  0x00000, 0x80000, CRC(8bb27659) SHA1(a3ffe3d8f21c261b36c7510d620d691a8bbf665b) )
	ROM_LOAD16_BYTE( "um311u63.bin",  0x00001, 0x80000, CRC(ea731783) SHA1(2915626090650c4b5adf5b26e736c3ec91ce81a6) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "umk-u133.bin",  0x0000000, 0x100000, CRC(79b94667) SHA1(31bba640c351fdccc6685cadb74dd79a3f910ce8) )
	ROM_LOAD( "umk-u132.bin",  0x0100000, 0x100000, CRC(13e95228) SHA1(405b05f5a5a55667c2be17d4b399129bdacefd90) )
	ROM_LOAD( "umk-u131.bin",  0x0200000, 0x100000, CRC(41001e30) SHA1(2cec91116771951c0380cec5debf4cbb40c14c61) )
	ROM_LOAD( "umk-u130.bin",  0x0300000, 0x100000, CRC(49379dd7) SHA1(e6dfab4e23d9cc38ae56c1bbf10ccd160e8fad5e) )

	ROM_LOAD( "umk-u129.bin",  0x0400000, 0x100000, CRC(a8b41803) SHA1(9697e35e8bb51d6d36b1d7ae47377b446e57682f) )
	ROM_LOAD( "umk-u128.bin",  0x0500000, 0x100000, CRC(b410d72f) SHA1(ac5c1c6f744186540f4ab100d9bd4ce6007e600b) )
	ROM_LOAD( "umk-u127.bin",  0x0600000, 0x100000, CRC(bd985be7) SHA1(f5183abea2e5eb2c2c8cefa72c9ed321679f5128) )
	ROM_LOAD( "umk-u126.bin",  0x0700000, 0x100000, CRC(e7c32cf4) SHA1(94ea7b2eed7dae66f5dd676c20d6b360140e3e0e) )

	ROM_LOAD( "umk-u125.bin",  0x0800000, 0x100000, CRC(9a52227e) SHA1(0474a14fa8dbfea0b0889c1d1756b86391683558) )
	ROM_LOAD( "umk-u124.bin",  0x0900000, 0x100000, CRC(5c750ebc) SHA1(45d68af1a56994376e086d840502453c8d6be700) )
	ROM_LOAD( "umk-u123.bin",  0x0a00000, 0x100000, CRC(f0ab88a8) SHA1(cdc9dc12e162255845c6627b1e35182b7e8502d0) )
	ROM_LOAD( "umk-u122.bin",  0x0b00000, 0x100000, CRC(9b87cdac) SHA1(a5f8db559293978f23e6f105543d8b2e170a2e0d) )

	ROM_LOAD( "umk-u121.bin",  0x0c00000, 0x100000, CRC(cc4b95db) SHA1(3d53180eec649e9616c4b87db55573f12d9bfee3) )
	ROM_LOAD( "umk-u120.bin",  0x0d00000, 0x100000, CRC(1c8144cd) SHA1(77cdc1eaf630ccb7233f5532f8b08191d00f0816) )
	ROM_LOAD( "umk-u119.bin",  0x0e00000, 0x100000, CRC(5f10c543) SHA1(24dc83b7aa531ebd399258ffa7b2e028f1c4a28e) )
	ROM_LOAD( "umk-u118.bin",  0x0f00000, 0x100000, CRC(de0c4488) SHA1(227cab34798c440b2a45223567113df5f17d913f) )

	ROM_LOAD( "umk-u113.bin",  0x1400000, 0x100000, CRC(99d74a1e) SHA1(ed3068afa98287ea290d1f537f5009d3b6d683da) )
	ROM_LOAD( "umk-u112.bin",  0x1500000, 0x100000, CRC(b5a46488) SHA1(dbf22e55d200eb9ff550f48b223cf0c6114a9357) )
	ROM_LOAD( "umk-u111.bin",  0x1600000, 0x100000, CRC(a87523c8) SHA1(e70b7599fef82001f762fc2c48f7b85474431ccc) )
	ROM_LOAD( "umk-u110.bin",  0x1700000, 0x100000, CRC(0038f205) SHA1(059c1c71a2d92ee6db36c09831d213a48a7e81d0) )
ROM_END


ROM_START( umk3r10 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "umk3-u2.bin",  0x000000, 0x100000, CRC(3838cfe5) SHA1(e3d2901f3bae1362742fc6ee0aa31c9f63b4dfa3) )
	ROM_LOAD16_BYTE( "umk3-u3.bin",  0x200000, 0x100000, CRC(856fe411) SHA1(6165ebecfce7500e948d84492ffa19eed7f47091) )
	ROM_LOAD16_BYTE( "umk3-u4.bin",  0x400000, 0x100000, CRC(428a406f) SHA1(e70ec83cd054de0da1e178720ed0035b8887f797) )
	ROM_LOAD16_BYTE( "umk3-u5.bin",  0x600000, 0x100000, CRC(3b98a09f) SHA1(edf1d02a56dcf3349e6b4bb4097acfe7592305f4) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "um310u54.bin",  0x00000, 0x80000, CRC(dfd735da) SHA1(bcb6d80dbde407d0042ec2f225b2f98740a79203) )
	ROM_LOAD16_BYTE( "um310u63.bin",  0x00001, 0x80000, CRC(2dff0c83) SHA1(8942ffa3addf134085ea8d77d56e82593312e7a5) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "umk-u133.bin",  0x0000000, 0x100000, CRC(79b94667) SHA1(31bba640c351fdccc6685cadb74dd79a3f910ce8) )
	ROM_LOAD( "umk-u132.bin",  0x0100000, 0x100000, CRC(13e95228) SHA1(405b05f5a5a55667c2be17d4b399129bdacefd90) )
	ROM_LOAD( "umk-u131.bin",  0x0200000, 0x100000, CRC(41001e30) SHA1(2cec91116771951c0380cec5debf4cbb40c14c61) )
	ROM_LOAD( "umk-u130.bin",  0x0300000, 0x100000, CRC(49379dd7) SHA1(e6dfab4e23d9cc38ae56c1bbf10ccd160e8fad5e) )

	ROM_LOAD( "umk-u129.bin",  0x0400000, 0x100000, CRC(a8b41803) SHA1(9697e35e8bb51d6d36b1d7ae47377b446e57682f) )
	ROM_LOAD( "umk-u128.bin",  0x0500000, 0x100000, CRC(b410d72f) SHA1(ac5c1c6f744186540f4ab100d9bd4ce6007e600b) )
	ROM_LOAD( "umk-u127.bin",  0x0600000, 0x100000, CRC(bd985be7) SHA1(f5183abea2e5eb2c2c8cefa72c9ed321679f5128) )
	ROM_LOAD( "umk-u126.bin",  0x0700000, 0x100000, CRC(e7c32cf4) SHA1(94ea7b2eed7dae66f5dd676c20d6b360140e3e0e) )

	ROM_LOAD( "umk-u125.bin",  0x0800000, 0x100000, CRC(9a52227e) SHA1(0474a14fa8dbfea0b0889c1d1756b86391683558) )
	ROM_LOAD( "umk-u124.bin",  0x0900000, 0x100000, CRC(5c750ebc) SHA1(45d68af1a56994376e086d840502453c8d6be700) )
	ROM_LOAD( "umk-u123.bin",  0x0a00000, 0x100000, CRC(f0ab88a8) SHA1(cdc9dc12e162255845c6627b1e35182b7e8502d0) )
	ROM_LOAD( "umk-u122.bin",  0x0b00000, 0x100000, CRC(9b87cdac) SHA1(a5f8db559293978f23e6f105543d8b2e170a2e0d) )

	ROM_LOAD( "umk-u121.bin",  0x0c00000, 0x100000, CRC(cc4b95db) SHA1(3d53180eec649e9616c4b87db55573f12d9bfee3) )
	ROM_LOAD( "umk-u120.bin",  0x0d00000, 0x100000, CRC(1c8144cd) SHA1(77cdc1eaf630ccb7233f5532f8b08191d00f0816) )
	ROM_LOAD( "umk-u119.bin",  0x0e00000, 0x100000, CRC(5f10c543) SHA1(24dc83b7aa531ebd399258ffa7b2e028f1c4a28e) )
	ROM_LOAD( "umk-u118.bin",  0x0f00000, 0x100000, CRC(de0c4488) SHA1(227cab34798c440b2a45223567113df5f17d913f) )

	ROM_LOAD( "umk-u113.bin",  0x1400000, 0x100000, CRC(99d74a1e) SHA1(ed3068afa98287ea290d1f537f5009d3b6d683da) )
	ROM_LOAD( "umk-u112.bin",  0x1500000, 0x100000, CRC(b5a46488) SHA1(dbf22e55d200eb9ff550f48b223cf0c6114a9357) )
	ROM_LOAD( "umk-u111.bin",  0x1600000, 0x100000, CRC(a87523c8) SHA1(e70b7599fef82001f762fc2c48f7b85474431ccc) )
	ROM_LOAD( "umk-u110.bin",  0x1700000, 0x100000, CRC(0038f205) SHA1(059c1c71a2d92ee6db36c09831d213a48a7e81d0) )
ROM_END


ROM_START( openice )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "oiceu2.bin",  0x000000, 0x100000, CRC(8adb5aab) SHA1(4c25bc051c90947f3366f83ac5ca8dc78e26b8a4) )
	ROM_LOAD16_BYTE( "oiceu3.bin",  0x200000, 0x100000, CRC(11c61ad6) SHA1(324621d6b486399b6d5ede1fed39d4e448cdeb32) )
	ROM_LOAD16_BYTE( "oiceu4.bin",  0x400000, 0x100000, CRC(04279290) SHA1(daf1e57137ae1c3434194054e69809bfe3ed1fc3) )
	ROM_LOAD16_BYTE( "oiceu5.bin",  0x600000, 0x100000, CRC(e90ad61f) SHA1(59eeabcae7e0e70cdb4472cde64b8a28b07ede98) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "oiceu54.bin",  0x00000, 0x80000, CRC(e4225284) SHA1(d5e267cf35826c106bb0a800363849ed4d489e56) )
	ROM_LOAD16_BYTE( "oiceu63.bin",  0x00001, 0x80000, CRC(97d308a3) SHA1(0a517fab77bc2277884587c7e29e392bb360d27b) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "oiceu133.bin",  0x0000000, 0x100000, CRC(8a81605c) SHA1(cf397b8da242566b21579b90528857ccd2f93141) )
	ROM_LOAD( "oiceu132.bin",  0x0100000, 0x100000, CRC(cfdd6702) SHA1(0198d2cc2de93a8aa345ba0af8d92713d798be8a) )
	ROM_LOAD( "oiceu131.bin",  0x0200000, 0x100000, CRC(cc428eb7) SHA1(ff2403077453f24bd1b176f57b17649b1b64bccf) )
	ROM_LOAD( "oiceu130.bin",  0x0300000, 0x100000, CRC(74c2d50c) SHA1(7880a28b003aa44878384efcb72b98833383f67e) )

	ROM_LOAD( "oiceu129.bin",  0x0400000, 0x100000, CRC(9e2ff012) SHA1(35160ab239f0d8efcb2dc67dee4bd8d204226e3d) )
	ROM_LOAD( "oiceu128.bin",  0x0500000, 0x100000, CRC(35d2e610) SHA1(c4bd18f44592299f120344ecaf1464a8b31d80c8) )
	ROM_LOAD( "oiceu127.bin",  0x0600000, 0x100000, CRC(bcbf19fe) SHA1(e28f0238ef020b75b10318e5c3dd4c5472b3638a) )
	ROM_LOAD( "oiceu126.bin",  0x0700000, 0x100000, CRC(8e3106ae) SHA1(58d1fd097e23578195d28671f22cfa3ed161c0f5) )

	ROM_LOAD( "oiceu125.bin",  0x0800000, 0x100000, CRC(a7b54550) SHA1(83e3627c4e84466ec10023b0e2259ad86b791fd7) )
	ROM_LOAD( "oiceu124.bin",  0x0900000, 0x100000, CRC(7c02cb50) SHA1(92d24bcfd66396c52c823b816118eed39c4ef9cd) )
	ROM_LOAD( "oiceu123.bin",  0x0a00000, 0x100000, CRC(d543bd9d) SHA1(a9ff8589fe185ea058b549c2ed4e71f6c50e9638) )
	ROM_LOAD( "oiceu122.bin",  0x0b00000, 0x100000, CRC(3744d291) SHA1(e4484f377a66c4c64b015ef461419d956b6e23e4) )

	ROM_LOAD( "oiceu121.bin",  0x0c00000, 0x100000, CRC(acd2f7c7) SHA1(82d6f09e63a825b118c36d668427011cd8892eaa) )
	ROM_LOAD( "oiceu120.bin",  0x0d00000, 0x100000, CRC(4295686a) SHA1(2522e57335bb8cca6d76942d2fd62560f88e37a6) )
	ROM_LOAD( "oiceu119.bin",  0x0e00000, 0x100000, CRC(948b9b27) SHA1(62d031410f491d557e27ba055d3db9d36d5a153c) )
	ROM_LOAD( "oiceu118.bin",  0x0f00000, 0x100000, CRC(9eaaf93e) SHA1(56bd881df5282f659ac68ace960a3b085c13dd9d) )
ROM_END


ROM_START( nbahangt )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "mhtu2.bin",  0x000000, 0x100000, CRC(3f0b0d0a) SHA1(e3b8a264686ce7359d86e4926237d8cf17612991) )
	ROM_LOAD16_BYTE( "mhtu3.bin",  0x200000, 0x100000, CRC(ec1db988) SHA1(1cf06d0b75f20ded7db648070e85c056043765bb) )
	ROM_LOAD16_BYTE( "mhtu4.bin",  0x400000, 0x100000, CRC(c7f847a3) SHA1(c50175dffa3563ccd5792c59a6b44523f4014544) )
	ROM_LOAD16_BYTE( "mhtu5.bin",  0x600000, 0x100000, CRC(ef19316a) SHA1(d41ae87ab45630a37c73684de42f7f6e0ed8f13b) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "htime54.bin",  0x00000, 0x80000, CRC(c2875d98) SHA1(3f88f6f5c15ae03bedda39f71a1deaf549a55516) )
	ROM_LOAD16_BYTE( "htime63.bin",  0x00001, 0x80000, CRC(6f4728c3) SHA1(c059f4aa72cc5c3edc41e72428b3ebba97cc9417) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "mhtu133.bin",  0x0000000, 0x100000, CRC(3163feed) SHA1(eb7f128de306933929a0933e36e57760459cb0a1) )
	ROM_LOAD( "mhtu132.bin",  0x0100000, 0x100000, CRC(428eaf44) SHA1(2897efef4ab1653870b5bebb2762ea85549da03a) )
	ROM_LOAD( "mhtu131.bin",  0x0200000, 0x100000, CRC(5f7c5111) SHA1(14337f50b7b98254b54250af00f8a4a46bd7ee8d) )
	ROM_LOAD( "mhtu130.bin",  0x0300000, 0x100000, CRC(c7c0c514) SHA1(49788ff885996d9c5909c0ecebe06b6abd4298ed) )

	ROM_LOAD( "mhtu129.bin",  0x0400000, 0x100000, CRC(b3d0daa0) SHA1(302208c30f2b0c4aead8cf9201ae4c9501f7f952) )
	ROM_LOAD( "mhtu128.bin",  0x0500000, 0x100000, CRC(3704ee69) SHA1(e57846e96380af480fd6851c5359f88d432ac7cc) )
	ROM_LOAD( "mhtu127.bin",  0x0600000, 0x100000, CRC(4ea64d5a) SHA1(e7054c3946898ab0c5b4c27244c2eb6b24eaced7) )
	ROM_LOAD( "mhtu126.bin",  0x0700000, 0x100000, CRC(0c5c19b7) SHA1(802a05f53fcc2827960a63ef5c32a884fc96aaee) )

	ROM_LOAD( "mhtu125.bin",  0x0800000, 0x100000, CRC(46c43d67) SHA1(13a4d924fed51c2db7f750436b0b7c6ef03d36a5) )
	ROM_LOAD( "mhtu124.bin",  0x0900000, 0x100000, CRC(ed495156) SHA1(40f67fb2ccdd185b444f1127b1ec2fa4b493d7d6) )
	ROM_LOAD( "mhtu123.bin",  0x0a00000, 0x100000, CRC(b48aa5da) SHA1(d8ee259a63dd7d997f2b99b73d0f11a277ba961d) )
	ROM_LOAD( "mhtu122.bin",  0x0b00000, 0x100000, CRC(b18cd181) SHA1(d86eb5c81b10ba112ed989ab581683ccb669c3b5) )

	ROM_LOAD( "mhtu121.bin",  0x0c00000, 0x100000, CRC(5acb267a) SHA1(9c7e55991f795f0deae5f5cada3d5f49b7da578e) )
	ROM_LOAD( "mhtu120.bin",  0x0d00000, 0x100000, CRC(28e05f86) SHA1(ab642e7525b8fe55aab79597fcf84c4a8265463f) )
	ROM_LOAD( "mhtu119.bin",  0x0e00000, 0x100000, CRC(b4f604ea) SHA1(3997acfc856eead321e98584f7cb21953c95951a) )
	ROM_LOAD( "mhtu118.bin",  0x0f00000, 0x100000, CRC(a257b973) SHA1(31b4e3cf4e93be673d9b32dc85d7be0edcf6234c) )

	ROM_LOAD( "mhtu113.bin",  0x1400000, 0x100000, CRC(d712a779) SHA1(ca0e25fbb570c28c9ac6674f35050152a9072a5b) )
	ROM_LOAD( "mhtu112.bin",  0x1500000, 0x100000, CRC(644e1bca) SHA1(447cc3b0a16aaff39b6cd095f1c255a91f235f7f) )
	ROM_LOAD( "mhtu111.bin",  0x1600000, 0x100000, CRC(10d3b768) SHA1(e22fcb02a17c78f963ee5d35a38a9ab14f3de450) )
	ROM_LOAD( "mhtu110.bin",  0x1700000, 0x100000, CRC(8575aeb2) SHA1(883acfc45416ab6e1ab77fc897638f89286bea7e) )
ROM_END


ROM_START( nbamht )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "mhtu2.bin",  0x000000, 0x100000, CRC(3f0b0d0a) SHA1(e3b8a264686ce7359d86e4926237d8cf17612991) )
	ROM_LOAD16_BYTE( "mhtu3.bin",  0x200000, 0x100000, CRC(ec1db988) SHA1(1cf06d0b75f20ded7db648070e85c056043765bb) )
	ROM_LOAD16_BYTE( "mhtu4.bin",  0x400000, 0x100000, CRC(c7f847a3) SHA1(c50175dffa3563ccd5792c59a6b44523f4014544) )
	ROM_LOAD16_BYTE( "mhtu5.bin",  0x600000, 0x100000, CRC(ef19316a) SHA1(d41ae87ab45630a37c73684de42f7f6e0ed8f13b) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "mhtu54_v103.bin",  0x00000, 0x80000, CRC(21b0d9e1) SHA1(34fa928bdb222fba1fec2a9f37b853f77922250f) )
	ROM_LOAD16_BYTE( "mhtu63_v103.bin",  0x00001, 0x80000, CRC(c6fdbb97) SHA1(e6cf0c6a94441befdde40b620a182877c11582a5) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "mhtu133.bin",  0x0000000, 0x100000, CRC(3163feed) SHA1(eb7f128de306933929a0933e36e57760459cb0a1) )
	ROM_LOAD( "mhtu132.bin",  0x0100000, 0x100000, CRC(428eaf44) SHA1(2897efef4ab1653870b5bebb2762ea85549da03a) )
	ROM_LOAD( "mhtu131.bin",  0x0200000, 0x100000, CRC(5f7c5111) SHA1(14337f50b7b98254b54250af00f8a4a46bd7ee8d) )
	ROM_LOAD( "mhtu130.bin",  0x0300000, 0x100000, CRC(c7c0c514) SHA1(49788ff885996d9c5909c0ecebe06b6abd4298ed) )

	ROM_LOAD( "mhtu129.bin",  0x0400000, 0x100000, CRC(b3d0daa0) SHA1(302208c30f2b0c4aead8cf9201ae4c9501f7f952) )
	ROM_LOAD( "mhtu128.bin",  0x0500000, 0x100000, CRC(3704ee69) SHA1(e57846e96380af480fd6851c5359f88d432ac7cc) )
	ROM_LOAD( "mhtu127.bin",  0x0600000, 0x100000, CRC(4ea64d5a) SHA1(e7054c3946898ab0c5b4c27244c2eb6b24eaced7) )
	ROM_LOAD( "mhtu126.bin",  0x0700000, 0x100000, CRC(0c5c19b7) SHA1(802a05f53fcc2827960a63ef5c32a884fc96aaee) )

	ROM_LOAD( "mhtu125.bin",  0x0800000, 0x100000, CRC(46c43d67) SHA1(13a4d924fed51c2db7f750436b0b7c6ef03d36a5) )
	ROM_LOAD( "mhtu124.bin",  0x0900000, 0x100000, CRC(ed495156) SHA1(40f67fb2ccdd185b444f1127b1ec2fa4b493d7d6) )
	ROM_LOAD( "mhtu123.bin",  0x0a00000, 0x100000, CRC(b48aa5da) SHA1(d8ee259a63dd7d997f2b99b73d0f11a277ba961d) )
	ROM_LOAD( "mhtu122.bin",  0x0b00000, 0x100000, CRC(b18cd181) SHA1(d86eb5c81b10ba112ed989ab581683ccb669c3b5) )

	ROM_LOAD( "mhtu121.bin",  0x0c00000, 0x100000, CRC(5acb267a) SHA1(9c7e55991f795f0deae5f5cada3d5f49b7da578e) )
	ROM_LOAD( "mhtu120.bin",  0x0d00000, 0x100000, CRC(28e05f86) SHA1(ab642e7525b8fe55aab79597fcf84c4a8265463f) )
	ROM_LOAD( "mhtu119.bin",  0x0e00000, 0x100000, CRC(b4f604ea) SHA1(3997acfc856eead321e98584f7cb21953c95951a) )
	ROM_LOAD( "mhtu118.bin",  0x0f00000, 0x100000, CRC(a257b973) SHA1(31b4e3cf4e93be673d9b32dc85d7be0edcf6234c) )

	ROM_LOAD( "mhtu113.bin",  0x1400000, 0x100000, CRC(d712a779) SHA1(ca0e25fbb570c28c9ac6674f35050152a9072a5b) )
	ROM_LOAD( "mhtu112.bin",  0x1500000, 0x100000, CRC(644e1bca) SHA1(447cc3b0a16aaff39b6cd095f1c255a91f235f7f) )
	ROM_LOAD( "mhtu111.bin",  0x1600000, 0x100000, CRC(10d3b768) SHA1(e22fcb02a17c78f963ee5d35a38a9ab14f3de450) )
	ROM_LOAD( "mhtu110.bin",  0x1700000, 0x100000, CRC(8575aeb2) SHA1(883acfc45416ab6e1ab77fc897638f89286bea7e) )
ROM_END

ROM_START( nbamht1 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "mhtu2.bin",  0x000000, 0x100000, CRC(3f0b0d0a) SHA1(e3b8a264686ce7359d86e4926237d8cf17612991) )
	ROM_LOAD16_BYTE( "mhtu3.bin",  0x200000, 0x100000, CRC(ec1db988) SHA1(1cf06d0b75f20ded7db648070e85c056043765bb) )
	ROM_LOAD16_BYTE( "mhtu4.bin",  0x400000, 0x100000, CRC(c7f847a3) SHA1(c50175dffa3563ccd5792c59a6b44523f4014544) )
	ROM_LOAD16_BYTE( "mhtu5.bin",  0x600000, 0x100000, CRC(ef19316a) SHA1(d41ae87ab45630a37c73684de42f7f6e0ed8f13b) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "mhtu54_v10.bin",  0x00000, 0x80000, CRC(dfb6b3ae) SHA1(1dc59a2d89bf9764a47cebf71b9657c6ae7ce959) )
	ROM_LOAD16_BYTE( "mhtu63_v10.bin",  0x00001, 0x80000, CRC(78da472c) SHA1(b4573ff19dc0d8a99f1bceace872e4999d53317a) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "mhtu133.bin",  0x0000000, 0x100000, CRC(3163feed) SHA1(eb7f128de306933929a0933e36e57760459cb0a1) )
	ROM_LOAD( "mhtu132.bin",  0x0100000, 0x100000, CRC(428eaf44) SHA1(2897efef4ab1653870b5bebb2762ea85549da03a) )
	ROM_LOAD( "mhtu131.bin",  0x0200000, 0x100000, CRC(5f7c5111) SHA1(14337f50b7b98254b54250af00f8a4a46bd7ee8d) )
	ROM_LOAD( "mhtu130.bin",  0x0300000, 0x100000, CRC(c7c0c514) SHA1(49788ff885996d9c5909c0ecebe06b6abd4298ed) )

	ROM_LOAD( "mhtu129.bin",  0x0400000, 0x100000, CRC(b3d0daa0) SHA1(302208c30f2b0c4aead8cf9201ae4c9501f7f952) )
	ROM_LOAD( "mhtu128.bin",  0x0500000, 0x100000, CRC(3704ee69) SHA1(e57846e96380af480fd6851c5359f88d432ac7cc) )
	ROM_LOAD( "mhtu127.bin",  0x0600000, 0x100000, CRC(4ea64d5a) SHA1(e7054c3946898ab0c5b4c27244c2eb6b24eaced7) )
	ROM_LOAD( "mhtu126.bin",  0x0700000, 0x100000, CRC(0c5c19b7) SHA1(802a05f53fcc2827960a63ef5c32a884fc96aaee) )

	ROM_LOAD( "mhtu125.bin",  0x0800000, 0x100000, CRC(46c43d67) SHA1(13a4d924fed51c2db7f750436b0b7c6ef03d36a5) )
	ROM_LOAD( "mhtu124.bin",  0x0900000, 0x100000, CRC(ed495156) SHA1(40f67fb2ccdd185b444f1127b1ec2fa4b493d7d6) )
	ROM_LOAD( "mhtu123.bin",  0x0a00000, 0x100000, CRC(b48aa5da) SHA1(d8ee259a63dd7d997f2b99b73d0f11a277ba961d) )
	ROM_LOAD( "mhtu122.bin",  0x0b00000, 0x100000, CRC(b18cd181) SHA1(d86eb5c81b10ba112ed989ab581683ccb669c3b5) )

	ROM_LOAD( "mhtu121.bin",  0x0c00000, 0x100000, CRC(5acb267a) SHA1(9c7e55991f795f0deae5f5cada3d5f49b7da578e) )
	ROM_LOAD( "mhtu120.bin",  0x0d00000, 0x100000, CRC(28e05f86) SHA1(ab642e7525b8fe55aab79597fcf84c4a8265463f) )
	ROM_LOAD( "mhtu119.bin",  0x0e00000, 0x100000, CRC(b4f604ea) SHA1(3997acfc856eead321e98584f7cb21953c95951a) )
	ROM_LOAD( "mhtu118.bin",  0x0f00000, 0x100000, CRC(a257b973) SHA1(31b4e3cf4e93be673d9b32dc85d7be0edcf6234c) )

	ROM_LOAD( "mhtu113.bin",  0x1400000, 0x100000, CRC(d712a779) SHA1(ca0e25fbb570c28c9ac6674f35050152a9072a5b) )
	ROM_LOAD( "mhtu112.bin",  0x1500000, 0x100000, CRC(644e1bca) SHA1(447cc3b0a16aaff39b6cd095f1c255a91f235f7f) )
	ROM_LOAD( "mhtu111.bin",  0x1600000, 0x100000, CRC(10d3b768) SHA1(e22fcb02a17c78f963ee5d35a38a9ab14f3de450) )
	ROM_LOAD( "mhtu110.bin",  0x1700000, 0x100000, CRC(8575aeb2) SHA1(883acfc45416ab6e1ab77fc897638f89286bea7e) )
ROM_END


ROM_START( rmpgwt )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "rwt.2",  0x000000, 0x100000, CRC(0e82f83d) SHA1(215eebb6c229ef9ad0fcbcbc6e4e07300c05654f) )
	ROM_LOAD16_BYTE( "rwt.3",  0x200000, 0x100000, CRC(3ff54d15) SHA1(827805602091313ec68ea1bccf667bd3b3fc6b8b) )
	ROM_LOAD16_BYTE( "rwt.4",  0x400000, 0x100000, CRC(5c7f5656) SHA1(6c9d692bad539fec8b5aa0bfb56de3ef3719c68a) )
	ROM_LOAD16_BYTE( "rwt.5",  0x600000, 0x100000, CRC(fd9aaf24) SHA1(d60dc076e72618c99ecac9d081d8c49d337b90c7) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "rwtr13.54",  0x00000, 0x80000, CRC(2a8f6e1e) SHA1(7a87ad37fa1d1228c4cdd4704ff0aee42e9c86cb) )
	ROM_LOAD16_BYTE( "rwtr13.63",  0x00001, 0x80000, CRC(403ae41e) SHA1(c08d9352efe63849f5d10c1bd1efe2b9dd7382e0) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "rwt.133",  0x0000000, 0x100000, CRC(5b5ac449) SHA1(1c01dde9a9dbd9f4a6cd30aea9f6410cab13c2c9) )
	ROM_LOAD( "rwt.132",  0x0100000, 0x100000, CRC(7b3f09c6) SHA1(477658481ee96d5ce462d5e198d80faff4d4352c) )
	ROM_LOAD( "rwt.131",  0x0200000, 0x100000, CRC(fdecf12e) SHA1(bcbd29009dabed484e2357dc75c38c7d7bade251) )
	ROM_LOAD( "rwt.130",  0x0300000, 0x100000, CRC(4a983b05) SHA1(022753e6fa3b3d74eff5b2ec835b07787f5473ab) )

	ROM_LOAD( "rwt.129",  0x0400000, 0x100000, CRC(dc495c6e) SHA1(7ec80d293f06a013c4f95bfb3fafe5b4a71cf170) )
	ROM_LOAD( "rwt.128",  0x0500000, 0x100000, CRC(5545503d) SHA1(75298ca742339d70f86460645a6145070737a883) )
	ROM_LOAD( "rwt.127",  0x0600000, 0x100000, CRC(6e1756ba) SHA1(6e88ee4f239d41430cdcacfbfe9dfe9e75c5e4ed) )
	ROM_LOAD( "rwt.126",  0x0700000, 0x100000, CRC(c300eb1b) SHA1(e2fffa31c773737d5f5f3f053f9afee4690fcd3d) )

	ROM_LOAD( "rwt.125",  0x0800000, 0x100000, CRC(7369bf5d) SHA1(edd84d4119d63263bc65adf953370a6e36c797d3) )
	ROM_LOAD( "rwt.124",  0x0900000, 0x100000, CRC(c0bf88c8) SHA1(5ea348bcd208c1c1f5f5943f1e1ef81d25d2b95d) )
	ROM_LOAD( "rwt.123",  0x0a00000, 0x100000, CRC(ac4c712a) SHA1(ccf40d004ddf1b62870b99da359b00d9fd702944) )
	ROM_LOAD( "rwt.122",  0x0b00000, 0x100000, CRC(609862a2) SHA1(366c70d5de9135c28934ed6ccf4f373c5a76c748) )

	ROM_LOAD( "rwt.121",  0x0c00000, 0x100000, CRC(f65119b7) SHA1(4c3110f3be370cad0b031a7aa8605559bdd14842) )
	ROM_LOAD( "rwt.120",  0x0d00000, 0x100000, CRC(6d643dee) SHA1(923ad01a0eb5a26d2a913f09fde254605113c868) )
	ROM_LOAD( "rwt.119",  0x0e00000, 0x100000, CRC(4e49c133) SHA1(0f83b0d645286f6c6196bcf316ac010378b96c43) )
	ROM_LOAD( "rwt.118",  0x0f00000, 0x100000, CRC(43a6f51e) SHA1(cb9d698a6a2ab9134339264e851b57e599db135c) )
ROM_END


ROM_START( rmpgwt11 )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "rwt.2",  0x000000, 0x100000, CRC(0e82f83d) SHA1(215eebb6c229ef9ad0fcbcbc6e4e07300c05654f) )
	ROM_LOAD16_BYTE( "rwt.3",  0x200000, 0x100000, CRC(3ff54d15) SHA1(827805602091313ec68ea1bccf667bd3b3fc6b8b) )
	ROM_LOAD16_BYTE( "rwt.4",  0x400000, 0x100000, CRC(5c7f5656) SHA1(6c9d692bad539fec8b5aa0bfb56de3ef3719c68a) )
	ROM_LOAD16_BYTE( "rwt.5",  0x600000, 0x100000, CRC(fd9aaf24) SHA1(d60dc076e72618c99ecac9d081d8c49d337b90c7) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "rwtr11.54",  0x00000, 0x80000, CRC(3aa514eb) SHA1(4ed8db55f257da6d872586d0f9f0cdf1c30e0d22) )
	ROM_LOAD16_BYTE( "rwtr11.63",  0x00001, 0x80000, CRC(031c908f) SHA1(531669b13c33921ff199be1e841dd337c86fec50) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "rwt.133",  0x0000000, 0x100000, CRC(5b5ac449) SHA1(1c01dde9a9dbd9f4a6cd30aea9f6410cab13c2c9) )
	ROM_LOAD( "rwt.132",  0x0100000, 0x100000, CRC(7b3f09c6) SHA1(477658481ee96d5ce462d5e198d80faff4d4352c) )
	ROM_LOAD( "rwt.131",  0x0200000, 0x100000, CRC(fdecf12e) SHA1(bcbd29009dabed484e2357dc75c38c7d7bade251) )
	ROM_LOAD( "rwt.130",  0x0300000, 0x100000, CRC(4a983b05) SHA1(022753e6fa3b3d74eff5b2ec835b07787f5473ab) )

	ROM_LOAD( "rwt.129",  0x0400000, 0x100000, CRC(dc495c6e) SHA1(7ec80d293f06a013c4f95bfb3fafe5b4a71cf170) )
	ROM_LOAD( "rwt.128",  0x0500000, 0x100000, CRC(5545503d) SHA1(75298ca742339d70f86460645a6145070737a883) )
	ROM_LOAD( "rwt.127",  0x0600000, 0x100000, CRC(6e1756ba) SHA1(6e88ee4f239d41430cdcacfbfe9dfe9e75c5e4ed) )
	ROM_LOAD( "rwt.126",  0x0700000, 0x100000, CRC(c300eb1b) SHA1(e2fffa31c773737d5f5f3f053f9afee4690fcd3d) )

	ROM_LOAD( "rwt.125",  0x0800000, 0x100000, CRC(7369bf5d) SHA1(edd84d4119d63263bc65adf953370a6e36c797d3) )
	ROM_LOAD( "rwt.124",  0x0900000, 0x100000, CRC(c0bf88c8) SHA1(5ea348bcd208c1c1f5f5943f1e1ef81d25d2b95d) )
	ROM_LOAD( "rwt.123",  0x0a00000, 0x100000, CRC(ac4c712a) SHA1(ccf40d004ddf1b62870b99da359b00d9fd702944) )
	ROM_LOAD( "rwt.122",  0x0b00000, 0x100000, CRC(609862a2) SHA1(366c70d5de9135c28934ed6ccf4f373c5a76c748) )

	ROM_LOAD( "rwt.121",  0x0c00000, 0x100000, CRC(f65119b7) SHA1(4c3110f3be370cad0b031a7aa8605559bdd14842) )
	ROM_LOAD( "rwt.120",  0x0d00000, 0x100000, CRC(6d643dee) SHA1(923ad01a0eb5a26d2a913f09fde254605113c868) )
	ROM_LOAD( "rwt.119",  0x0e00000, 0x100000, CRC(4e49c133) SHA1(0f83b0d645286f6c6196bcf316ac010378b96c43) )
	ROM_LOAD( "rwt.118",  0x0f00000, 0x100000, CRC(43a6f51e) SHA1(cb9d698a6a2ab9134339264e851b57e599db135c) )
ROM_END


ROM_START( wwfmania )
	ROM_REGION16_LE( 0x800000, REGION_SOUND1, ROMREGION_ERASEFF )	/* sound data */
	ROM_LOAD16_BYTE( "wwf.2",  0x000000, 0x100000, CRC(a9acb250) SHA1(c1a7773ffdb86dc2c1c90c220482ed6330fcbb55) )
	ROM_LOAD16_BYTE( "wwf.3",  0x200000, 0x100000, CRC(9442b6c9) SHA1(1f887c05ab9ca99078be584d7e9e6c59c8ec1818) )
	ROM_LOAD16_BYTE( "wwf.4",  0x400000, 0x100000, CRC(cee78fac) SHA1(c37d3b4aef47dc80d864497b3013f03220d45482) )
	ROM_LOAD16_BYTE( "wwf.5",  0x600000, 0x100000, CRC(5b31fd40) SHA1(35dcf19b223029e17616357d29dd04bbfeb83491) )

	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )	/* 34010 code */
	ROM_LOAD16_BYTE( "wwf.54",  0x00000, 0x80000, CRC(eeb7bf58) SHA1(d93df59aed1672ab38af231d909d9df1a8e30f44) )
	ROM_LOAD16_BYTE( "wwf.63",  0x00001, 0x80000, CRC(09759529) SHA1(cf548ff199428a93b9bc5f4fc1347c4a3cbdf106) )

	ROM_REGION( 0x2000000, REGION_GFX1, 0 )
	ROM_LOAD( "wwf.133",  0x0000000, 0x100000, CRC(5e1b1e3d) SHA1(55f54e4b0dc775058699b1c0abdd7241ffca0e76) )
	ROM_LOAD( "wwf.132",  0x0100000, 0x100000, CRC(5943b3b2) SHA1(8ba0b20e7993769736c961d0fda97b2850d1446b) )
	ROM_LOAD( "wwf.131",  0x0200000, 0x100000, CRC(0815db22) SHA1(ebd6a8c4f0e8d979af7f173b3f139d91e4857f6b) )
	ROM_LOAD( "wwf.130",  0x0300000, 0x100000, CRC(9ee9a145) SHA1(caeb8506e1414e8c58e3031d4a2e0619ef3922b7) )

	ROM_LOAD( "wwf.129",  0x0400000, 0x100000, CRC(c644c2f4) SHA1(9094452eb37ec92932109ab2b209e12074111dd7) )
	ROM_LOAD( "wwf.128",  0x0500000, 0x100000, CRC(fcda4e9a) SHA1(a05a12f606632034eae662cccfee5aaaffe0348b) )
	ROM_LOAD( "wwf.127",  0x0600000, 0x100000, CRC(45be7428) SHA1(a5d3e37c64cac03139028fe998494b76e6b6a7ae) )
	ROM_LOAD( "wwf.126",  0x0700000, 0x100000, CRC(eaa276a8) SHA1(d0c2f4d4409830355c6e112e3eafb4d3a1b8c22e) )

	ROM_LOAD( "wwf.125",  0x0800000, 0x100000, CRC(a19ebeed) SHA1(cf51bca29fd39c6189c2b431eb718a6341781d1f) )
	ROM_LOAD( "wwf.124",  0x0900000, 0x100000, CRC(dc7d3dbb) SHA1(8982d9a1babce57ae7465bce3f4863dd336c20ac) )
	ROM_LOAD( "wwf.123",  0x0a00000, 0x100000, CRC(e0ade56f) SHA1(a15c672a45f39c0232d678e71380d4f58c4659ae) )
	ROM_LOAD( "wwf.122",  0x0b00000, 0x100000, CRC(2800c78d) SHA1(8012785f1c1eaf8d533a98e0a521a5d31efc7a42) )

	ROM_LOAD( "wwf.121",  0x0c00000, 0x100000, CRC(a28ffcba) SHA1(f66be0793b12a7f04e32d3db8311d1f33b0c3fbe) )
	ROM_LOAD( "wwf.120",  0x0d00000, 0x100000, CRC(3a05d371) SHA1(4ed73e1c06ea7bd33e6c72a6a752960ba55d1975) )
	ROM_LOAD( "wwf.119",  0x0e00000, 0x100000, CRC(97ffa659) SHA1(986f8ec57085b808d33c85ed55b35a5e1cadf3b6) )
	ROM_LOAD( "wwf.118",  0x0f00000, 0x100000, CRC(46668e97) SHA1(282ca2e561f7553717d60b5a745f8e3fc1bda610) )
ROM_END



/*************************************
 *
 *  Game drivers
 *
 *************************************/

GAME( 1994, mk3,     0,         wunit, mk3,     mk3,     ROT0, "Midway", "Mortal Kombat 3 (rev 2.1)", 0 )
GAME( 1994, mk3r20,  mk3,       wunit, mk3,     mk3r20,  ROT0, "Midway", "Mortal Kombat 3 (rev 2.0)", 0 )
GAME( 1994, mk3r10,  mk3,       wunit, mk3,     mk3r10,  ROT0, "Midway", "Mortal Kombat 3 (rev 1.0)", 0 )
GAME( 1994, umk3,    mk3,       wunit, mk3,     umk3,    ROT0, "Midway", "Ultimate Mortal Kombat 3 (rev 1.2)", 0 )
GAME( 1994, umk3r11, mk3,       wunit, mk3,     umk3r11, ROT0, "Midway", "Ultimate Mortal Kombat 3 (rev 1.1)", 0 )
GAME( 1994, umk3r10, mk3,       wunit, mk3,     umk3r11, ROT0, "Midway", "Ultimate Mortal Kombat 3 (rev 1.0)", 0 )

GAME( 1995, wwfmania,0,         wunit, wwfmania,wwfmania,ROT0, "Midway", "WWF: Wrestlemania (rev 1.30 08/10/95)", 0 )
GAME( 1995, openice, 0,         wunit, openice, openice, ROT0, "Midway", "2 On 2 Open Ice Challenge (rev 1.21)", 0 )
GAME( 1996, nbahangt,0,         wunit, nbahangt,nbahangt,ROT0, "Midway", "NBA Hangtime (rev L1.1 04/16/96)", 0 )
GAME( 1996, nbamht,  nbahangt,  wunit, nbahangt,nbahangt,ROT0, "Midway", "NBA Maximum Hangtime (rev 1.03 06/09/97)", 0 )
GAME( 1996, nbamht1, nbahangt,  wunit, nbahangt,nbahangt,ROT0, "Midway", "NBA Maximum Hangtime (rev 1.0 11/08/96)", 0 )
GAME( 1997, rmpgwt,  0,         wunit, rmpgwt,  rmpgwt,  ROT0, "Midway", "Rampage: World Tour (rev 1.3)", 0 )
GAME( 1997, rmpgwt11,rmpgwt,    wunit, rmpgwt,  rmpgwt,  ROT0, "Midway", "Rampage: World Tour (rev 1.1)", 0 )
