/*
    GQ -- a GTK-based LDAP client
    Copyright (C) 1998-2001 Bert Vermeulen

    This program is released under the Gnu General Public License with
    the additional exemption that compiling, linking, and/or using
    OpenSSL is allowed.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#include <config.h>

#include <stdio.h>
#include <string.h>

#if defined(HAVE_LIBCRYPTO)
#include <openssl/rand.h>
#include <openssl/des.h>
#include <openssl/md5.h>
#include <openssl/sha.h>
#endif /* HAVE_LIBCRYPTO */

#include <glib.h>
#include <gtk/gtk.h>

#if defined(HAVE_ICONV_H)
#include <iconv.h>
#endif /* HAVE_ICONV_H */

#include "common.h"
#include "util.h"
#include "formfill.h"
#include "ldif.h"
#include "encode.h"


#define LDAP_CODESET "UTF-8"
#define GQ_CODESET "ISO-8859-1"

extern struct tokenlist cryptmap[];


#if defined(HAVE_LIBCRYPTO)

/*
 * iterate through the hash's formlist and find any password
 * entries that need an encoding applied (and apply it)
 */
void encode_password(GHashTable *hash)
{
     GList *formlist;
     struct formfill *form;
     char *password;

     formlist = g_hash_table_lookup(hash, "formlist");

     while(formlist) {
	  form = (struct formfill *) formlist->data;
	  if(form->displaytype == DISPLAYTYPE_PASSWORD && form->flags & ENCODING_MASK && form->values) {
	       password = form->values->data;
	       if(password[0] != '{') {
		    switch(form->flags & ENCODING_MASK) {
		    case FLAG_ENCODE_CRYPT:
			 encode_password_crypt(form);
			 break;
		    case FLAG_ENCODE_MD5:
			 encode_password_md5(form);
			 break;
		    case FLAG_ENCODE_SHA:
			 encode_password_sha1(form);
			 break;
		    default:
                         /*
                          * Something's very wrong!  This should complain
                          * and exit.  (syl)
                          */
                         break;
		    }
	       }

	  }

	  formlist = formlist->next;
     }

}


void encode_password_crypt(struct formfill *form)
{
     GString *salt;
     unsigned char *password, rand[16], password_out[64], *crypt_out;

     password = form->values->data;
     salt = g_string_sized_new(32);
     RAND_pseudo_bytes(rand, 8);
     b64_encode(salt, rand, 8);
     /* crypt(3) says [a-zA-Z0-9./] while base64 has [a-zA-Z0-9+/] */
     if(salt->str[0] == '+') salt->str[0] = '.';
     if(salt->str[1] == '+') salt->str[1] = '.';
     salt->str[2] = 0;
     crypt_out = des_crypt(password, salt->str);
     strcpy(password_out, "{crypt}");
     strncat(password_out, crypt_out, 63);
     g_string_free(salt, TRUE);
     password_out[63] = 0;
     g_free(form->values->data);
     form->values->data = g_strdup(password_out);

}


void encode_password_md5(struct formfill *form)
{
     int i;
     unsigned char *password, password_out[64], md5_out[MD5_DIGEST_LENGTH];

     password = form->values->data;
     MD5(password, strlen(password), md5_out);
     strcpy(password_out, "{md5}");
     for(i = 0; i < MD5_DIGEST_LENGTH; i++)
	  sprintf(password_out + strlen(password_out), "%.02x", md5_out[i]);
     g_free(form->values->data);
     form->values->data = g_strdup(password_out);

}


void encode_password_sha1(struct formfill *form)
{
     int i;
     unsigned char *password, password_out[64], sha1_out[SHA_DIGEST_LENGTH];

     password = form->values->data;
     SHA1(password, strlen(password), sha1_out);
     strcpy(password_out, "{sha}");
     for(i = 0; i < SHA_DIGEST_LENGTH; i++)
	  sprintf(password_out + strlen(password_out), "%.02x", sha1_out[i]);
     g_free(form->values->data);
     form->values->data = g_strdup(password_out);

}
#endif /* HAVE_LIBCRYPTO */


const char *decode_string(char *native_string, const gchar *ldap_string)
{
#if defined(HAVE_ICONV_H)
     char *in, *out;
     size_t len, outlen;
     iconv_t conv;
#endif /* HAVE_ICONV_H */

#if defined(HAVE_ICONV_H)
     in = (char *) ldap_string;
     out = native_string;
     len = strlen(in);
     outlen = len;
     conv = iconv_open(GQ_CODESET, LDAP_CODESET);
  
     while(len > 0 && outlen > 0) {
	  if(iconv(conv, &in, &len, &out, &outlen) != 0) {
	       in++;		/* *** */
	       len--;
	  }
     }
     iconv_close(conv);
     *out = '\0';
#else /* HAVE_ICONV_H */
     strcpy(native_string, ldap_string);
#endif /* HAVE_ICONV_H */

#ifdef DEBUG
     fprintf(stderr, "decode_string \"%s\" -> \"%s\"\n", 
	     ldap_string, native_string);
#endif /* DEBUG */

     return native_string;
}


const gchar *encode_string(gchar *ldap_string, const gchar *native_string)
{
#if defined(HAVE_ICONV_H)
     char *in, *out;
     size_t len, outlen;
     iconv_t conv;
#endif /* HAVE_ICONV_H */

#if defined(HAVE_ICONV_H)
     in = (char *) native_string;
     out = ldap_string;
     len = strlen(in);
     outlen = len * 2; /* Worst case */
     conv = iconv_open(LDAP_CODESET, GQ_CODESET);

     while(len > 0 && outlen > 0) {
	  if(iconv(conv, &in, &len, &out, &outlen) != 0) {
	       in++;		/* *** */
	       len--;
	  }
     }
     iconv_close(conv);
     *out = '\0';
#else /* HAVE_ICONV_H */
     strcpy(ldap_string, native_string);
#endif /* HAVE_ICONV_H */
  
#ifdef DEBUG
     fprintf(stderr, "encode_string \"%s\" -> \"%s\"\n", 
	     native_string, ldap_string);
#endif /* DEBUG */

     return ldap_string;
}


gchar *decoded_string(const gchar *string)
{
     char *native_string;

     native_string = calloc(strlen(string) + 1, sizeof(char));
  
     decode_string(native_string, string);
  
     return native_string;
}


gchar *encoded_string(const gchar *string)
{
     char *ldap_string;

     ldap_string = calloc(2 * strlen(string) + 1, sizeof(char));
  
     encode_string(ldap_string, string);

     return ldap_string;
}

