/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  1/23/2001 Hajimu UMEMOTO improvements with read_freebsd_cpu() and
|			register_freebsd_cpus().
| 10/12/2000  NetBSD code contributed by Anthony Mallet <metall@ficus.yi.org>
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "cpu_disk.h"


GList	*cpu_mon_list;

void	(*read_system_cpu_info)();

static gint		n_cpus;
static gint		smp_mode;
gint			smp_cpus;


/* ====== System dependent interface ====================================== */

/* ----- FreeBSD ---------------------------------------------------- */

#if defined(__FreeBSD__) || defined(__OpenBSD__)
#include <sys/dkstat.h>
#include <kvm.h>

static struct nlist nl[] = {
#define N_CP_TIME	0
	{ "_cp_time" },
	{ "" }
};

extern	kvm_t	*kvmd;

#if defined(__FreeBSD__)
#include <sys/param.h>
#include <sys/sysctl.h>

static int	oid_cp_time[CTL_MAXNAME + 2];
static size_t	oid_cp_time_len = sizeof(oid_cp_time);
static gint	have_cp_time;

void
read_freebsd_cpu()
	{
	static gint	data_read_tick = -1;
	long		cp_time[CPUSTATES];
	int		len = sizeof(cp_time);
	CpuMon		*cpu;

	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (have_cp_time)
		{
		if (sysctl(oid_cp_time, oid_cp_time_len,
			   cp_time, &len, 0, 0) < 0)
			return;
		}
	else
		{
		if (kvmd == NULL)
			return;
		if (nl[0].n_type == 0)
			if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
				return;
		if (kvm_read(kvmd, nl[N_CP_TIME].n_value, (char *)&cp_time,
			     sizeof(cp_time)) != sizeof(cp_time))
			return;
		}

	/* Currently, SMP is not supported */
	cpu = (CpuMon *) cpu_mon_list->data;
	cpu->user = cp_time[CP_USER];
	cpu->nice = cp_time[CP_NICE];
	cpu->sys = cp_time[CP_SYS];
	cpu->idle = cp_time[CP_IDLE];
	}

static void register_glibtop_cpus();

static void
register_freebsd_cpus()
	{
	static int	oid_name2oid[2] = { 0, 3 };
	static char	*name = "kern.cp_time";

	register_glibtop_cpus();

	if (sysctl(oid_name2oid, 2, oid_cp_time, &oid_cp_time_len,
		   (void *)name, strlen(name)) < 0)
		return;
	oid_cp_time_len /= sizeof(int);
	++have_cp_time;
	}
#else
void
read_openbsd_cpu()
	{
	long		cp_time[CPUSTATES];
	CpuMon		*cpu;
	static gint	data_read_tick	= -1;

	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (kvmd == NULL)
		return;
	if (nl[0].n_type == 0)
		if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
			return;
	if (kvm_read(kvmd, nl[N_CP_TIME].n_value,
		     (char *)&cp_time, sizeof(cp_time)) != sizeof(cp_time))
		return;

	/* Currently, SMP is not supported */
	cpu = (CpuMon *) cpu_mon_list->data;
	cpu->user = cp_time[CP_USER];
	cpu->nice = cp_time[CP_NICE];
	cpu->sys = cp_time[CP_SYS];
	cpu->idle = cp_time[CP_IDLE];
	}
#endif
#endif


/* ----- Linux ---------------------------------------------------- */
#if defined(__linux__)
#define	PROC_STAT_FILE	"/proc/stat"

static gulong	swapin,
				swapout;

  /* CPU, and Disk monitors call this in their update routine. 
  |  Whoever calls it first will read the data for everyone.
  |
  | /proc/stat has cpu entries like:
  |		cpu		total_user	total_nice	total_sys	total_idle
  |		cpu0	cpu0_user	cpu0_nice	cpu0_sys	cpu0_idle
  |			...
  |		cpuN	cpuN_user	cpuN_nice	cpuN_sys	cpuN_idle
  |  where ticks for cpu are jiffies * smp_num_cpus
  |  and ticks for cpu[i] are jiffies (1/CLK_TCK)
  */
void
read_proc_stat()
	{
	FILE		*f;
	GList		*list;
	CpuMon		*cpu;
	DiskMon		*disk;
	gchar		*item, *arg;
	gchar		buf[1024];
	static gint	data_read_tick	= -1;

	if (   data_read_tick == GK.timer_ticks		/* Only one read per tick */
		|| (f = fopen(PROC_STAT_FILE, "r")) == NULL
	   )
		return;
	data_read_tick = GK.timer_ticks;

	composite_disk->rblk = 0;
	composite_disk->wblk = 0;
	while ((fgets(buf, sizeof(buf), f)) != NULL)
		{
		item = strtok(buf, " \t\n");
		arg = strtok(NULL, "\n");
		if (item == NULL || arg == NULL)
			continue;
		if (item[0] == 'c' && item[1] == 'p')
			{
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, item) == 0)
					{
					sscanf(arg,"%lu %lu %lu %lu", &cpu->user, &cpu->nice,
								&cpu->sys, &cpu->idle);
					break;
					}
				}
			}
		else if (strcmp("disk_rblk", item) == 0)	/* Pre kernel 2.4 format */
			{
			for (list = disk_mon_list->next; list; list = list->next)
				{
				disk = (DiskMon *) list->data;
				disk->rblk = strtoul(arg, &arg, 0);
				composite_disk->rblk += disk->rblk;
				}
			}
		else if (strcmp("disk_wblk", item) == 0)	/* Pre kernel 2.4 format */
			{
			for (list = disk_mon_list->next; list; list = list->next)
				{
				disk = (DiskMon *) list->data;
				disk->wblk = strtoul (arg, &arg, 0);
				composite_disk->wblk += disk->wblk;
				}
			}
		/* Read swap data for the meminfo monitor
		*/
		else if (strcmp("swap", item) == 0)
			sscanf(arg, "%lu %lu", &swapin, &swapout);

		else if (strcmp("disk_io:", item) == 0)	/* Kernel 2.4 format */
			{
			gint	major, i_disk, n;
			gulong	rblk, wblk, rb1, rb2, wb1, wb2;
			DiskMon	*disk;

			item = strtok(arg, " \t\n");
			while (item)
				{
				/* It may evolve - won't know till the 2.4 dust settles */
				n = sscanf(item, "(%d,%d):(%*d,%lu,%lu,%lu,%lu)",
						&major, &i_disk, &rb1, &rb2, &wb1, &wb2);
				if (n == 6)	/* patched as of 2.4.0-test1-ac9 */
					{		/* (major,disk):(total,rio,rblk,wio,wblk) */
					rblk = rb2;
					wblk = wb2;
					}
				else	/* 2.3.99-pre8 to 2.4.0-testX */
					{		/* (major,disk):(rio,rblk,wio,wblk) */
					rblk = rb1;
					wblk = wb1;
					}
				if ((disk = lookup_disk_by_device(major, i_disk)) != NULL)
					{
					disk->rblk = rblk;
					disk->wblk = wblk;
					}
				composite_disk->rblk += rblk;
				composite_disk->wblk += wblk;
				item = strtok(NULL, " \t\n");
				}
			break;
			}
		}
	fclose(f);
	}

void
read_stat_swap(gulong *sin, gulong *sout)
	{
	*sin = swapin;
	*sout = swapout;
	}

static void
register_stat_cpus()
	{
	FILE	*f;
	CpuMon	*cpu;
	GList	*list;
	gchar	buf[1024], *s;

	if ((f = fopen(PROC_STAT_FILE, "r")) == NULL)
		return;

	while (fgets(buf, sizeof(buf), f))
		{
		if (strncmp(buf, "cpu", 3) != 0)
			continue;
		s = strtok(buf, " \t\n");
		cpu = g_new0(CpuMon, 1);
		cpu->name = g_strdup(s);
		cpu_mon_list = g_list_append(cpu_mon_list, cpu);
		++n_cpus;
		}
	fclose(f);

	/* There can be cpu and cpu0 only (single cpu machines with kernel 2.4
	|  or 2.2 compiled for SMP), and in this case, eliminate cpu0.
	*/
	smp_cpus = n_cpus - 1;
	if (smp_cpus == 1)	/* cpu and cpu0 but no cpu1 (not really SMP)	*/
		{
		smp_cpus = 0;
		n_cpus = 1;
		list = cpu_mon_list->next;	/* cpu0 */
		cpu = (CpuMon *) list->data;
		g_free(cpu->name);
		g_free(cpu);
		cpu_mon_list = g_list_remove_link(cpu_mon_list, list);
		}
	if (smp_cpus)			/* The first one is the composite cpu	*/
		((CpuMon *) cpu_mon_list->data)->is_composite = TRUE;
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__)

#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/sched.h>

void
read_netbsd_cpu()
{
   static gint data_read_tick = -1;
   static int mib[] = { CTL_KERN, KERN_CP_TIME };
   u_int64_t cp_time[CPUSTATES];
   int len = sizeof(cp_time);
   CpuMon *cpu;

   if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
      return;
   data_read_tick = GK.timer_ticks;

   if (sysctl(mib, 2, cp_time, &len, NULL, 0) < 0) return;

   /* Currently, SMP is not supported */
   cpu = (CpuMon *) cpu_mon_list->data;
   cpu->user = cp_time[CP_USER];
   cpu->nice = cp_time[CP_NICE];
   cpu->sys = cp_time[CP_SYS];
   cpu->idle = cp_time[CP_IDLE];
}

#endif /* __NetBSD__ */


/* ----- Others ------------------------------------------------------- */
#if defined(USE_LIBGTOP)

#include <glibtop/cpu.h>

static void
read_glibtop_cpu()
	{
	CpuMon		*cpu;
	GList		*list;
	glibtop_cpu	glt_cpu;

	for (list = cpu_mon_list; list; list = list->next)	/* Only one! */
		{
		glibtop_get_cpu(&glt_cpu);
		cpu = (CpuMon *) list->data;
		cpu->user = (gulong) glt_cpu.user;
		cpu->nice = (gulong) glt_cpu.nice;
		cpu->sys  = (gulong) glt_cpu.sys;
		cpu->idle = (gulong) glt_cpu.idle;
		}
	}
#endif


#if !defined(__linux__)
static void
register_glibtop_cpus()
	{
	CpuMon	*cpu;

	cpu = g_new0(CpuMon, 1);
	cpu->name = g_strdup("cpu");
	cpu_mon_list = g_list_append(cpu_mon_list, cpu);
	n_cpus = 1;
	smp_cpus = 0;
	}
#endif

/* ----- Pick a system interface ----------------------------------------- */

static gint
setup_cpu_interface()
	{
#if defined(__FreeBSD__)
	register_freebsd_cpus();
	read_system_cpu_info = read_freebsd_cpu;
#elif defined(__linux__)
	register_stat_cpus();
	read_system_cpu_info = read_proc_stat;
#elif defined(__NetBSD__)
	register_glibtop_cpus();
	read_system_cpu_info = read_netbsd_cpu;
#elif defined(__OpenBSD__)
	register_glibtop_cpus();
	read_system_cpu_info = read_openbsd_cpu;
#else
	register_glibtop_cpus();
	read_system_cpu_info = read_glibtop_cpu;
#endif
	return TRUE;
	}

/* ======================================================================== */

static gboolean	cpu_enabled  = TRUE;
static gint		add_nice_mode;
static gint		ascent;
static gint		style_id;
static gint		mapped_sensors;
static gint		sensor_separate_mode;


static void
draw_cpu_extra(Chart *cp, unsigned long l, unsigned long total)
	{
	TextStyle	*ts;
	gint		n, y;
	gchar		buf[32];

	if (!cp || !total)
		return;
	ts = gkrellm_chart_textstyle(style_id);
	if (ascent == 0)
		ascent = gdk_char_height(ts->font, '8');
	y = 4;
	if (   cp->panel && cp->panel->label && sensor_separate_mode
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& (mapped_sensors & SENSOR_FAN)
	   )
		{
		y += ascent;
		gkrellm_draw_chart_label(cp, ts, 4, y, cp->panel->label->string);
		y += 2;
		}
	y += ascent;
	n = ((200 * l / total) + 1) / 2;
	if (n > 100)
		n = 100;
	snprintf(buf, sizeof(buf), "%d%%", n);
	gkrellm_draw_chart_label(cp, ts, 4, y, buf);
	}


static void
refresh_cpu_chart(CpuMon *cpu)
	{
	Chart	*cp		= cpu->chart;

	gkrellm_draw_chart(cp);

	/* prevTotal is really current total.  It just will be prevTotal next
	|  time store_chart_data() is called.
	*/
	if (cpu->extra_info)
		draw_cpu_extra(cp, cpu->sys_user,cp->prevTotal - cpu->prev_total_save);
	cp->need_redraw = FALSE;
	}

static void
draw_sensor_decals(CpuMon *cpu)
	{
	Panel	*p		= cpu->chart->panel;
	gchar	*name	= cpu->chart->name;
	gchar	units;
	gfloat	t, f;
	gint	toggle;

	if (   sensor_separate_mode
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& (mapped_sensors & SENSOR_FAN)
	   )
		{
		gkrellm_sensor_read_temperature(name, &t, &units);
		gkrellm_sensor_draw_temperature_decal(p, cpu->sensor_decal, t, units);
		gkrellm_sensor_read_fan(name, &f);
		gkrellm_sensor_draw_fan_decal(p, cpu->fan_decal, f);
		}
	else
		{
		toggle = time_now & 2;
		if (   (mapped_sensors & SENSOR_FAN)
			&& (toggle || !(mapped_sensors & SENSOR_TEMPERATURE))
		   )
			{
			gkrellm_sensor_read_fan(name, &f);
			gkrellm_sensor_draw_fan_decal(p, cpu->sensor_decal, f);
			}
		else if (   (mapped_sensors & SENSOR_TEMPERATURE)
				 && (!toggle || !(mapped_sensors & SENSOR_FAN))
				)
			{
			gkrellm_sensor_read_temperature(name, &t, &units);
			gkrellm_sensor_draw_temperature_decal(p, cpu->sensor_decal, t,
							units);
			}
		}
	}

static void
update_cpu()
	{
	GList		*list;
	CpuMon		*cpu;
	Chart		*cp;
	Panel		*p;
	Krell		*krell;
	gulong		total;

	GK.cpu_sys_activity = 0;
	(*read_system_cpu_info)();

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (!cpu->enabled)
			continue;
		cp = cpu->chart;
		p = cp->panel;
		if (smp_mode == 0)
			GK.cpu_sys_activity += (int)(cpu->sys - cp->prevOut);
		else if (list == cpu_mon_list)
			GK.cpu_sys_activity = (int)(cpu->sys - cp->prevOut);
		total = cpu->user + cpu->nice + cpu->sys + cpu->idle;
		if (add_nice_mode)
			cpu->user += cpu->nice;
		krell = KRELL(cp->panel);
		if (GK.second_tick)
			{
			cpu->sys_user = (cpu->sys - cp->prevOut) + (cpu->user - cp->prevIn);
			cpu->prev_total_save = cp->prevTotal;
			gkrellm_store_chart_data(cp, cpu->sys, cpu->user, total);
			refresh_cpu_chart(cpu);
			}

		if (   (GK.two_second_tick && !sensor_separate_mode)
			|| (GK.five_second_tick && sensor_separate_mode)
		   )
			draw_sensor_decals(cpu);

		if (cp->need_redraw)
			refresh_cpu_chart(cpu);
		krell->full_scale = (gint) (total - krell->previous_total);
		krell->full_scale *= krell->full_scale_expand;
		if (krell->previous_total > 0)
			{
			gkrellm_update_krell(p, krell, cpu->sys + cpu->user);
			gkrellm_draw_layers(p);
			}
		krell->previous_total = total;
		}
	}


static gint
cpu_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GList		*list;
	Chart		*cp;
	GdkPixmap	*pixmap	= NULL;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cp = ((CpuMon *) list->data)->chart;
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		if (pixmap)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}


static gint
cb_cpu_extra(GtkWidget *widget, GdkEventButton *event)
	{
	GList	*list;
	CpuMon	*cpu;
	Chart	*cp;

	if (event->button == 1)
		for (list = cpu_mon_list; list; list = list->next)
			{
			cpu = (CpuMon *) list->data;
			cp = cpu->chart;
			if (widget == cpu->chart->drawing_area)
				{
				cpu->extra_info = 1 - cpu->extra_info;
				gkrellm_config_modified();
				}
			refresh_cpu_chart(cpu);
			}
	return TRUE;
	}

static void
setup_cpu_scaling(Chart *cp)
	{
	Krell	*krell;
	gint	grids;

	krell = KRELL(cp->panel);
	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;

	cp->scale_min = CPU_TICKS_PER_SECOND / grids;
	cp->scale_max = 0;	/* Force a chart rescale */

	/* krell->full_scale can be small (as low as 10), so map it to a bigger
	|  number so ema calc will be smooth.   This is accounted for in
	|  update_krell()
	*/
	krell->full_scale_expand = 10;
	}

static gboolean
enable_cpu_visibility(CpuMon *cpu)
	{
	gint	enabled = cpu_enabled;

	if (smp_cpus > 0)
		{
		if (   (cpu->is_composite && smp_mode == 0)		/* real cpu's only */
			|| (! cpu->is_composite && smp_mode == 1)	/* Compos cpu only */
		   )
			enabled = FALSE;
		}
	return
		gkrellm_enable_visibility(enabled, &cpu->enabled, cpu->vbox,
				cpu->height);
	}


  /* How to decide when to make the sensor_decal and fan_decal visible.
  |  The sensor_decal can show temp values, fan values, or alternating
  |  temp/fan values.  The fan_decal only shows fan values when non
  |  alternating mode is selected. The sensors are mapped in sensors.c
  |
  |   Sensor and fan decal display truth table:
  |   |-----decals visible----||--sensors mapped--|   separate  |
  |	  |sensor_decal  fan_decal||   temp     fan   |    mode     |
  |   |-----------------------||--------------------------------|
  |   |     0           0     ||     0       0          0       |
  |   |     1           0     ||     1       0          0       |
  |   |     1           0     ||     0       1          0       |
  |   |     1           0     ||     1       1          0       |
  |   |     0           0     ||     0       0          1       |
  |   |     1           0     ||     1       0          1       |
  |   |     1           0     ||     0       1          1       |
  |   |     1           1     ||     1       1          1       |
  |   |----------------------------------------------------------
  */
static gint
adjust_sensors_display(CpuMon *cpu, gint force)
	{
	Panel	*p;
	Decal	*ds, *df;
	gint	position	= 0;

	ds = cpu->sensor_decal;
	df = cpu->fan_decal;
	if (!ds || !df)
		return FALSE;
	/* The test for state change is decal state vs success at reading
	|  a temperature.
	*/
	p = cpu->chart->panel;
	mapped_sensors = 0;
	if (get_mapped_sensor(cpu->chart->name, SENSOR_TEMPERATURE) || GK.demo)
		mapped_sensors |= SENSOR_TEMPERATURE;
	if (get_mapped_sensor(cpu->chart->name, SENSOR_FAN) || GK.demo)
		mapped_sensors |= SENSOR_FAN;

	if (mapped_sensors & (SENSOR_TEMPERATURE | SENSOR_FAN))
		{
		if (! gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_visible(p, ds);
		position = 0;
		}
	else
		{
		if (gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_invisible(p, ds);
		position = cpu->save_label_position;
		}
	if (   (mapped_sensors & SENSOR_FAN)
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& sensor_separate_mode
	   )
		{
		if (! gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_visible(p, df);
		position = -1;
		}
	else
		{
		if (gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_invisible(p, df);
		}
	if (position != p->label->position || force)
		{
		if (cpu->save_label_position >= 0)	/* Reassign position only if the */
			p->label->position = position;	/* original label was visible.   */
		gkrellm_draw_panel_label(p, gkrellm_bg_panel_image(style_id));
		draw_sensor_decals(cpu);
		gkrellm_draw_layers(p);
		return TRUE;
		}
	return FALSE;
	}


  /* I want cpu labels in upper case.
  */
static gchar	mapname[5] ;
static void
create_cpu(GtkWidget *vbox, gint first_create)
	{
	CpuMon	*cpu;
	Chart	*cp;
	Panel	*p;
	Style	*style;
	GList	*list;

	ascent = 0;
	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;

		/* Default to all cpu charts visible.  Correct this as last step.
		*/
		if (first_create)
			{
			cpu->vbox = gtk_vbox_new(FALSE, 0);
			gtk_container_add(GTK_CONTAINER(vbox), cpu->vbox);
			gtk_widget_show(cpu->vbox);
			cpu->chart = gkrellm_chart_new0();
			cpu->chart->panel = gkrellm_panel_new0();
			cpu->chart->name = g_strdup(cpu->name);
			cpu->enabled = TRUE;
			}
		else
			{
			gkrellm_destroy_decal_list(cpu->chart->panel);
			gkrellm_destroy_krell_list(cpu->chart->panel);
			}
		cp = cpu->chart;
		p = cp->panel;

		style = gkrellm_panel_style(style_id);
		gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

		setup_cpu_scaling(cp);

		if (GK.debug_level & DEBUG_MISC)
			printf(_("creating cpu <%s>, scale_min=%d krell->full_scale=%d\n"),
					cp->name, cp->scale_min, KRELL(p)->full_scale);

		cp->h = UC.chart_height[MON_CPU];
		gkrellm_create_chart(cpu->vbox, cp, style_id);

		sprintf( mapname, _("CPU%c"), (cp->name)[3] );
		p->textstyle = gkrellm_panel_textstyle(style_id);

		if (p->label->string)
			g_free(p->label->string);

		if (! (!strcmp(cp->name, "cpu") && cpu->is_composite))
			sensor_create_decals(p, style_id,
					&cpu->sensor_decal, &cpu->fan_decal);

		gkrellm_configure_panel(p, g_strdup(mapname), style);
		gkrellm_create_panel(cpu->vbox, p, gkrellm_bg_panel_image(style_id));
		cpu->height = cp->h + p->h;
		if (cpu->enabled)
			gkrellm_monitor_height_adjust(cpu->height);

		cpu->save_label_position = p->label->position;
		if (cpu->sensor_decal)
			adjust_sensors_display(cpu, TRUE);

		gkrellm_alloc_chart_data(cp);
		enable_cpu_visibility(cpu);

		if (first_create)
			{
			gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
					(GtkSignalFunc) cpu_expose_event, NULL);
			gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
					(GtkSignalFunc) cpu_expose_event, NULL);

			gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
					"button_press_event", (GtkSignalFunc) cb_cpu_extra, NULL);
			}
		else
			refresh_cpu_chart(cpu);
		gkrellm_setup_launcher(p, &cpu->launch, CHART_PANEL_TYPE, 4);
		}
	}



/* ------------------------------------------------------------------ */
#define	CPU_CONFIG_KEYWORD	"cpu"

static GtkWidget	*cpu_enabled_button;
static GtkWidget	*smp_button[3];
static GtkWidget	*add_nice_button;
static GtkWidget	*sensor_separate_button;


static void
save_cpu_config(FILE *f)
	{
	GList	*list;
	CpuMon	*cpu;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (*(cpu->launch.command) != '\0')
			fprintf(f, "%s launch %s %s\n", CPU_CONFIG_KEYWORD,
						cpu->name, cpu->launch.command);
		if (*(cpu->launch.tooltip_comment) != '\0')
			fprintf(f, "%s tooltip_comment %s %s\n", CPU_CONFIG_KEYWORD,
						cpu->name, cpu->launch.tooltip_comment);
		fprintf(f, "%s extra_info %s %d\n", CPU_CONFIG_KEYWORD,
					cpu->name, cpu->extra_info);

		}
	fprintf(f, "%s enable %d\n", CPU_CONFIG_KEYWORD, cpu_enabled);
	fprintf(f, "%s smp_mode %d\n", CPU_CONFIG_KEYWORD, smp_mode);
	fprintf(f, "%s nice_mode %d\n", CPU_CONFIG_KEYWORD, add_nice_mode);
	fprintf(f, "%s sensor_mode %d\n", CPU_CONFIG_KEYWORD,
			sensor_separate_mode);
	}

static void
load_cpu_config(gchar *arg)
	{
	GList	*list;
	CpuMon	*cpu;
	gchar	config[32], item[CFG_BUFSIZE],
			cpu_name[32], command[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "enable") == 0)
			sscanf(item, "%d", &cpu_enabled);
		else if (strcmp(config, "smp_mode") == 0)
			sscanf(item, "%d\n", &smp_mode);
		else if (strcmp(config, "nice_mode") == 0)
			sscanf(item, "%d\n", &add_nice_mode);
		else if (strcmp(config, "sensor_mode") == 0)
			sscanf(item, "%d\n", &sensor_separate_mode);
		else if (strcmp(config, "extra_info") == 0)
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					sscanf(command, "%d\n", &cpu->extra_info);
				}
			}
		else if (strcmp(config, "launch") == 0)
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					cpu->launch.command = g_strdup(command);
				}
			}
		else if (strcmp(config, "tooltip_comment") == 0)
			{
			sscanf(item, "%31s %[^\n]", cpu_name, command);
			for (list = cpu_mon_list; list; list = list->next)
				{
				cpu = (CpuMon *) list->data;
				if (strcmp(cpu->name, cpu_name) == 0)
					cpu->launch.tooltip_comment = g_strdup(command);
				}
			}
		}
	}


static void
apply_cpu_config()
	{
	GList	*list;
	CpuMon	*cpu;
	gint	i;

	cpu_enabled = GTK_TOGGLE_BUTTON(cpu_enabled_button)->active;
	if (smp_cpus > 0)
		for (i = 0; i < 3; ++i)
			if (GTK_TOGGLE_BUTTON(smp_button[i])->active)
				smp_mode = i;
	add_nice_mode = GTK_TOGGLE_BUTTON(add_nice_button)->active;
	if (sensor_separate_button)
		sensor_separate_mode =
				GTK_TOGGLE_BUTTON(sensor_separate_button)->active;

	for (list = cpu_mon_list; list; list = list->next)
		{
		cpu = (CpuMon *) list->data;
		if (enable_cpu_visibility(cpu) && cpu->enabled)
			gkrellm_reset_chart(cpu->chart);
		setup_cpu_scaling(cpu->chart);
		gkrellm_apply_launcher(&cpu->launch_entry, &cpu->tooltip_entry,
					cpu->chart->panel, &cpu->launch, gkrellm_launch_button_cb);
		if (adjust_sensors_display(cpu, FALSE) && cpu->launch.button)
			{
			gkrellm_destroy_button(cpu->launch.button);
			cpu->launch.button = 
				gkrellm_put_label_in_panel_button(cpu->chart->panel,
					gkrellm_launch_button_cb, &cpu->launch, cpu->launch.pad);
			}
		}
	}

static void
create_cpu_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*button;
	GtkWidget	*hbox;
	GtkWidget	*table;
	GSList		*group;
	GList		*list;
	CpuMon		*cpu;
	gchar		buf[128];
	gint		i;

	gkrellm_check_button(tab_vbox, &cpu_enabled_button, cpu_enabled,
			FALSE, 10, _("Enable CPU"));
	if (smp_cpus > 0)
		{
		insert_expanded_filler(tab_vbox);
		hbox = gtk_hbox_new (FALSE, 3);
		gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);
		gtk_container_add(GTK_CONTAINER(tab_vbox), hbox);

		button = gtk_radio_button_new_with_label(NULL, _("Real CPUs."));
		gtk_box_pack_start(GTK_BOX (hbox), button, TRUE, TRUE, 0);
		smp_button[0] = button;
		group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));

		button = gtk_radio_button_new_with_label(group, _("Composite CPU."));
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		group = gtk_radio_button_group(GTK_RADIO_BUTTON (button));
		smp_button[1] = button;

		button = gtk_radio_button_new_with_label(group, _("Composite and real"));
		gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
		smp_button[2] = button;

		button = smp_button[smp_mode];
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
		}
	insert_expanded_filler(tab_vbox);
	gkrellm_check_button(tab_vbox, &add_nice_button, add_nice_mode, FALSE, 0,
		_("Add nice CPU time to user time."));
	if (gkrellm_sensors_available())
		gkrellm_check_button(tab_vbox, &sensor_separate_button,
					sensor_separate_mode, FALSE, 0,
		_("Draw fan and temperature values separately (not alternating)."));
	insert_expanded_filler(tab_vbox);

	table = gkrellm_launcher_table_new(tab_vbox, n_cpus);
	for (i = 0, list = cpu_mon_list; list; list = list->next, ++i)
		{
		cpu = (CpuMon *) list->data;
		snprintf(buf, sizeof(buf), _("%s"), cpu->name);
		gkrellm_config_launcher(table, i,  &cpu->launch_entry,
				&cpu->tooltip_entry, buf, &cpu->launch);
		}
	}

static Monitor	monitor_cpu =
	{
	N_("CPU"),				/* Name, for config tab.	*/
	MON_CPU,			/* Id,  0 if a plugin		*/
	create_cpu,			/* The create function		*/
	update_cpu,			/* The update function		*/
	create_cpu_tab,		/* The config tab create function	*/
	apply_cpu_config,	/* Apply the config function		*/

	save_cpu_config,	/* Save user conifg			*/
	load_cpu_config,	/* Load user config			*/
	CPU_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_cpu_monitor(void)
	{
	monitor_cpu.name = _(monitor_cpu.name);
	style_id = gkrellm_add_chart_style(&monitor_cpu, CPU_STYLE_NAME);
	if (setup_cpu_interface())
		return &monitor_cpu;
	return NULL;
	}

