/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: print.c,v 1.1.1.1 2003/08/14 03:28:52 cosine Exp $
 */

#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <setjmp.h>
#include <signal.h>

#include "alddefs.h"
#include "command.h"
#include "defs.h"
#include "main.h"
#include "print.h"
#include "scroll.h"
#include "terminal.h"
#include "window.h"

/*
 * libString includes
 */
#include "Strn.h"

/*
initPrint()
  Initialize a print workspace

Return: pointer to print workspace
*/

struct printWorkspace *
initPrint()

{
  struct printWorkspace *ws;

  ws = (struct printWorkspace *) malloc(sizeof(struct printWorkspace));
  if (!ws)
  {
    fprintf(stderr, "initPrint: malloc failed: %s\n",
      strerror(errno));
    return (0);
  }

  memset(ws, '\0', sizeof(struct printWorkspace));

  ws->PrintBurst = 0;
  ws->PrintLineCnt = 0;
  ws->PausePrint = 1;
  ws->file_p = 0;
  ws->filename = 0;

  return (ws);
} /* initPrint() */

/*
termPrint()
  Terminate a print workspace

Inputs: ws - print workspace

Return: none
*/

void
termPrint(struct printWorkspace *ws)

{
  if (!ws)
    return;

  if (ws->file_p)
    fclose(ws->file_p);

  if (ws->filename)
    free(ws->filename);

  free(ws);
} /* termPrint() */

/*
startPrintBurst()
  This function should be called when we are about to print a lot
of lines of information. It will set up the necessary variables to
pause the printing when a pageful of information has been printed,
so the user can decide whether to continue or stop.
*/

void
startPrintBurst(struct printWorkspace *ws)

{
  if (ws->PausePrint)
  {
    ws->PrintBurst = 1;
    ws->PrintLineCnt = 0;
  }
} /* startPrintBurst() */

/*
endPrintBurst()
  This function should be called after the print burst has completed.
*/

void
endPrintBurst(struct printWorkspace *ws)

{
  ws->PrintBurst = 0;
} /* endPrintBurst() */

/*
Print()
 Print the specified string to the appropriate window

Inputs: main_p - main workspace
        flags  - for curses mode, to specify what window to print to
        format - formatted string to print

NOTE: All routines which print to the output windows
      (OutputWindow1 and OutputWindow2) *MUST* use this routine
      so the window buffers are updated correctly.
*/

void
Print(struct aldWorkspace *main_p, int flags, const char *format, ...)

{
  va_list args;

  va_start(args, format);

  PrintWindow(main_p, 0, flags, format, args);

  va_end(args);
} /* Print() */

/*
RawPrint()
  Similar to Print(), except do not print \n characters on the
end of the string
*/

void
RawPrint(struct aldWorkspace *main_p, int flags, const char *format, ...)

{
  va_list args;

  va_start(args, format);

  PrintWindow(main_p, 1, flags, format, args);

  va_end(args);
} /* RawPrint() */

/*
PrintWindow()

Inputs: main_p - main workspace
        raw    - to specify whether to add a \n to the string
        flags  - for curses mode, to specify which window to print to
        format - string to print
        args   - list of args

Return: none
*/

void
PrintWindow(struct aldWorkspace *main_p, int raw, int flags, const char *format,
            va_list args)

{
  struct printWorkspace *printWorkspace_p = main_p->printWorkspace_p;
  char rawbuf[MAXLINE]; /* raw buffer (format + args) */
  int rawlen;           /* length of raw buffer */

#ifdef USE_CURSES

  char *rawend;         /* end of raw buffer */
  char *buffer,         /* current buffer we are printing */
       *tmp;
  int buflen;           /* length of current buffer */

#endif /* USE_CURSES */

/*  rawlen = vsprintf(rawbuf, format, args); */
  rawlen = vSnprintf(rawbuf, sizeof(rawbuf), (char *) format, args);

  if (ModeConsole)
  {
    /*
     * We are in console mode - just print it to stdout
     */
    fwrite(rawbuf, sizeof(char), rawlen, stdout);
    if (!raw)
      fputc((unsigned char) '\n', stdout);
    else
      fflush(stdout);

    if (printWorkspace_p->file_p)
    {
      fwrite(rawbuf, sizeof(char), rawlen, printWorkspace_p->file_p);
      if (!raw)
        fputc((unsigned char) '\n', printWorkspace_p->file_p);
      else
        fflush(printWorkspace_p->file_p);
    }

    if (printWorkspace_p->PrintBurst)
    {
      if (raw)
      {
        if (strchr(rawbuf, '\n'))
          ++(printWorkspace_p->PrintLineCnt);
      }
      else
        ++(printWorkspace_p->PrintLineCnt);

      if (printWorkspace_p->PrintLineCnt ==
          (unsigned long) (main_p->terminalWorkspace_p->LinesPerPage - 1))
      {
        char str[MAXLINE];

        /*
         * We have filled up the screen, so pause and let the user
         * hit a key to continue
         */
        fprintf(stdout, "Hit <return> to continue, or <q> to quit");

        fgets(str, MAXLINE, stdin);
        if (*str == 'q')
        {
          /*
           * They want to stop printing - cleanup and simulate a SIGINT
           */
          printWorkspace_p->PrintBurst = 0;
          longjmp(main_p->commandWorkspace_p->CmdParserEnv, SIGINT);
        }
        else
        {
          /*
           * They want to continue printing
           */
          printWorkspace_p->PrintLineCnt = 0;
        }
      }
    } /* if (printWorkspace_p->PrintBurst) */

    return;
  }

#ifdef USE_CURSES

  if (ModeCurses)
  {
    WINDOW *outwin = 0;
    struct Frame *frame = 0; /* scrolling window ptr */

    if (flags & P_DEBUG)
    {
      frame = DebugFrame;
    }
    else if (flags & P_DISASSEMBLY)
    {
      /*
       * This data is coming from c_disassemble() - do not print
       * it to the window because we need to highlight the top
       * line when all the data has been stored. c_disassemble()
       * will call DrawWindow() to accomplish this when it is done.
       */
      frame = DisassemblyFrame;
    }
    else if (flags & (P_COMMAND | P_ERROR | P_MEMORY))
    {
      /*
       * We are writing this data to the command output window -
       * store it and write immediately, but keep CurrentElement
       * and TopElement updated correctly
       */
      outwin = CommandOutputFrame->window;
      frame = CommandOutputFrame;
    }
    else if (flags & P_REGISTER)
    {
      outwin = RegistersFrame->window;
      frame = RegistersFrame;
    }
    else if (flags & P_OUTPUT)
    {
      outwin = ProcessOutputFrame->window;
      frame = ProcessOutputFrame;
    }

    assert(frame != 0);

    if ((frame->flags & W_AUTORAISE) &&
        !(frame->flags & W_RAISED) &&
        !(CurrentFrame->flags & W_STAYONTOP))
    {
      /*
       * Window is not currently visible, but is set to auto-raise
       * when something is written to it - so raise it
       */
      RaiseFrame(frame);
    }

    buffer = rawbuf;

    rawend = rawbuf + rawlen;

    /*
     *  This loop will take the string 'rawbuf' (which possibly
     * contains multiple \n characters), and add the lines
     * one by one, using the \n character as a line separator.
     * This way, PrintWindow() can be called to print multiple
     * lines at once and still keep the 'data' array of the
     * scrolling window intact, with one line per index.
     *  Add a check here for rawlen == 0, in case we want
     * to print a blank line.
     */
    while ((buffer < rawend) || (rawlen == 0))
    {
      if ((tmp = strchr(buffer, '\n')))
      {
        *tmp++ = '\0';
        buflen = tmp - buffer - 1;
      }
      else if (raw)
      {
        /*
         * No ending \n character - assume it is a partial
         * line which will be completed later - store the line
         * in frame's buffer. Otherwise, we would have 1 line
         * taking up multiple indices in our 'data' array,
         * which would be a nightmare when we want to scroll
         * the window up/down.
         */
        frame->bufptr += Sprintf(frame->bufptr, "%s", buffer);

        if (frame->flags & W_SCROLL)
        {
          /*
           * Flush any other lines this loop may have added
           * to frame->data before coming across this partial
           * line
           */
          DrawWindow(frame);
        }

        return;
      }
      else
        buflen = rawlen;

      if (*(frame->buffer))
      {
        /*
         * This scrolling window has a partial line stored from
         * past prints, and this time we have an ending \n character,
         * so add it to the buffer and flush it
         */
        Sprintf(frame->bufptr, "%s", buffer);
        AddScrollData(frame, frame->buffer);

        frame->buffer[0] = '\0';
        frame->bufptr = frame->buffer;
      }
      else
      {
        /*
         * 'buffer' contains a complete line - add it to our
         * data array
         */
        AddScrollData(frame, buffer);
      }

      if (!(frame->flags & W_SCROLL) && outwin)
      {
        /*
         * Print the line to the window
         */
        waddstr(outwin, buffer);
        if (!raw)
          waddch(outwin, '\n');
      }

      buffer += buflen + 1;
      rawlen = rawlen - buflen - 1;

      if (!rawlen)
        break;
    } /* while (buffer) */

    if (frame->flags & W_SCROLL)
    {
      /*
       * We have just written to a scrolling window which
       * is marked with W_SCROLL. This means we have to take
       * care of the scrolling ourselves, so call DrawWindow()
       * to redraw the window with the correct information
       * displayed.
       */
      DrawWindow(frame);
    }

    /*
     * If we just wrote to a frame which was not raised, it
     * could have possibly overwritten a frame on top of it,
     * so re-raise the current frame - this is also the only
     * way to "refresh" the frame if there is another frame
     * on top of it.
     */
    if (!IsHidden(frame))
      ReRaiseFrame(frame);
  } /* if (ModeCurses) */

#endif /* USE_CURSES */

} /* PrintWindow() */
