/*
 * Assembly Language Debugger
 *
 * Copyright (C) 2000 Patrick Alken
 * This program comes with absolutely NO WARRANTY
 *
 * Should you choose to use and/or modify this source code, please
 * do so under the terms of the GNU General Public License under which
 * this program is distributed.
 *
 * $Id: c_examine.c,v 1.1.1.1 2003/08/14 03:28:52 cosine Exp $
 */

#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include "alddefs.h"
#include "defs.h"
#include "load.h"
#include "main.h"
#include "memory.h"
#include "msg.h"
#include "print.h"

#include "libDebug.h"

/*
 * libString includes
 */
#include "alloc.h"
#include "Strn.h"

/*
c_examine()
  Examine the debugged process' memory

Format for this command:
  examine [startaddr] [stopaddr] [-num <num>] [-size <val>]
          [-output <letter>]

Options:
  -num <value>         = number of elements to dump
  -size <value/letter> = size of each element
    Possible letter values: b (byte), w (word), h (halfword),
                            d (dword), g (giant - 8 bytes)
  -output <letter>     = output format
    Possible letter values: o (octal), d (decimal), h (hex - default)
                            b (binary)

Return: 0 upon failure
        1 upon success
*/

int
c_examine(struct aldWorkspace *ws, int ac, char **av)

{
  int cnt,
      alen;
  char *endptr;
  unsigned long startaddr, /* address to begin memory dump */
                stopaddr;  /* address to stop memory dump */
  int gotaddr;             /* did user supply start address? */
  long elnum,              /* number of elements in memory to dump */
       elsize,             /* size of each element */
       elbytes;            /* number of bytes for all elements */
  unsigned char output;    /* output type (hex, binary, etc) */
  unsigned char *membuf;   /* where to store dumped memory */
  long numbytes;           /* number of bytes of memory to dump */
  long ndumped;            /* number of bytes actually dumped */
  unsigned long deladdr;   /* delta address (stopaddr - startaddr) */

  /*
   * Defaults: dump 64 elements, each 1 byte, in hex format
   */
  elnum = 64;
  elsize = 1;
  output = (unsigned char) 'h';
  startaddr = 0;
  stopaddr = 0;
  gotaddr = 0;

  for (cnt = 1; cnt < ac; ++cnt)
  {
    alen = strlen(av[cnt]);

    if (!Strncasecmp(av[cnt], "-num", alen))
    {
      if (++cnt >= ac)
      {
        Print(ws, P_ERROR, "No element number specified");
        return (0);
      }

      elnum = strtol(av[cnt], &endptr, 0);
      if ((endptr == av[cnt]) || (*endptr != '\0') || (elnum <= 0))
      {
        Print(ws, P_ERROR, "Invalid element number: %s", av[cnt]);
        return (0);
      }
    }
    else if (!Strncasecmp(av[cnt], "-size", alen))
    {
      if (++cnt >= ac)
      {
        Print(ws, P_ERROR, "No element size specified");
        return (0);
      }

      elsize = GetElementSize(av[cnt]);
      if (!elsize)
      {
        Print(ws, P_ERROR, "Invalid element size: %s", av[cnt]);
        return (0);
      }
    }
    else if (!Strncasecmp(av[cnt], "-output", alen))
    {
      if (++cnt >= ac)
      {
        Print(ws, P_ERROR, "No element size specified");
        return (0);
      }

      output = (unsigned char) *av[cnt];

      /*
       * We'll let OutputMemory() do any error checking on the
       * output type
       */
    }
    else
    {
      if (!gotaddr)
      {
        gotaddr = 1;
        startaddr = strtoul(av[cnt], &endptr, 0);

        if ((endptr == av[cnt]) || (*endptr != '\0'))
        {
          int ridx;

          /*
           * It is not a valid numerical address, but they
           * may have specified a register name wanting to
           * start the dump at the address in the register
           * so check for that before giving an error
           */
          ridx = findRegisterDebug(ws->debugWorkspace_p, av[cnt]);
          if (ridx != (-1))
            startaddr = (unsigned long) readRegisterDebug(ws->debugWorkspace_p, ridx);
          else
          {
            Print(ws, P_ERROR, "Invalid address: %s", av[cnt]);
            return (0);
          }
        }
      }
      else
      {
        stopaddr = strtol(av[cnt], &endptr, 0);

        if ((endptr == av[cnt]) || (*endptr != '\0'))
        {
          Print(ws, P_ERROR, "Invalid address: %s", av[cnt]);
          return (0);
        }
      }
    }
  } /* for (cnt = 1; cnt < ac; ++cnt) */

  if (!gotaddr)
  {
    /*
     * If no start address was specified, the first choice is
     * the address of the next instruction about to be executed.
     * Second choice is the program's entry point, since if
     * nextInstruction is 0, the program hasn't started yet.
     */
    if (ws->nextInstruction)
      startaddr = ws->nextInstruction;
    else
      startaddr = ws->virtualEntryPoint;
  }

  membuf = 0;

  /*
   * Determine exactly how many bytes of memory we want to
   * dump
   */
  elbytes = elnum * elsize;
  if (stopaddr >= startaddr)
    deladdr = stopaddr - startaddr;
  else
    deladdr = 0;

  if (deladdr > (unsigned long) elbytes)
    numbytes = deladdr;
  else
    numbytes = elbytes;

  ndumped = dumpMemoryDebug(ws->debugWorkspace_p,
                            (char **) &membuf,
                            startaddr,
                            numbytes);

  if (membuf)
  {
    if (ndumped > 0)
      OutputMemory(ws, membuf, startaddr, ndumped, elsize, output);
    MyFree(membuf);
  }

  if (ndumped < numbytes)
    Print(ws, P_ERROR, MSG_NOACCESS, startaddr + ndumped, strerror(errno));

  return (1);
} /* c_examine() */
