<?php
/*********************************************************************************
 * The contents of this file are subject to the TimeTrex Public License Version
 * 1.1.0 ("License"); You may not use this file except in compliance with the
 * License. You may obtain a copy of the License at http://www.TimeTrex.com/TPL
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * All copies of the Covered Code must include on each user interface screen:
 *    (i) the "Powered by TimeTrex" logo and
 *    (ii) the TimeTrex copyright notice
 * in the same form as they appear in the distribution.  See full license for
 * requirements.
 *
 * The Original Code is: TimeTrex Open Source
 * The Initial Developer of the Original Code is TimeTrex Payroll Services
 * Portions created by TimeTrex are Copyright (C) 2004-2007 TimeTrex Payroll Services;
 * All Rights Reserved.
 *
 ********************************************************************************/
/*
 * $Revision: 2158 $
 * $Id: HierarchyListFactory.class.php 2158 2008-09-15 23:26:15Z ipso $
 * $Date: 2008-09-15 16:26:15 -0700 (Mon, 15 Sep 2008) $
 */

/**
 * @package Module_Hierarchy
 */
class HierarchyListFactory extends HierarchyFactory implements IteratorAggregate {

	protected $fasttree_obj = NULL;

	function getFastTreeObject() {

		if ( is_object($this->fasttree_obj) ) {
			return $this->fasttree_obj;
		} else {
			global $fast_tree_options;
			$this->fasttree_obj = new FastTree($fast_tree_options);

			return $this->fasttree_obj;
		}
	}

	function getChildLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id, $recurse = FALSE) {
		//This only gets the immediate children
		//Used for authorization list when they just want to see immediate children.

		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		//Get current level IDs first, then get children of all of them.
		$ids = $this->getCurrentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id, FALSE);
		Debug::Arr($ids ,' zzNodes at the same level: User ID: '. $user_id, __FILE__, __LINE__, __METHOD__,10);

		if ( $ids === FALSE ) {
			return FALSE;
		}

		$hslf = new HierarchyShareListFactory();

		$retarr = array();
		foreach ( $ids as $id ) {
			//Debug::Text(' Getting Children of ID: '. $id, __FILE__, __LINE__, __METHOD__,10);
			$children = $this->getFastTreeObject()->getAllChildren( $id, $recurse );
			//Debug::Arr($children ,' ccNodes at the same level', __FILE__, __LINE__, __METHOD__,10);

			if ( $children === FALSE ) {
				continue;
			}

			//Remove $user_id from final array, otherwise permission checks will think the user doing the permission
			//check is a child of themself, preventing users from view/editing children but not themselves.
			/*
			if ( isset($children[$user_id]) ) {
				unset($children[$user_id]);
			}
			*/
			$child_ids = array_keys( $children );

			$retarr = array_merge($retarr, $child_ids);
			unset($child_ids);
		}

		return $retarr;
	}

	function getAllParentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id) {
		//This only gets the immediate parents
		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		$ids = $this->getFastTreeObject()->getAllParents( $user_id );
		Debug::Arr($ids ,' Parent Nodes', __FILE__, __LINE__, __METHOD__,10);

		//Find out if any of the parents are shared.
		$hslf = new HierarchyShareListFactory();

		$retarr = array();
		foreach ( $ids as $id ) {

			$hierarchy_share = $hslf->getByHierarchyControlIdAndUserId( $tree_id, $id )->getCurrent()->isNew();

			if ( $hierarchy_share === FALSE ) {
				Debug::Text(' Node IS shared:  '. $id, __FILE__, __LINE__, __METHOD__,10);

				//Get current level IDs
				$current_level_ids = $this->getCurrentLevelIdArrayByHierarchyControlIdAndUserId( $tree_id, $id );
				$retarr = array_merge($retarr, $current_level_ids);
				unset($current_level_ids);
			} else {
				Debug::Text(' Node isnt shared:  '. $id, __FILE__, __LINE__, __METHOD__,10);
				$retarr[] = (int)$id;
			}
		}

		Debug::Arr($retarr ,' Final Parent Nodes including shared', __FILE__, __LINE__, __METHOD__,10);
		return array_unique($retarr);
	}


	function getParentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id) {
		//This only gets the immediate parents
		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		//Get the parent, then get the current level from that.
		$parent_id = $this->getFastTreeObject()->getParentId( $user_id );

		$retarr = array();

		$parent_nodes = $this->getCurrentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $parent_id);
		//Debug::Arr($parent_nodes ,' Parent Nodes', __FILE__, __LINE__, __METHOD__,10);
		if ( is_array($parent_nodes) ) {
			$retarr = $parent_nodes;
		}

		return $retarr;
	}

	function getCurrentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id, $ignore_self = FALSE ) {
		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		$parent_id = $this->getFastTreeObject()->getParentId( $user_id);

		$children = $this->getFastTreeObject()->getAllChildren( $parent_id );
		if ( $children === FALSE ) {
			return FALSE;
		}

		$ids = array_keys( $children );
		Debug::Arr($ids ,' zNodes at the same level', __FILE__, __LINE__, __METHOD__,10);

		$hslf = new HierarchyShareListFactory();

		//Check if current user is shared, because if it isn't shared, then we can ignore
		//all other shared users in the tree.
		$root_user_id_shared = $hslf->getByHierarchyControlIdAndUserId( $tree_id, $user_id )->getRecordCount();
		Debug::Text('Root User ID: '. $user_id .' Shared: '. (int)$root_user_id_shared, __FILE__, __LINE__, __METHOD__,10);

		$retarr[] = (int)$user_id;
		foreach ( $ids as $id ) {

			$hierarchy_share = $hslf->getByHierarchyControlIdAndUserId( $tree_id, $id )->getCurrent()->isNew();

			if ( $root_user_id_shared == TRUE AND $hierarchy_share === FALSE ) {
				Debug::Text(' Node IS shared:  '. $id, __FILE__, __LINE__, __METHOD__,10);
				$retarr[] = $id;
			} else {
				//Debug::Text(' Node isnt shared:  '. $id, __FILE__, __LINE__, __METHOD__,10);
			}
		}

		return array_unique($retarr);
	}

	function getLevelsByHierarchyControlIdAndUserId( $tree_id, $user_id ) {
		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		$node_level = $this->getFastTreeObject()->getLevel( $user_id );
		//Debug::Text(' Node Level: '. $node_level, __FILE__, __LINE__, __METHOD__,10);

		$current_level_nodes = $this->getCurrentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id);
		//Debug::Arr($current_level_nodes, ' Current Level Nodes  ', __FILE__, __LINE__, __METHOD__,10);

		//Do this because of shared hierarchy levels. Only one user can have unique children, so we
		//need to go through each user at the same level and collect all children.
		if ( is_array($current_level_nodes) AND count($current_level_nodes) > 0 ) {
			foreach($current_level_nodes as $tmp_user_id ) {
				//Debug::Text(' Getting Children for tmp user ID: '. $tmp_user_id, __FILE__, __LINE__, __METHOD__,10);

				$children = $this->getFastTreeObject()->getAllChildren( $tmp_user_id, 'RECURSE' );

				if ( $children === FALSE ) {
					//Debug::Text(' No children found, continuing...', __FILE__, __LINE__, __METHOD__,10);
					continue;
				}

				//Debug::Text(' Children FOUND, NOT returning FALSE:  ', __FILE__, __LINE__, __METHOD__,10);
				//Debug::Arr($children ,' aChildren', __FILE__, __LINE__, __METHOD__,10);
				$levels = array_unique( $children );

				foreach( $levels as $key => $value ) {
					$retarr[$key] = $value + $node_level;
				}
				unset($children, $levels, $key, $value);
			}

			if ( isset($retarr) ) {
				//Need to unique the final array once more so we don't get multiple entries
				//for each level, otherwise the following array_pop is useless.
				$retarr = array_unique($retarr);
				//Debug::Arr($retarr ,' RetArr', __FILE__, __LINE__, __METHOD__,10);

				//Ignore the highest level.
				asort($retarr);
				array_pop($retarr);

				//$retarr = array_flip($retarr);
				$final_retarr = array();
				$i=0;
				foreach( $retarr as $key => $value ) {
					$final_retarr[$i] = $value;

					$i++;
				}

				return $final_retarr;
			}
		}

		return FALSE;
	}

	function getByHierarchyControlIdAndUserId( $tree_id, $user_id ) {
		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		$node = $this->getFastTreeObject()->getNode( $user_id );

		if ($node === FALSE ) {
			return FALSE;
		}

		$ulf = new UserListFactory();
		$user_obj = $ulf->getById( $node['object_id'] )->getCurrent();

		$hslf = new HierarchyShareListFactory();
		$hierarchy_share = $hslf->getByHierarchyControlIdAndUserId( $tree_id, $user_id )->getCurrent()->isNew();

		if ( $hierarchy_share === FALSE ) {
			$shared = TRUE;
		} else {
			$shared = FALSE;
		}

		$retarr = array(
						'id' => $node['object_id'],
						'parent_id' => $node['parent_id'],
						'name' => $user_obj->getFullName(),
						'level' => $node['level'],
						'shared' => $shared
					);

		return $retarr;
	}

	function getByHierarchyControlIdAndUserIdAndLevel( $tree_id, $user_id, $level = 0 ) {
		if ( $tree_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		if ( !is_numeric($level) ) {
			return FALSE;
		}
		Debug::Text(' User ID: '. $user_id .' Level: '. $level, __FILE__, __LINE__, __METHOD__,10);

		$current_level_ids = $this->getCurrentLevelIdArrayByHierarchyControlIdAndUserId($tree_id, $user_id);
		Debug::Arr($current_level_ids ,' Nodes at the same level', __FILE__, __LINE__, __METHOD__,10);

		$retarr = array( 'current_level' => array(), 'parent_level' => array(), 'child_level' => array() );

		//Get children of all current levels.
		if ( is_array($current_level_ids) ) {

			$children = array();

			$this->getFastTreeObject()->setTree( $tree_id );

			foreach( $current_level_ids as $shared_user_id ) {
				$retarr['current_level'][] = $shared_user_id;

				$shared_children = $this->getFastTreeObject()->getAllChildren( $shared_user_id, 'RECURSE' );
				//Debug::Arr($shared_children ,' aChildren of User: '. $shared_user_id, __FILE__, __LINE__, __METHOD__,10);

				if ( is_array($shared_children) ) {
					foreach( $shared_children as $tmp_child_id => $child_level  ) {
						if ( $level == $child_level ) {
							//Debug::Text(' Current Level: '. $tmp_child_id, __FILE__, __LINE__, __METHOD__,10);
							$retarr['current_level'][] = $tmp_child_id;
						} elseif ( ($level-1) == $child_level ) {
							//Debug::Text(' Parent Level: '. $tmp_child_id, __FILE__, __LINE__, __METHOD__,10);
							$retarr['parent_level'][] = $tmp_child_id;
						} elseif ( ($level+1) == $child_level ) {
							//Debug::Text(' Child Level: '. $tmp_child_id, __FILE__, __LINE__, __METHOD__,10);
							$retarr['child_level'][] = $tmp_child_id;
						} else {
							//Debug::Text(' No Level: '. $tmp_child_id, __FILE__, __LINE__, __METHOD__,10);
						}
					}
				}
			}

			$retarr['current_level'] = array_unique( $retarr['current_level'] );
			$retarr['parent_level'] = array_unique( $retarr['parent_level'] );
			$retarr['child_level'] = array_unique( $retarr['child_level'] );

			//Debug::Arr($retarr ,' aChildren of User: '. $user_id .' At Level: '. $level, __FILE__, __LINE__, __METHOD__,10);

			return $retarr;
		}


		return FALSE;
	}

	function getByCompanyIdAndHierarchyControlId( $company_id, $tree_id ) {
		if ( $company_id == '' ) {
			return FALSE;
		}

		if ( $tree_id == '' ) {
			return FALSE;
		}

		$hclf = new HierarchyControlListFactory();
		$hclf->getByIdAndCompanyId($tree_id, $company_id);
		if ( $hclf->getRecordCount() == 0 ) {
			return FALSE;
		}

		return $this->getByHierarchyControlId( $tree_id );
	}

	function getByHierarchyControlId( $tree_id ) {
		if ( $tree_id == '' ) {
			return FALSE;
		}

		$this->getFastTreeObject()->setTree( $tree_id );

		$children = $this->getFastTreeObject()->getAllChildren(NULL, 'RECURSE');

		$ulf = new UserListFactory();
		$hslf = new HierarchyShareListFactory();
		$hslf->getByHierarchyControlId( $tree_id );
		$shared_user_ids = array();
		foreach( $hslf as $hierarchy_share ) {
			$shared_user_ids[] = $hierarchy_share->getUser();
		}

		if ( $children !== FALSE ) {
			foreach ($children as $object_id => $level ) {

				if ( $object_id !== 0 ) {
					$user_obj = $ulf->getById ( $object_id )->getCurrent();

					unset($shared);
					if ( in_array( $object_id, $shared_user_ids) === TRUE ) {
						$shared = TRUE;
					} else {
						$shared = FALSE;
					}

					$nodes[] = array(
									'id' => $object_id,
									'name' => $user_obj->getFullName(),
									'level' => $level,
									'shared' => $shared
									);
				}

			}

			if ( isset($nodes) ) {
				return $nodes;
			}
		}

		return FALSE;
	}

	function getHierarchyChildrenByCompanyIdAndUserIdAndObjectTypeID( $company_id, $user_id, $object_type_id = 100 ) {
		global $profiler;
		$profiler->startTimer( "getPermissionHierarchyChildrenByCompanyIdAndUserId" );

		if ( $company_id == '' ) {
			return FALSE;
		}

		if ( $user_id == '' ) {
			return FALSE;
		}

		if ( $object_type_id == '' ) {
			return FALSE;
		}

		$hotlf = new HierarchyObjectTypeListFactory();
		$hotlf->getByCompanyIdAndObjectTypeId($company_id, $object_type_id );
		if ( $hotlf->getRecordCount() > 0 ) {
			$hierarchy_control_id = $hotlf->getCurrent()->getHierarchyControl();
		} else {
			//No Permission hierarhcy.
			$profiler->stopTimer( "getPermissionHierarchyChildrenByCompanyIdAndUserId" );
			return FALSE;
		}
		Debug::Text(' Hierarchy Control ID: '. $hierarchy_control_id , __FILE__, __LINE__, __METHOD__,10);

		$cache_id = $hierarchy_control_id.$user_id;

		$retval = $this->getCache($cache_id);
		if ( $retval === FALSE ) {
			$hlf = new HierarchyListFactory();
			$retval = $hlf->getChildLevelIdArrayByHierarchyControlIdAndUserId( $hierarchy_control_id, $user_id, 'RECURSE' );

			$this->saveCache($retval,$cache_id);
		}

		$profiler->stopTimer( "getPermissionHierarchyChildrenByCompanyIdAndUserId" );
		return $retval;
	}
}
?>
