/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/

#define FILECODE "F(303)"

/*
 * Partly derived from:
 */

/*
 * freebsd/hostif.c
 * 
 * Copyright by Vladimir N. Silyaev 1999-2000
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions 
 * are met: 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE 
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * $vmFreeBSD: vmware/vmmon-only/freebsd/hostif.c,v 1.13 2000/01/23 22:29:19 vsi
lyaev Exp $
 * 
 *
 * NetBSD modifications by Frank van der Linden (fvdl@wasabisystems.com)
 */

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/lock.h>
#include <sys/malloc.h>
#include <sys/proc.h>

#include <machine/pio.h>

#if __NetBSD_Version__ > 105009900
#include <uvm/uvm_extern.h>
#include <uvm/uvm_param.h>
#else
#include <vm/vm.h>
#include <uvm/uvm_page.h>
extern vm_map_t kernel_map;
#endif

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "vm_asm.h"
#include "modulecall.h"
#include "hostif.h"
#include "memtrack.h"
#include "phystrack.h"

#define HOST_LOCK_PFN(_vm,_pfn) {		\
	host_lock_ppn(_pfn);			\
	PhysTrack_Add(_vm->physTracker,_pfn);	\
}

#define HOST_UNLOCK_PFN(_vm,_pfn) {		\
	host_unlock_ppn(_pfn);				\
	PhysTrack_Remove(_vm->physTracker,_pfn);	\
}

#define HOST_ISTRACKED_PFN(_vm, _pfn)		\
	(PhysTrack_Test(_vm->physTracker, _pfn))

#ifdef SMP_GLOBAL_VMLOCK
static struct simplelock vmmon_lock;
static int vmmon_lock_holder;
#endif

static int host_lock_ppn(PPN);
static int host_unlock_ppn(PPN);
static void UnlockEntry(void *, MemTrackEntry *);
static void * FindFunc(void *, MemTrackEntry *);


static
int host_lock_ppn(PPN ppn)
{
	paddr_t paddr = (paddr_t)PPN_2_PA(ppn);
	struct vm_page *pg;

	pg = PHYS_TO_VM_PAGE(paddr);
	uvm_pagewire(pg);
	return 0;
}     

static
int host_unlock_ppn(PPN ppn)
{
	paddr_t paddr = (paddr_t)PPN_2_PA(ppn);
	struct vm_page *pg;

	pg = PHYS_TO_VM_PAGE(paddr);
	uvm_pageunwire(pg);
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * FindMPN --
 *
 *      walks through the hardware defined page tables
 *      to find a valid mpn.
 *      
 * Results:
 *      mpn on success, 0 on failure
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */    
static INLINE MPN
FindMPN(PPN ppn)
{
	vaddr_t vaddr = (vaddr_t)VPN_2_VA(ppn);
	pt_entry_t *pteptr = (pt_entry_t *)vtopte(vaddr);
	PTE pte;
   
	pte = *pteptr;
	if (pte & PTE_P)
		return PTE_2_PFN(pte);
	else
		return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_InitSpinLock --
 *
 *      Initialize the vm spin lock
 *
 * Results:
 *
 *     none
 *
 * Side effects:
 *     None
 *      
 *----------------------------------------------------------------------
 */     
void
HostIF_InitSpinLock(void)
{
#ifdef SMP_GLOBAL_VMLOCK
	simple_lock_init(&vmmon_lock);
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_Init --
 *      
 *      Initialize the host depandent part of the driver.
 * 
 * Results: 
 *
 *     zero on success, non-zero on error.
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
int
HostIF_Init(VMDriver *vm)
{  
	vm->memtracker = MemTrack_Init(sizeof(MemTrackEntry));
	vm->physTracker = PhysTrack_Init();
	return (vm->memtracker == NULL || vm->physTracker==NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_IsSystemSMP --
 *
 *      define if we are running on a SMP machine
 *
 * Results:
 *      
 *      TRUE if SMP
 *      
 *----------------------------------------------------------------------
 */
Bool    
HostIF_IsSystemSMP(void)
{
	return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LookupUserMPN --
 *
 *      Lookup the MPN of a locked user page.
 *
 * Results:
 *      
 *      Returned page is a valid MPN, zero on error. 
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
MPN 
HostIF_LookupUserMPN(VMDriver *vm, char * addr)
{
	MPN mpn;
	mpn = FindMPN(PTR_2_VPN(addr));
	if (mpn == 0) {
		NOT_TESTED();
	}
	return mpn;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_InitFP --
 *
 *      masks IRQ13 if not previously the case.
 *
 * Results:
 *      
 *      prevents INTR #0x2d (IRQ 13) from being generated --
 *      assume that Int16 works for interrupt reporting
 *      
 *
 * Side effects:
 *      PIC
 *
 *----------------------------------------------------------------------
 */
void
HostIF_InitFP(VMDriver *vm)
{
	/*
	 * Since a working int16 means that IRQ 13 will not be used
	 * on NetBSD, this should not be needed.
	 */
#if 1
	int mask = (1<<(0xd-0x8));
	uint8 val = inb(0xA1);

	if (!(val & mask)) {
		val = val | mask;
		outb(val,0xA1);
	}
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LockPage --
 *
 *      Lockup the MPN of an pinned user-level address space 
 *
 * Results:
 *      
 *      The MPN or zero on an error. 
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
MPN
HostIF_LockPage(VMDriver *vm, void *addr)
{
	MPN mpn;
	VPN vpn;
	MemTrackEntry *entryPtr;
	int c;

	vpn = PTR_2_VPN(addr);
	entryPtr = (MemTrackEntry *) MemTrack_LookupVPN(vm->memtracker, vpn);

	if ((entryPtr == NULL) || (entryPtr->mpn == 0)) {
		/*
		 * If the page is not in the memory tracker or it has been
		 * unlocked. 
		 */

		c = fubyte(addr);

		/*
		 * now locate it and write the results back to the pmap
		 *
		 * Under extreme memory pressure, the page may be gone again.
		 * Just fail the lock in that case.
		 * (It was ASSERT_BUG(6339, mpn).)
		 * -- edward
		 */
		mpn = FindMPN(vpn);
		if (mpn == 0) {
			return 0;
		}
     	 
		/*
		 * XXX SMP.
		 * XXX critical region
		 */
		if (HOST_ISTRACKED_PFN(vm, mpn)) { 
			Warning("HostIF_LockPage vpn=0x%06x mpn=%06x" 
				 "already tracked\n", vpn, mpn);
	
			return 0;
		}
      
		HOST_LOCK_PFN(vm, mpn);
		/*
		 * If the entry doesn't exist, add it to the memtracker
		 * otherwise we just update the mpn.
		 */
		if (entryPtr == NULL) {
			entryPtr = MemTrack_Add(vm->memtracker, vpn, mpn);
			if (entryPtr == NULL) {
				HOST_UNLOCK_PFN(vm, mpn);
				return 0;
			}
		} else {
			entryPtr->mpn = mpn;  
		}
	} else {
		/*
		 * Already locked. 
		 */

		mpn = FindMPN(vpn);
		ASSERT(mpn);
		ASSERT(entryPtr->mpn == mpn);
		ASSERT(HOST_ISTRACKED_PFN(vm, mpn));
	}
	return mpn;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_UnlockPage --
 *
 *      Unlock an pinned user-level page.
 *
 * Results:
 *      0 if successful, otherwise non-zero
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

int
HostIF_UnlockPage(VMDriver *vm, void *addr)
{
	VPN vpn;
	MemTrackEntry *entryPtr;

	vpn = PTR_2_VPN(addr);
	entryPtr = (MemTrackEntry *) MemTrack_LookupVPN(vm->memtracker, vpn);

	if ((entryPtr == NULL) || (entryPtr->mpn == 0)) {
		Warning("HostIF_UnlockPage vpn=0x%06x not tracked!\n", vpn);
		/* 
		 * dangerous, we potentially leak memory here since
		 * the use count remains high.
		 * This only happens after the watchdog runs and unwedges a 
		 * process anyway
		 */
		return 1;
	}

	HOST_UNLOCK_PFN(vm, entryPtr->mpn);
	entryPtr->mpn = 0;
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_FreeAllResources --
 *
 *      Unlock all the pages pinned for a vm. 
 *
 * Results:
 *      0 if successful, otherwise non-zero
 *      
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

static void 
UnlockEntry(void *clientData, MemTrackEntry *entryPtr)
{
	VMDriver *vm = (VMDriver *)clientData;

	if (entryPtr->mpn) {
		if (HOST_ISTRACKED_PFN(vm, entryPtr->mpn)) {
			vm->releaseCount++;
			HOST_UNLOCK_PFN(vm, entryPtr->mpn);
		} else { 
			Warning("UnlockEntry vpn=0x%06x mpn=0x%06x not owned\n",
			    entryPtr->vpn, entryPtr->mpn);
		}
		entryPtr->mpn = 0;
	}
}

int
HostIF_FreeAllResources(VMDriver *vm)
{
	vm->releaseCount = 0;
#if 0
	/* XXXX causes a panic? */
	if (vm->crossvaddr != NULL) {
		uvm_km_free_wakeup(kernel_map, (vaddr_t)vm->crossvaddr,
		    PAGE_SIZE);
		uvm_vsunlock(curproc, vm->crossuaddr, PAGE_SIZE);
		vm->crossvaddr = NULL;
	}
#endif

	if (vm->memtracker) {
		MemTrack_Cleanup(vm->memtracker, UnlockEntry,vm);
		vm->memtracker = 0;
	}

	if (vm->physTracker) {
		PhysTrack_Cleanup(vm->physTracker);
		vm->physTracker = 0;
	}

	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_AllocKernelMem
 *      
 *      Allocate some kernel memory for the driver.
 *    
 * Results:
 *      The address allocated or NULL on error.
 *     
 * 
 * Side effects:
 *      memory is malloced
 *----------------------------------------------------------------------
 */

void *      
HostIF_AllocKernelMem(int size, int wired)
{
	void *ptr = malloc(size, M_DEVBUF, M_WAITOK);
   
	if (ptr==NULL) {
		Warning("HostIF_AllocKernelMem failed (size=0%x)\n",size);
	}     
      
	return ptr;
}      
      
void *
HostIF_AllocPage(int wired)
{
	void *addr;

	addr = malloc(PAGE_SIZE, M_DEVBUF, M_WAITOK);
	if (addr==0) {
		Warning("HostIF_AllocPage failed\n");
	}
	memset(addr, 0, PAGE_SIZE);
	return addr;
}

/* 
 *----------------------------------------------------------------------
 * 
 * HostIF_FreeKernelMem
 * 
 *      Free kernel memory allocated for the driver.
 * 
 * Results:
 *      None.
 * 
 * Side effects:
 *      memory is freed.
 *----------------------------------------------------------------------
 */
   
void
HostIF_FreeKernelMem(void *ptr)
{  
	free(ptr, M_DEVBUF);
}
  
 
  
void
HostIF_FreePage(void *ptr)
{
	free(ptr, M_DEVBUF);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_RealTime
 * 
 *      Read the systems real time clock.
 * 
 * Results:
 *      Current real time.
 *    
 * Side effects:
 *----------------------------------------------------------------------
 */   
      
VmTimeRealClock
HostIF_ReadTime(void)
{
	struct timeval curtime;

	microtime(&curtime);
	return (((VmTimeRealClock)curtime.tv_sec) * 1000000 + curtime.tv_usec);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_CopyFromUser
 * 
 *      Copy memory from the user application into a kernel buffer.
 * 
 * Results:
 *      0 if sucessful and non-zero or error.
 * 
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */
 
int     
HostIF_CopyFromUser(void *dst, void *src, int len)
{  
	return copyin(src, dst, len);
}

/* 
 *----------------------------------------------------------------------
 *
 * HostIF_CopytoUser
 *
 *      Copy memory to the user application from a  kernel buffer.
 *  
 * Results:
 *      0 if sucessful and non-zero or error.
 * 
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */   
   
int
HostIF_CopyToUser(void *dst, void *src, int len)
{
	return copyout(src, dst, len);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_UserToDriverPtr 
 *
 *      Convert user address into an address that can be used by the
 *      driver to access the memory. 
 *
 * Results:
 *      A pointer or NULL on error. 
 *
 * Side effects:
 *
 * Bugs:
 *      anyone wants to lock the page?
 *----------------------------------------------------------------------
 */

char userpage[NBPG];

void *
HostIF_UserToDriverPtr(VMDriver *vm, void *addr)
{
	paddr_t paddr;
	vaddr_t kvaddr, uaddr;
#ifdef DEBUG
	int error;
#endif

	if (vm->crossvaddr != NULL)
		Warning("KernelAddr already allocated\n");

	PHOLD(curproc);
	uvm_vslock(curproc, addr, PAGE_SIZE,
	    VM_PROT_READ|VM_PROT_WRITE|VM_PROT_EXECUTE);

	uaddr = (vaddr_t)addr;
	kvaddr = uvm_km_valloc_wait(kernel_map, PAGE_SIZE);
	pmap_extract(vm_map_pmap(&curproc->p_vmspace->vm_map), uaddr, &paddr);
	pmap_kenter_pa(kvaddr, paddr,
	    VM_PROT_READ | VM_PROT_WRITE | VM_PROT_EXECUTE);
	PRELE(curproc);
	vm->crossvaddr = (void *)kvaddr;
	vm->crossuaddr = addr;
#ifdef DEBUG
	printf("cross user vaddr %p kernel vaddr %p\n", addr, (void *)kvaddr);
	printf("cross user paddr %p kernel paddr %p\n", (void *)paddr,
	    (void *)vtophys(kvaddr));
	error = copyin(addr, userpage, PAGE_SIZE);
	if (error != 0)
		panic("user cross page inaccessible\n");
	if (memcmp(userpage, (void *)kvaddr, PAGE_SIZE) != 0)
		panic("cross page doesn't compare");
#endif
	return (void *)kvaddr;
}	     

/*
 *----------------------------------------------------------------------
 *
 * HostIF_IsMPNLocked
 *
 *      Return entryPtr if MPN is locked into memory
 *
 * Results:
 *      0 if sucessful and non-zero or error.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static void *
FindFunc(void *arg, MemTrackEntry *entryPtr)
{
	MPN mpn = (MPN) arg;
   
	if (entryPtr->mpn == mpn)
		return entryPtr;
	return NULL;
}

int
HostIF_IsMPNLocked(VMDriver *vm, MPN mpn)
{
	MemTrackEntry *entry;

	entry = MemTrack_Scan(vm->memtracker,vm, FindFunc);
	return (entry != NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_CheckMemory
 *
 *      Make sure memory locking is OK.
 *
 *      This function runs as part of the application
 *      process, as such FindMPN, get_user and put_user are fine
 *
 * Results:
 *      None
 *
 * Side effects:
 *      None.
 *---------------------------------------------------------------------- 
 */
 
static void *
CheckFunc(void *arg, MemTrackEntry *entryPtr)  
{ 
	VMDriver *vm = (VMDriver *)arg;
	MPN mpn;
 
	if (entryPtr->mpn == 0)
	return NULL;
 
	mpn = FindMPN(entryPtr->vpn);
	if (mpn == 0) {
		/* unmapped */
		return entryPtr; 
	}
 
	if (entryPtr->mpn != mpn) {
		Warning("CheckFunc vpn 0x%x mpn 0x%x and not 0x%x \n",
		    entryPtr->vpn, mpn, entryPtr->mpn);
		vm->checkFuncFailed = TRUE;
	}
 
	return NULL;
}

VA 
HostIF_CheckMemory(VMDriver *vm)
{  
	MemTrackEntry *entryPtr;
 
	vm->checkFuncFailed = FALSE;
	entryPtr = MemTrack_Scan(vm->memtracker,vm, CheckFunc);
   
	if (vm->checkFuncFailed)
		return 1;
	else if (entryPtr)
		return (VPN_2_VA(entryPtr->vpn));
	return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_MhzRating
 *
 *      Return an estimate the of the processor's MHZ rating, based on
 *      the ratio of the cycle counter and gettimeofday
 *      
 * Results: 
 *
 * 
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */     
uint32
HostIF_MhzRating(VMDriver *vm)
{

	VmTimeType vDiff, rDiff;
	uint32 flags;

	/* 
	 * Compute the cycle rate based on the change in the
	 * cycle counter and real time clock since we open the
	 * device.
	 */
   
	SAVE_FLAGS(flags);
	CLEAR_INTERRUPTS();
	vDiff = GET_TSC() - vm->startTime.count;
	rDiff = HostIF_ReadTime() - vm->startTime.time;
	RESTORE_FLAGS(flags);
       
	if (rDiff == 0)
		return 0;

	/* Need to do 32bit divide since 64bit one doesn't seem to
	 * work in a linux device driver. Scale the numbers back into
	 * 32bit numbers. 
	 * XXX doesn't apply to NetBSD -- redo this one.
	*/
	while (vDiff > (0xffffffff/10)) {
		vDiff >>= 1; 
		rDiff >>= 1;
	}
	return ((10*(uint32)vDiff)/(uint32)rDiff + 5)/10;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LockKernel --
 *
 *      grabs the global kernel lock
 *      For the UP driver, do nothing
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      lock held
 *
 *----------------------------------------------------------------------
 */
void
HostIF_LockKernel(void)
{
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_UnLockKernel --
 *
 *      releases the global kernel lock
 *      For the UP driver, do nothing
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      lock released
 *
 *----------------------------------------------------------------------
 */
void
HostIF_UnLockKernel(void)
{
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_GlobalVMLock --
 *
 *      grabs the global data structure lock.
 *      For the UP driver, assert that the lock counter is zero and increment
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      Should be a very low contention lock.
 *
 *----------------------------------------------------------------------
 */
void
HostIF_GlobalVMLock(int callerID)
{
#ifdef SMP_GLOBAL_VMLOCK
	if (!simple_lock_try(&vmmon_lock)) {
		printf("vmmon: %d wait for global VM lock %d\n",
		    callerID, vm_lock_holder);
		simple_lock(&vmmon_lock);
	}
	vmmon_lock_holder = callerID;
#endif
}

/*
 *----------------------------------------------------------------------
 * 
 * HostIF_GlobalVMUnlock --
 *      
 *      releases the global data structure lock.
 *      For the UP driver, assert that the lock counter is 1 and decrement
 *
 * Results:
 *
 *      void
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */ 
void
HostIF_GlobalVMUnLock(int callerID)
{
#ifdef SMP_GLOBAL_VMLOCK
	if (vm_lock_holder != callerID) {
		printf("/dev/vmmon: %d releasing global VM lock %d\n",
		    callerID, vm_lock_holder);
	}
	vm_lock_holder = -1;
	simple_unlock(&vm_lock); 
#endif
}

MA
HostIF_APIC_Base(VMDriver *vm, Bool setVMPtr)
{
	return 0;
}

/*    
 *----------------------------------------------------------------------
 *  
 * HostIF_IOAPICBase --
 *
 *      Lookup the IO-APIC physical address
 *    
 * Results:
 * 
 *      Returned value is IO-APIC physical address
 * 
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
 
MA      
HostIF_IOAPIC_Base(VMDriver *vm) 
{
	return 0;
}
