// -*- mode: c++; indent-tabs-mode: nil; -*-
#include "qmousewsmouse_qws.h"

#ifndef QT_NO_QWS_MOUSE_WSMOUSE

#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#include <dev/wscons/wsconsio.h>

#include "qsocketnotifier.h"
#include "qscreen_qws.h"


QT_BEGIN_NAMESPACE

class QWSWSMouseMousePrivate : public QObject
{
    Q_OBJECT

public:
    QWSWSMouseMousePrivate(QWSWSMouseMouseHandler &handler,
                           const QString &device);
    ~QWSWSMouseMousePrivate();

    void suspend();
    void resume();


private:
    Qt::MouseButton buttonWStoQ(int wsmouse_button);

private:
    QWSWSMouseMouseHandler &handler;

    int fd;
    QSocketNotifier *notifier;

    int bstate;

private slots:
    void readMouseData();
};


QWSWSMouseMouseHandler::QWSWSMouseMouseHandler(const QString &driver,
					       const QString &device)
  : QWSMouseHandler(driver, device),
    d(new QWSWSMouseMousePrivate(*this, device))
{
    return;
}


QWSWSMouseMouseHandler::~QWSWSMouseMouseHandler()
{
    delete d;
}


void QWSWSMouseMouseHandler::suspend()
{
    d->suspend();
}


void QWSWSMouseMouseHandler::resume()
{
    d->resume();
}


QWSWSMouseMousePrivate::QWSWSMouseMousePrivate(
    QWSWSMouseMouseHandler &handler,
    const QString &device)
  : handler(handler),
    fd(-1),
    notifier(NULL),
    bstate(0)
{
    QString mousedev;
    if (device.isEmpty()) {
        mousedev = QLatin1String("/dev/wsmouse");
    }
    else {
        mousedev = device;
    }

    fd = open(mousedev.toLatin1().constData(), O_RDONLY | O_NDELAY);
    if (fd < 0) {
        qWarning("Cannot open %s (%s)", qPrintable(mousedev), strerror(errno));
        return;
    }
    fcntl(fd, F_SETFD, FD_CLOEXEC | fcntl(fd, F_GETFD, 0));

#ifdef WSMOUSEIO_SETVERSION
    int version = WSMOUSE_EVENT_VERSION;
    if (ioctl(fd, WSMOUSEIO_SETVERSION, &version) < 0) {
        qWarning("WSMOUSEIO_SETVERSION: %s", strerror(errno));
        return;
    }
#endif

    notifier = new QSocketNotifier(fd, QSocketNotifier::Read, this);
    connect(notifier, SIGNAL(activated(int)), this, SLOT(readMouseData()));
}


QWSWSMouseMousePrivate::~QWSWSMouseMousePrivate()
{
    if (notifier != NULL) {
        notifier->setEnabled(false);
        delete notifier;
    }

    if (fd >=0) {
        close(fd);
    }
}


void QWSWSMouseMousePrivate::suspend()
{
    if (notifier != NULL)
        notifier->setEnabled(false);
}


void QWSWSMouseMousePrivate::resume()
{
    if (notifier != NULL)
        notifier->setEnabled(true);
}


Qt::MouseButton QWSWSMouseMousePrivate::buttonWStoQ(int wsmouse_button)
{
    switch (wsmouse_button) {
    case 0:
        return Qt::LeftButton;
    case 1:
        return Qt::MidButton;
    case 2:
        return Qt::RightButton;
    case 3:
        return Qt::XButton1;
    case 4:
        return Qt::XButton2;
    default:
        return Qt::NoButton;
    }
}


void QWSWSMouseMousePrivate::readMouseData()
{
    if(!qt_screen)
        return;

    for (;;) {
        struct wscons_event wsevent;
        ssize_t nread = read(fd, &wsevent, sizeof(wsevent));
        if (nread != sizeof(wsevent)) {
            return;
        }

        QPoint opos(handler.pos());
        QPoint npos = opos;
        int obstate = bstate;

        switch (wsevent.type) {
        case WSCONS_EVENT_MOUSE_DOWN:
        {
            // qDebug("WSCONS_EVENT_MOUSE_DOWN %d", wsevent.value);
            Qt::MouseButton pressed = buttonWStoQ(wsevent.value);
            bstate |= pressed;
            break;
        }

        case WSCONS_EVENT_MOUSE_UP:
        {
            // qDebug("WSCONS_EVENT_MOUSE_UP %d", wsevent.value);
            Qt::MouseButton released = buttonWStoQ(wsevent.value);
            bstate &= ~released;
            break;
        }

        case WSCONS_EVENT_MOUSE_DELTA_X:
            // qDebug("dx=%d", wsevent.value);
            npos.setX(npos.x() + wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_ABSOLUTE_X:
            // qDebug("X=%d", wsevent.value);
            npos.setX(wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_DELTA_Y:
            // qDebug("dy=%d", wsevent.value);
            npos.setY(npos.y() - wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_ABSOLUTE_Y:
            // qDebug("Y=%d", wsevent.value);
            npos.setY(wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_DELTA_Z:
            // qDebug("dz=%d", wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_ABSOLUTE_Z:
            // qDebug("Z=%d", wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_DELTA_W:
            // qDebug("dw=%d", wsevent.value);
            break;

        case WSCONS_EVENT_MOUSE_ABSOLUTE_W:
            // qDebug("W=%d", wsevent.value);
            break;

        default:
            // qDebug("Unknown WSEVENT %u/%d", wsevent.type, wsevent.value);
            break;
        }

        handler.limitToScreen(npos);
        if (npos != opos || bstate != obstate) {
            // qDebug("> x=%d, y=%d, b=%x", npos.x(), npos.y(), bstate);
            handler.mouseChanged(npos, bstate);
        }
    }
}

QT_END_NAMESPACE

#include "qmousewsmouse_qws.moc"

#endif // QT_NO_QWS_MOUSE_WSMOUSE
